/*
 * Copyright (c) 2001, 2002 The XDoclet team
 * All rights reserved.
 */
package xdoclet.modules.ibm.websphere.ejb;

import xdoclet.XDocletException;

import xdoclet.modules.ejb.dd.AbstractEjbDeploymentDescriptorSubTask;

/**
 * Generates WebSphere specific deployment descriptors for EJB modules. The following files are generated:
 * ibm-ejb-jar-bnd.xmi, ibm-ejb-jar-ext.xmi and Schema.dbxmi. <p>
 *
 * NOTE: Since the WebSphere specific deployment descriptors depend on that id's are set in the ejb-jar.xml file you
 * must genererate it with useIds set to true, e.g. <code>&lt;deploymentdescriptor useIds="true"/&gt;</code>. </p> <p>
 *
 * This Ant task defines the following attributes (the rest of the attributes in the list below comes from its
 * baseclass):
 * <ul>
 *   <li> datasource</li>
 *   <li> currentBackendId</li>
 * </ul>
 * </p>
 *
 * @author        Minh Yie
 * @author        <a href="mailto:ml at callista.se">Magnus Larsson</a>
 * @created       15 August 2001
 * @ant.element   display-name="WebSphere" name="websphere" parent="xdoclet.modules.ejb.EjbDocletTask"
 * @version       $Revision: 1.10 $
 */
public class WebSphereSubTask extends AbstractEjbDeploymentDescriptorSubTask
{
    private static String WEBSPHERE_DEFAULT_BND_TEMPLATE_FILE = "resources/ibm-ejb-jar-bnd_xmi.xdt";

    private static String WEBSPHERE_DD_BND_FILE_NAME = "ibm-ejb-jar-bnd.xmi";

    private static String WEBSPHERE_DEFAULT_EXT_TEMPLATE_FILE = "resources/ibm-ejb-jar-ext_xmi.xdt";

    private static String WEBSPHERE_DD_EXT_FILE_NAME = "ibm-ejb-jar-ext.xmi";

    private static String WEBSPHERE_SCHEMA_TEMPLATE_FILE = "resources/Schema_dbxmi.xdt";

    private static String WEBSPHERE_DD_SCHEMA_FILE_NAME = "Schema.dbxmi";

    /**
     * A member for the default CMP DataSource
     */
    private String  datasource;

    /**
     * A member for the current backend id
     */
    private String  currentBackendId;

    public WebSphereSubTask()
    {
        setUseIds(true);
    }

    /**
     * Gets the default CMP Datasource
     *
     * @return   The default CMP Datasource
     */
    public String getDatasource()
    {
        return datasource;
    }

    /**
     * Gets the current backend id
     *
     * @return   The current backend id
     */
    public String getCurrentBackendId()
    {
        return currentBackendId;
    }

    /**
     * Sets the default CMP Datasource
     *
     * @param datasource   The new default CMP Datasource
     * @ant.not-required   No.
     */
    public void setDatasource(String datasource)
    {
        this.datasource = datasource;
    }

    /**
     * Sets the current backend id. This parameter is used to point out the id of the current CMP mapping.
     *
     * @param backendId    The current backend id
     * @ant.not-required   No.
     */
    public void setCurrentBackendId(String backendId)
    {
        this.currentBackendId = backendId;
    }

    /**
     * Called to validate configuration parameters.
     *
     * @exception XDocletException  Description of Exception
     */
    public void validateOptions() throws XDocletException
    {
        // WebSphere does not require a template url or a destination file
        //
        // super.validateOptions();
    }

    /**
     * Describe what the method does
     *
     * @exception XDocletException  Describe the exception
     */
    public void execute() throws XDocletException
    {
        setTemplateURL(getClass().getResource(WEBSPHERE_DEFAULT_BND_TEMPLATE_FILE));
        setDestinationFile(WEBSPHERE_DD_BND_FILE_NAME);
        startProcess();

        setTemplateURL(getClass().getResource(WEBSPHERE_DEFAULT_EXT_TEMPLATE_FILE));
        setDestinationFile(WEBSPHERE_DD_EXT_FILE_NAME);
        startProcess();

        /*
         * TODO: Conflicts with WSAD so we better simply rely on the CMP mapping capabilities in WSAD and WebSphere!
         */
        if (atLeastOneCmpEntityBeanExists()) {
            setTemplateURL(getClass().getResource(WEBSPHERE_SCHEMA_TEMPLATE_FILE));
            setDestinationFile(WEBSPHERE_DD_SCHEMA_FILE_NAME);
            startProcess();
        }
    }

    /**
     * Describe what the method does
     *
     * @exception XDocletException  Describe the exception
     */
    protected void engineStarted() throws XDocletException
    {
    }
}
