/*
	Copyright (C) 2009 -=[dxp]=-

	This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include "globals.h"

/* ----------------------------------- Error () ---------------------------- */
/*
 * Print error code and message to stderr and abort upon request.
 */
void
Error (const int quit, const int errcode, const char *msg)
{
	if (msg)
		fprintf(stderr, "%i\t%s\n", errcode, msg);

	if (quit)
		exit(errcode);
}
/* ----------------------------------- Usage () ---------------------------- */
void
Usage (const uint32_t found)
{
	unsigned long int	sec, micro;
	int			rv;
	

	rv = getrusage (RUSAGE_SELF, &usage);
	if (rv == -1)
		Error (1, errno, "getrusage()");

	sec = usage.ru_utime.tv_sec + usage.ru_stime.tv_sec;
	micro = usage.ru_utime.tv_usec + usage.ru_stime.tv_usec;
	
	if (micro > 1000000)
		{
			sec++;
			micro -= 1000000;
		}
	
	if (found)
		fprintf (stderr, "Total Time: %li.%03lis\n", sec, micro / 1000);
	else
		fprintf (stderr, " %li.%03lis\n", sec, micro / 1000);
}
/* ----------------------------------- found () ---------------------------- */
void
Found (const char *word, const char *chash, const char *ihash, const int mode)
{	
	uint32_t x;
	
	//signal (SIGIO, SIG_IGN);
/*	
	printf ("\n***CRACKED ");
	switch (mode)
		{
			case 1:	printf ("MD5");	break;
			case 2:	printf ("SHA1");	break;
			default:	printf ("Unknown");	break;
		}
	printf (" HASH***\n");
*/
	printf ("%s %s\n", chash, word);
	
	if ( ! list_removenode(&list, list_tmp) )
		Error (1, 31, "Failed to remove node");
	
	x = list_countnodes (list);
	if (x == 1)
		fprintf (stderr, "[i]... %i node left\n", x);
	else if (x > 1)
		fprintf (stderr, "[i]... %i nodes left\n", x);
	
	//printf ("String:        \t%s\n", word);
	//printf ("Computed HASH: \t%s\n", chash);
	//printf ("Input HASH:    \t%s\n", ihash);
	//printf ("Words: %.0f Chars: %.0f", (double) cnt_words, (double) cnt_chars);
	//printf("Words tried: %.0f\nTotal characters: %.0f\n", (double) cnt_words, (double) cnt_chars);	

#ifdef DEBUG
	Usage (1);
	print_stats ();
#endif
}
/* ----------------------------------- hash2digest () ---------------------- */
unsigned char *
hash2digest (const char *hash)
{
	static unsigned char digest[21];
	unsigned int len = strnlen(hash, MAX_HASH_LEN); 
	unsigned int i, di = 0;
	unsigned char x;

	/* Sanity check */
	if ( MAX_HASH_LEN / 2 > sizeof(digest) )
		Error(1, 0, "Hash length is greater then allocated buffer");
	
	for (i = 0; i < len; i += 2) {
		x = 16 * hex2int (hash[i]);
		x = x + hex2int (hash[i + 1]);
		digest[di++] = x;
	}
	
	digest[di] = '\0';
	pd_len = di - 1;
	
	return digest;
}
/* ----------------------------------- hex2int () ---------------------- */
unsigned char
hex2int (const char hex)
{
	static unsigned char	binary;
	unsigned char		uppercase;

	uppercase = toupper (hex);
	if ( uppercase > 47 && uppercase < 58)
		binary = uppercase - 48;
	else
		binary = uppercase - 55;

	return binary;
}	
/* ----------------------------------- digestcmp () ---------------------- */
uint32_t
digestcmp (const unsigned char *digest)
{
	unsigned int	i;

	// Compare given digest against the computed, check in blocks of 4 bytes.
	for (i = 0; i < pd_len; i += 4)
		if ( *(unsigned int *) (password_digest + i) != *(unsigned int *) (digest + i) )
			return 1;

	return 0;
}
/* ----------------------------------- digest2hash () ---------------------- */
/*
 * Convert given digest into hex represantation for given type of hash (md5,sha1).
 */
char *
digest2hash (const unsigned char *digest)
{
	int	j;
	static char output[41];

/*
	for (i = 0; i < x; i++)
		{
			j = digest[i];
			output[i << 1] = fasthex[ (j >> 4) & 0x0f ];
			output[ (i << 1) + 1 ] = fasthex[j & 0x0f];
		}
	
	output[x << 1] = 0;
*/
/*
	We'll use a macro to convert with constant index,
	which will be faster then variable incrementation.
*/
	switch (hash_mode)
		{
			case 1:
				convert (0);
				convert (1);
				convert (2);
				convert (3);
				convert (4);
				convert (5);
				convert (6);
				convert (7);
				convert (8);
				convert (9);
				convert (10);
				convert (11);
				convert (12);
				convert (13);
				convert (14);
				convert (15);
				output[32] = 0;
				break;

			case 2:
				convert (0);
				convert (1);
				convert (2);
				convert (3);
				convert (4);
				convert (5);
				convert (6);
				convert (7);
				convert (8);
				convert (9);
				convert (10);
				convert (11);
				convert (12);
				convert (13);
				convert (14);
				convert (15);
				convert (16);
				convert (17);
				convert (18);
				convert (19);
				output[40] = 0;
				break;
		}

	return output;
}

/* ----------------------------------- brute2 () ---------------------- */
/*
 *	Main loop which calls the recursive function while advancing
 *	characters in the word arrary.
*/
void
brute2 (void)
{
	uint32_t pos, x;

	x = wlen;
	memset(word_rf, '\0', MAX_WORD_SIZE);
	
	/* init word array with first char from alphabet */
	for (pos = 0; pos < wlen; pos++)
		word_rf[pos] = alphabet[0];
	
	// launch threads
	for (pos = wlen; pos < MAX_WORD_SIZE; pos++) {
#ifdef DEBUG		
		if (pos < wlen)
			fprintf (stderr, "\t\t Total Time after Length of %i: ", x++);
		else
			fprintf (stderr, "\t\t Total Time after Length of %i: ", pos);
#endif		
		rf(pos);
		wlen++;
		
#ifdef DEBUG
		Usage(0);
#endif
	}	
}
/* ----------------------------------- rf () ---------------------- */
/*
 *	Recursive function which calls the calculation() for every
 *	character position within a word or itself when every
 *	character in a position has been tried and we need to advance.
*/
void
rf (const int pos)
{
	uint32_t i, j = 0, alphabet_len;
	uint32_t found;
	
	alphabet_len = strlen(alphabet);
	
	if (pos == 1) {
		for (i = wlen - 1; j < alphabet_len; j++) {
			// if I locked then adjust word and clear my local lock, otherwise Lock
			word_rf[i] = alphabet[j];
			cnt_words++;
			word_rf_len = strlen((char *)word_rf);
			// Unlock
			cnt_chars += word_rf_len;
			
			list_tmp = list_pvt = list;
			
			calculation(); // generate digest for current word
			
			// check linked list for a matching node
			while (list_pvt) {
				found = 1;
				
				/* compare in four byte chunks */
				for (dig_off = 0; dig_off < pd_len; dig_off += 4) {
					if ( *(unsigned int *) (list_pvt->digest + dig_off) != *(unsigned int *) (ssl_digest + dig_off) ) {
						found = 0;
						break;
					}
				}
				
				list_tmp = list_pvt;
				list_pvt = list_pvt->next;
				
				if (found)
					Found((char *) word_rf, digest2hash(ssl_digest), password_hash, 1);	
			}
			
			if (! list)
				Error(1, 0, NULL);
		}
	}
	else {
		for (i = wlen - pos; j < alphabet_len; j++) {
			// Lock
			word_rf[i] = alphabet[j];
			rf(pos - 1);
		}
	}	
}
/* ----------------------------------- signal_handler () ---------------------- */
/*
 * SIGIO and SIGINT handler
 */
void
signal_handler (int sig)
{
	ssize_t bytes = 0;
	uint8_t c;

	/* Ignore the signal while it's being handled */
	signal(sig, SIG_IGN);
	
#ifdef DEBUG
	if (sig == SIGFPE) {
		printf ("\nfault\n");
		signal (SIGFPE, SIG_DFL);
		printf ("Words: %lu ; Chars: %lu\n", (long unsigned int) cnt_words, (long unsigned int) cnt_chars);
	}

	if (input_mode == 0)
		printf ("single\n");
#endif

	if (sig == SIGINT)
		Exit(2);
	
	/*
	bytes = read(STDIN_FILENO, &c, sizeof(uint8_t));
	if (bytes > 0)
		print_stats();
	*/		

	do {
		bytes = read(STDIN_FILENO, &c, sizeof(uint8_t));
		printf("bytes\n");
		if (c == '\n')
			print_stats();
	}
	while (c != '\n');
	
	/* Restore signal handling */
	signal(sig, signal_handler);
}
/* ----------------------------------- reset_input_mode () ---------------------- */
void
reset_input_mode (void)
{
	int	x, flags;

	/* Restore previous descriptor flags */
	flags = fcntl(STDIN_FILENO, F_GETFL);
	if (flags != -1) {
		flags |= ~O_ASYNC;
		fcntl(STDIN_FILENO, F_SETFL, flags);
	}
	
	/* Drain the queues */
	tcdrain(STDIN_FILENO);
	tcdrain(STDOUT_FILENO);	
	
	/* Restore previous terminal attributes	*/
	x = tcsetattr(STDIN_FILENO, TCSANOW, &tio_isaved);
	if (x)
		Error (0, 0, "Error restoring input I/O terminal attributes");
	
	x = tcsetattr(STDOUT_FILENO, TCSANOW, &tio_osaved);
	if (x)
		Error (0, 0, "Error restoring output I/O terminal attributes");
}
/* ----------------------------------- print_stats () ---------------------- */
/*
 * Display various runtime statistics.
 */
void
print_stats (void)
{
	struct rusage usg;
	long unsigned int sec, micro;
	int rv;
	
	// Populate the usage structure
	rv = getrusage (RUSAGE_SELF, &usg);
	if (rv == -1)
		Error (1, errno, "print_stats: getrusage()");

	// Calculate elapsed amount of seconds
	sec = usg.ru_utime.tv_sec + usg.ru_stime.tv_sec;		// combine user and os time (seconds)
	micro = usg.ru_utime.tv_usec + usg.ru_stime.tv_usec;	// combine user and os time (micro seconds)
	if (micro > 1000000)									// adjust seconds if there's enough micro seconds
		sec++;

	if (sec < 1)
		sec = 1;
		
	fprintf (stderr, "Time: %lu sec ; Words: %.0f (%.0u K/s) ;  Characters: %.0f (%.0u K/s)\n", sec, \
		(double) cnt_words, (unsigned int) (cnt_words / sec) / 1024, (double) cnt_chars, (unsigned int) (cnt_chars / sec) / 1024);
}
/* ----------------------------------- Exit () ---------------------- */
/*
 * Reset IO mode and exit
 */
void
Exit (const int status)
{
	if (status == 0)
		print_stats();

#ifdef DEBUG
	fprintf (stderr, "Exit status code: %i\n", status);
#endif

	reset_input_mode();
	exit(status);
}
/* ----------------------------------- get_index () ---------------------- */
/*
unsigned int
get_index (const char ch)
{
	unsigned int	i = 0;

	while (1) {
		if(ch == alphabet[i++])
			break;
	}
	return i - 1;
}
*/
/* ----------------------------------- list_addnode () ---------------------- */
/*
 * Add a node to the end of a linked list.
 * Returns boolean true upon success.
 */
unsigned int
list_addnode (NODE_T **head, NODE_T *node)
{
	NODE_T *tmp;
	
#ifdef DEBUG
	fprintf (stderr, "%s: list_addnode(): list = %p  node = %p  %s\n", __FILE__, *head, node, digest2hash(node->digest));
#endif

	if ( ! node )
		return 0;
	
	node->next = NULL;
	
	if ( ! (*head) )
		*head = node;
	else {
		tmp = *head;
	
		while (tmp->next)
			tmp = tmp->next;
	
		tmp->next = node;
	}
		
	return 1;
}
/* ----------------------------------- list_removenode () ---------------------- */
/*
 * Remove a node from a linked list.
 * Returns boolean true upon success.
 */
unsigned int
list_removenode (NODE_T **head, NODE_T *node)
{
	NODE_T *tmp, *prev;

#ifdef DEBUG
	fprintf (stderr, "%s: list_removenode(): list = %p  node = %p\n", __FILE__, *head, node);
#endif
	
	if ( ! (*head) || ! node )
		return 0;
	
	// point to start of list
	tmp = prev = *head;
	
	// list consists of single node
	if ( tmp == node && node->next == NULL )
		*head = NULL;
	// multiple nodes
	else {
		// locate desired node and point 'tmp' to it and 'prev' to node back
		while ( tmp != node && tmp ) {
			prev = tmp;
			tmp = tmp->next;
		}
		
		if ( ! tmp )
			return 0;
		// last node
		if ( ! tmp->next )
			prev->next = NULL;
		// node somewhere in the middle
		else {
			prev->next = tmp->next;
			*head = tmp->next;
		}
	}
	
	free (node);
	return 1;
}
/* ----------------------------------- bye () ---------------------- */
/*
 * Exit handler for normal process termination.
 */
void
bye (void)
{
	Exit(0);
}
/* ----------------------------------- list_findnode () ---------------------- */
/*
 * Search the list for the given node.
 * Returns boolean true (1) if node was found.
 */
uint32_t
list_findnode (NODE_T **list, NODE_T *node)
{
	NODE_T *tmp;
	uint32_t len;
	
	if ( ! (*list) || ! node )
		return 0;
	
	(hash_mode == HASH_MD5) ? (len = 16) : (len = 20);
	tmp = *list;
	
	while (tmp) {
		if ( ! memcmp(tmp->digest, node->digest, len) )
			return 1;
		tmp = tmp->next;
	}
	
	return 0;
}
/* ----------------------------------- hashtype () ---------------------- */
/*
 * Determine type of hash.
 * Returns 'hash_mode' value or '0' on error.
 */
uint32_t
hashtype (const char *str)
{
	uint32_t mode = 0;
	
	if ( ! str )
		return mode;
	
	switch ( strlen(str) ) {
		case 32:	mode = HASH_MD5; break;
		case 40:	mode = HASH_SHA1; break;
	}
	
	return mode;
}
/* ----------------------------------- list_countnodes () ---------------------- */
/*
 * Count number of nodes in the linked list.
 */
uint32_t
list_countnodes (NODE_T *list)
{
	uint32_t count = 0;
	
	if ( ! list )
		return count;
	
	while (list) {
		count++;
		list = list->next;
	}
	
	return count;
}
