import mock, os, unittest
from mercurial import ui
from tortoisehg.hgqt import thgrepo

def mockrepo(ui, path):
    m = mock.MagicMock(ui=ui, root=path)
    m.unfiltered = lambda: m
    return m

if os.name == 'nt':
    def fspath(s):
        if s.startswith('/'):
            s = 'C:' + s
        return s.replace('/', os.sep)
else:
    def fspath(s):
        return s

LOCAL_SIGNALS = ['repositoryOpened', 'repositoryClosed']
MAPPED_SIGNALS = ['configChanged', 'repositoryChanged', 'repositoryDestroyed']

class RepoManagerMockedTest(unittest.TestCase):
    def setUp(self):
        self.hgrepopatcher = mock.patch('mercurial.hg.repository', new=mockrepo)
        self.watcherpatcher = mock.patch('tortoisehg.hgqt.thgrepo.RepoWatcher')
        self.hgrepopatcher.start()
        self.watcherpatcher.start()
        self.repoman = thgrepo.RepoManager(ui.ui())

        for signame in LOCAL_SIGNALS + MAPPED_SIGNALS:
            slot = mock.Mock()
            setattr(self, signame, slot)
            getattr(self.repoman, signame).connect(slot)

    def tearDown(self):
        self.watcherpatcher.stop()
        self.hgrepopatcher.stop()
        thgrepo._repocache.clear()

    def test_cached(self):
        a1 = self.repoman.openRepoAgent('/a')
        a2 = self.repoman.openRepoAgent('/a')
        self.assertTrue(a1 is a2)

    def test_release(self):
        self.repoman.openRepoAgent('/a')
        self.repoman.openRepoAgent('/a')

        self.repoman.releaseRepoAgent('/a')
        self.assertTrue(self.repoman.repoAgent('/a'))

        self.repoman.releaseRepoAgent('/a')
        self.assertFalse(self.repoman.repoAgent('/a'))

    def test_sub_release(self):
        a = self.repoman.openRepoAgent('/a')
        b1 = a.subRepoAgent('b')
        b2 = a.subRepoAgent('b')
        self.assertTrue(b1 is b2)
        self.assertTrue(self.repoman.repoAgent('/a/b'))

        self.repoman.releaseRepoAgent('/a')
        self.assertFalse(self.repoman.repoAgent('/a/b'))

    def test_sub_already_open(self):
        a = self.repoman.openRepoAgent('/a')
        b1 = self.repoman.openRepoAgent('/a/b')
        b2 = a.subRepoAgent('b')
        self.assertTrue(b1 is b2)

        self.repoman.releaseRepoAgent('/a')
        self.assertTrue(self.repoman.repoAgent('/a/b'))
        self.repoman.releaseRepoAgent('/a/b')
        self.assertFalse(self.repoman.repoAgent('/a/b'))

    def test_sub_invalidpath(self):
        a = self.repoman.openRepoAgent('/a')
        a.subRepoAgent('/a/b')  # ok
        a.subRepoAgent('../a/b')  # ok
        self.assertRaises(ValueError, lambda: a.subRepoAgent('/a'))
        self.assertRaises(ValueError, lambda: a.subRepoAgent('/b'))
        self.assertRaises(ValueError, lambda: a.subRepoAgent('../a'))

    def test_sub_rootedpath(self):
        a = self.repoman.openRepoAgent('/')
        a.subRepoAgent('/a')  # ok
        self.assertRaises(ValueError, lambda: a.subRepoAgent('/'))

    def test_signal_map(self):
        p = fspath('/a')
        a = self.repoman.openRepoAgent(p)
        for signame in MAPPED_SIGNALS:
            getattr(a, signame).emit()
            getattr(self, signame).assert_called_once_with(p)

    def test_disconnect_signal_on_close(self):
        a = self.repoman.openRepoAgent('/a')
        self.repoman.releaseRepoAgent('/a')
        for signame in MAPPED_SIGNALS:
            getattr(a, signame).emit()
            self.assertFalse(getattr(self, signame).called)

    def test_opened_signal(self):
        p = fspath('/a')
        self.repoman.repositoryOpened.connect(
            lambda: self.assertTrue(self.repoman.repoAgent(p)))
        self.repoman.openRepoAgent(p)
        self.repositoryOpened.assert_called_once_with(p)
        self.repositoryOpened.reset_mock()
        # emitted only if repository is actually instantiated (i.e. not cached)
        self.repoman.openRepoAgent(p)
        self.assertFalse(self.repositoryOpened.called)

    def test_closed_signal(self):
        p = fspath('/a')
        self.repoman.repositoryClosed.connect(
            lambda: self.assertFalse(self.repoman.repoAgent(p)))
        self.repoman.openRepoAgent(p)
        self.repoman.openRepoAgent(p)
        self.repoman.releaseRepoAgent(p)
        self.assertFalse(self.repositoryClosed.called)
        self.repoman.releaseRepoAgent(p)
        self.repositoryClosed.assert_called_once_with(p)
