import mock, unittest

from PyQt4.QtCore import QCoreApplication, QEventLoop, QObject, QTimer
from PyQt4.QtCore import pyqtSlot

from tortoisehg.hgqt import cmdcore

import helpers

class CmdWaiter(QObject):
    def __init__(self, session):
        super(CmdWaiter, self).__init__()
        self._session = session
        self._outputs = []
        self._session.outputReceived.connect(self._captureOutput)

    def outputString(self):
        return ''.join(self._outputs).rstrip()

    def wait(self, timeout=5000):
        if self._session.isFinished():
            return
        loop = QEventLoop()
        self._session.commandFinished.connect(loop.quit)
        QTimer.singleShot(timeout, loop.quit)
        loop.exec_()

    @pyqtSlot(unicode, unicode)
    def _captureOutput(self, msg, label):
        if not label:
            self._outputs.append(unicode(msg))


class CmdAgentTest(unittest.TestCase):
    @classmethod
    def setUpClass(cls):
        tmpdir = helpers.mktmpdir(cls.__name__)
        cls.hg = hg = helpers.HgClient(tmpdir)
        hg.init()
        hg.ftouch('foo')
        hg.commit('-Am', 'add foo')

    def setUp(self):
        self.agent = agent = cmdcore.CmdAgent()
        agent.setWorkingDirectory(self.hg.path)
        self.busyChanged = mock.Mock()
        agent.busyChanged.connect(self.busyChanged)

    def test_runcommand(self):
        sess = self.agent.runCommand(['root'])
        self._check_runcommand(sess, self.hg.path)

    def test_runcommandseq(self):
        sess = self.agent.runCommandSequence([['root'], ['id', '-i']])
        self._check_runcommand(sess, '\n'.join([self.hg.path, '53245c60e682']))

    def test_runcommandseq_firsterror(self):
        sess = self.agent.runCommandSequence([['id', '-r100'], ['root']])
        self._check_runcommand(sess, '', 255)

    def test_runcommand_proc(self):
        sess = self.agent.runCommand(['root'], worker='proc')
        self._check_runcommand(sess, self.hg.path)

    def test_runcommand_delayedstart(self):
        sess = self.agent.runCommand(['root'])
        self.assertFalse(sess.isRunning())
        QCoreApplication.processEvents()
        self.assertTrue(sess.isRunning())
        self._check_runcommand(sess, self.hg.path)

    def test_runcommand_queued(self):
        sess1 = self.agent.runCommand(['id', '-i'])
        QCoreApplication.processEvents()
        self.assertTrue(sess1.isRunning())
        sess2 = self.agent.runCommand(['id', '-n'])
        self.assertFalse(sess2.isRunning())

        self._check_runcommand(sess1, '53245c60e682')
        QCoreApplication.processEvents()
        self.assertTrue(sess2.isRunning())
        self._check_runcommand(sess2, '0')

    def test_runcommand_signal_chain(self):
        sess = self.agent.runCommand(['id', '-i'])
        sess.commandFinished.connect(self._chained_runcommand)
        self._check_runcommand(sess, '53245c60e682')

    @pyqtSlot()
    def _chained_runcommand(self):
        sess = self.agent.runCommand(['id', '-n'])
        QCoreApplication.processEvents()
        self.assertTrue(sess.isRunning())
        self._check_runcommand(sess, '0')

    def _check_runcommand(self, sess, expectedout, expectedcode=0):
        self.assertFalse(sess.isFinished())
        waiter = CmdWaiter(sess)
        waiter.wait()
        self.assertTrue(sess.isFinished())
        self.assertFalse(sess.isRunning())
        self.assertEqual(expectedcode, sess.exitCode())
        self.assertEqual(expectedout, waiter.outputString())

    def test_abort_session(self):
        sess = self.agent.runCommand(['log'])
        finished = mock.Mock()
        sess.commandFinished.connect(finished)
        QCoreApplication.processEvents()
        sess.abort()
        self._check_abort_session(sess)
        self.assertEqual(1, finished.call_count)

    def test_abort_session_not_running(self):
        sess1 = self.agent.runCommand(['id', '-i'])
        sess2 = self.agent.runCommand(['id', '-n'])
        finished = mock.Mock()
        sess1.commandFinished.connect(finished.sess1)
        sess2.commandFinished.connect(finished.sess2)

        sess2.abort()
        self.assertFalse(sess2.isFinished())
        CmdWaiter(sess1).wait()
        self._check_abort_session(sess2)
        # finished signals should be emitted in order
        self.assertEqual(['sess1', 'sess2'],
                         [x[0] for x in finished.method_calls])

    def test_abortcommands(self):
        sessions = map(self.agent.runCommand,
                       [['id', '-i'], ['id', '-n'], ['root']])
        self.agent.abortCommands()
        for sess in sessions:
            self._check_abort_session(sess)
        self.assertFalse(self.agent.isBusy())

    def _check_abort_session(self, sess):
        waiter = CmdWaiter(sess)
        waiter.wait()
        self.assertTrue(sess.isAborted())
        self.assertTrue(sess.isFinished())
        self.assertFalse(sess.isRunning())

    def test_busystate(self):
        self.assertFalse(self.agent.isBusy())
        sess = self.agent.runCommand(['id'])
        self.assertTrue(self.agent.isBusy())
        self.busyChanged.assert_called_once_with(True)
        self.busyChanged.reset_mock()

        CmdWaiter(sess).wait()
        self.assertFalse(self.agent.isBusy())
        self.busyChanged.assert_called_once_with(False)

    def test_busystate_queued(self):
        sess1 = self.agent.runCommand(['id'])
        self.busyChanged.assert_called_once_with(True)
        self.busyChanged.reset_mock()

        sess2 = self.agent.runCommand(['id'])
        self.assertTrue(self.agent.isBusy())

        CmdWaiter(sess1).wait()
        self.assertTrue(self.agent.isBusy())
        self.assertFalse(self.busyChanged.called)

        CmdWaiter(sess2).wait()
        self.assertFalse(self.agent.isBusy())
        self.busyChanged.assert_called_once_with(False)

    def test_busycount_first(self):
        stubsess = cmdcore.runningCmdSession()
        self.agent._enqueueSession(stubsess)
        self.assertTrue(self.agent.isBusy())

        sess = self.agent.runCommand(['id'])
        self.assertFalse(sess.isRunning())  # queued
        self.assertTrue(self.agent.isBusy())

        self.agent._dequeueSession(stubsess)
        QCoreApplication.processEvents()
        self.assertTrue(self.agent.isBusy())
        self.assertTrue(sess.isRunning())

        CmdWaiter(sess).wait()
        self.assertFalse(self.agent.isBusy())

    def test_busycount_intermediate(self):
        sess = self.agent.runCommand(['id'])

        stubsess = cmdcore.runningCmdSession()
        self.agent._enqueueSession(stubsess)
        self.assertTrue(self.agent.isBusy())

        self.agent._dequeueSession(stubsess)
        self.assertTrue(self.agent.isBusy())

        CmdWaiter(sess).wait()
        self.assertFalse(self.agent.isBusy())

    def test_busycount_intersect(self):
        sess = self.agent.runCommand(['id'])

        stubsess = cmdcore.runningCmdSession()
        self.agent._enqueueSession(stubsess)
        self.assertTrue(self.agent.isBusy())

        CmdWaiter(sess).wait()
        self.assertTrue(self.agent.isBusy())

        self.agent._dequeueSession(stubsess)
        self.assertFalse(self.agent.isBusy())
