/*
 *  This file is part of timespan.
 *
 *  Timespan is the legal property of its authors.  Refer to the file
 *  AUTHORS in the top-level directory of the source distribution for
 *  details.
 *  
 *  Timespan is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Timespan is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with timespan; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "timespan.hpp"

bool isleap(unsigned int year)
{ //see if year is a leap year or not

	/*
	 * After some research, we've discovered that leap years were handled
	 * somewhat differently before the 1600's.  Prior to the 1600's ALL years
	 * that were divisible by 4 were considered leap years, whether or not they
	 * were divisible by 100.  After 1600, the current system of years divisible
	 * by 4 but not by 100 or years divisible by 4 and by 400 being leap years
	 * took effect.  Therefore, prior to 1700, all years divisible by 100 are
	 * leap yars.  Now we test to see if the year is before or after 1601 and
	 * adjust the test accordingly.
	 */

	//if the year is prior to 1601, do the simpler test
	if(year < 1601)
	{ //braces needed to separate if from else
		//if the year is divisible by 4, it's a leap year
		if(!(year % 4))
			return true;
	}
	else //year is 1601 or later
	{ //braces not necessary, but in place for ease of reading
		//if the year is divisible by 4 and either not divisible by
		//100 or divisible by 400, then it is a leap year
		if(!(year % 4) && ((year % 100) || !(year % 400)))
			return true; //indicate leap year
		//otherwise the year is not a leap year
		else
			return false; //indicate a normal year
	}

	return false; //avoid MSVC6 warning
}

unsigned int get_month_num(char *s)
{ //take a string and get a month number based on it
	//if the first character of the string is alphabetic try looking for
	//the month names
	if(isalpha(s[0]))
	{
		//since few systems have case-insensitive string comparisons,
		//simulate it by forcing lowercase
		for(unsigned int i = 0; i < strlen(s); i++)
			if(isalpha(s[i]))
				s[i] = tolower(s[i]);

		//now that we can do "case insensitive" comparisons, find the month num
		for(unsigned int j = 0; j < 12; j++)
			if(!strcmp(s, lmonths[j]) || !strcmp(s, lsmonths[j]))
				return j + 1;

		//if execution reaches here the month given is invalid
		//explain invaild monhts
		cout << s << " is an invalid month." << endl;

		usage();
	}
	//if not, check to see if a valid month number was given
	else if(isdigit(s[0]))
	{
		unsigned int temp = atoi(s);

		//1 thru 12 are valid months, return valid months
		if(temp >= 1 && temp <= 12)
			return temp;
		//explain invalid months
		else
		{
			cout << s << " is an invalid month." << endl;

			usage();
		}
	}
	//explain invalid months
	else
	{
		cout << s << " is an invalid month." << endl;

		usage();
	}

	return 0; //execution should never reach here; just shut the compiler up
}

unsigned int get_day_num(char *s, unsigned int month, unsigned int year)
{
	if(isdigit(s[0]))
	{
		unsigned int temp = atoi(s);

		bool valid_leap = isleap(year) && temp <= monthlengthleap[month - 1],
			  valid_norm = !isleap(year) && temp <= monthlength[month - 1];

		//if temp is nonzero there was a number to convert
		if(temp && (valid_leap || valid_norm))
			return temp;
		//explain invalid days
		else
		{
			cout << s << " is an invalid day." << endl;

			usage();
		}
	}
	//explain invalid days
	else
	{
		cout << s << " is an invalid day." << endl;

		usage();
	}

	return 0; //execution should never reach here; just shut the compiler up
}

unsigned int get_year_num(char *s)
{ //take string and get a year number based on it
	//if the first character of the string is a number try to get the number
	if(isdigit(s[0]))
	{
		unsigned int temp = atoi(s);

		//if temp is nonzero there was a number to convert
		if(temp)
			return temp;
		//explain invalid years
		else
		{
			cout << s << " is an invalid year." << endl;

			usage();
		}
	}
	//explain invalid years
	else
	{
		cout << s << " is an invalid year." << endl;

		usage();
	}

	return 0; //execution should never reach here; just shut the compiler up
}

void print_date(const Date date, const Flag flags)
{
	if(flags.wantlongdate)
	{
		if(flags.usout)
			cout << months[date.imonth - 1] << " " << date.day << " "
				  << date.year << endl;
		else
			cout << date.day << " " << months[date.imonth - 1] << " "
				  << date.year << endl;
	}
	else
	{
		if(flags.usout)
			cout << date.imonth << " " << date.day << " " << date.year << endl;
		else
			cout << date.day << " " << date.imonth << " " << date.year << endl;
	}

	return;
}

Date get_todays_date()
{ //get today's date and return it

	const time_t thissec = time(NULL); //get the current time exactly

	tm *today = localtime(&thissec); //convert the time to user's local time

	//set up the Date to return
	Date temp = { today->tm_mon + 1, today->tm_mday, today->tm_year + 1900 };
	
	return temp; //return the Date
}

Date get_date(char *argv[], const int i, const Flag flags)
{
	//private helper functions
	unsigned int get_month_num(char *);
	unsigned int get_day_num(char *, unsigned int, unsigned int);
	unsigned int get_year_num(char *);

	Date temp = { 0, 0, 0 };
	
	if(flags.usin)
	{
		temp.imonth = get_month_num(argv[i]);
		temp.year = get_year_num(argv[i + 2]);
		temp.day = get_day_num(argv[i + 1], temp.imonth, temp.year);
	}
	else
	{
		temp.imonth = get_month_num(argv[i + 1]);
		temp.year = get_year_num(argv[i + 2]);
		temp.day = get_day_num(argv[i], temp.imonth, temp.year);
	}

	return temp;
}

#ifndef HAVE_BASENAME
char *basename (char *s)
{ //if a system doesn't have the basename function, then we must implement a
  //version of it if we want to use it.  all but the if statement with the
  //second call to strrchr() come from GNU libc.

	char *p = strrchr(s, '/'); //find the last occurance of '/' in the string

	//if there wasn't a '/' in the string, look for a backslash ('\\')
	if(p == NULL) p = strrchr(s, '\\');

	//if p points to a character, return a pointer to the char after it,
	//else return the same pointer that came in
	return p ? (p + 1) : s;
}
#endif

