/*
 *  This file is part of timespan.
 *
 *  Timespan is the legal property of its authors.  Refer to the file
 *  AUTHORS in the top-level directory of the source distribution for
 *  details.
 *
 *  Timespan is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Timespan is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with timespan; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "timespan.hpp"

#ifndef HAVE_IOSTREAM_64BIT_IO

namespace {

	// determine numeric base for i/o 
	int get_base(ios &s)
	{
		if((s.flags() & ios::hex) == ios::hex)
			return 16;
		if((s.flags() & ios::oct) == ios::oct)
			return 8;
		
		return 10;
	}

	// Fill a buffer with digits making up an int, while stripping leading zeros.
	// Return a pointer to the first non-zero within the supplied buffer.
	char *format_text(char *text, uint_least64_t v, int base, ostream &os)
	{
		*--text = '\0';
		
		if(v == 0)
		{
			// special case -- the number is 0
			*--text = '0';
			
			return text;
		}

		char *pc = text;
		uint_least64_t pow = base, prev_pow = 1, rem;
		char coeff;
		bool want_uppercase = ((os.flags () & ios::uppercase) == ios::uppercase);
	 
		do
		{
			rem = v % pow;
			coeff = rem / prev_pow;
			if(16 == base)
			{
				if(coeff < 10)
					*--pc = coeff + '0';
				else if(want_uppercase)
					*--pc = coeff - 10 + 'A';
				else
					*--pc = coeff - 10 + 'a';
			}
			else
				*--pc = coeff + '0';
		  
			if(coeff != 0)
				text = pc;
		  
			prev_pow = pow;
		  
			pow = pow * base;
		  
			v -= rem;
		} while(pow > prev_pow);
		
		return text;
	}

	// Get base/sign prefix '0x', '0', etc.
	size_t get_unsigned_prefix(char *prefix, uint_least64_t v, int base,
									  ostream &os)
	{
		if((os.flags() & ios::showbase) != ios::showbase)
			return 0;

		if(base == 16)
		{
			prefix[0] = '0';
			prefix[1] = 'x';
			prefix[2] = '\0';
			return 2;
		}

		if(base == 8 && v != 0)
		{
			prefix[0] = '0';
			prefix[1] = '\0';
			return 1;
		}

		return 0;
  }

	// Calculate fill length
	size_t get_fill(size_t prefix_leng, size_t text_leng, ostream &os)
	{
		if((prefix_leng + text_leng) < os.width())
			return os.width() - prefix_leng - text_leng;

		return 0;
	}

	// Write out prefix, fill and text
	void format_int(const char *prefix, size_t prefix_leng, size_t fill_leng,
						 const char *text, size_t text_leng, ostream &os)
	{
		if((os.flags() & ios::internal) == ios::internal)
		{
			os.write (prefix, prefix_leng);
			
			size_t i;
			
			for(i = 0; i < fill_leng; ++i)
				os.put (os.fill());
			
			os.write (text, text_leng);
		}
		else if((os.flags() & ios::right) == ios::right)
		{
			os.write (prefix, prefix_leng);
			
			os.write (text, text_leng);
			
			size_t i;
			
			for(i = 0; i < fill_leng; ++i)
				os.put (os.fill());
		}
		else
		{
			size_t i;
			
			for(i = 0; i < fill_leng; ++i)
				os.put (os.fill());
			
			os.write (prefix, prefix_leng);
			
			os.write (text, text_leng);
		}

		if((os.flags () & ios::unitbuf) == ios::unitbuf)
			os.flush();
	}

} // namespace <anonymous>


// print out a large unsigned integer
ostream &operator << (ostream &os, uint_least64_t v)
{
	if(!os) return os;

	char buf[64];
	char prefix[3];
	size_t prefix_leng, text_leng, fill_leng;
	char *text;
	int base;

	base = get_base (os);
	text = format_text (buf + sizeof (buf) - 1, v, base, os);
	prefix_leng = get_unsigned_prefix (prefix, v, base, os);
	text_leng = sizeof (buf) - (text - buf) - 2;
	fill_leng = get_fill (prefix_leng, text_leng, os);

	format_int (prefix, prefix_leng, fill_leng, text, text_leng, os);

	return os;
}

// read a large unsigned integer
istream &operator >> (istream &is, uint_least64_t &v)
{
	int base;
	int C;
	base = get_base(is);
	uint_least64_t value = 0, digit;
	int input_count = 0;

	if(base == 16)
	{
		if((C = is.get()) == '0')
		{
			if((C = is.get()) != 'x')
			{
				is.unget();
				input_count = 1;
			}
		}
		else
			is.unget();
	}

	while((C = is.get()) != -1)
	{
		if(base == 16)
		{
			if(C >= '0' && C <= '9')
				digit = C - '0';
			else if(C >= 'a' && C <= 'f')
				digit = C - 'a' + 10;
			else if(C >= 'A' && C <= 'F')
				digit = C - 'A' + 10;
			else
				break;
		}
		else if(base == 8)
		{
			if(C >= '0' && C <= '7')
				digit = C - '0';
			else
				break;
		}
		else if(C >= '0' && C <= '9')
			digit = C - '0';
		else
			break;

		value = value * base + digit;
		++input_count;
	}
	
	if(C != -1)
		is.unget();
	if(input_count)
		v = value;
	else
		is.setstate (ios::failbit);

	return is;
}

#endif

