/*
 *  This file is part of timespan.
 *
 *  Timespan is the legal property of its authors.  Refer to the file
 *  AUTHORS in the top-level directory of the source distribution for
 *  details.
 *  
 *  Timespan is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Timespan is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with timespan; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "timespan.hpp"

uint_least64_t calc_days_between(Date first_date, Date second_date)
{ //calculate the number of days between two specified dates

	uint_least64_t first_date_days = calcdays(first_date, true),
						second_date_days = calcdays(second_date, true);

	//because we're dealing with unsigned data (uint_least64_t is an unsigned
	//type) we have to make sure we can't possibly get a negative value
	if(first_date_days > second_date_days)
		return first_date_days - second_date_days;
	else if(first_date_days < second_date_days)
		return second_date_days - first_date_days;
	else
		return 0;
}

uint_least64_t calcdays(Date date, bool year_start_1)
{ //calculate the number of days between 1/1/1970 and specified date,
  //including the specified date in the calculation but not 1/1/1970

	unsigned int year; //year, used for loop control
	uint_least64_t lmonth,		  //numerical value of month for loop control
						totaldays = 0; //calculated number of days so far
	
	//loop through years from 1 or 1970 to one less than the year in date.
	for(year_start_1 ? year = 1 : year = 1970; year < date.year; year++)
	{
		//when starting in 1, 1/1/1 is not counted in calculations
		//when starting in 1970, 1/1/1970 is not counted in calculations
		if((year == 1970 && !year_start_1) || (year == 1 && year_start_1))
			totaldays += 364;
		//if the year is a leap year, add 366 days because of 2/29/year
		else if(isleap(year))
			totaldays += 366;
		//otherwise add 365 because all other years have 365 days
		else
			totaldays += 365;

		/* BUGFIX -- inaccuracy
		 * In the year 1582, Pope Gregory XIII decreed that October 4 would
		 * become October 15 to correct discrepancies in the calendar, thus the
		 * year lost 11 days.  If we have to come to the year 1582 in this loop,
		 * we need to subtract 11 days for accuracy.  This part of the fix will
		 * only execute when the year of the specified date is AFTER 1582.
		 */
		if(year == 1582)
			totaldays -= 11;
	}

	//now calculate the number of days in the current year and add to the
	//total.  this is done with the previously found month number
	//BUGFIX -- inaccuracy -- see above bugfix comment
	for(lmonth = 1; lmonth < date.imonth; lmonth++)
	{
		if(isleap(date.year))
			totaldays += monthlengthleap[lmonth - 1];
		//if we're going past October 1582, then we must account for that month
		//being 11 days shorter than normal!  this part of the fix will only
		//execute when the year of the specified date is 1582 AND the specified
		//date is ON OR AFTER November 1.
		else if(date.year == 1582 && date.imonth == 10)
			totaldays += monthlengthleap[9] - 11;
		else
			totaldays += monthlength[lmonth - 1];
	}

	//now all that's left is the number of days into the month in the
	//specified date, but check for some conditions first.
	if(date.year == 1582 && date.imonth == 10)
	{
		if(date.day <= 14)
			//october 4 thru october 14 did not exist in 1582, therefore we should
			//map october 4 to october 15, october 5 to october 16, etc., as some
			//calendaring officials have suggested for dealing with dates in that
			//range.
			totaldays += date.day;
		else
			//the date given was October 15 or later, so we must correct for the
			//11-day shortage
			totaldays += date.day - 11;
	}
	else
		//normal--add number of days into the month in the specified date
		totaldays += date.day;
			

	//it the year is 1970 then we need to subtract 1 because 1/1/1970 isn't
	//counted as a day.  since the year is 1970 the first loop above didn't do
	//`any work
	if(date.year == 1970 && !year_start_1)
		totaldays--;

	//now send back the calculated number of days
	return totaldays;
}

uint_least64_t calc_hours_between(Date dates, Date customdates)
{ //calculate the number of hours between two specified dates
	return calc_days_between(dates, customdates) * hours_in_day;
}

uint_least64_t calc_minutes_between(Date dates, Date customdates)
{ //calculate the number of minutes between two specified dates
	return calc_hours_between(dates, customdates) * minutes_in_hour;
}

uint_least64_t calc_seconds_between(Date dates, Date customdates)
{ //calculate the number of seconds between two specified dates
	return calc_minutes_between(dates, customdates) * seconds_in_minute;
}

uint_least64_t calchours(Date date)
{ //calculate the number of hours the specified date is from 1/1/1970
	return calcdays(date) * hours_in_day;
}

uint_least64_t calcminutes(Date date)
{ //calculate the number of minutes a specified date is from 1/1/1970
	return calchours(date) * minutes_in_hour;
}

uint_least64_t calcseconds(Date date)
{ //calculate the number of seconds a specified date is from 1/1/1970
	return calcminutes(date) * seconds_in_minute;
}

Date calcdate(uint_least64_t days)
{ //calculate the date given a number of days since 1/1/1970

	unsigned int i = 0,		 //loop control
					 year = 1970; //counts the current year
	//this is called days_converted because when subtracted from days, the number
	//of days stored here have been converted to years
	uint_least64_t days_converted = 0;
	Date date = { 0, 0, 0 };

	//accumulate days_converted while incrementing year so we can get the year
	//the date falls in here, then get the month and date below.
	while(days_converted < days)
	{
		//if the year is 1970, add 364 days to total
		if(year == 1970)
			days_converted += 364;
		//if the year is a leap year, add 366 days to total
		else if(isleap(year))
			days_converted += 366;
		//otherwise the year is normal; add 365 days
		else
			days_converted += 365;

		year++; //increment year 
	}

	//if days_converted > days, then the loop above went one year too far
	if(days_converted > days) //so correct the mistake
	{
		year--; //decrement yearbecause it's one year too late

		//if year is a leapyear, subtract 366 days, otherwise subtract 365
		if(isleap(year))
			days_converted -= 366;
		else
			days_converted -= 365;
		
		days -= days_converted;
	}
	//if days == days_converted then subtract from days
	else if(days == days_converted)
		days -= days_converted;

	//figure out what month the date is in
	if(days > 31) //we're somewhere after January
	{
		if(isleap(year))
		{
			while(days > monthlengthleap[i])
			{
				days -= monthlengthleap[i];
				i++;
			}
		}
		else
		{
			while(days > monthlength[i])
			{
				days -= monthlength[i];
				i++;
			}
		}

		//whatever the contents of month[i] are is the month we ended up in
		date.imonth = i + 1;
		date.day = days;
		date.year = year;
	}
	else if(days <= 31 && days > 0) //we're in January
	{
		date.imonth = 1;
		date.day = days;
		date.year = year;
	}
	else //days == 0, but January 0 is bad, so fix it
	{
		date.imonth = 12;
		date.day = 31;
		date.year = year - 1;
	}

	return date; //return the date calculated
}

