/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 4 -*-
 * vim: set ts=8 sts=4 et sw=4 tw=99:
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

/*

   IonPower (C)2015 Contributors to TenFourFox. All rights reserved.

   Authors: Cameron Kaiser <classilla@floodgap.com>
   with thanks to Ben Stuhl and David Kilbridge
   and the authors of the ARM and MIPS ports

 */

#include "jit/osxppc/Assembler-ppc.h"

#include "mozilla/DebugOnly.h"
#include "mozilla/MathAlgorithms.h"

#include "jscompartment.h"
#include "jsutil.h"

#include "gc/Marking.h"
#include "jit/ExecutableAllocator.h"
#include "jit/JitCompartment.h"

using mozilla::DebugOnly;

#if JS_CODEGEN_PPC_OSX
/* AsmJS stubs for non-AsmJS (big endian, basically). */
namespace js {

bool
IsAsmJSModule(HandleFunction fun)
{
    return false;
}

JSString *
AsmJSModuleToString(JSContext *cx, HandleFunction fun, bool addParenToLambda)
{
    return nullptr;
}

} // namespace js
#endif

using namespace js;
using namespace js::jit;

ABIArgGenerator::ABIArgGenerator()
  : stackOffset_(0),
    usedGPRs_(0),
    usedFPRs_(0),
    current_()
{}

ABIArg
ABIArgGenerator::next(MIRType type)
{
    // TODO:
    // This does not yet handle the situation where we overflow the argregs.
    switch (type) {
      case MIRType_Int32:
      case MIRType_Pointer:
		if (usedGPRs_ == 8) // i.e., we already allocated r10
			MOZ_CRASH("ABIArgGenerator overflowed (GPR)");
		current_ = ABIArg(Register::FromCode((Register::Code)(usedGPRs_ + 3)));
		usedGPRs_++;
		break;
      case MIRType_Float32:
      case MIRType_Double:
		if (usedFPRs_ == 12) // i.e., we already allocated f13
			MOZ_CRASH("ABIArgGenerator overflowed (FPR)");
		current_ = ABIArg(FloatRegister::FromCode((Register::Code)(usedFPRs_ + 1)));
		usedGPRs_ += 2;
		usedFPRs_ ++;
		break;
      default:
		MOZ_CRASH("Unexpected argument type");
    }
    return current_;
}
// XXX?
const Register ABIArgGenerator::NonArgReturnReg0 = r9;
const Register ABIArgGenerator::NonArgReturnReg1 = r10;
const Register ABIArgGenerator::NonArg_VolatileReg = r13;
const Register ABIArgGenerator::NonReturn_VolatileReg0 = r14;
const Register ABIArgGenerator::NonReturn_VolatileReg1 = r15;

void
jit::PatchJump(CodeLocationJump &jump_, CodeLocationLabel label, ReprotectCode reprotect)
{
    Instruction *inst1 = (Instruction *)jump_.raw();
    Instruction *inst2 = inst1->next();
    uint32_t target = (uint32_t)label.raw();

    // If we optimized this to a naked b, then try to keep it that
    // way first.
    if (inst1[0].isOpcode(PPC_b)) {
        int32_t diff = target - (uint32_t)(uintptr_t(inst1));
//fprintf(stderr, "patch: 0x%08x target %08x diff %i\n", uintptr_t(inst1),
//target, diff);
        if (diff > -0x02000000 && diff < 0x02000000) {
            // Patch in place.
            MaybeAutoWritableJitCode awjc(inst1, 4, reprotect);
            inst1[0].setData(PPC_b | (diff & 0x03ffffff) |
                (inst1[0].encode() & Assembler::LinkB));
            AutoFlushICache::flush(uintptr_t(inst1), 4);
            return;
        }
        // Demote to a long branch. It will stay this way.
        MaybeAutoWritableJitCode awjc(inst1, 16, reprotect);
        inst1[3].assertOpcode(PPC_nop);
	inst1[3].makeOp_bctr((Assembler::LinkBit)(inst1[0].encode() &
            Assembler::LinkB));
        inst1[2].assertOpcode(PPC_nop);
	inst1[2].makeOp_mtctr(r0);
        inst1[1].assertOpcode(PPC_nop);
	Assembler::WriteLisOriInstructions(&inst1[0], &inst1[1], r0, target);
        AutoFlushICache::flush(uintptr_t(inst1), 16);
        return;
    }

    // Patch long branches.
    MaybeAutoWritableJitCode awjc(inst1, 8, reprotect);
    Assembler::UpdateLisOriValue(inst1, inst2, target);
    AutoFlushICache::flush(uintptr_t(inst1), 8);
}

/*
 
 Backedges look like this (generated by MacroAssembler's backedgeJump).
 
 b (target)
    Target is either: the target
                  or: .+4
                  or: .+20
 lis/ori BackedgeLoopHeaderTarget
 mtctr
 bctr
 lis/ori OtherTarget (interrupt loop, only occurs in AsmJS)
 mtctr
 bctr
 
 The reason for this format is to avoid patching lis/ori while execution is
 occuring; we only have to patch a single instruction (the lead branch).
 
 */

void
jit::PatchBackedge(CodeLocationJump &jump, CodeLocationLabel label,
                   JitRuntime::BackedgeTarget target)
{
    uint32_t sourceAddr = (uint32_t)jump.raw();
    uint32_t targetAddr = (uint32_t)label.raw();
    InstImm *branch = (InstImm *)jump.raw();

#if DEBUG
	JitSpew(JitSpew_Codegen, "# ::PatchBackedge evaluating %08x %08x %08x\n",
		(uint32_t)branch, sourceAddr, targetAddr);
    branch->assertOpcode(PPC_b);
    MOZ_ASSERT(!(branch->encode() & Assembler::LinkB)); // Shouldn't get here!
#endif

// TODO: Handle relocation if the target address itself fits.
#if(0)
    if (JOffImm26::IsInRange(targetAddr - sourceAddr) || JOffImm26::IsInRange(targetAddr)) {
    	// The branch is within range; set it directly.
    	// Rebuild the branch instruction to clear all necessary bits.
    	if (JOffImm26::IsInRange(targetAddr)) {
    		// Absolute branch.
       		JOffImm26(targetAddr) timm;
    		branch->setData(PPC_b | timm.encode() | AbsoluteBranch);
    	} else {
    		JOffImm26(targetAddr - sourceAddr) jimm;
        	branch->setData(PPC_b | jimm.encode());
        }
#endif
    if (JOffImm26::IsInRange(targetAddr - sourceAddr)) {
    	// The branch is within range; set it directly.
    	// Rebuild the branch instruction to clear all necessary bits.
    	JOffImm26 jimm = JOffImm26(targetAddr - sourceAddr);
        branch->setData(PPC_b | jimm.encode());
    } else {
    	// The branch is not. Set either the first or second target.
        if (target == JitRuntime::BackedgeLoopHeader) {
            Instruction *lis = &branch[1];
            Assembler::UpdateLisOriValue(lis, lis->next(), targetAddr);
            branch->setData(PPC_b | 0x04);
        } else {
            Instruction *lis = &branch[5];
            Assembler::UpdateLisOriValue(lis, lis->next(), targetAddr);
            branch->setData(PPC_b | 0x14);
        }
    }
    // Flush everything.
    AutoFlushICache::flush(uintptr_t(branch), 36);
}

// Debugging convenience macros (copied over from JM/PPCBC).
#if defined(DEBUG)
#define JSPW(x, y, ...)  { \
    spew_with_address(y, x, ##__VA_ARGS__); \
    return writeInst(x); \
    }

#define JSPWS(x, y) { \
    spew_with_address(y, x); \
    return writeInst(x); \
    }
#else
#define JSPW(x, y, ...) return writeInst(x);
#define JSPWS(x, y) return writeInst(x);
#endif

#if defined(DEBUG)
        // Specialized spewer to show addresses so we can monitor codegen.
        // Ion doesn't do this for us, apparently?
        void Assembler::spew_with_address(const char *fmt, uint32_t ins, ...) {
            char buf[200];
            // We emit this *before* we emit the instruction so it matches up.
            intptr_t code = (intptr_t)(m_buffer.size());

            va_list va;
            va_start(va, ins);
            (void)vsnprintf(buf, sizeof(buf), fmt, va);
            va_end(va);

            if (code == 0x00000000) 
                JitSpew(JitSpew_Codegen, "////////");
            JitSpew(JitSpew_Codegen, "%08x --- %08x %s", code, ins, buf);
        }
#endif

void
Assembler::finish()
{
    MOZ_ASSERT(!isFinished);
    isFinished = true;
}

void
Assembler::executableCopy(uint8_t *buffer)
{
#if DEBUG
	JitSpew(JitSpew_Codegen,"## executableCopy to 0x%08x 0x%08x+0x%x\n", (uint32_t)buffer, (uint32_t)buffer, m_buffer.size());
#endif
    MOZ_ASSERT(isFinished);
    m_buffer.executableCopy(buffer);

    // Patch all long jumps during code copy.
    for (size_t i = 0; i < longJumps_.length(); i++) {
#if DEBUG
        JitSpew(JitSpew_Codegen,"### examining longJump @ %08x (location: 0x%08x)",
            (uint32_t)longJumps_[i], ((uint32_t)buffer + longJumps_[i]));
        // Look for duplicates.
        for (size_t j = 0; j < i; j++) {
            MOZ_ASSERT(longJumps_[j] != longJumps_[i]);
        }
#endif
        Instruction *inst1 = (Instruction *) ((uint32_t)buffer + longJumps_[i]);
        uint32_t value = ExtractLisOriValue(inst1, inst1->next());
#if DEBUG
        JitSpew(JitSpew_Codegen,"### ... value: %08x", (uint32_t)value);
#endif

        // Can we optimize further?
        // Branches to JitCode may be already in lis/ori/mtctr/bctr format
        // if the branch was stowed earlier. In that case, we have another
        // chance to save them. (JitCode calls have the return address push
        // preceding the branch stanza, so this catches them too.)
        // Note that PatchJump may patch these multiple times.
        int32_t diff = value - longJumps_[i];
        if(inst1[2].isOpcode(PPC_mtspr) && inst1[3].isOpcode(PPC_bctr) &&
            // Displacement can be 26 full bits.
            diff > -0x02000000 && diff < 0x02000000) {
            // Pull branch up to lead the stanza.
            inst1[0].setData(PPC_b | (diff & 0x03ffffff) |
                // Save the link bit!
                (inst1[3].encode() & LinkB)); // bctr(l) -> b(l)
            inst1[1].setData(PPC_nop);
            inst1[2].setData(PPC_nop);
            inst1[3].setData(PPC_nop);
        } else {
            UpdateLisOriValue(inst1, inst1->next(), (uint32_t)buffer + value);
        }
    }

    AutoFlushICache::setRange(uintptr_t(buffer), m_buffer.size());
}

uint32_t
Assembler::actualOffset(uint32_t off_) const
{
    return off_;
}

uint32_t
Assembler::actualIndex(uint32_t idx_) const
{
    return idx_;
}

uint8_t *
Assembler::PatchableJumpAddress(JitCode *code, uint32_t pe_)
{
    return code->raw() + pe_;
}

class RelocationIterator
{
    CompactBufferReader reader_;
    // offset in bytes
    uint32_t offset_;

  public:
    RelocationIterator(CompactBufferReader &reader)
      : reader_(reader)
    { }

    bool read() {
        if (!reader_.more())
            return false;
        offset_ = reader_.readUnsigned();
        return true;
    }

    uint32_t offset() const {
        return offset_;
    }
};

uintptr_t
Assembler::GetPointer(uint8_t *instPtr)
{
    Instruction *inst = (Instruction*)instPtr;
    return Assembler::ExtractLisOriValue(inst, inst->next());
}

static JitCode *
CodeFromJump(Instruction *jump)
{
    uint8_t *target = (uint8_t *)Assembler::ExtractLisOriValue(jump, jump->next());
    return JitCode::FromExecutable(target);
}

void
Assembler::TraceJumpRelocations(JSTracer *trc, JitCode *code, CompactBufferReader &reader)
{
    RelocationIterator iter(reader);
    while (iter.read()) {
        JitCode *child = CodeFromJump((Instruction *)(code->raw() + iter.offset()));
        TraceManuallyBarrieredEdge(trc, &child, "rel32");
    }
}

static void
TraceOneDataRelocation(JSTracer *trc, Instruction *inst)
{
    void *ptr = (void *)Assembler::ExtractLisOriValue(inst, inst->next());
    void *prior = ptr;
    
    // The low bit should not be set.
    MOZ_ASSERT(!(uintptr_t(ptr) & 0x1));

    // No barrier needed since these are constants.
    TraceManuallyBarrieredGenericPointerEdge(trc, reinterpret_cast<gc::Cell**>(&ptr),
                                                 "ion-masm-ptr");
    if (ptr != prior) {
        Assembler::UpdateLisOriValue(inst, inst->next(), uint32_t(ptr));
        AutoFlushICache::flush(uintptr_t(inst), 8);
    }
}

static void
TraceDataRelocations(JSTracer *trc, uint8_t *buffer, CompactBufferReader &reader)
{
    while (reader.more()) {
        size_t offset = reader.readUnsigned();
        Instruction *inst = (Instruction*)(buffer + offset);
        TraceOneDataRelocation(trc, inst);
    }
}

static void
TraceDataRelocations(JSTracer *trc, PPCBuffer *buffer, CompactBufferReader &reader)
{
    while (reader.more()) {
        BufferOffset bo (reader.readUnsigned());
        PPCBuffer::AssemblerBufferInstIterator iter(bo, buffer);
        TraceOneDataRelocation(trc, iter.cur());
    }
}

void
Assembler::TraceDataRelocations(JSTracer *trc, JitCode *code, CompactBufferReader &reader)
{
    ::TraceDataRelocations(trc, code->raw(), reader);
}

#if(0)
void
Assembler::FixupNurseryObjects(JSContext* cx, JitCode* code, CompactBufferReader& reader,
                               const ObjectVector& nurseryObjects)
{
    MOZ_ASSERT(!nurseryObjects.empty());

    uint8_t* buffer = code->raw();
    bool hasNurseryPointers = false;

    while (reader.more()) {
        size_t offset = reader.readUnsigned();
        Instruction* inst = (Instruction*)(buffer + offset);

        void* ptr = (void*)Assembler::ExtractLisOriValue(inst, inst->next());
        uintptr_t word = uintptr_t(ptr);

        if (!(word & 0x1))
            continue;

        uint32_t index = word >> 1;
        JSObject* obj = nurseryObjects[index];

        Assembler::UpdateLisOriValue(inst, inst->next(), uint32_t(obj));
        AutoFlushICache::flush(uintptr_t(inst), 8);

        // Either all objects are still in the nursery, or all objects are
        // tenured.
        MOZ_ASSERT_IF(hasNurseryPointers, IsInsideNursery(obj));

        if (!hasNurseryPointers && IsInsideNursery(obj))
            hasNurseryPointers = true;
    }

    if (hasNurseryPointers)
        cx->runtime()->gc.storeBuffer.putWholeCellFromMainThread(code);
}
#endif

void
Assembler::copyJumpRelocationTable(uint8_t *dest)
{
    if (jumpRelocations_.length())
        memcpy(dest, jumpRelocations_.buffer(), jumpRelocations_.length());
}

void
Assembler::copyDataRelocationTable(uint8_t *dest)
{
    if (dataRelocations_.length())
        memcpy(dest, dataRelocations_.buffer(), dataRelocations_.length());
}

void
Assembler::copyPreBarrierTable(uint8_t *dest)
{
    if (preBarriers_.length())
        memcpy(dest, preBarriers_.buffer(), preBarriers_.length());
}

void
Assembler::trace(JSTracer *trc)
{
    for (size_t i = 0; i < jumps_.length(); i++) {
        RelativePatch &rp = jumps_[i];
        if (rp.kind == Relocation::JITCODE) {
            JitCode *code = JitCode::FromExecutable((uint8_t *)rp.target);
            TraceManuallyBarrieredEdge(trc, &code, "masmrel32");
            MOZ_ASSERT(code == JitCode::FromExecutable((uint8_t *)rp.target));
        }
    }
    if (dataRelocations_.length()) {
        CompactBufferReader reader(dataRelocations_);
        ::TraceDataRelocations(trc, &m_buffer, reader);
    }
}

void
Assembler::processCodeLabels(uint8_t *rawCode)
{
    for (size_t i = 0; i < codeLabels_.length(); i++) {
        CodeLabel label = codeLabels_[i];
        Bind(rawCode, label.patchAt(), rawCode + label.target()->offset());
    }
}

int32_t
Assembler::ExtractCodeLabelOffset(uint8_t *code) {
    InstImm *inst = (InstImm *)code;
    return Assembler::ExtractLisOriValue(inst, inst->next());
}

void
Assembler::bindS(BufferOffset s, BufferOffset i) {
    // Given instruction i, set its low 16-bits to the difference between
    // i and buffer offset s as a fixed short branch.

    // If the offset of i is not yet assigned, then this is defective, and
    // we should not patch it (such as a hanging branch).
    if (!i.assigned()) return;

    int32_t offs = (s.getOffset() - i.getOffset());
    MOZ_ASSERT((offs < 32768) && (offs > -32768));

    InstImm *inst = (InstImm *)editSrc(i);
    MOZ_ASSERT(inst->isOpcode(PPC_b) || inst->isOpcode(PPC_bc));
#if _PPC970_
    // Translate the likely y-bit to likely at-bits.
    // TODO: In this version, we assume no hint set means no hint,
    // so we only act if LikelyB is set (bdnz handled separately).
    // This can miss possibly half our hinting, so we should overhaul
    // the assembler to have Likely, NotLikely and NoHint options with
    // conventional semantics and translate to the proper encoding at
    // runtime.
    uint32_t op = inst->encode();
    uint32_t bo = (op & 0x03FFFFFF) >> 21;
    op &= 0xFC1FFFFF; // clear bo bits
    
    // bc with likely bit for cr[bi]
    if (bo == 5 || bo == 13) {
        if (bo == 5) // branch forward is likely, cr[bi]==1
            bo = (offs < 0) ? 6 : 7; // 00110 vs 00111
        else // cr[bi]==0
            bo = (offs < 0) ? 14 : 15;
        inst->setData(op | (bo<<21) | ((uint16_t)offs & 0xFFFC));
    } else // can't ascertain, don't muck with it
#endif
    inst->setImm16((int16_t)offs);
}

void
Assembler::bindSS(BufferOffset i) { bindS(nextOffset(), i); }

void
Assembler::Bind(uint8_t *rawCode, CodeOffset *label, const void *address)
{
    if (label->bound()) {
        int32_t src = label->offset();
#if DEBUG
	JitSpew(JitSpew_Codegen,"## ::Bind %08x+%08x to %08x", (uint32_t)rawCode, (uint32_t)src, (uint32_t)address);
#endif
        Instruction *inst = (Instruction *) (rawCode + src);
        Assembler::UpdateLisOriValue(inst, inst->next(), (uint32_t)address);
    }
}

Assembler::Condition
Assembler::InvertCondition(Condition cond)
{
    switch (cond) {
      case Equal:
        return NotEqual;
      case NotEqual:
        return Equal;
      /* Zero == Equal, etc.
      case Zero:
        return NonZero;
      case NonZero:
        return Zero; */
      case LessThan:
        return GreaterThanOrEqual;
      case LessThanOrEqual:
        return GreaterThan;
      case GreaterThan:
        return LessThanOrEqual;
      case GreaterThanOrEqual:
        return LessThan;
      case Above:
        return BelowOrEqual;
      case AboveOrEqual:
        return Below;
      case Below:
        return AboveOrEqual;
      case BelowOrEqual:
        return Above;
      /* Signed == LessThan, etc.
      case Signed:
        return NotSigned;
      case NotSigned:
        return Signed; */
      default:
        MOZ_CRASH("unexpected condition");
    }
}

Assembler::DoubleCondition
Assembler::InvertCondition(DoubleCondition cond)
{
    switch (cond) {
      case DoubleOrdered:
        return DoubleUnordered;
      case DoubleEqual:
        return DoubleNotEqualOrUnordered;
      case DoubleNotEqual:
        return DoubleEqualOrUnordered;
      case DoubleGreaterThan:
        return DoubleLessThanOrEqualOrUnordered;
      case DoubleGreaterThanOrEqual:
        return DoubleLessThanOrUnordered;
      case DoubleLessThan:
        return DoubleGreaterThanOrEqualOrUnordered;
      case DoubleLessThanOrEqual:
        return DoubleGreaterThanOrUnordered;
      case DoubleUnordered:
        return DoubleOrdered;
      case DoubleEqualOrUnordered:
        return DoubleNotEqual;
      case DoubleNotEqualOrUnordered:
        return DoubleEqual;
      case DoubleGreaterThanOrUnordered:
        return DoubleLessThanOrEqual;
      case DoubleGreaterThanOrEqualOrUnordered:
        return DoubleLessThan;
      case DoubleLessThanOrUnordered:
        return DoubleGreaterThanOrEqual;
      case DoubleLessThanOrEqualOrUnordered:
        return DoubleGreaterThan;
      default:
        MOZ_CRASH("unexpected condition");
    }
}

BOffImm16::BOffImm16(InstImm inst)
  : data(inst.encode() & 0xffff)
{
}

bool
Assembler::oom() const
{
    return AssemblerShared::oom() ||
           m_buffer.oom() ||
           jumpRelocations_.oom() ||
           dataRelocations_.oom() ||
           preBarriers_.oom();
}

void
Assembler::addCodeLabel(CodeLabel label)
{
    return propagateOOM(codeLabels_.append(label));
}

// Size of the instruction stream, in bytes.
size_t
Assembler::size() const
{
    return m_buffer.size();
}

// Size of the relocation table, in bytes.
size_t
Assembler::jumpRelocationTableBytes() const
{
    return jumpRelocations_.length();
}

size_t
Assembler::dataRelocationTableBytes() const
{
    return dataRelocations_.length();
}

size_t
Assembler::preBarrierTableBytes() const
{
    return preBarriers_.length();
}

// Size of the data table, in bytes.
size_t
Assembler::bytesNeeded() const
{
    return size() +
           jumpRelocationTableBytes() +
           dataRelocationTableBytes() +
           preBarrierTableBytes();
}

// Write a 32-bit word into the buffer.
BufferOffset
Assembler::writeInst(uint32_t x, uint32_t *dest)
{
    if (MOZ_LIKELY(dest == nullptr))
        return m_buffer.putInt(x);

    WriteInstStatic(x, dest);
    return BufferOffset();
}

void
Assembler::WriteInstStatic(uint32_t x, uint32_t *dest)
{
    MOZ_ASSERT(dest != nullptr);
    *dest = x;
}

BufferOffset
Assembler::align(int alignment, bool useTrap)
{
    BufferOffset ret;
    MOZ_ASSERT(m_buffer.isAligned(4));
    if (alignment == 8) {
        if (!m_buffer.isAligned(alignment)) {
            BufferOffset tmp = (useTrap) ? x_trap() : x_nop();
            if (!ret.assigned())
                ret = tmp;
        }
    } else {
        MOZ_ASSERT((alignment & (alignment - 1)) == 0);
        while (size() & (alignment - 1)) {
            BufferOffset tmp = (useTrap) ? x_trap() : x_nop();
            if (!ret.assigned())
                ret = tmp;
        }
    }
    return ret;
}

// Convert a synthetic condition code into a sequence of instructions needed to interpret
// it, returning a native condition for a terminal branch with the proper condition register.
uint32_t Assembler::computeConditionCode(Condition op, CRegisterID cr) {
	// Mask off the unsigned bit, if present. Hopefully we handled this already!
	uint32_t newop = (uint32_t)op & 255;
	
	// Currently only one condition code (Overflow) uses XER bits. We
	// load that into cr and set the appropriate code.
	if (op & ConditionOnlyXER) {
		MOZ_ASSERT(op == Overflow);
		// Get XER into CR.
		x_mcrxr(cr);
		// Convert op to GT.
		newop = (uint32_t)GreaterThan;
	}
	
	// Set BIF to the proper CR. In cr0, the normal state, this just returns newop.
	return (newop + ((uint8_t)cr << 6));
}
uint32_t Assembler::computeConditionCode(DoubleCondition op, CRegisterID cr) {
	// Use condition register logic to combine the FU (FUUUU-! I mean, unordered)
	// bit with the actual condition bit.
	const uint8_t condBit = crBit(cr, op);
	const uint8_t fuBit = crBit(cr, DoubleUnordered);
	uint32_t newop = (uint32_t)op & 255;
	
	if (op & DoubleConditionUnordered) {
		// branch if condition true OR Unordered
		if ((op & BranchOptionMask) == BranchOnClear) {
			// invert the condBit, or it with fuBit, and branch on Set
			crorc(condBit, fuBit, condBit);
			newop |= BranchOnSet;
		} else {
			// or the condBit with fuBit, and then branch on Set
			if (condBit != fuBit)
				cror(condBit, fuBit, condBit);
		}
	} else {
		// branch if condition true AND ordered
		if ((op & BranchOptionMask) == BranchOnClear) {
			// or the condBit with fuBit, and branch on Clear
			if (condBit != fuBit)
				cror(condBit, fuBit, condBit);
		} else {
			// and the condBit with (!fuBit), and branch on Set, but
			// don't clear SO if this is actually DoubleUnordered
			// (fuBit == condBit), which is NOT a synthetic condition.
			if (condBit != fuBit)
				crandc(condBit, condBit, fuBit);
		}
	}
	
	// Set BIF to the proper CR. In cr0, the normal state, this just returns newop.
	return (newop + ((uint8_t)cr << 6));
}

/* INSTRUCTIONS AWESOME YAY */
BufferOffset Assembler::x_nop() { JSPWS(PPC_nop, "nop"); }

// Common convenience macros for GPRs since we don't control the Register struct.
// FloatRegister doesn't need these because it has an operator int().
#define _R(x) ((uint32_t)x.code())
#define _RA ((uint32_t)ra.code())
#define _RB ((uint32_t)rb.code())
#define _RD ((uint32_t)rd.code())
#define _RS ((uint32_t)rs.code())

// Branch and jump instructions.
// b() and bc() to labels are considered part of the MacroAssembler.
// XXX: See MacroAssembler-mips.cpp::branchWithCode   
BufferOffset Assembler::b(JOffImm26 off, BranchAddressType bat, LinkBit lb) {
	// The JOffImm26 has already done the masking and validation.
	JSPW(PPC_b | off.encode() | bat | lb,
		"b%s%s .%+d",
		(lb ? "l" : ""),
                (bat == AbsoluteBranch ? "a" : ""),
                off.decode());
}
BufferOffset Assembler::b(int32_t off, BranchAddressType bat, LinkBit lb) {
	return b(JOffImm26(off), bat, lb);
}
BufferOffset Assembler::blr(LinkBit lb) {
	JSPW(PPC_blr | (lb ? 1 : 0), "blr%s", ((lb) ? "l" : ""));
}
BufferOffset Assembler::bctr(LinkBit lb) {
	JSPW(PPC_bctr | (lb ? 1 : 0), "bctr%s", ((lb) ? "l" : ""));
}

// Conditional branches.
BufferOffset Assembler::bc(BOffImm16 off, Condition cond, CRegisterID cr, LikelyBit lkb, LinkBit lb) {
    MOZ_ASSERT(BOffImm16::IsInSignedRange(off.decode()));
	uint32_t op = computeConditionCode(cond, cr); // This may emit instructions
	int likely = (int)lkb;
	int is_link = (int)lb;
    int bi = (op >> 4);
    int bo = (op & 15);
    JSPW((PPC_bc | (bo | likely)<<21 | bi<<16 | off.encode() | is_link),
        "bc%s%s %d,%d,%d", (is_link)?"l":"",
        (likely)?"+":"", bo, bi, off.decode());
}
BufferOffset Assembler::bc(int16_t off, Condition cond, CRegisterID cr, LikelyBit lkb, LinkBit lb) {
	return bc(BOffImm16(off), cond, cr, lkb, lb);
}
BufferOffset Assembler::bc(BOffImm16 off, DoubleCondition cond, CRegisterID cr, LikelyBit lkb, LinkBit lb) {
	uint32_t op = computeConditionCode(cond, cr); // This may emit instructions
	int likely = (int)lkb;
	int is_link = (int)lb;
    int bi = (op >> 4);
    int bo = (op & 15);
    JSPW((PPC_bc | (bo | likely)<<21 | bi<<16 | off.encode() | is_link),
        "bc%s%s %d,%d,%d", (is_link)?"l":"",
        (likely)?"+":"", bo, bi, off.decode());
}
BufferOffset Assembler::bc(int16_t off, DoubleCondition cond, CRegisterID cr, LikelyBit lkb, LinkBit lb) {
	return bc(BOffImm16(off), cond, cr, lkb, lb);
}
BufferOffset Assembler::bcctr(Condition cond, CRegisterID cr, LikelyBit lkb, LinkBit lb) {
	uint32_t op = computeConditionCode(cond, cr); // This may emit instructions
	int likely = (int)lkb;
	int is_link = (int)lb;
    int bi = (op >> 4);
    int bo = (op & 15);
    JSPW((PPC_bcctr | (bo | likely)<<21 | bi<<16 | is_link),
                "bcctr%s %d,%d",  (likely)?"+":"", bo, bi);
}
BufferOffset Assembler::bcctr(DoubleCondition cond, CRegisterID cr, LikelyBit lkb, LinkBit lb) {
	uint32_t op = computeConditionCode(cond, cr); // This may emit instructions
	int likely = (int)lkb;
	int is_link = (int)lb;
    int bi = (op >> 4);
    int bo = (op & 15);
    JSPW((PPC_bcctr | (bo | likely)<<21 | bi<<16 | is_link),
                "bcctr%s %d,%d",  (likely)?"+":"", bo, bi);
}

// Unsafe raw variants used by the MacroAssembler directly as conveniences.
BufferOffset Assembler::bc(int16_t off, uint32_t op, LikelyBit lkb, LinkBit lb) {
	int likely = (int)lkb;
	int is_link = (int)lb;
    int bi = (op >> 4);
    int bo = (op & 15);
    JSPW((PPC_bc | (bo | likely)<<21 | bi<<16 | (off & 0xfffc) | is_link),
        "bc%s%s %d,%d,%d", (is_link)?"l":"",
        (likely)?"+":"", bo, bi, off);
}
BufferOffset Assembler::bcctr(uint32_t op, LikelyBit lkb, LinkBit lb) {
	int likely = (int)lkb;
	int is_link = (int)lb;
    int bi = (op >> 4);
    int bo = (op & 15);
    JSPW((PPC_bcctr | (bo | likely)<<21 | bi<<16 | is_link),
                "bcctr%s %d,%d",  (likely)?"+":"", bo, bi);
}


// SPR operations.
BufferOffset Assembler::mtspr(SPRegisterID spr, Register ra) {
	JSPW(PPC_mtspr | _RA<<21 | PPC_SPR(spr) << 11,
		"mtspr %s,%s", nSPR(spr), nGPR(ra));
}
BufferOffset Assembler::mfspr(Register rd, SPRegisterID spr) {
	JSPW(PPC_mfspr | _RD<<21 | PPC_SPR(spr) << 11,
		"mfspr %s,%s", nGPR(rd), nSPR(spr));
}

// CR operations.
#define DEF_CRCR(op) \
    BufferOffset Assembler::op(uint8_t t, uint8_t a, uint8_t b) {       \
        JSPW(PPC_##op | t << 21 | a << 16 | b << 11,                    \
            "%s %i,%i,%i", #op, t, a, b);                               \
    }
        DEF_CRCR(crand)
        DEF_CRCR(crandc)
        DEF_CRCR(cror)
        DEF_CRCR(crorc)
        DEF_CRCR(crxor)
#undef DEF_CRCR

BufferOffset Assembler::mtcrf(uint32_t mask, Register rs) {
	JSPW(PPC_mtcrf | _RS << 21 | mask << 12, "mtcrf %d,%s", mask, nGPR(rs));
}
BufferOffset Assembler::mfcr(Register rd) {
	JSPW(PPC_mfcr | _RD << 21, "mfcr %s", nGPR(rd));
}
BufferOffset Assembler::mfocrf(Register rd, CRegisterID crfs) { // G5 only
	JSPW(PPC_mfcr | _RD << 21 | 1 << 20 | 1 << (19-(int)crfs), "mfocrf %s,%s", nGPR(rd), nCR(crfs));
}
BufferOffset Assembler::x_mcrxr(CRegisterID crt, Register temp) {	// Emulated on G5! EEEK!
#if defined(_PPC970_) || defined(DEBUG)
	// A temp register is required. r12 is the default.
	// Use alternate code in DEBUG mode too since I usually debug on a G5 and I'd
	// like my test runs to finish sometime before the heat death of the universe.
	//
	// Manually copy the bits out of the XER into the CR field we want.
	// This is slower than the G3/G4 instruction, but it's sure faster
	// than having to let the OS X kernel emulate it via instruction fault.
	mfspr(temp, xer);
	// Rotate the bits into position.
	if ((int)crt > 0) // Rotation not needed if CR0 is the destination.
		rlwinm(temp, temp, 32-(4*(int)crt), 0, 31);
	// Copy the bits to the right field.
	mtcrf((1 << (7-(int)crt)), temp);
	// Rotate back (if required) and mask off the copied bits.
	rlwinm(temp, temp, 4*(int)crt, 4, 31);
	return mtspr(xer, temp);
#else
	// The temp register is not used by G3/G4, since they can execute mcrxr directly.
	JSPW(PPC_mcrxr | (int)crt << 23, "mcrxr %s", nCR(crt));
#endif
}

// GPR operations and load-stores.
BufferOffset Assembler::neg(Register rd, Register rs) {
	JSPW(PPC_neg | _RD << 21 | _RS << 16, "neg %s,%s", nGPR(rd), nGPR(rs));
}
BufferOffset Assembler::cmpw(CRegisterID cr, Register ra, Register rb) {
	JSPW(PPC_cmpw | cr << 23 | _RA << 16 | _RB << 11, "cmpw %s,%s,%s", nCR(cr), nGPR(ra), nGPR(rb));
}
BufferOffset Assembler::cmpwi(CRegisterID cr, Register ra, int16_t im) {
	JSPW(PPC_cmpwi | cr << 23 | _RA << 16 | uint16_t(im), "cmpwi %s,%s,%i (0x%x)", nCR(cr), nGPR(ra), im, im);
}
BufferOffset Assembler::cmplw(CRegisterID cr, Register ra, Register rb) {
	JSPW(PPC_cmplw | cr << 23 | _RA << 16 | _RB << 11, "cmplw %s,%s,%s", nCR(cr), nGPR(ra), nGPR(rb));
}
BufferOffset Assembler::cmplwi(CRegisterID cr, Register ra, int16_t im) {
	JSPW(PPC_cmplwi | cr << 23 | _RA << 16 | uint16_t(im), "cmplwi %s,%s,%u (0x%x)", nCR(cr), nGPR(ra), im, im);
}
BufferOffset Assembler::cmpw(Register ra, Register rb) { return cmpw(cr0, ra, rb); }
BufferOffset Assembler::cmpwi(Register ra, int16_t im) { return cmpwi(cr0, ra, im); }
BufferOffset Assembler::cmplw(Register ra, Register rb) { return cmplw(cr0, ra, rb); }
BufferOffset Assembler::cmplwi(Register ra, int16_t im) { return cmplwi(cr0, ra, im); }
BufferOffset Assembler::srawi(Register rd, Register rs, uint8_t n) {
	JSPW(PPC_srawi | _RS << 21 | _RD << 16 | n << 11, "srawi %s,%s,%d", nGPR(rd), nGPR(rs), n);
}
BufferOffset Assembler::rlwinm(Register rd, Register rs, uint8_t sh, uint8_t mb, uint8_t me) {
	JSPW(PPC_rlwinm | _RS << 21 | _RD << 16 | (sh) << 11 | (mb) << 6 | (me) << 1,
		"rlwinm %s,%s,%d,%d,%d", nGPR(rd), nGPR(rs), sh, mb, me);
}
BufferOffset Assembler::rlwimi(Register rd, Register rs, uint8_t sh, uint8_t mb, uint8_t me) {
	JSPW(PPC_rlwimi | _RS << 21 | _RD << 16 | (sh) << 11 | (mb) << 6 | (me) << 1,
		"rlwimi %s,%s,%d,%d,%d", nGPR(rd), nGPR(rs), sh, mb, me);
}
#define DEF_ALU2(op) \
    BufferOffset Assembler::op(Register rd, Register ra, Register rb) { \
        JSPW(PPC_##op | _RD<< 21 | _RA << 16 | _RB << 11,               \
            "%s %s,%s,%s", #op, nGPR(rd), nGPR(ra), nGPR(rb));          \
    }                                                                   \
    BufferOffset Assembler::op##_rc(Register rd, Register ra, Register rb) { \
        JSPW(PPC_##op | _RD << 21 | _RA << 16 | _RB << 11 | 1, \
            "%s. %s,%s,%s", #op, nGPR(rd), nGPR(ra), nGPR(rb));         \
    }

        DEF_ALU2(add)
        DEF_ALU2(addc)
        DEF_ALU2(adde)
        DEF_ALU2(addo)
        DEF_ALU2(subf)
        DEF_ALU2(subfc)
        DEF_ALU2(subfe)
        DEF_ALU2(subfo)
        DEF_ALU2(divw)
        DEF_ALU2(divwo)
        DEF_ALU2(divwu)
        DEF_ALU2(divwuo)
        DEF_ALU2(mullw)
        DEF_ALU2(mulhw)
        DEF_ALU2(mulhwu)
        DEF_ALU2(mullwo)
#undef DEF_ALU2
// eqv is a little weird.
BufferOffset Assembler::eqv(Register ra, Register rs, Register rb) {
	JSPW(PPC_eqv | _RS << 21 | _RA << 16 | _RB << 11, "eqv %s,%s,%s", nGPR(ra), nGPR(rs), nGPR(rb));
}
BufferOffset Assembler::eqv_rc(Register ra, Register rs, Register rb) {
	JSPW(PPC_eqv | _RS << 21 | _RA << 16 | _RB << 11 | 1, "eqv. %s,%s,%s", nGPR(ra), nGPR(rs), nGPR(rb));
}

/* Let's make the runtime help us out -- don't use addi r0,r0,... when we really
   should use li to make it clear we do mean 0. Love those mscdfr (Means Something
   Completely Different For r0) #ppcinstructions! */
#define DEF_ALUI_mscdfr(op) \
    BufferOffset Assembler::op(Register rd, Register ra, int16_t im) {                 \
        JSPW(PPC_##op | _RD << 21 | _RA << 16 | uint16_t(im),   \
            "%s %s,%s,%i (0x%x)", #op, nGPR(rd), nGPR(ra), im, im);     \
    }
        DEF_ALUI_mscdfr(addi)
        DEF_ALUI_mscdfr(addis)
#undef DEF_ALUI_mscdfr
// No r0 restriction for these, oddly.
#define DEF_ALUI(op) \
    BufferOffset Assembler::op(Register rd, Register ra, int16_t im) {                 \
        JSPW(PPC_##op | _RD << 21 | _RA << 16 | uint16_t(im),   \
            "%s %s,%s,%i (0x%x)", #op, nGPR(rd), nGPR(ra), im, im);     \
    }
        DEF_ALUI(addic)
        // mulli is usually strength-reduced, since it can take up to five
        // cycles in the worst case. See x_sr_mulli.
        DEF_ALUI(mulli)
#undef DEF_ALUI
#define DEF_ALUE(op) \
    BufferOffset Assembler::op(Register rd, Register ra) {                             \
        JSPW(PPC_##op | _RD<< 21 | _RA << 16,                  \
            "%s %s,%s", #op, nGPR(rd), nGPR(ra));                       \
    } \
    BufferOffset Assembler::op##_rc(Register rd, Register ra) {                        \
        JSPW(PPC_##op | _RD<< 21 | _RA << 16 | 1,              \
            "%s. %s,%s", #op, nGPR(rd), nGPR(ra));                      \
    }
        DEF_ALUE(addme)
        DEF_ALUE(addze)
        DEF_ALUE(subfze)
#undef DEF_ALUE
// cntlzw is "special" -- its encoded operand order is reversed, so the macro doesn't work.
BufferOffset Assembler::cntlzw(Register ra, Register rs) {
	JSPW(PPC_cntlzw | _RA<< 16 | _RS<< 21, "cntlzw %s,%s", nGPR(ra), nGPR(rs));
}
BufferOffset Assembler::cntlzw_rc(Register ra, Register rs) {
	JSPW(PPC_cntlzw | _RA<< 16 | _RS<< 21 | 1, "cntlzw %s,%s", nGPR(ra), nGPR(rs));
}
// See side effect in Assembler-ppc.h.
#define DEF_BITALU2(op) \
    BufferOffset Assembler::op(Register rd, Register rs, Register rb) {              \
        JSPW(PPC_##op | _RD<< 16 | _RS<< 21 | _RB<< 11,  \
            "%s %s,%s,%s", #op, nGPR(rd), nGPR(rs), nGPR(rb));          \
    }                                                                   \
    BufferOffset Assembler::op##_rc(Register rd, Register rs, Register rb) {         \
        JSPW(PPC_##op | _RD<< 16 | _RS<< 21 | _RB<< 11 | 1, \
            "%s. %s,%s,%s", #op, nGPR(rd), nGPR(rs), nGPR(rb));         \
    }                                                                   \

        DEF_BITALU2(andc)
        DEF_BITALU2(nand)
        DEF_BITALU2(nor)
        DEF_BITALU2(slw)
        DEF_BITALU2(srw)
        DEF_BITALU2(sraw)
        DEF_BITALU2(sld)
        DEF_BITALU2(srd)
        DEF_BITALU2(srad)
        DEF_BITALU2(and_)
        DEF_BITALU2(or_)
        DEF_BITALU2(xor_)
#undef DEF_BITALU2
#define DEF_BITALUI(op) \
    BufferOffset Assembler::op(Register rd, Register ra, uint16_t im) {                \
        JSPW(PPC_##op | _RA<< 21 | _RD << 16 | im,             \
            "%s %s,%s,%u (0x%x)", #op, nGPR(rd), nGPR(ra), im, im);     \
    }
        DEF_BITALUI(ori)
        DEF_BITALUI(oris)
        DEF_BITALUI(xori)
        DEF_BITALUI(xoris)
#undef DEF_BITALUI
// There are no Rc-free versions of andi/andis.
BufferOffset Assembler::andi_rc(Register rd, Register ra, uint16_t im) {
	JSPW(PPC_andi | _RA<< 21 | _RD<< 16 | im, "andi. %s,%s,%u (0x%x)", nGPR(rd), nGPR(ra), im, im);
}
BufferOffset Assembler::andis_rc(Register rd, Register ra, uint16_t im) {
	JSPW(PPC_andis | _RA<< 21 | _RD<< 16 | im, "andis. %s,%s,%u (0x%x)", nGPR(rd), nGPR(ra), im, im);
}
#define DEF_ALUEXT(op) \
    BufferOffset Assembler::op(Register rd, Register rs) {             \
        JSPW(PPC_##op | _RD << 16 | _RS << 21,  \
            "%s %s,%s", #op, nGPR(rd), nGPR(rs));       \
    }
        DEF_ALUEXT(extsb)
        DEF_ALUEXT(extsh)
        DEF_ALUEXT(extsw)
#undef DEF_ALUEXT

/* Another mscdfr (Means Something Completely Different For r0) #ppcinstructions
   situation, so put in a hazard assertion since we don't use addresses of this sort. */
#define DEF_MEMd(op) \
    BufferOffset Assembler::op(Register rd, Register rb, int16_t off) {                 \
    	MOZ_ASSERT(rb != r0); \
        JSPW(PPC_##op | _RD<< 21 | _RB<< 16 | uint16_t(off),   \
            "%s %s,%d(%s)", #op, nGPR(rd), off, nGPR(rb));               \
    }
        DEF_MEMd(lbz)
        DEF_MEMd(lha)
        DEF_MEMd(lhz)
        DEF_MEMd(lwz)
        DEF_MEMd(ld)

        DEF_MEMd(stb)
        DEF_MEMd(stw)
        DEF_MEMd(stwu)
        DEF_MEMd(sth)
        DEF_MEMd(std)
        DEF_MEMd(stdu)
#undef DEF_MEMd
#define DEF_MEMx(op) \
    BufferOffset Assembler::op(Register rd, Register ra, Register rb) {            \
    	MOZ_ASSERT(ra != r0); \
        JSPW(PPC_##op | _RD<< 21 | _RA<< 16 | _RB<< 11,         \
            "%s %s,%s,%s", #op, nGPR(rd), nGPR(ra), nGPR(rb));  \
    }
        DEF_MEMx(lbzx)
        DEF_MEMx(lhax)
        DEF_MEMx(lhzx)
        DEF_MEMx(lhbrx)
        DEF_MEMx(lwzx)
        DEF_MEMx(lwbrx)
        DEF_MEMx(ldx)

        DEF_MEMx(stbx)
        DEF_MEMx(stwx)
        DEF_MEMx(stwux)
        DEF_MEMx(stwbrx)
        DEF_MEMx(sthx)
        DEF_MEMx(sthbrx)
        DEF_MEMx(stdx)
        DEF_MEMx(stdux)
#undef DEF_MEMx

// FPR operations and load-stores.
BufferOffset Assembler::fcmpu(CRegisterID cr, FloatRegister ra, FloatRegister rb) {
            JSPW(PPC_fcmpu | cr << 23 | (int)ra << 16 | (int)rb << 11,
                "fcmpu %s,%s,%s", nCR(cr), nFPR(ra), nFPR(rb));
}
BufferOffset Assembler::fcmpu(FloatRegister ra, FloatRegister rb) { return fcmpu(cr0, ra, rb); }
#define DEF_FPUAC(op) \
    BufferOffset Assembler::op(FloatRegister rd, FloatRegister ra, FloatRegister rc) {        \
        JSPW(PPC_##op | (int)rd << 21 | (int)ra << 16 | (int)rc << 6,   \
            "%s %s,%s,%s", #op, nFPR(rd), nFPR(ra), nFPR(rc));          \
    }                                                                   \
    BufferOffset Assembler::op##_rc(FloatRegister rd, FloatRegister ra, FloatRegister rc) {   \
        JSPW(PPC_##op | (int)rd << 21 | (int)ra << 16 | (int)rc << 6 | 1, \
            "%s. %s,%s,%s", #op, nFPR(rd), nFPR(ra), nFPR(rc));         \
    }
        DEF_FPUAC(fmul)
        DEF_FPUAC(fmuls)
#undef DEF_FPUAC
#define DEF_FPUAB(op) \
    BufferOffset Assembler::op(FloatRegister rd, FloatRegister ra, FloatRegister rb) {        \
        JSPW(PPC_##op | (int)rd << 21 | (int)ra << 16 | (int)rb << 11,  \
            "%s %s,%s,%s", #op, nFPR(rd), nFPR(ra), nFPR(rb));          \
    }                                                                   \
    BufferOffset Assembler::op##_rc(FloatRegister rd, FloatRegister ra, FloatRegister rb) {   \
        JSPW(PPC_##op | (int)rd << 21 | (int)ra << 16 | (int)rb << 11 | 1, \
            "%s. %s,%s,%s", #op, nFPR(rd), nFPR(ra), nFPR(rb));         \
    }

        DEF_FPUAB(fadd)
        DEF_FPUAB(fdiv)
        DEF_FPUAB(fsub)
        DEF_FPUAB(fadds)
        DEF_FPUAB(fdivs)
        DEF_FPUAB(fsubs)
#undef DEF_FPUAB
#define DEF_FPUDS(op) \
    BufferOffset Assembler::op(FloatRegister rd, FloatRegister rs) {         \
        JSPW(PPC_##op | (int)rd << 21 | (int)rs << 11,  \
            "%s %s,%s", #op, nFPR(rd), nFPR(rs));       \
    } \
    BufferOffset Assembler::op##_rc(FloatRegister rd, FloatRegister rs) {         \
        JSPW(PPC_##op | (int)rd << 21 | (int)rs << 11 | 1,  \
            "%s. %s,%s", #op, nFPR(rd), nFPR(rs));       \
    }
        DEF_FPUDS(fabs)
        DEF_FPUDS(fneg)
        DEF_FPUDS(fmr)
        DEF_FPUDS(fctiw)
        DEF_FPUDS(fctiwz)
        DEF_FPUDS(frsp)
        DEF_FPUDS(frsqrte)

        // G5 only
        DEF_FPUDS(fcfid)
        DEF_FPUDS(fsqrt)
#undef DEF_FPUDS
// In Ion, the semantics for this macro are now corrected compared to JM/PPCBC.
// (See OPPCC p.432, etc.) 
#define DEF_FPUABC(op) \
    BufferOffset Assembler::op(FloatRegister rd, FloatRegister ra, FloatRegister rc, FloatRegister rb) {\
        JSPW(PPC_##op | (int)rd << 21 | (int)ra << 16 | (int)rb << 11   \
                | (int)rc << 6,                                         \
            "%s %s,%s,%s,%s",#op, nFPR(rd), nFPR(ra), nFPR(rb), nFPR(rc)); \
    } \
    BufferOffset Assembler::op##_rc(FloatRegister rd, FloatRegister ra, FloatRegister rc, FloatRegister rb) {\
        JSPW(PPC_##op | (int)rd << 21 | (int)ra << 16 | (int)rb << 11   \
                | (int)rc << 6 | 1,                                     \
            "%s. %s,%s,%s,%s",#op, nFPR(rd), nFPR(ra), nFPR(rb), nFPR(rc)); \
    }
        DEF_FPUABC(fmadd)
        DEF_FPUABC(fnmsub)
        DEF_FPUABC(fsel)
#undef DEF_FPUABC
#define DEF_FMEMd(op) \
    BufferOffset Assembler::op(FloatRegister rd, Register rb, int16_t off) {               \
        JSPW(PPC_##op | (int)rd << 21 | _RB << 16 | uint16_t(off),   \
            "%s %s,%d(%s)", #op, nFPR(rd), off, nGPR(rb));               \
    }
        DEF_FMEMd(lfd)
        DEF_FMEMd(lfs)
        DEF_FMEMd(stfd)
        DEF_FMEMd(stfs)
        DEF_FMEMd(stfdu)
#undef DEF_FMEMd
#define DEF_FMEMx(op) \
    BufferOffset Assembler::op(FloatRegister rd, Register ra, Register rb) {            \
        JSPW(PPC_##op | (int)rd << 21 | _RA << 16 | _RB << 11,  \
            "%s %s,%s,%s", #op, nFPR(rd), nGPR(ra), nGPR(rb));          \
    }
        DEF_FMEMx(lfdx)
        DEF_FMEMx(lfsx)
        DEF_FMEMx(stfdx)
        DEF_FMEMx(stfsx)
#undef DEF_FMEMx
BufferOffset Assembler::mtfsb0(uint8_t bt) { JSPW(PPC_mtfsb0 | (bt << 21), "mtfsb0 %d", bt); }
BufferOffset Assembler::mtfsb1(uint8_t bt) { JSPW(PPC_mtfsb1 | (bt << 21), "mtfsb1 %d", bt); }
BufferOffset Assembler::mtfsfi(uint8_t fi, uint8_t imm) {
	MOZ_ASSERT(fi < 8);
	MOZ_ASSERT(imm < 16);
	JSPW(PPC_mtfsfi | fi<<23 | imm<<12,
		"mtfsfi %d,%d", fi, imm);
}
BufferOffset Assembler::mcrf(CRegisterID bt, CRegisterID bs) {
            JSPW(PPC_mcrf | (int)bt<<23 | (int)bs<<18,
                "mcrf %s,%s", nCR(bt), nCR(bs));
}

BufferOffset Assembler::mcrfs(CRegisterID bf, uint8_t bfa) {
            JSPW(PPC_mcrfs | (int)bf<<23 | (bfa << 18),
                "mcrfs %s,%d", nCR(bf), bfa);
}

// Conveniences and generally accepted alternate mnemonics.
// These are not true opcodes in the strictest sense, and some may map to multiple instructions.
BufferOffset Assembler::x_trap() { JSPWS(PPC_trap, "trap"); }
BufferOffset Assembler::x_mtrap() { JSPWS(PPC_trap, "\n\n === MARKED TRAP ===\n"); }
BufferOffset Assembler::x_mr(Register rd, Register ra) { return or_(rd, ra, ra); }
BufferOffset Assembler::x_beq(CRegisterID cr, int16_t off, LikelyBit lkb, LinkBit lb) {
	return bc(BOffImm16(off), Equal, cr, lkb, lb);
}
BufferOffset Assembler::x_bne(CRegisterID cr, int16_t off, LikelyBit lkb, LinkBit lb) {
	return bc(BOffImm16(off), NotEqual, cr, lkb, lb);
}
BufferOffset Assembler::x_bdnz(int16_t off, LikelyBit lkb, LinkBit lb) {
	// Fixed BO = 16, BI = 0 (from OPPCC appendix A, page 370).
	BOffImm16 offs = BOffImm16(off);
#if _PPC970_
	// Convert likely in y-bit encoding to at-bit encoding.
	// Here, we assume backwards branches without likely bit are
	// taken, and hint the branch accordingly. If we get here with
	// an offset of zero, set no hinting, because wtf did I do.
	// TODO: Convert to conventional hinting (q.v.).
	int likely;
	if (lkb == LikelyB) {
		// The branch is likely forward.
		likely = (off < 0) ? 8 : (off > 0) ? 9 : /* ?! */ 0;
	} else {
		// The branch is likely backwards.
		likely = (off > 0) ? 8 : (off < 0) ? 9 : /* ?!?! */ 0;
	}
#else
	int likely = (int)lkb;
#endif
	int is_link = (int)lb;
    JSPW((PPC_bc | (16 | likely)<<21 | offs.encode() | is_link),
    	"bdnz%s%s %d", (is_link)?"l":"", (likely)?"+":"", offs.decode());
}
BufferOffset Assembler::x_mtctr(Register ra) { return mtspr(ctr, ra); }
BufferOffset Assembler::x_mtlr(Register ra) { return mtspr(lr_spr, ra); }
BufferOffset Assembler::x_mflr(Register rd) { return mfspr(rd, lr_spr); }
BufferOffset Assembler::x_mtcr(Register rs) { JSPW(PPC_mtcrf | _RS << 21 | 0xff << 12, "mtcr %s", nGPR(rs)); }
BufferOffset Assembler::x_insertbits0_15(Register rd, Register rs) {
	// fill bits 0-15 of rd with bits 16-31 of rs
	return rlwimi(rd, rs, 16, 0, 15);
}
BufferOffset Assembler::x_bit_value(Register rd, Register rs, unsigned bit) {
	// set rd to 1 if the given bit in rs is set, 0 otherwise
	MOZ_ASSERT(bit < 32);
	// rotate the given bit to bit 31 and then mask with 0x00000001
	if (bit < 31) {
		JSPW(PPC_rlwinm |
			_RS << 21 | _RD << 16 | (bit+1) << 11 |
				31 << 6 | 31 << 1,
			"x_bit_value %s,%s,%d", nGPR(rd), nGPR(rs), bit);
	} else {
		return andi_rc(rd, rs, 0x00000001);
	}
}
BufferOffset Assembler::x_slwi(Register rd, Register rs, int n) {
	MOZ_ASSERT(n < 32 && n > 0);
	JSPW((PPC_rlwinm | (_RS<< 21) | (_RD<< 16) | (n << 11) | (0 << 6) | ((31-n) << 1)),
		"slwi %s,%s,%d", nGPR(rd), nGPR(rs), n);
}
BufferOffset Assembler::x_srwi(Register rd, Register rs, int n) {
	MOZ_ASSERT(n < 32 && n > 0);
	JSPW((PPC_rlwinm | (_RS << 21) | (_RD << 16) | ((32-n) << 11) | (n << 6) | (31<<1)),
		"srwi %s,%s,%d", nGPR(rd), nGPR(rs), n);
}
BufferOffset Assembler::x_subi(Register rd, Register ra, int16_t im) {
	JSPW(PPC_addi | _RD << 21 | _RA << 16 | uint16_t(-im),
		"subi %s,%s,%d", nGPR(rd), nGPR(ra), im);
}
BufferOffset Assembler::x_sr_mulli(Register rd, Register ra, int16_t im) {
	// Strength-reduced immediate multiply. mulli, in the worst case,
	// can take up to five cycles (most implementations range from three
	// to four). Try to prevent this by turning into series of additions
	// without exceeding the expected cycle count.
	//
	// TODO: This is limited to adds. Explore if rlwimi + extsw can
	// still be faster in the general case or in cases of 2^x & 2^x+1.
	//
	// In the current implementation, only Baseline uses this; Ion uses
	// an expanded version in visitMulI.
#if DEBUG
	spew_with_address("#x_sr_mulli %s,%s,%d", 0, nGPR(rd), nGPR(ra), im);
#endif
	switch (im) {
		case 0:
			// Load zero.
			x_li32(rd, 0);
			break;
		case 1:
			// Load ra, unless rd == ra.
			if (rd != ra)
				x_mr(rd, ra);
			break;
		case 2:
			// rd = ra + ra
			add(rd, ra, ra);
			break;
		case 3:
			// This is on SunSpider.
			// The trick here is that rd may equal ra. If so,
			// we can't do this optimization because storing and
			// restoring the register won't save any time.
			if (rd != ra) {
				add(rd, ra, ra);
				add(rd, rd, ra);
			} else {
				// Oh well.
				mulli(rd, ra, im);
			}
			break;
		case 4:
			// Shift left
			x_slwi(rd, ra, 2);
			break;
		case 5:
			// Same limitation as 3.
			if (rd != ra) {
				x_slwi(rd, ra, 2);
				add(rd, rd, ra);
			} else {
				// Oh well.
				mulli(rd, ra, im);
			}
			break;
		case 8:
			// Shift left
			x_slwi(rd, ra, 3);
			break;
		case 10:
			// *5 then *2
			if (rd != ra) {
				x_slwi(rd, ra, 2);
				add(rd, rd, ra);
				add(rd, rd, rd);
			} else {
				// Oh well.
				mulli(rd, ra, im);
			}
			break;
		case 12:
			// *3 then *4
			if (rd != ra) {
				add(rd, ra, ra);
				add(rd, rd, ra);
				x_slwi(rd, rd, 2);
			} else {
				// Oh well.
				mulli(rd, ra, im);
			}
			break;
		case 16:
			// Shift left
			x_slwi(rd, ra, 4);
			break;
		
		default:
			// Oh well.
			mulli(rd, ra, im);
			break;
	}
	return BufferOffset();
}

// Large loads.
BufferOffset Assembler::x_li(Register rd, int16_t im) {
	JSPW(PPC_addi | _RD << 21 | uint16_t(im), "li %s,%i (0x%x)", nGPR(rd), im, im);
}
BufferOffset Assembler::x_lis(Register rd, int16_t im) {
	JSPW(PPC_addis | _RD << 21 | uint16_t(im), "lis %s,%d (0x%x)", nGPR(rd), im, im << 16);
}
BufferOffset Assembler::x_p_li32(Register rd, int32_t im) {
	x_lis(rd, uint16_t(im >> 16));
	return ori(rd, rd, uint16_t(im & 0xffff));
}
BufferOffset Assembler::x_li32(Register rd, int32_t im) {
	if ((im & 0xffff8000) == 0 || (im & 0xffff8000) == 0xffff8000) {
		// fits in an int16_t: li
		return x_li(rd, uint16_t(im & 0xffff));
	} else if (!(im & 0xffff)) {
		// doesn't have a lower 16 bits: only lis
		return x_lis(rd, uint16_t(im >> 16));
	} else {
		// 32 bits: lis for the upper 16 bits, then ori the lower
		return x_p_li32(rd, im);
	}
}

void
Assembler::bind(Label *label, BufferOffset boff)
{
    // If our caller didn't give us an explicit target to bind to
    // then we want to bind to the location of the next instruction
    BufferOffset dest = boff.assigned() ? boff : nextOffset();
    if (label->used()) {
        int32_t next;

        // A used label holds a link to branch that uses it.
        BufferOffset b(label);
        do {
            if (MOZ_UNLIKELY(oom())) return;

            Instruction *inst = editSrc(b);
            inst->assertOpcode(PPC_trap);

            // Second word holds a pointer to the next branch in the label's chain.
            next = inst[1].encode();
            bind(reinterpret_cast<InstImm *>(inst), b.getOffset(), dest.getOffset());

            b = BufferOffset(next);
        } while (next != LabelBase::INVALID_OFFSET);
    }
    label->bind(dest.getOffset());
}

/*

 Branches could take several forms.
 
 Unconditional branches are, when short (generated by MacroAssembler's b):
 
 [unpatched]
 trap
 (offset word)
 nop
 b(l) .+4
 
 [patched]
 nop
 nop
 nop
 b(l) target
 
 Long,
 
 lis/ori target
 mtctr
 bctr(l)
 
 Conditional branches (MA's bc), however, have an extra word for a diversion. When short,
 
 [unpatched]
 trap
 (offset word)
 nop
 nop
 bc(l) .+4
 
 [patched]
 nop
 nop
 nop
 nop
 bc(l) target
 
 When long, however, we invert the sense to divert around the branch if false.
 
 bcinverted .+20
 lis/ori target
 mtctr
 bctr(l)
 
 ::bind() has to handle all of these.
 
 Only backedges lead with an unconditional b, which is easy to test for.
 
 */
 
void
Assembler::bind(InstImm *inst, uint32_t branch, uint32_t target)
{
    int32_t offset = target - branch;
    int32_t moffset = offset;
    uint32_t stanzend = 0;

#if DEBUG
	// Determine length of stanza. Assume nothing about our input.
	JitSpew(JitSpew_Codegen, "# ::bind evaluating %08x", branch);
	MOZ_ASSERT(!(offset & 0x03));
#endif

	if (inst[0].encode() == PPC_trap || inst[0].encode() == PPC_nop) {
		// Short stanza, could be patched or unpatched. Count nops. Be explicit.
		// We need the computed stanza length for the speculative offset below.
		if (inst[0].encode() == PPC_trap)
			stanzend = 2; // Skip trap word and offset word.
		while (inst[stanzend].encode() == PPC_nop && stanzend < 6) { stanzend++; }
                if (MOZ_UNLIKELY(oom())) return; // we're generating incorrect code anyway
		MOZ_ASSERT(stanzend == 3 || stanzend == 4);
		// Fall through.
	} else if (inst[0].isOpcode(PPC_addis)) {
		// Long unconditional branch. Sanity check, then patch in place.
#if DEBUG
		JitSpew(JitSpew_Codegen, "# ::bind long unconditional %08x -> %08x offset %i %s",
			branch, target, offset, (BOffImm16::IsInSignedRange(offset)) ? "SHORT" : "");
		inst[3].assertOpcode(PPC_bctr); // This works for bctrl also.
#endif
		addLongJump(BufferOffset(branch));
		UpdateLisOriValue(inst, &inst[1], target);
		return;
	} else if (inst[0].isOpcode(PPC_bc)) {
		// Long conditional branch. Sanity check, then patch in place as well.
#if DEBUG
		JitSpew(JitSpew_Codegen, "# ::bind long conditional %08x -> %08x offset %i %s",
			branch, target, offset, (BOffImm16::IsInSignedRange(offset)) ? "SHORT" : "");
		inst[4].assertOpcode(PPC_bctr); // This works for bctrl also.
#endif
                // Make sure the long jump is the lis/ori, not the bc.
		addLongJump(BufferOffset(branch+4));
		UpdateLisOriValue(&inst[1], &inst[2], target);
		return;
	} else {
		// Binding a backedge here? That should have been handled below.
		MOZ_CRASH("unexpected branch type");
	}
	
	// Fallen through; we must have a short branch at this point. Since we know we will
	// patch to a known location, write in the nops now if this is an unpatched branch;
	// bind() above already got the information it needed.
	if (inst[0].encode() == PPC_trap) {
		inst[0].makeOp_nop();
		inst[1].makeOp_nop();
	}
	
	// Now we're in the situation of patching a short branch that can stay short,
	// or having to lengthen it. TODO: patch back to a short branch? See debugging above.
	moffset -= stanzend*4; // Speculative offset. Walk back by length of stanza.
		
	if (BOffImm16::IsInSignedRange(moffset)) {
		BOffImm16 bimm = BOffImm16(moffset);
		
		// This is always short. Preserve the link bit and patch the immediate word,
		// preserving BI, BO and CR, if any. This is always relative.
#if DEBUG
		JitSpew(JitSpew_Codegen, "# ::bind short jump %i %08x -> %08x offset %i",
			stanzend, branch, target, offset);
#endif
		LinkBit lb = (LinkBit)(inst[stanzend].encode() & LinkB);
		if (inst[stanzend].isOpcode(PPC_b)) {
			// Rebuild the opcode, because we need to ensure the upper bits are clear.
			inst[stanzend].setData(PPC_b | bimm.encode() | lb);
		} else {
			inst[stanzend].assertOpcode(PPC_bc);
			inst[stanzend].setImm16(Imm16(bimm.encode() | lb));
		}
		return;
	}
	
	// Possibly must patch long.
	
	if (stanzend == 3) {
		// Unconditional. Keep the link bit, though.
#if DEBUG
		JitSpew(JitSpew_Codegen, "# ::bind new long unconditional %08x -> %08x offset %i",
			branch, target, offset);
		inst[stanzend].assertOpcode(PPC_b);
#endif

		LinkBit lb = (LinkBit)(inst[stanzend].encode() & LinkB);
		// We have one option to consider. If the branch can be wangled into 26 bits, then
		// we don't have to go long. TODO: While it would be nice to allow absolute quantities,
		// that makes relocation tricky.
		if (JOffImm26::IsInRange(moffset)) {
			JOffImm26 jimm = JOffImm26(moffset);
			
			inst[stanzend].setData(PPC_b | jimm.encode() | lb);
			return;
		} /* else if (JOffImm26::IsInRange(target)) {
			JOffImm26(target) timm;
			
			inst[stanzend].setData(PPC_b | timm.encode() | AbsoluteBranch | lb);
			return;
		} */
		// Nope, go long.
		addLongJump(BufferOffset(branch));
		WriteLisOriInstructions(inst, &inst[1], r0, target);
		inst[2].makeOp_mtctr(r0);
		inst[3].makeOp_bctr(lb);
		return;
	}
	
	// Conditional. This is the tricky one. We need to invert the sense of the branch (but
	// keep its CR and BI), while moving its link bit, if any, to the bctr at the end.
	// We also need to be mindful of any optimizations on it, too.

#if DEBUG
	JitSpew(JitSpew_Codegen, "# ::bind new long conditional %08x -> %08x offset %i",
		branch, target, offset);
	inst[stanzend].assertOpcode(PPC_bc);
#endif
        // Make sure the long jump is the lis/ori, not the new bc.
	addLongJump(BufferOffset(branch+4));

        // The new branch is always forward, so the likely bit needs to
        // be twiddled. (Remember: the bit means invert normal behaviour.)
        // Backwards old branch had Likely: set NotLikely on new branch
        // Forwards old branch had Likely: set Likely
        // Backwards old branch had NotLikely: set Likely
        // Forwards old branch had NotLikely: set NotLikely
        LikelyBit lkb = (LikelyBit)(inst[stanzend].encode() & LikelyB);
        // TODO: Fix this for G5. For now, do NOT flip the likely t-bit on G5
        // (it does nothing if the a-bit isn't set, and it's wrong if it is).
#ifndef _PPC970_
        if (lkb == LikelyB)
            lkb = (moffset < 0) ? NotLikelyB : LikelyB;
        else
            lkb = (moffset < 0) ? LikelyB : NotLikelyB;
#endif

	// Flip the sense bits and rewrite the branch.
	LinkBit lb = (LinkBit)(inst[stanzend].encode() & LinkB);
	inst[0].setData(((inst[stanzend].encode() & 0xFFFF0000) ^ 0x01000000) | 20 | ((int)lkb << 21)); // bc (inverted) .+20
	WriteLisOriInstructions(&inst[1], &inst[2], r0, target);
	inst[3].makeOp_mtctr(r0);
	inst[4].makeOp_bctr(lb);
	return;
}

void
Assembler::bind(RepatchLabel *label)
{
    BufferOffset dest = nextOffset();
    if (label->used() && MOZ_LIKELY(!oom())) {
        // If the label has a use, then change this use to refer to
        // the bound label;
        BufferOffset b(label->offset());
        InstImm *inst1 = (InstImm *)editSrc(b);
#if DEBUG
		JitSpew(JitSpew_Codegen,"# ::bind repatch %08x %08x\n", (uint32_t)inst1, label->offset());
#endif

        // If first instruction is a branch, then this must be a loop backedge.
        if (inst1->isOpcode(PPC_b)) {
        	MOZ_ASSERT(!(inst1->encode() & LinkB));
            // Backedges are short jumps when bound, but can become long
            // when patched.
            uint32_t offset = dest.getOffset() - label->offset();
            JOffImm26 offs = JOffImm26(offset);
            inst1->setData(PPC_b | offs.encode());
            AutoFlushICache::flush(uintptr_t(inst1), 4);
        } else {
            UpdateLisOriValue(inst1, inst1->next(), dest.getOffset());
            AutoFlushICache::flush(uintptr_t(inst1), 8);
        }
    }
    label->bind(dest.getOffset());
}

void
Assembler::retarget(Label *label, Label *target)
{
    if (label->used() && MOZ_LIKELY(!oom())) {
        if (target->bound()) {
            bind(label, BufferOffset(target));
        } else if (target->used()) {
            // The target is not bound but used. Prepend label's branch list
            // onto target's.
            int32_t next;
            BufferOffset labelBranchOffset(label);

            // Find the head of the use chain for label.
            do {
                Instruction *inst = editSrc(labelBranchOffset);
                inst->assertOpcode(PPC_trap);

                // Second word holds a pointer to the next branch in chain.
                next = inst[1].encode();
                labelBranchOffset = BufferOffset(next);
            } while (next != LabelBase::INVALID_OFFSET);

            // Then patch the head of label's use chain to the tail of the
            // target's use chain, prepending the entire use chain of the target.
            Instruction *inst = editSrc(labelBranchOffset);
            int32_t prev = target->use(label->offset());
            inst[1].setData(prev);
        } else {
            // The target is unbound and unused.  We can just take the head of
            // the list hanging off of label, and dump that into target.
            DebugOnly<uint32_t> prev = target->use(label->offset());
            MOZ_ASSERT((int32_t)prev == Label::INVALID_OFFSET);
        }
    }
    label->reset();
}

void dbg_break() {}

BufferOffset
Assembler::haltingAlign(int alignment)
{
	// This is used in the CodeGenerator to force a bomb if code is unaligned.
	// For our purposes, a sufficient number of trap words will work.
	return align(alignment, true);
}

BufferOffset
Assembler::nopAlign(int alignment)
{
	// Basically the old ::align.
	return align(alignment);
}

uint32_t
Assembler::PatchWrite_NearCallSize()
{
    return 4 * sizeof(uint32_t);
}

void
Assembler::PatchWrite_NearCall(CodeLocationLabel start, CodeLocationLabel toCall)
{
	// Normally we don't use the link bit. This is only used by the code generator to
	// call the invalidation epilogue, which gets the OsiPoint from LR and assumes there is
	// actually no return address on the stack.
    Instruction *inst = (Instruction *) start.raw();
    uint8_t *dest = toCall.raw();

#if DEBUG
        JitSpew(JitSpew_Codegen,
                "## PatchWrite_NearCall: %p to %p\n",
                        start.raw(), toCall.raw());
#endif

    // Overwrite whatever instruction used to be here with a call.
    // Always use a long stanza:
    // - The jump must be the same size because of PatchWrite_NearCallSize.
    // - The return address must be at the end of the block.
    // A short stanza wouldn't be any more efficient, even if the offset fit.
    WriteLisOriInstructions(inst, &inst[1], r0, (uint32_t)dest);
    inst[2] = Instruction(PPC_mtspr | PPC_SPR(ctr) << 11); // mtctr r0
    inst[3] = Instruction(PPC_bctr | LinkB); // bctrl
    
    // Ensure everyone sees the code that was just written into memory.
    AutoFlushICache::flush(uintptr_t(inst), PatchWrite_NearCallSize());
}

uint32_t
Assembler::ExtractLisOriValue(Instruction *inst0, Instruction *inst1)
{
    InstImm *i0 = (InstImm *) inst0;
    InstImm *i1 = (InstImm *) inst1;
    i0->assertOpcode(PPC_addis);
    i1->assertOpcode(PPC_ori);

    uint32_t value = i0->extractImm16Value() << 16;
    value = value | i1->extractImm16Value();
    return value;
}

// WARNING: Does not flush icache itself. You need to do so after calling!
void
Assembler::UpdateLisOriValue(Instruction *inst0, Instruction *inst1, uint32_t value)
{
	inst0->assertOpcode(PPC_addis);
	inst1->assertOpcode(PPC_ori);

    ((InstImm *) inst0)->setImm16(Imm16::Upper(Imm32(value)));
    ((InstImm *) inst1)->setImm16(Imm16::Lower(Imm32(value)));
}

void
Assembler::WriteLisOriInstructions(Instruction *inst0, Instruction *inst1,
                                   Register reg, uint32_t value)
{
    *inst0 = InstImm(PPC_addis);
    ((InstImm *) inst0)->setUpperReg(reg); // Lower reg defaults to zero, making this lis.
    ((InstImm *) inst0)->setImm16(Imm16::Upper(Imm32(value)));
    *inst1 = InstImm(PPC_ori);
    ((InstImm *) inst1)->setUpperReg(reg);
    ((InstImm *) inst1)->setLowerReg(reg);
    ((InstImm *) inst1)->setImm16(Imm16::Lower(Imm32(value)));
}

void
Assembler::PatchDataWithValueCheck(CodeLocationLabel label, PatchedImmPtr newValue,
                                   PatchedImmPtr expectedValue)
{
    Instruction *inst = (Instruction *) label.raw();

    // Extract and verify old value.
    DebugOnly<uint32_t> value = Assembler::ExtractLisOriValue(&inst[0], &inst[1]);
    MOZ_ASSERT(value == uint32_t(expectedValue.value));

    // Replace with new value.
    Assembler::UpdateLisOriValue(inst, inst->next(), uint32_t(newValue.value));

    AutoFlushICache::flush(uintptr_t(inst), 8);
}

void
Assembler::PatchDataWithValueCheck(CodeLocationLabel label, ImmPtr newValue, ImmPtr expectedValue)
{
    PatchDataWithValueCheck(label, PatchedImmPtr(newValue.value),
                            PatchedImmPtr(expectedValue.value));
}

// This just stomps over memory with 32 bits of raw data. Its purpose is to
// overwrite the call of JITed code with a 32-bit offset. This will
// only ever run on code that was just invalidated and thrown away.
// Since that instruction will never be executed again, no cache flush is required.
void
Assembler::PatchWrite_Imm32(CodeLocationLabel label, Imm32 imm)
{
    // This is the pointer to where it will go.
    uint32_t *raw = (uint32_t*)label.raw();
    // Overwrite the 4 bytes before the return address, which will
    // end up being the call instruction.
    *(raw - 1) = imm.value;
#if DEBUG
    JitSpew(JitSpew_Codegen, "## PatchWrite_Imm32(0x%08x -> %08x)",
        (uint32_t)(raw - 1), (uint32_t)imm.value);
#endif 
}

void
Assembler::PatchInstructionImmediate(uint8_t *code, PatchedImmPtr imm)
{
    InstImm *inst = (InstImm *)code;
    Assembler::UpdateLisOriValue(inst, inst->next(), (uint32_t)imm.value);
}

uint8_t *
Assembler::NextInstruction(uint8_t *inst_, uint32_t *count)
{
    Instruction *inst = reinterpret_cast<Instruction*>(inst_);
    if (count != nullptr)
        *count += sizeof(Instruction);
    return reinterpret_cast<uint8_t*>(inst->next());
}

// Since there are no pools, this should be simple.
Instruction *
Instruction::next()
{
    return this + 1;
}

// These work on a toggledJump(). Because we don't know what the label will be patched
// to, assume the jump is always one stanza long for the branch-around.
void
Assembler::ToggleToCmp(CodeLocationLabel inst_)
{
	// Set the nop to b, disabling the toggled jump.
    InstImm * inst = (InstImm *)inst_.raw();
#if DEBUG
    JitSpew(JitSpew_Codegen, "## ToggleToCmp(0x%08x)", (uintptr_t)inst);
#endif
    inst->assertOpcode(PPC_nop);
    // XXX: This should use PPC_B_STANZA_LENGTH
    inst->setData(PPC_b | 0x14); // Skip over the branch.

    AutoFlushICache::flush(uintptr_t(inst), 4);
}
void
Assembler::ToggleToJmp(CodeLocationLabel inst_)
{
	// Set the b to nop, enabling the toggled jump.
    InstImm * inst = (InstImm *)inst_.raw();
#if DEBUG
    JitSpew(JitSpew_Codegen, "## ToggleToJmp(0x%08x)", (uintptr_t)inst);
#endif
    inst->assertOpcode(PPC_b);
    inst->makeOp_nop();

    AutoFlushICache::flush(uintptr_t(inst), 4);
}

// This works on a toggledCall(). This is always long, so we keep the lis/ori
// and patch the second half to nop nop if disabled and mtctr bctrl if enabled.
// This is only used for generateDebugTrapHandler, so it's okay to set Lk,
// because we know it will handle LR correctly (it's our code).
void
Assembler::ToggleCall(CodeLocationLabel inst_, bool enabled)
{
    Instruction *inst = (Instruction *)inst_.raw();
#if DEBUG
    JitSpew(JitSpew_Codegen, "## ToggleCall(0x%08x)", (uintptr_t)inst);
#endif
    InstImm *i0 = (InstImm *) inst;
    InstImm *i1 = (InstImm *) i0->next();
    Instruction *i2 = (Instruction *) i1->next();
    Instruction *i3 = (Instruction *) i2->next();

    i0->assertOpcode(PPC_addis);
    i1->assertOpcode(PPC_ori);
    if (enabled) {
    	//i2->assertOpcode(PPC_nop);
    	//i3->assertOpcode(PPC_nop);
    	
    	i2->makeOp_mtctr(r0);
    	i3->makeOp_bctr(LinkB); // On purpose. generateDebugTrapHandler blr()s back.
    } else {
    	//i2->assertOpcode(PPC_mtspr);
    	//i3->assertOpcode(PPC_bctr);
    	
        i2->makeOp_nop();
        i3->makeOp_nop();
    }

    AutoFlushICache::flush(uintptr_t(i2), 8);
}

void Assembler::UpdateBoundsCheck(uint32_t heapSize, Instruction *inst)
{
    InstImm *i0 = (InstImm *) inst;
    InstImm *i1 = (InstImm *) i0->next();

    // Replace with new value
    Assembler::UpdateLisOriValue(i0, i1, heapSize);
}

