/*
 * Copyright (c) 2012 SURFnet
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
 * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER
 * IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
 * IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*****************************************************************************
 SymmetricAlgorithmTests.cpp

 Contains test cases for symmetrical algorithms (i.e., AES and DES)
 *****************************************************************************/

#include <config.h>
#include <stdlib.h>
#include <string.h>
#include <climits>
#include <array>
//#include <iomanip>
#include "SymmetricAlgorithmTests.h"

// CKA_TOKEN
const CK_BBOOL IN_SESSION = CK_FALSE;

// CKA_PRIVATE
const CK_BBOOL IS_PUBLIC = CK_FALSE;

#define NR_OF_BLOCKS_IN_TEST 0x10001


CPPUNIT_TEST_SUITE_REGISTRATION(SymmetricAlgorithmTests);

// the next instances of WrappedMaterial are used for checking ckm_xxx_cbc_pad
// with keys wrapped under another PKCS#11 token


#ifndef WITH_FIPS

WrappedMaterial rsa2048underdes56
{
	"RSA2048 wrapped under DES56 using CKM_DES_CBC_PAD",
	CKK_DES,		// wrapping key type
	CKM_DES_CBC_PAD,	// mechanism
	CKO_PRIVATE_KEY,	// wrapped object class
	CKK_RSA,		// wrapped key type
	{
		{		// key
			0xdf, 0xba, 0xd0, 0xdf, 0x91, 0xda, 0x3e, 0xf1,
		},
		{		// IV
			0x48, 0x66, 0xc8, 0x2c, 0xe1, 0x40, 0x52, 0xa9, 
		},
		{		// wrapped RSA key
			0x74, 0x55, 0xe1, 0x60, 0x4e, 0x73, 0x91, 0x7f, 0x2c, 0xd9, 0x8a, 0xb7, 0x0c, 0x0e, 0xc6, 0xa7,
			0x15, 0x3f, 0x9d, 0x72, 0x26, 0x7f, 0xb2, 0x02, 0x49, 0xe3, 0x43, 0x1f, 0x14, 0xba, 0xdb, 0xf0, 
			0xbb, 0x12, 0x66, 0x68, 0xba, 0xc3, 0x13, 0x4a, 0x5f, 0x6c, 0x50, 0x5c, 0x8c, 0xf6, 0x2e, 0xe6, 
			0xa1, 0xc6, 0x27, 0xc9, 0x6a, 0x5c, 0x80, 0xf6, 0x4c, 0x33, 0x2f, 0x5e, 0x00, 0xba, 0x94, 0x9a, 
			0xa4, 0xbf, 0x92, 0x7b, 0xb3, 0x47, 0x33, 0x6a, 0x6e, 0xdd, 0xb2, 0xfa, 0x10, 0x66, 0x92, 0x1c, 
			0x1d, 0x41, 0xaa, 0x50, 0x5f, 0x6a, 0xe6, 0xe1, 0x2a, 0x17, 0x69, 0x63, 0x92, 0x41, 0x34, 0xeb, 
			0x83, 0xfe, 0x08, 0x7c, 0x3a, 0x90, 0xe3, 0xdf, 0xd2, 0x5f, 0xfc, 0x3c, 0x25, 0x44, 0xd6, 0x11, 
			0x89, 0xa5, 0x33, 0x7f, 0xf5, 0x5f, 0xe8, 0xc9, 0xdf, 0x21, 0x6c, 0xc7, 0xe1, 0x01, 0xc7, 0x7b, 
			0x9e, 0xbc, 0x64, 0xe4, 0x72, 0x85, 0x2b, 0x04, 0x56, 0x7a, 0x72, 0xfa, 0x15, 0x65, 0x4f, 0xa9, 
			0xac, 0x96, 0x94, 0x5c, 0x66, 0x34, 0xec, 0x44, 0xc3, 0x68, 0x64, 0xa5, 0x6c, 0x6c, 0xeb, 0x61, 
			0x6d, 0x36, 0xf5, 0x47, 0x89, 0x64, 0xb9, 0x63, 0xaf, 0x2b, 0x61, 0xfe, 0x30, 0xb3, 0x09, 0x1b, 
			0x02, 0x70, 0x1f, 0x5e, 0xab, 0xaf, 0x97, 0xa9, 0xf5, 0x44, 0xc4, 0x4d, 0x1b, 0x23, 0x4d, 0xfc, 
			0xb3, 0x8c, 0xf4, 0xe5, 0x20, 0xc3, 0x94, 0xcb, 0x85, 0x1f, 0x48, 0xb1, 0x2b, 0xbd, 0xfd, 0x04, 
			0xec, 0x97, 0x1e, 0xd5, 0xcb, 0x43, 0x16, 0x32, 0xe3, 0xbf, 0x54, 0x38, 0x2b, 0xb4, 0x6f, 0x2b, 
			0x70, 0xa3, 0x13, 0x2c, 0xe0, 0xae, 0x41, 0x6c, 0x58, 0x5d, 0x04, 0x1d, 0xf9, 0x50, 0x62, 0x48, 
			0x1d, 0x95, 0xc8, 0x8e, 0x8d, 0xa1, 0xd5, 0xac, 0x34, 0xfe, 0x69, 0x00, 0x89, 0xef, 0x53, 0x58, 
			0x8e, 0x3b, 0xa4, 0xa7, 0x69, 0xbc, 0xb3, 0x9d, 0x2d, 0x83, 0x18, 0xed, 0xe2, 0xbd, 0x72, 0x03, 
			0xda, 0x70, 0x9c, 0x53, 0x7e, 0x2a, 0x5f, 0x63, 0x6f, 0xe5, 0xd1, 0x4e, 0xf7, 0x23, 0x89, 0xc2, 
			0x2d, 0x17, 0x5e, 0x98, 0xf8, 0xc0, 0x58, 0x26, 0xee, 0xe0, 0x7f, 0x2f, 0xd0, 0x0c, 0xb2, 0x50, 
			0xa1, 0xd1, 0x0e, 0x8e, 0x97, 0x8e, 0x72, 0x79, 0x18, 0x64, 0x05, 0xde, 0x14, 0x51, 0x30, 0x37, 
			0xf3, 0x71, 0x34, 0x3d, 0x81, 0x94, 0xe3, 0xef, 0x55, 0x54, 0xf6, 0xd6, 0x5c, 0x9b, 0x57, 0x45, 
			0xe8, 0x4c, 0xcf, 0x94, 0x42, 0x10, 0xef, 0xdf, 0x9c, 0x47, 0xc5, 0x9f, 0xcf, 0x50, 0xd4, 0xc1, 
			0x46, 0xee, 0x13, 0xfd, 0xa5, 0x7a, 0x4c, 0xea, 0xcd, 0xdd, 0x34, 0x31, 0xc0, 0xfc, 0x9e, 0xcc, 
			0x57, 0x1d, 0x23, 0xf9, 0x70, 0x75, 0xb9, 0x79, 0x8c, 0xd1, 0xe0, 0x5d, 0x41, 0x2b, 0xcd, 0x5c, 
			0xc9, 0x7c, 0xb8, 0x7c, 0x59, 0x6b, 0xfe, 0x97, 0xbc, 0x43, 0x56, 0x9b, 0xcd, 0x68, 0x76, 0x71, 
			0xc1, 0x0a, 0x74, 0xfa, 0x79, 0x71, 0x92, 0xb3, 0xd0, 0x16, 0xe9, 0xad, 0x5f, 0x76, 0xfc, 0x9a, 
			0x2a, 0x8b, 0x52, 0xf3, 0xa1, 0x6b, 0xa6, 0xdb, 0xdd, 0x06, 0x47, 0xae, 0x39, 0x3a, 0xd4, 0x26, 
			0x4e, 0x3f, 0x5e, 0x12, 0xd1, 0x7c, 0x01, 0x9e, 0x0c, 0x52, 0x92, 0x44, 0xd2, 0xc5, 0x18, 0x0c, 
			0xc4, 0xab, 0x8b, 0x08, 0xeb, 0xbc, 0xfe, 0x7b, 0x31, 0xe1, 0x8b, 0x35, 0x0d, 0xf3, 0xae, 0x47, 
			0x1b, 0xc7, 0x90, 0xb6, 0xdf, 0xfe, 0xa7, 0x89, 0x36, 0xcf, 0xf9, 0x64, 0x7c, 0x01, 0x7e, 0xdf, 
			0x09, 0xa9, 0xf9, 0xdf, 0x19, 0xb2, 0xc2, 0xb1, 0x60, 0x08, 0xa5, 0x81, 0xf4, 0xbe, 0x2d, 0xea, 
			0x03, 0x37, 0x10, 0xfe, 0x00, 0xe8, 0x1e, 0xed, 0xfc, 0xaa, 0x4d, 0x61, 0x6a, 0xd5, 0x80, 0x4a, 
			0xec, 0x75, 0x20, 0x50, 0x56, 0x9f, 0x1f, 0x00, 0xd3, 0x20, 0x7a, 0x74, 0x29, 0x02, 0xa3, 0xb6, 
			0x3c, 0xc2, 0xd7, 0x71, 0x39, 0xa4, 0x5b, 0x82, 0x60, 0xf5, 0x1d, 0xa2, 0xea, 0x29, 0x42, 0xc8, 
			0x49, 0x78, 0x5e, 0xc5, 0xb2, 0x28, 0x97, 0x59, 0x04, 0x8a, 0x2e, 0x1e, 0x27, 0xe4, 0x1d, 0x59, 
			0x89, 0x7d, 0xf2, 0x47, 0xad, 0xa7, 0x90, 0xd3, 0xff, 0xe0, 0x2d, 0xe0, 0x05, 0xf5, 0x7b, 0x38, 
			0x4d, 0x65, 0xc0, 0x5f, 0x30, 0x6b, 0x06, 0x97, 0x85, 0xfd, 0x7f, 0x73, 0xce, 0x02, 0x7c, 0x66, 
			0x6f, 0xcb, 0x90, 0x7a, 0x2c, 0x83, 0x8a, 0x3a, 0x0b, 0xd4, 0x33, 0xb0, 0xe2, 0x6c, 0x12, 0xc5, 
			0x57, 0x53, 0xa1, 0x8b, 0x73, 0xb6, 0x18, 0xa5, 0x0c, 0xf7, 0x8c, 0xb2, 0xde, 0x34, 0xdc, 0x63, 
			0x73, 0x0d, 0x53, 0x95, 0x8c, 0x22, 0x58, 0xc1, 0x63, 0xbb, 0xa6, 0x3a, 0x49, 0x34, 0xd1, 0x1d, 
			0x6b, 0x17, 0x37, 0xe3, 0x7b, 0x7d, 0x8c, 0xa4, 0xef, 0xd8, 0x47, 0x71, 0x3d, 0x4b, 0xc8, 0x55, 
			0xd4, 0x6a, 0xc5, 0xc9, 0xfb, 0xd7, 0x83, 0xf5, 0xda, 0x8f, 0xa1, 0x7c, 0x0a, 0xa4, 0x4b, 0xa3, 
			0x6b, 0xa3, 0x32, 0xd3, 0x08, 0x45, 0x28, 0xf9, 0x9d, 0x66, 0x21, 0x98, 0x3d, 0xd8, 0x0c, 0x00, 
			0xbe, 0xeb, 0x10, 0xd4, 0x55, 0x0b, 0xba, 0x43, 0x0f, 0xa9, 0x1a, 0x10, 0xa5, 0x48, 0x2e, 0xf3, 
			0x7f, 0xcf, 0x72, 0x86, 0x16, 0x9d, 0xb0, 0x20, 0x40, 0x84, 0x28, 0x88, 0x81, 0x19, 0x2b, 0x48, 
			0xeb, 0x31, 0x0e, 0xd3, 0x2f, 0x80, 0xbf, 0xc7, 0xf1, 0xae, 0x67, 0xac, 0x7c, 0x1f, 0x85, 0x47, 
			0xf6, 0x10, 0x64, 0x72, 0x46, 0x22, 0xad, 0xeb, 0xc9, 0x3f, 0x2b, 0xc7, 0x67, 0x64, 0x03, 0x9a, 
			0x3a, 0x8e, 0x2e, 0xd9, 0x30, 0x94, 0x98, 0x7a, 0xd1, 0xb2, 0x8e, 0x17, 0x83, 0x4b, 0x67, 0x1b, 
			0x86, 0x37, 0x67, 0x5f, 0x22, 0x32, 0xdf, 0x9e, 0xe2, 0xd7, 0xae, 0x2e, 0xec, 0x94, 0x3a, 0x87, 
			0x1f, 0xed, 0x7e, 0x33, 0xae, 0xdd, 0x4d, 0xae, 0x15, 0x83, 0x98, 0xa7, 0xdc, 0x0f, 0x96, 0x8c, 
			0x6b, 0xfb, 0xe7, 0x58, 0x8a, 0x92, 0xe6, 0x8a, 0x4c, 0xbd, 0xfd, 0x1b, 0xc9, 0xc8, 0x60, 0x99, 
			0xf5, 0x56, 0xd8, 0x3f, 0x9b, 0x63, 0xb9, 0xc2, 0x0e, 0x22, 0xbf, 0xb1, 0xef, 0x5a, 0xb6, 0x4f, 
			0x02, 0xdb, 0x19, 0xbf, 0x41, 0xe6, 0xad, 0x9e, 0x0b, 0x9b, 0x98, 0x0d, 0xad, 0xb6, 0x98, 0xcc, 
			0xff, 0x97, 0xe0, 0xb1, 0xcd, 0x96, 0xf7, 0x73, 0x9f, 0xb7, 0xf4, 0x66, 0x4f, 0xd4, 0x9f, 0xf9, 
			0x41, 0x20, 0x69, 0x10, 0xbb, 0x6c, 0xe9, 0xe0, 0xfb, 0x1c, 0x9b, 0xcd, 0xfd, 0x84, 0x65, 0xc3, 
			0x75, 0x1c, 0x2c, 0x6d, 0xbf, 0x46, 0x4a, 0x23, 0x77, 0xc3, 0xb0, 0x38, 0x54, 0x8c, 0x12, 0xb3, 
			0x99, 0x67, 0x53, 0x94, 0x3f, 0x25, 0x4d, 0x16, 0x38, 0xb5, 0xc0, 0xaa, 0x63, 0xd1, 0xe0, 0x5a, 
			0x77, 0xce, 0x51, 0xdb, 0x21, 0x96, 0x65, 0x95, 0x03, 0x87, 0x86, 0xb5, 0xcd, 0xf6, 0xd5, 0x3e, 
			0x92, 0x05, 0x04, 0xca, 0x0a, 0x23, 0xf9, 0x5c, 0xe9, 0xc7, 0x58, 0x21, 0x1d, 0x08, 0x84, 0x22, 
			0x79, 0x6f, 0xb7, 0x79, 0x16, 0xba, 0x69, 0x2d, 0x14, 0x8d, 0xc9, 0xc9, 0xb5, 0x72, 0xba, 0x90, 
			0x1c, 0x0b, 0x81, 0x71, 0x51, 0x94, 0x17, 0x00, 0x0b, 0x2a, 0x17, 0xa2, 0x46, 0x82, 0x59, 0xb0, 
			0x53, 0x5b, 0x9a, 0x57, 0xf2, 0x5a, 0x8f, 0x43, 0x89, 0x68, 0x30, 0xb0, 0x86, 0x6f, 0x1c, 0x84, 
			0x24, 0x32, 0x09, 0xf2, 0xf1, 0xcd, 0x2a, 0xc9, 0xf0, 0xca, 0x9a, 0x2f, 0xe2, 0xc3, 0x6c, 0x04, 
			0xa0, 0x68, 0x1b, 0x8b, 0x81, 0xf4, 0x25, 0x91, 0x26, 0xcc, 0x8d, 0xa1, 0x37, 0xf1, 0xdc, 0x53, 
			0xd2, 0x60, 0xad, 0xeb, 0x15, 0xc3, 0x7d, 0xa6, 0xd6, 0xb1, 0xed, 0xb7, 0x35, 0xac, 0x50, 0x60, 
			0x67, 0x5f, 0x2f, 0xbe, 0xb2, 0xc4, 0x11, 0x9b, 0x9a, 0xfc, 0x7a, 0x75, 0x85, 0xfb, 0xc9, 0xcf, 
			0x8a, 0xca, 0xcb, 0xd3, 0x92, 0x5b, 0xfc, 0x52, 0xae, 0x1f, 0x6c, 0xda, 0x61, 0xd1, 0x33, 0x84, 
			0x8f, 0x74, 0x0d, 0xd9, 0xa3, 0x16, 0xbe, 0x5c, 0x5e, 0x12, 0xcf, 0xf2, 0x5a, 0x18, 0x2e, 0x3f, 
			0xf7, 0x48, 0x05, 0xca, 0x15, 0xbe, 0x25, 0xa3, 0xd5, 0x06, 0xe5, 0xe0, 0x00, 0xb0, 0x89, 0xc1, 
			0x93, 0x56, 0x10, 0xbd, 0xfc, 0x3c, 0xa2, 0xb8, 0xe2, 0xcc, 0xe4, 0xc3, 0xe6, 0x2c, 0x85, 0x9e, 
			0x78, 0xec, 0x72, 0x36, 0xcc, 0x56, 0xec, 0x63, 0x9c, 0x42, 0xb8, 0xc9, 0xf5, 0x25, 0x94, 0xd9, 
			0x0f, 0xf6, 0x75, 0x0f, 0x19, 0xf6, 0x23, 0x8a, 0x73, 0xb1, 0xd1, 0x6f, 0xcb, 0x66, 0xc1, 0x46, 
			0x1a, 0xbe, 0xf6, 0xfd, 0xb9, 0x47, 0x90, 0xa9, 0x0a, 0x3b, 0xd0, 0x79, 0x4d, 0x5a, 0x93, 0xe3, 
			0x5e, 0xf3, 0x3f, 0xc3, 0x84, 0xc0, 0xbc, 0xad, 0xee, 0x2b, 0xa2, 0x00, 0xc7, 0x6e, 0x29, 0x94, 
			0x14, 0xea, 0xb9, 0x6b, 0xda, 0x9a, 0x1d, 0x94, 0xf5, 0x72, 0x95, 0x9f, 0x22, 0x84, 0x4e, 0xbb, 
			0x9b, 0x39, 0x5f, 0x1e, 0x97, 0x7d, 0xf2, 0xcb, 0x19, 0x77, 0xc0, 0x7e, 0x96, 0xc0, 0xef, 0x53, 
			0x81, 0x34, 0xe9, 0x10, 0xaf, 0xa6, 0xbc, 0xc7, 			
		},
	},
};


WrappedMaterial rsa2048underdes112
{
	"RSA2048 wrapped under DES112 using CKM_DES3_CBC_PAD",
	CKK_DES2,		// wrapping key type
	CKM_DES3_CBC_PAD,	// mechanism
	CKO_PRIVATE_KEY,	// wrapped object class
	CKK_RSA,		// wrapped key type
	{
		{		// key
			0xef, 0x4c, 0x9d, 0x73, 0xc8, 0x45, 0x0d, 0x31, 0x97, 0x4a, 0xfe, 0x2f, 0x9d, 0x94, 0x98, 0x54,
		},
		{		// IV
			0x74, 0x83, 0xdb, 0xb8, 0xb3, 0x9f, 0x55, 0xdc, 
		},
		{		// wrapped RSA key
			0xca, 0x83, 0xa0, 0x2f, 0x06, 0xab, 0xc5, 0xf3, 0xa6, 0x10, 0x93, 0x75, 0x93, 0x8f, 0xa0, 0x9f, 
			0x6e, 0x71, 0xbc, 0xa7, 0x69, 0xf8, 0x6c, 0xff, 0x92, 0xab, 0x6f, 0x08, 0xdd, 0x4b, 0xd3, 0x90, 
			0x66, 0x1c, 0x94, 0xf6, 0x8b, 0xc5, 0xb7, 0xfb, 0x1c, 0xea, 0x17, 0xe6, 0xce, 0xb0, 0xe1, 0x6a, 
			0x89, 0x0e, 0x32, 0xa8, 0xe9, 0x52, 0xad, 0x7a, 0x1f, 0x00, 0x79, 0x32, 0xd9, 0x06, 0xad, 0x00, 
			0xeb, 0xbf, 0x97, 0x67, 0xca, 0x67, 0x6a, 0xb8, 0xcf, 0xc8, 0xfc, 0xd7, 0x23, 0xa3, 0x04, 0x2c, 
			0xed, 0x0c, 0xf1, 0xc5, 0x0d, 0x4a, 0x29, 0xe3, 0x1d, 0x0c, 0xec, 0xae, 0x05, 0xd9, 0x39, 0x48, 
			0x45, 0x20, 0x7e, 0x40, 0xf7, 0xa7, 0x68, 0xb5, 0x1a, 0xc6, 0x75, 0x97, 0x4d, 0xfd, 0x37, 0xcc, 
			0xb1, 0xdb, 0x6d, 0x8a, 0x46, 0x2d, 0x3a, 0x59, 0xee, 0x4c, 0x21, 0x31, 0x92, 0x25, 0x3a, 0x1d, 
			0x30, 0xc9, 0x65, 0x21, 0x8e, 0xd8, 0xfb, 0xdb, 0xa5, 0xc4, 0x70, 0x9d, 0x03, 0x0e, 0xb7, 0x48, 
			0x13, 0xaa, 0xa8, 0xf5, 0x0f, 0x71, 0xcf, 0xfc, 0xc4, 0x0b, 0xf0, 0xa1, 0xbf, 0x64, 0x77, 0x32, 
			0xa6, 0x72, 0xb5, 0xa7, 0x73, 0x4a, 0x63, 0x63, 0x77, 0x71, 0x49, 0x78, 0x7e, 0x2b, 0xc9, 0x4c, 
			0x01, 0x1a, 0x42, 0x49, 0x5d, 0x7d, 0x8c, 0x70, 0x9a, 0x8c, 0x65, 0x3d, 0x53, 0x36, 0xe5, 0xfb, 
			0x95, 0x98, 0x2a, 0xdc, 0xf4, 0xc6, 0x2d, 0xcd, 0x79, 0xd0, 0x06, 0x16, 0x8c, 0xe3, 0xe5, 0x75, 
			0x6d, 0x0a, 0x9e, 0xd2, 0x4c, 0x36, 0x13, 0xf5, 0x2c, 0x55, 0x85, 0x6e, 0x6a, 0x19, 0x64, 0x4c, 
			0xf8, 0x9b, 0x0b, 0x0a, 0x59, 0x5e, 0x7f, 0x96, 0x4b, 0x6b, 0x27, 0xa0, 0xd4, 0x09, 0x31, 0xbd, 
			0xca, 0xf8, 0xc8, 0x7a, 0xa5, 0xe9, 0xa3, 0x51, 0xe9, 0xfe, 0xbf, 0x95, 0xad, 0x85, 0x1e, 0x03, 
			0x62, 0xb3, 0xdf, 0xf7, 0xe3, 0x43, 0xe9, 0xf6, 0x68, 0xed, 0x3b, 0x3b, 0x06, 0x10, 0x81, 0xfd, 
			0x45, 0x79, 0xfa, 0xa2, 0x47, 0xc3, 0xad, 0x28, 0x77, 0xca, 0xe7, 0x44, 0x10, 0x6f, 0xde, 0x5a, 
			0x4f, 0xf2, 0xd2, 0x7f, 0x7a, 0x10, 0xb9, 0xc6, 0x61, 0x1f, 0xb3, 0x5b, 0xf3, 0xcd, 0x22, 0x8f, 
			0xc7, 0xae, 0x5f, 0x5c, 0xdc, 0x41, 0xd6, 0x85, 0x4a, 0xe9, 0x14, 0x0c, 0x6a, 0xe6, 0xdd, 0x6a, 
			0x41, 0x04, 0x5c, 0x72, 0x99, 0xee, 0xf9, 0xce, 0x8d, 0x80, 0x96, 0xb5, 0x1c, 0x93, 0x03, 0xb8, 
			0x0e, 0xbe, 0x16, 0x16, 0xa1, 0xf7, 0x5e, 0x14, 0xa4, 0xb5, 0x7d, 0xe0, 0xde, 0x99, 0xc7, 0x51, 
			0x81, 0xe8, 0x9e, 0xd1, 0xdf, 0x44, 0xc8, 0x93, 0xc5, 0x11, 0x8f, 0xd2, 0x5f, 0xd5, 0x53, 0xa8, 
			0x25, 0x63, 0x6c, 0xfa, 0x6c, 0x2c, 0xe7, 0xcc, 0x88, 0x25, 0xe8, 0x4a, 0x15, 0x5e, 0x44, 0x22, 
			0x01, 0x2d, 0x19, 0x16, 0x81, 0xbe, 0x21, 0x59, 0x3d, 0x86, 0x36, 0xf3, 0xe0, 0x7a, 0x73, 0x11, 
			0x4d, 0x4a, 0x35, 0xcc, 0x54, 0xbc, 0x90, 0x03, 0x9b, 0x62, 0xbe, 0x83, 0xee, 0x61, 0x45, 0xd8, 
			0x53, 0x25, 0xa2, 0x46, 0x8c, 0xf9, 0xf2, 0x15, 0x8d, 0x6f, 0x51, 0x2a, 0x3b, 0x0d, 0x11, 0xb5, 
			0x9d, 0x55, 0x20, 0x06, 0xed, 0xee, 0xc5, 0xb7, 0x08, 0xe7, 0x14, 0x1a, 0xee, 0x44, 0xef, 0x37, 
			0xb6, 0xbf, 0x27, 0x68, 0x46, 0x35, 0x89, 0x84, 0x6a, 0x9c, 0x07, 0x2e, 0x87, 0xc5, 0x11, 0xf6, 
			0xc4, 0x9b, 0x36, 0x10, 0xa1, 0xdd, 0x66, 0x85, 0x33, 0x72, 0x03, 0x41, 0x75, 0xa4, 0x04, 0x6b, 
			0x17, 0x10, 0x55, 0xc6, 0xd8, 0x68, 0x9a, 0x55, 0xcb, 0xd7, 0xcd, 0x1d, 0x85, 0x3e, 0xe8, 0xe7, 
			0x2e, 0x5b, 0x25, 0x24, 0xf6, 0x54, 0x60, 0xb2, 0x17, 0xcd, 0xe0, 0x11, 0x8f, 0xca, 0x24, 0x32, 
			0x0c, 0x63, 0x9f, 0x2d, 0x7c, 0xe8, 0x7f, 0x53, 0x58, 0x7c, 0xeb, 0xd4, 0x16, 0x7b, 0x78, 0x50, 
			0xf4, 0x75, 0xe5, 0x25, 0xf4, 0x47, 0xa1, 0x08, 0x82, 0x68, 0xd6, 0xb1, 0xbd, 0x9a, 0xa2, 0xa5, 
			0x22, 0x7d, 0x43, 0xff, 0x9d, 0xdc, 0x7f, 0x48, 0x29, 0x22, 0x3e, 0xb8, 0xf5, 0xe2, 0x26, 0x16, 
			0x85, 0xb8, 0x01, 0x4a, 0x1f, 0x81, 0xe0, 0x54, 0xde, 0xc0, 0x3e, 0x57, 0x33, 0x0f, 0x23, 0x31, 
			0xae, 0x88, 0x3a, 0x42, 0x14, 0x24, 0xc1, 0x6f, 0x34, 0xb8, 0x37, 0x14, 0x29, 0x04, 0x01, 0x54, 
			0xa1, 0x62, 0x89, 0x51, 0x35, 0xfe, 0x34, 0xa3, 0x19, 0x45, 0x23, 0x3a, 0x2a, 0x55, 0x9a, 0x51, 
			0xec, 0x1e, 0x5a, 0x30, 0x3f, 0xce, 0xf2, 0x76, 0x0c, 0x9a, 0x87, 0xba, 0xc3, 0x6f, 0x71, 0x2e, 
			0x0e, 0x2b, 0x00, 0x05, 0xfb, 0xa1, 0x10, 0x35, 0x4c, 0x27, 0x11, 0x2f, 0xfb, 0x45, 0x34, 0x2c, 
			0x8d, 0x57, 0x2f, 0x80, 0x95, 0x46, 0x2a, 0xef, 0x09, 0x82, 0x12, 0x17, 0x62, 0xc8, 0x7d, 0xf5, 
			0x51, 0x05, 0xb5, 0xfe, 0x2b, 0xcf, 0x46, 0xba, 0xd6, 0x89, 0x49, 0x90, 0xdc, 0x47, 0x13, 0x77, 
			0xa6, 0xd8, 0x03, 0x68, 0x68, 0x6b, 0x40, 0xa4, 0x93, 0x74, 0x2c, 0x75, 0xdc, 0xfa, 0x31, 0xb1, 
			0xaa, 0xc2, 0xc7, 0x77, 0x23, 0xcd, 0x7c, 0x2c, 0x9c, 0x1b, 0xe1, 0xe9, 0x09, 0x90, 0x5c, 0xbe, 
			0xa6, 0xfc, 0x48, 0x8c, 0xf5, 0xce, 0xcd, 0x43, 0xa9, 0x96, 0x19, 0xb0, 0x47, 0x80, 0xf8, 0x04, 
			0x40, 0x8f, 0x51, 0x5f, 0x34, 0x4a, 0x18, 0x6e, 0x07, 0xea, 0x12, 0x84, 0x71, 0xe4, 0x0c, 0xbd, 
			0x5a, 0x04, 0x82, 0x99, 0xaf, 0x97, 0x8f, 0x03, 0x1d, 0xb3, 0x2a, 0x0e, 0x3b, 0x11, 0x48, 0xa0, 
			0x8a, 0xa3, 0xe3, 0xa3, 0xd0, 0x7d, 0x4a, 0x91, 0xc2, 0x58, 0xdf, 0x98, 0x3e, 0x50, 0xb0, 0xd6, 
			0x5b, 0x0b, 0x0d, 0xaa, 0xed, 0x01, 0x79, 0x63, 0x37, 0x21, 0x8a, 0xbe, 0x10, 0x71, 0xbe, 0xa7, 
			0x5f, 0xda, 0x3a, 0x2b, 0x66, 0x29, 0x6c, 0x05, 0x60, 0x13, 0x9e, 0x24, 0xcc, 0x8e, 0x50, 0xd5, 
			0x73, 0x62, 0x2f, 0xec, 0x8e, 0x9b, 0xdc, 0x5b, 0x36, 0xc5, 0x06, 0xf1, 0xe5, 0xac, 0xde, 0x33, 
			0xda, 0x04, 0x99, 0x4d, 0x25, 0x33, 0x06, 0x5d, 0x45, 0x61, 0xd0, 0xa5, 0xed, 0xd5, 0xa8, 0x94, 
			0x19, 0x06, 0x03, 0x58, 0xb9, 0x7e, 0xd4, 0xab, 0x53, 0xfa, 0x74, 0xcb, 0xb9, 0x15, 0x78, 0x1b, 
			0x62, 0x79, 0x62, 0x4a, 0x23, 0x5e, 0x86, 0x1d, 0xde, 0xe5, 0xbf, 0x00, 0x88, 0x24, 0xcd, 0xdd, 
			0x44, 0x8c, 0xcd, 0xd3, 0xf9, 0x50, 0xa6, 0x98, 0xb1, 0xd7, 0xe6, 0x2e, 0x7f, 0x59, 0xb9, 0xb9, 
			0xc9, 0x3c, 0x64, 0xb3, 0x1a, 0x85, 0x67, 0x1d, 0x85, 0xa0, 0xd2, 0xbd, 0x19, 0x2f, 0x9b, 0x8e, 
			0xa4, 0xf8, 0x24, 0x69, 0xc7, 0xc8, 0x10, 0x19, 0x74, 0x53, 0x50, 0x3b, 0x00, 0x45, 0x33, 0x8f, 
			0xac, 0xea, 0x77, 0x44, 0x50, 0xe3, 0xde, 0x86, 0xbc, 0x00, 0x05, 0xfa, 0xd6, 0xc8, 0xef, 0x95, 
			0x60, 0xec, 0x61, 0x70, 0xf3, 0xec, 0x0b, 0xf3, 0x32, 0x33, 0x10, 0xf6, 0x1c, 0xe9, 0xab, 0xe1, 
			0x09, 0xad, 0x74, 0x3c, 0x23, 0x39, 0x86, 0xa6, 0xbf, 0x90, 0xcf, 0xae, 0xd5, 0xe2, 0xcd, 0xe6, 
			0xcb, 0x38, 0x79, 0xe7, 0xd3, 0x1e, 0xd9, 0x3e, 0x13, 0x96, 0xf0, 0x49, 0x35, 0xdf, 0x94, 0x3c, 
			0x60, 0xfe, 0xd1, 0xba, 0xd4, 0xcc, 0x5f, 0x9b, 0x38, 0xc5, 0x48, 0x67, 0x4f, 0xa3, 0xb5, 0x34, 
			0x29, 0x74, 0xbc, 0x90, 0xbf, 0x8a, 0xbf, 0xb1, 0x2c, 0x43, 0x3c, 0xfd, 0x72, 0x06, 0x4e, 0xff, 
			0x46, 0x96, 0x63, 0x9a, 0x65, 0x69, 0x91, 0x4c, 0x14, 0x6a, 0x60, 0xe5, 0x6c, 0x0b, 0x76, 0xa2, 
			0x64, 0x7d, 0x36, 0x8f, 0x8d, 0xbe, 0x67, 0x89, 0x0a, 0x51, 0x6a, 0x2e, 0x63, 0x60, 0x31, 0x73, 
			0xf0, 0x27, 0x41, 0xc7, 0xd2, 0x3f, 0x52, 0xd3, 0x20, 0x8e, 0xf3, 0xee, 0x77, 0xcc, 0x10, 0xc2, 
			0x2f, 0x60, 0xf1, 0x7a, 0x3d, 0xd2, 0xee, 0x58, 0xd6, 0x68, 0xf8, 0xa0, 0x87, 0x88, 0x69, 0xf9, 
			0xb4, 0x00, 0xa7, 0xa6, 0x2a, 0xe8, 0x35, 0xb1, 0x8d, 0x91, 0x27, 0xd9, 0x64, 0x0c, 0xd3, 0xe9, 
			0xeb, 0xef, 0x35, 0x10, 0x51, 0xdc, 0x42, 0xfe, 0x7b, 0xf0, 0x55, 0x7e, 0x0c, 0x62, 0xd7, 0xc8, 
			0x7d, 0xfa, 0x5e, 0xde, 0x05, 0x84, 0x3d, 0x1d, 0xda, 0x42, 0xc8, 0xba, 0x50, 0x0a, 0x5b, 0xab, 
			0xa4, 0xdd, 0x02, 0x4f, 0xa9, 0x73, 0xd0, 0xeb, 0x78, 0x4c, 0x24, 0xd2, 0xe1, 0x84, 0x07, 0x75, 
			0xad, 0x4d, 0x8b, 0x00, 0x7b, 0x4e, 0x63, 0x1f, 0xeb, 0xe4, 0x42, 0x5e, 0xca, 0xe8, 0x8e, 0xcb, 
			0xe4, 0x6f, 0x10, 0xd6, 0xe0, 0x4a, 0xd7, 0xea, 0x44, 0xd5, 0xaf, 0xb7, 0x7a, 0xdb, 0x33, 0x4b, 
			0xfd, 0x8a, 0x7e, 0x28, 0x52, 0x61, 0xbe, 0x58, 0xb9, 0x03, 0xf7, 0x2d, 0x2a, 0x1c, 0xa8, 0x73, 
			0x8f, 0xfd, 0x15, 0x77, 0xca, 0x02, 0xc1, 0x14, 0x3c, 0x15, 0xf9, 0xd6, 0xda, 0xdc, 0xfc, 0x1c, 
			0x89, 0x0f, 0x19, 0xb6, 0x68, 0x80, 0x78, 0x09, 0x43, 0x4e, 0xd0, 0xf7, 0x61, 0xe7, 0x4f, 0x9b, 
			0x6d, 0x38, 0x37, 0x74, 0x14, 0x4e, 0x6c, 0x7b, 
		},
	},
};

#endif // WITH_FIPS

WrappedMaterial rsa2048underdes168
{
	"RSA2048 wrapped under DES168 using CKM_DES3_CBC_PAD",
	CKK_DES3,		// wrapping key type
	CKM_DES3_CBC_PAD,	// mechanism
	CKO_PRIVATE_KEY,	// wrapped object class
	CKK_RSA,		// wrapped key type
	{
		{		// key
			0xfe, 0x0e, 0x7f, 0x68, 0x9e, 0xcb, 0x23, 0x68, 0xb3, 0x51, 0xea, 0x49, 0x26, 0xf2, 0xf8, 0x49,
			0x4c, 0xe0, 0x58, 0x25, 0xf7, 0x64, 0xc1, 0xf7,
		},
		{		// IV
			0x1e, 0x06, 0xbb, 0xa6, 0xd4, 0x3e, 0x1d, 0x05, 
		},
		{		// wrapped RSA key
			0x45, 0x84, 0xa6, 0xc7, 0xec, 0x3c, 0x6b, 0x38, 0xbe, 0x6f, 0x0d, 0x9d, 0x3a, 0x80, 0x17, 0x2d, 
			0xab, 0x6b, 0x6f, 0x82, 0x5a, 0x6d, 0x8e, 0x50, 0x84, 0xf2, 0x61, 0x29, 0xc8, 0x99, 0x97, 0xac, 
			0x2a, 0xca, 0x1c, 0xb3, 0x45, 0x7c, 0xcf, 0xd8, 0xcb, 0xf8, 0x65, 0x85, 0xaf, 0x1f, 0x1f, 0xb0, 
			0x60, 0x02, 0x8e, 0xa4, 0x21, 0x7c, 0x95, 0xd9, 0x37, 0x78, 0xad, 0x9e, 0x02, 0xbf, 0x97, 0xab, 
			0x9c, 0x28, 0x55, 0xdc, 0x90, 0x8c, 0x74, 0xd2, 0x35, 0xdf, 0x15, 0x7f, 0x11, 0x70, 0x64, 0x91, 
			0xb7, 0x43, 0x2b, 0x41, 0x8b, 0x23, 0x12, 0x3b, 0xc4, 0x68, 0x46, 0x39, 0x25, 0xaa, 0x22, 0xf9, 
			0x42, 0x10, 0x4e, 0x1a, 0xda, 0xef, 0x25, 0x43, 0x1b, 0x43, 0x8c, 0x5e, 0xe5, 0x81, 0x93, 0x11, 
			0x67, 0xa4, 0x4a, 0xb2, 0x47, 0x8e, 0x35, 0x80, 0xa1, 0xf6, 0x3a, 0x0c, 0x02, 0x1e, 0x5f, 0x6e, 
			0x54, 0xd6, 0x45, 0xe3, 0xd7, 0x0f, 0x39, 0xc6, 0x76, 0x41, 0xf7, 0xcc, 0xd8, 0x28, 0xec, 0xe3, 
			0x53, 0xfe, 0x86, 0x33, 0x62, 0x50, 0x36, 0xa6, 0x7a, 0xac, 0x24, 0x10, 0xcb, 0x4d, 0x2b, 0x76, 
			0x44, 0xe9, 0x65, 0xa0, 0x13, 0xe8, 0x92, 0x9c, 0x0a, 0xe2, 0x66, 0x8a, 0x46, 0x7d, 0x9e, 0xd4, 
			0x76, 0xdc, 0x30, 0xc7, 0xd8, 0x23, 0x72, 0x53, 0xaf, 0x1f, 0xe7, 0xdf, 0xb3, 0x93, 0xea, 0xf8, 
			0x09, 0xc3, 0xca, 0xa4, 0xc8, 0x52, 0x05, 0xd3, 0xc2, 0x08, 0x14, 0xd5, 0x50, 0x36, 0x68, 0xdc, 
			0x9e, 0xdc, 0x1d, 0xca, 0x33, 0xbd, 0x64, 0xc7, 0x9f, 0x14, 0x51, 0x15, 0xd0, 0xc7, 0x2f, 0xef, 
			0x20, 0x14, 0x2e, 0x89, 0xc0, 0x28, 0xd4, 0xb5, 0x60, 0xfb, 0x9d, 0x6c, 0x2e, 0x4b, 0xf3, 0xb3, 
			0x11, 0xef, 0xf4, 0xd3, 0xbb, 0x63, 0x8a, 0x9b, 0xfe, 0x9d, 0x49, 0xb6, 0xad, 0x44, 0xd8, 0xed, 
			0x23, 0xd4, 0x8e, 0x71, 0x82, 0x6c, 0x13, 0xcd, 0x18, 0x98, 0xd1, 0xf2, 0x8e, 0x42, 0xa4, 0x91, 
			0x46, 0x41, 0x90, 0x29, 0xf2, 0x0b, 0xf8, 0x4d, 0xc0, 0x51, 0x5f, 0x8f, 0xf8, 0x81, 0x1b, 0xf3, 
			0x23, 0xb2, 0xf9, 0x6c, 0x2f, 0x20, 0xa8, 0xb7, 0x52, 0xb5, 0x4b, 0xce, 0xd8, 0xd2, 0x65, 0x4c, 
			0xc1, 0xd8, 0xc4, 0x91, 0xc0, 0x75, 0x8b, 0x6a, 0x8e, 0x11, 0xbe, 0x68, 0x61, 0xff, 0xc8, 0xb8, 
			0x74, 0x6e, 0x84, 0x04, 0x65, 0xe0, 0xd8, 0x71, 0x76, 0x8a, 0x87, 0xae, 0xc6, 0x68, 0x3b, 0x48, 
			0xe5, 0xed, 0x4e, 0x6c, 0xbf, 0x42, 0xb4, 0xc9, 0x60, 0x9d, 0xe2, 0xb4, 0x61, 0xce, 0xd7, 0x80, 
			0x75, 0x6f, 0x7d, 0x8d, 0xf3, 0xde, 0x09, 0xa7, 0xcb, 0xc0, 0xca, 0xa3, 0x98, 0xca, 0x3b, 0x95, 
			0x37, 0x4d, 0xae, 0x8f, 0x15, 0xb4, 0x74, 0xd1, 0xb4, 0x91, 0xef, 0x10, 0xae, 0x0e, 0x22, 0x05, 
			0xba, 0xd0, 0xd4, 0x28, 0x6c, 0x24, 0x7c, 0x6e, 0xfc, 0xc1, 0x5b, 0x8c, 0x9c, 0x1c, 0xfc, 0xea, 
			0x20, 0x89, 0x94, 0x33, 0xfb, 0x1b, 0x44, 0xd6, 0x76, 0xf8, 0xbb, 0x0c, 0x08, 0x43, 0xd4, 0x0d, 
			0xda, 0x4c, 0x64, 0x7f, 0x49, 0x06, 0xfd, 0x65, 0xc1, 0x14, 0xfd, 0x96, 0xa3, 0xca, 0x76, 0x0d, 
			0x89, 0xf2, 0xef, 0xce, 0x62, 0x5f, 0x35, 0x91, 0x3f, 0x3a, 0x81, 0x88, 0x5a, 0x7e, 0x09, 0xde, 
			0x97, 0x49, 0x64, 0xf8, 0x64, 0xbf, 0xf4, 0xc2, 0x89, 0x87, 0x4f, 0xc3, 0x14, 0x17, 0x3b, 0x47, 
			0xde, 0x2b, 0xe7, 0x06, 0x9d, 0x06, 0x60, 0xbe, 0xaa, 0x84, 0x37, 0xbf, 0x3a, 0xd8, 0x99, 0x36, 
			0xde, 0x64, 0x37, 0x94, 0x52, 0x53, 0x3c, 0x39, 0xe7, 0x35, 0x7c, 0xbd, 0x27, 0xec, 0xf1, 0xbe, 
			0xa5, 0x4e, 0x83, 0xf7, 0x66, 0x95, 0xe8, 0x7f, 0x7e, 0x31, 0xad, 0xac, 0xbb, 0x8c, 0x1a, 0x02, 
			0xbb, 0xa3, 0x75, 0xd3, 0x54, 0xae, 0x96, 0x4b, 0xb9, 0x2e, 0x3f, 0x09, 0xd4, 0x5b, 0xaa, 0xc1, 
			0xf7, 0x68, 0xcc, 0x3b, 0x91, 0x51, 0xaf, 0x96, 0xf3, 0x8b, 0xb1, 0x0f, 0x3b, 0x06, 0xe5, 0xba, 
			0xb0, 0xb8, 0x57, 0x7d, 0xe1, 0xcd, 0x53, 0x75, 0xb0, 0xbb, 0xa3, 0x81, 0x97, 0x3b, 0x41, 0x1c, 
			0xfd, 0x7d, 0x62, 0x87, 0x91, 0xde, 0x07, 0xa5, 0xe5, 0xba, 0xf5, 0x4b, 0x61, 0x64, 0x89, 0x0c, 
			0xf3, 0x54, 0xce, 0x70, 0xd2, 0x77, 0x66, 0x5b, 0x7c, 0x26, 0x4d, 0x0f, 0x53, 0x64, 0x4b, 0xbb, 
			0x5d, 0x10, 0xee, 0xe8, 0x8e, 0x07, 0x4b, 0x6e, 0x24, 0x3f, 0x36, 0x2f, 0x7a, 0x0b, 0xd9, 0x46, 
			0x40, 0xae, 0xd5, 0x5e, 0x43, 0x79, 0x71, 0x7f, 0x78, 0xf0, 0xbf, 0xe2, 0x82, 0xbb, 0x52, 0x80, 
			0xa3, 0xfc, 0xab, 0x3e, 0x7a, 0x16, 0x79, 0xbe, 0x1a, 0x9f, 0x97, 0x6e, 0xc0, 0xb7, 0x61, 0x91, 
			0x8b, 0xca, 0x1f, 0x11, 0xdb, 0xb2, 0xa8, 0xa8, 0x92, 0x64, 0xf6, 0xab, 0x9a, 0x92, 0x3f, 0x33, 
			0xdb, 0x1d, 0x06, 0xf3, 0x53, 0x78, 0xb0, 0x11, 0x70, 0x4f, 0xca, 0x69, 0x6b, 0xf7, 0xc6, 0x6e, 
			0xc1, 0x13, 0x03, 0x02, 0x59, 0x9d, 0xfd, 0x9f, 0xa5, 0xd1, 0x1e, 0x3e, 0x33, 0xb1, 0x25, 0x18, 
			0xd9, 0x69, 0x43, 0x37, 0x34, 0x48, 0x2e, 0xa1, 0xe3, 0x20, 0xdf, 0x08, 0x8e, 0xc7, 0xdc, 0x0a, 
			0xc2, 0x01, 0x23, 0xa2, 0xa5, 0x93, 0xe9, 0xe0, 0x6d, 0x2a, 0x8f, 0x9a, 0xc6, 0x95, 0xc3, 0xe1, 
			0xe9, 0xca, 0x26, 0x08, 0x5f, 0x20, 0xc2, 0x57, 0xc6, 0xcf, 0x54, 0x2e, 0x9a, 0x2d, 0xd8, 0x1a, 
			0x13, 0x8b, 0xb2, 0x68, 0x08, 0x3f, 0x3f, 0x69, 0xf3, 0x57, 0x75, 0x46, 0xe6, 0x82, 0x9f, 0xae, 
			0xfd, 0x19, 0x05, 0x64, 0x16, 0x1c, 0x90, 0x4f, 0xd0, 0xcb, 0x05, 0xed, 0x80, 0x8f, 0x82, 0xdf, 
			0x77, 0xa7, 0xa9, 0x59, 0xe4, 0xd4, 0x92, 0x76, 0x7b, 0xf0, 0x6c, 0x15, 0xa5, 0x4d, 0x3c, 0x93, 
			0xf8, 0x29, 0xa7, 0x0d, 0xa0, 0x8e, 0x6e, 0x85, 0x23, 0x8e, 0xf3, 0x4d, 0x3c, 0x93, 0x9d, 0x55, 
			0xc3, 0x81, 0xf8, 0xab, 0x8a, 0x03, 0xf4, 0xde, 0x21, 0xa7, 0x46, 0x8a, 0x03, 0xee, 0xce, 0xad, 
			0x9e, 0x27, 0x1a, 0x20, 0x4f, 0xa1, 0x17, 0xbc, 0x83, 0xbf, 0x87, 0x76, 0xf0, 0x22, 0x2e, 0xbb, 
			0x20, 0x5b, 0x7d, 0xe9, 0xac, 0x16, 0xe5, 0x2e, 0x0f, 0x75, 0x96, 0x56, 0xe6, 0x81, 0xd6, 0x6f, 
			0xa0, 0x42, 0xae, 0x8c, 0x6d, 0xe2, 0xfd, 0x4f, 0xf2, 0x8c, 0xea, 0xcc, 0xb7, 0x74, 0x5a, 0xb7, 
			0x3b, 0xb0, 0xaa, 0x6e, 0xb0, 0xd5, 0xd0, 0xb2, 0x68, 0x63, 0xad, 0xc1, 0x20, 0x32, 0xb8, 0x65, 
			0xb9, 0xfd, 0x94, 0x48, 0x8a, 0x34, 0xaa, 0x28, 0x6f, 0x3c, 0x00, 0x4f, 0x01, 0x41, 0xdd, 0x10, 
			0x6b, 0x19, 0x92, 0xbc, 0x73, 0x72, 0xad, 0xcb, 0xba, 0x78, 0x20, 0x2e, 0x86, 0xc4, 0xbf, 0xed, 
			0x91, 0x62, 0x49, 0xd3, 0xcf, 0x3d, 0xdb, 0xbd, 0x66, 0xb5, 0x69, 0x4c, 0xd6, 0xa9, 0x17, 0xbc, 
			0xba, 0xb7, 0x71, 0x40, 0x02, 0x2d, 0x7e, 0xa6, 0x33, 0x40, 0x03, 0xbd, 0x5b, 0xf5, 0xd2, 0x5d, 
			0x68, 0x19, 0x55, 0xe4, 0xb4, 0x46, 0xbd, 0xc3, 0x21, 0x8f, 0x75, 0xd6, 0xb3, 0x91, 0xaf, 0x38, 
			0xb6, 0x21, 0x9f, 0x0e, 0xd2, 0x7c, 0xb2, 0x7d, 0x87, 0x65, 0xc4, 0xe5, 0x6a, 0x8b, 0x4d, 0x42, 
			0x68, 0x6b, 0x1a, 0xc8, 0xf5, 0x47, 0xd0, 0x66, 0x67, 0xc4, 0x45, 0xc0, 0xb9, 0xbe, 0x24, 0x49, 
			0xc9, 0x7e, 0xcd, 0xff, 0x0f, 0x23, 0x17, 0x90, 0xe2, 0x6c, 0x96, 0xa0, 0xfd, 0x5b, 0x21, 0x0e, 
			0x7c, 0xfa, 0xa7, 0x32, 0x16, 0xbe, 0x64, 0x1f, 0xa1, 0x89, 0x61, 0xeb, 0x19, 0x9a, 0xa3, 0xbf, 
			0xe2, 0x35, 0xf6, 0x02, 0x86, 0x42, 0xb7, 0xa6, 0x85, 0xff, 0xa9, 0x19, 0xe9, 0x65, 0x64, 0x37, 
			0x11, 0xe9, 0x84, 0xba, 0x3f, 0x87, 0x25, 0x08, 0xb0, 0xcb, 0x7c, 0x3d, 0x66, 0x4b, 0xad, 0x38, 
			0x8f, 0x38, 0x9f, 0x60, 0xce, 0x08, 0xb3, 0x50, 0x29, 0xe4, 0x30, 0x13, 0xc0, 0x64, 0x5a, 0xc1, 
			0x9a, 0xaf, 0x68, 0x46, 0x56, 0x5f, 0xca, 0x1d, 0xc7, 0x4e, 0xb0, 0x5d, 0x59, 0x78, 0xbd, 0x7b, 
			0x09, 0x13, 0x08, 0x3b, 0x64, 0x02, 0x36, 0x22, 0xef, 0x00, 0xb0, 0xc7, 0x1e, 0x35, 0x5d, 0x37, 
			0x6c, 0x56, 0x1e, 0xa5, 0x42, 0x16, 0x18, 0xb4, 0xbf, 0x13, 0x9d, 0xf7, 0x5a, 0x50, 0x29, 0x84, 
			0xb2, 0xee, 0x9a, 0x4a, 0xcf, 0xce, 0x75, 0xcb, 0x8a, 0x9e, 0x83, 0x95, 0xd8, 0x48, 0xd7, 0x85, 
			0x26, 0xdb, 0x62, 0xa5, 0xb5, 0x1c, 0xd8, 0x16, 0x59, 0xb9, 0x20, 0x40, 0x99, 0x53, 0x16, 0xc5, 
			0x79, 0xe0, 0x40, 0x74, 0x5f, 0xa8, 0x21, 0x3c, 0x30, 0xbb, 0x3a, 0x11, 0x25, 0x0b, 0xb0, 0x77, 
			0x94, 0xd7, 0x6c, 0x22, 0x34, 0x14, 0x7c, 0xec, 0x93, 0xdd, 0x04, 0xb8, 0x4a, 0x8b, 0x9d, 0xaf, 
			0xc1, 0xe6, 0x5c, 0x28, 0xfc, 0x68, 0x68, 0x2a, 0x2f, 0x31, 0xd5, 0x52, 0xf0, 0xd0, 0x57, 0x1c, 
			0x02, 0x65, 0xa9, 0xd0, 0x6b, 0x62, 0xb1, 0xed, 0xc0, 0x4e, 0x4b, 0x79, 0x46, 0x6e, 0xf6, 0x15, 
			0x27, 0x44, 0xff, 0x0e, 0x89, 0xc1, 0xb2, 0xbe, 			
		},
	},
};

WrappedMaterial rsa2048underaes128 
{
	"RSA2048 wrapped under AES128 using CKM_AES_CBC_PAD",
	CKK_AES,		// wrapping key type
	CKM_AES_CBC_PAD,	// mechanism
	CKO_PRIVATE_KEY,	// wrapped object class
	CKK_RSA,		// wrapped key type
	{
		{		// key
			0xd5, 0x20, 0xbc, 0xaa, 0x9c, 0x3f, 0x1f, 0x32,
			0x51, 0x87, 0x64, 0x34, 0x5a, 0xd6, 0x4f, 0x9e
		},
		{		// IV
			0x55, 0x7c, 0xc6, 0xab, 0xdf, 0xee, 0xa1, 0x10,
			0xeb, 0x89, 0x42, 0x31, 0x72, 0x82, 0xbc, 0xeb
		},
		{		// wrapped RSA key
			0x7d, 0xa7, 0xa9, 0xe9, 0x8f, 0x73, 0x76, 0xde,	0x66, 0x9d, 0x81, 0x0b, 0x6f, 0xcb, 0x17, 0x87,
			0xb7, 0x2d, 0xb8, 0xc0, 0x9e, 0xec, 0x1f, 0xd5,	0x4a, 0x3a, 0x82, 0x72, 0xe4, 0x65, 0x55, 0x1e,
			0x83, 0x8c, 0xde, 0x6e, 0x3e, 0xef, 0x29, 0xdf,	0x90, 0x5c, 0xb0, 0xae, 0x32, 0x33, 0xe8, 0x9c,
			0xd1, 0x75, 0x16, 0xd6, 0xca, 0x50, 0x28, 0xec,	0x68, 0x9c, 0xe9, 0x1a, 0x3d, 0x1f, 0xe7, 0x18,
			0x5e, 0xa6, 0x65, 0x6b, 0xeb, 0xc4, 0x6c, 0x19,	0xfc, 0xd3, 0x88, 0x49, 0xe1, 0x77, 0x2a, 0x6a,
			0x0a, 0xba, 0x56, 0x93, 0x27, 0x56, 0xb4, 0x24,	0x89, 0x80, 0xc5, 0xec, 0xeb, 0x3e, 0x92, 0x25,
			0xeb, 0x5b, 0xf6, 0xf1, 0x5e, 0xd8, 0xc1, 0x9d,	0x7b, 0x46, 0xcc, 0x5d, 0xb5, 0xe2, 0x28, 0x6a,
			0x7d, 0x9b, 0x88, 0x43, 0x74, 0xa0, 0x33, 0xf9,	0xa8, 0xb4, 0xa6, 0x43, 0xf2, 0x4e, 0x0d, 0x57,
			0x4d, 0x87, 0x6e, 0x84, 0xc5, 0xe8, 0xbb, 0x9b,	0x5c, 0x15, 0x51, 0xef, 0xe6, 0x84, 0x28, 0x80,
			0xe3, 0xba, 0xad, 0xa3, 0x72, 0x93, 0xd2, 0x49,	0x20, 0x0f, 0x5c, 0xf9, 0xbb, 0x00, 0xf1, 0xbd,
			0x98, 0x68, 0x6f, 0x03, 0x5f, 0x5a, 0xd5, 0xd0,	0x71, 0x9c, 0xcb, 0x37, 0x8b, 0xa7, 0x02, 0xdb,
			0x41, 0x96, 0x24, 0x91, 0xf6, 0xaa, 0xa4, 0xba,	0x31, 0x97, 0x86, 0x4f, 0x02, 0xa9, 0xe3, 0xfd,
			0x61, 0x2d, 0x97, 0xf0, 0xa0, 0x77, 0x97, 0xb4,	0x67, 0x82, 0xd9, 0x17, 0x60, 0xd3, 0x47, 0x42,
			0x7e, 0xa5, 0xa9, 0xcb, 0x0b, 0x44, 0xe1, 0x94,	0xd9, 0x62, 0xd8, 0xee, 0x5e, 0x65, 0xa7, 0xb1,
			0xe6, 0x2c, 0x50, 0xcd, 0x10, 0x84, 0x58, 0xca,	0xfc, 0x9f, 0xd4, 0x92, 0x65, 0x42, 0xad, 0x08,
			0x98, 0x9e, 0x6f, 0xe1, 0x88, 0xc4, 0x65, 0x94,	0xd4, 0x39, 0x90, 0xca, 0x47, 0x09, 0x15, 0xa2,
			0x45, 0xad, 0x53, 0xab, 0xc5, 0x0c, 0xff, 0x05,	0x72, 0x66, 0xbf, 0x33, 0xbc, 0x9f, 0x01, 0xea,
			0xa8, 0xe0, 0x84, 0xa4, 0x8c, 0x5a, 0xc8, 0xbb,	0x7a, 0x74, 0xcb, 0xdc, 0xaf, 0x8c, 0x4b, 0x77,
			0xf2, 0x1f, 0xbe, 0xa7, 0x87, 0xe1, 0x51, 0x6c,	0xe8, 0x59, 0x7a, 0x15, 0x81, 0x96, 0x19, 0x6a,
			0xbc, 0xdc, 0xca, 0xb6, 0x61, 0x3b, 0x0d, 0x0e,	0xff, 0x96, 0x90, 0x40, 0x0c, 0xc4, 0x7f, 0x4c,
			0x58, 0xeb, 0xb9, 0xd6, 0x88, 0xe5, 0xdd, 0x69,	0x94, 0x7a, 0x3e, 0x64, 0xc4, 0xd6, 0x54, 0xf4,
			0x52, 0xe8, 0x5e, 0x72, 0x2d, 0x81, 0x66, 0x09,	0x7a, 0xf6, 0x64, 0x34, 0xf4, 0x39, 0xb3, 0xc4,
			0x50, 0x5c, 0x02, 0xd4, 0x27, 0xe2, 0x56, 0xf5,	0xa0, 0x80, 0x05, 0x96, 0xc1, 0xef, 0x44, 0x4e,
			0xe3, 0x91, 0x6d, 0x5d, 0xe7, 0xa5, 0x86, 0x0c,	0xa7, 0x8a, 0x95, 0x81, 0x01, 0xd8, 0x2e, 0x03,
			0x09, 0x21, 0x10, 0x77, 0x63, 0xed, 0x89, 0x78,	0x6c, 0x9d, 0x1e, 0xba, 0xb8, 0x64, 0xdd, 0x41,
			0x70, 0x75, 0x7f, 0x7c, 0x37, 0x6d, 0xc0, 0x66,	0xb2, 0xe1, 0x32, 0xb0, 0xa3, 0x99, 0xf0, 0xe9,
			0xac, 0x00, 0xe8, 0x9a, 0x0e, 0x31, 0x3e, 0xd5,	0xae, 0x9a, 0x3a, 0x56, 0x47, 0x1a, 0xd3, 0xde,
			0x0e, 0xa2, 0xba, 0xfb, 0x5a, 0xf1, 0xf7, 0x5d,	0x01, 0x32, 0x75, 0xfe, 0x1e, 0x53, 0xc0, 0xb5,
			0x43, 0x17, 0x58, 0x13, 0x7a, 0x58, 0x19, 0x7a,	0x29, 0xfc, 0x1f, 0xe6, 0xf2, 0xc9, 0xbb, 0x23,
			0x1b, 0x1a, 0x38, 0xa5, 0xdd, 0x38, 0x5b, 0x19,	0x2e, 0x32, 0x60, 0xaf, 0x6a, 0x84, 0xd0, 0xef,
			0x82, 0x20, 0x2d, 0xbb, 0xec, 0xcb, 0xf1, 0x98,	0xf3, 0x8a, 0x3d, 0x03, 0x17, 0x9d, 0xf9, 0x8b,
			0x9d, 0x65, 0xcf, 0x61, 0x78, 0x22, 0x68, 0x5a,	0x2e, 0x53, 0x35, 0xed, 0x34, 0xc2, 0xa1, 0x8f,
			0x84, 0xf6, 0x0c, 0x46, 0x8f, 0x0c, 0xa4, 0xdb,	0x5d, 0x24, 0x51, 0xfe, 0x7d, 0xba, 0xa8, 0xdd,
			0xae, 0x0a, 0xd3, 0x65, 0xef, 0x4b, 0x58, 0x8a,	0xa2, 0x4b, 0x56, 0xc2, 0x75, 0xb3, 0x9a, 0x8c,
			0x72, 0x0e, 0x67, 0xd7, 0x22, 0xdc, 0xbb, 0xa7,	0xb2, 0x8e, 0x24, 0x41, 0x44, 0xcb, 0x8b, 0x5b,
			0xf2, 0x45, 0xba, 0xd5, 0x2a, 0x5e, 0xdc, 0xb2,	0x5e, 0x62, 0x7d, 0xbf, 0xa2, 0x9b, 0xc0, 0x05,
			0x79, 0xdc, 0x9c, 0x2b, 0x45, 0x74, 0x7b, 0xd6,	0x50, 0x5c, 0x12, 0x6f, 0xed, 0xa3, 0xb8, 0x7e,
			0x6b, 0xc1, 0xe7, 0xfe, 0x00, 0x37, 0x3f, 0xe4,	0x5b, 0xb2, 0xf9, 0xd4, 0x75, 0xe6, 0xf2, 0x01,
			0x90, 0x16, 0x72, 0x06, 0xe6, 0xc0, 0x89, 0x59,	0x4c, 0xce, 0xdd, 0xc2, 0x3c, 0x90, 0xde, 0xce,
			0x36, 0x91, 0xc7, 0xda, 0xf5, 0x7d, 0xb0, 0x1f,	0xfc, 0xb6, 0x39, 0x54, 0x43, 0x09, 0x0d, 0x75,
			0xbc, 0xb2, 0xcd, 0xf0, 0x0d, 0xc3, 0x9b, 0x1b,	0xf3, 0x90, 0xeb, 0x5a, 0xe3, 0x06, 0xac, 0xc3,
			0xda, 0x13, 0x1e, 0x1a, 0x90, 0x30, 0xc4, 0xbf,	0xbc, 0x40, 0x0f, 0xd8, 0xe4, 0x11, 0xed, 0xc2,
			0x79, 0xc5, 0xcd, 0xee, 0xce, 0x39, 0xed, 0x2b,	0xdf, 0xcf, 0xa3, 0x01, 0xf7, 0x97, 0xbf, 0xef,
			0x99, 0xaf, 0xa3, 0xc6, 0x51, 0x15, 0xfa, 0x64,	0x8f, 0xb9, 0xbe, 0xe3, 0xd8, 0x09, 0x8d, 0xd9,
			0x32, 0xd8, 0x7d, 0xbf, 0x8b, 0x77, 0x61, 0x4f,	0x97, 0x5c, 0x50, 0x76, 0xf6, 0x84, 0x2d, 0x72,
			0x35, 0xfb, 0x18, 0x25, 0x06, 0xdc, 0x9c, 0x6b,	0x13, 0xf5, 0x96, 0xe2, 0xa9, 0xbf, 0x42, 0xde,
			0xf7, 0x41, 0xe2, 0xf2, 0xab, 0xa5, 0x35, 0x7d,	0xec, 0xe4, 0xd0, 0x8a, 0xcf, 0x6b, 0x3e, 0x52,
			0x3d, 0x53, 0x11, 0x15, 0xda, 0xb0, 0x44, 0x5e,	0xb2, 0x36, 0x30, 0x5e, 0x1f, 0x89, 0x39, 0xed,
			0x0f, 0x61, 0x41, 0xba, 0xb4, 0x60, 0xc1, 0x2b,	0xf7, 0x11, 0x46, 0xc1, 0xe5, 0xdb, 0x8c, 0xfc,
			0x4d, 0xcc, 0xee, 0x16, 0x1a, 0x0e, 0xef, 0xfb,	0x98, 0x7f, 0xa9, 0xb6, 0xd2, 0x1e, 0x46, 0x18,
			0x61, 0xec, 0x27, 0xee, 0xbb, 0x9a, 0x30, 0xc2,	0x89, 0x99, 0xd9, 0x30, 0xe0, 0xc0, 0xdb, 0xcd,
			0xeb, 0xf8, 0x5b, 0xc0, 0x55, 0xbb, 0x3e, 0xf5,	0x06, 0xa7, 0x0a, 0x38, 0x02, 0x77, 0x59, 0xaf,
			0x8f, 0xf5, 0x3b, 0x8f, 0xd8, 0x0a, 0xfa, 0x43,	0x05, 0xd0, 0xcd, 0xa2, 0xed, 0x70, 0xcd, 0x12,
			0xa3, 0xd3, 0x8a, 0xa9, 0xee, 0xa0, 0x00, 0x2f,	0x72, 0x88, 0xa0, 0xa7, 0xd3, 0x3f, 0xa2, 0xaf,
			0xc2, 0xa9, 0x66, 0x7a, 0xae, 0x02, 0x5c, 0x26,	0x9b, 0xe6, 0xc6, 0x87, 0xef, 0x76, 0x35, 0xd9,
			0xa1, 0xcd, 0xd1, 0xcb, 0xad, 0x61, 0xd0, 0xc0,	0xb1, 0xdc, 0x68, 0x6d, 0x45, 0x6d, 0x4a, 0x81,
			0x6b, 0xa1, 0xe0, 0x53, 0xfe, 0x33, 0xac, 0x8f,	0x4c, 0xb6, 0xb8, 0x96, 0x6e, 0xea, 0xf2, 0x51,
			0x15, 0xc5, 0xcf, 0x9c, 0x98, 0xca, 0xcc, 0xb7,	0xbc, 0x4e, 0x27, 0x27, 0x85, 0x77, 0x3d, 0x37,
			0x58, 0xe3, 0x10, 0xb9, 0x61, 0x4f, 0x2a, 0x5d,	0x2a, 0x30, 0xfa, 0xd9, 0x8f, 0xf8, 0x9d, 0x60,
			0xf0, 0x83, 0x64, 0x6e, 0x9c, 0x55, 0xaf, 0x42,	0xc0, 0x50, 0xf5, 0xd7, 0x79, 0xac, 0x15, 0x8c,
			0x83, 0x7c, 0xa8, 0x10, 0xbe, 0xe8, 0x77, 0xb5,	0xee, 0x6f, 0xc9, 0x0f, 0xc9, 0x8f, 0x78, 0xc5,
			0x72, 0x4c, 0x40, 0xda, 0xee, 0x1d, 0x02, 0xe7,	0x71, 0xa9, 0xb6, 0x97, 0x22, 0x2e, 0xe5, 0x4b,
			0xdb, 0x61, 0xb6, 0x04, 0xc8, 0xcc, 0x98, 0x13,	0x8a, 0x0a, 0x43, 0xfd, 0xb1, 0x8d, 0xe1, 0x81,
			0x50, 0xf6, 0xf8, 0x5d, 0x75, 0x1d, 0x0b, 0x40,	0x3e, 0x01, 0x09, 0x54, 0x4e, 0xb7, 0xc9, 0x02,
			0xc4, 0x8b, 0xda, 0x64, 0xd1, 0x74, 0xd5, 0x58,	0x5d, 0x75, 0x54, 0x3a, 0x35, 0x01, 0x93, 0xc3,
			0x1c, 0x60, 0x59, 0x5b, 0x68, 0xed, 0x58, 0xee,	0xd1, 0xd3, 0xa8, 0x65, 0x36, 0xf3, 0xcb, 0xd4,
			0x7e, 0x6d, 0x89, 0x86, 0x90, 0x32, 0x80, 0x1c,	0x5a, 0x2a, 0xac, 0xa8, 0x92, 0x9a, 0xdc, 0x17,
			0x16, 0x5d, 0x8c, 0xe3, 0xad, 0x66, 0x10, 0x3f,	0x68, 0xa1, 0xf5, 0x29, 0x93, 0x4b, 0xb3, 0xd6,
			0x18, 0x6a, 0x36, 0xb5, 0x8b, 0xbb, 0x05, 0xc0,	0x53, 0xe3, 0xf7, 0x56, 0xbd, 0xa4, 0xba, 0xf5,
			0xa1, 0xbd, 0xb8, 0x4c, 0xe7, 0x3d, 0x62, 0x66,	0xcb, 0xc7, 0xd7, 0x60, 0x1a, 0xe9, 0x10, 0x6a,
			0xd8, 0xb1, 0xc8, 0x10, 0x87, 0x33, 0x26, 0x6b,	0xd3, 0x9d, 0x52, 0x28, 0x25, 0x30, 0x06, 0x1c,
			0x1f, 0x9a, 0xb2, 0x79, 0xe3, 0xb6, 0x0b, 0x13,	0xf7, 0xb5, 0x7d, 0x97, 0xa6, 0x60, 0x5e, 0x5f,
			0xcc, 0x44, 0xd2, 0x51, 0x9e, 0xbd, 0x5e, 0xfc,	0xdd, 0x89, 0xad, 0xfa, 0x9f, 0xf6, 0xdf, 0x47,
			0x0e, 0x9c, 0xc3, 0x13, 0xc8, 0xe5, 0x33, 0xe6,	0x29, 0x4a, 0x94, 0x2d, 0xe0, 0x56, 0x92, 0xd6,
			0x7a, 0x92, 0x6c, 0x6f, 0xff, 0x94, 0x13, 0x58,	0x43, 0x5e, 0x23, 0x65, 0xc8, 0xdf, 0xee, 0x73,
			0x6a, 0x5b, 0x04, 0xe3, 0xdb, 0x7a, 0xa1, 0x43,	0x09, 0x14, 0xd4, 0x0f, 0x5c, 0x76, 0xbd, 0x7e,
			0xaa, 0x00, 0x64, 0x09, 0x00, 0x3b, 0x7d, 0x77,	0xfd, 0x17, 0x57, 0xfa, 0xfe, 0x0b, 0x40, 0xfb,
		},
	},
};

	
WrappedMaterial rsa2048underaes192
{
	"RSA2048 wrapped under AES192 using CKM_AES_CBC_PAD",
	CKK_AES,		// wrapping key type
	CKM_AES_CBC_PAD,	// mechanism
	CKO_PRIVATE_KEY,	// wrapped object class
	CKK_RSA,		// wrapped key type
	{
		{		// key
			0xe3, 0x38, 0xb0, 0xa3, 0xa9, 0x49, 0x66, 0xf3, 0x1b, 0xe8, 0xc6, 0x3c, 0x40, 0x9d, 0xe3, 0xf2, 
			0xaa, 0x07, 0xae, 0x37, 0xea, 0x28, 0xbc, 0xde,
		},
		{		// IV
			0x48, 0x5c, 0x96, 0x7e, 0x41, 0x47, 0xcf, 0x17, 0x35, 0xa3, 0x12, 0xf1, 0x76, 0x2e, 0x12, 0xdc, 
		},
		{	        // wrapped RSA key
			0x90, 0xe2, 0x9d, 0x99, 0xcb, 0xc3, 0xfe, 0x34, 0x4f, 0xe1, 0x73, 0x7b, 0xc2, 0xde, 0xed, 0xf3,
			0x13, 0x4c, 0x0e, 0xa1, 0x2d, 0xfe, 0x82, 0xf3, 0x69, 0x04, 0xad, 0x8e, 0xcc, 0x23, 0x35, 0x0a,
			0x97, 0x8b, 0xc4, 0x9d, 0x32, 0x97, 0x2c, 0x32, 0x43, 0xc9, 0x07, 0x99, 0x78, 0x7c, 0x6c, 0xc6,
			0xa3, 0xd4, 0xf2, 0x88, 0xf8, 0xb2, 0x77, 0xcb, 0xab, 0xdf, 0x64, 0x46, 0x00, 0xd1, 0x6e, 0x47,
			0x90, 0xc8, 0x21, 0x5e, 0x40, 0x5c, 0xe3, 0x37, 0x74, 0x74, 0x00, 0x91, 0x78, 0xb6, 0x8a, 0x0b,
			0x5e, 0xf9, 0x01, 0x66, 0xd6, 0xcb, 0x44, 0x68, 0xfe, 0xcc, 0x1a, 0x9c, 0x38, 0x0f, 0x4a, 0x4a,
			0x37, 0xee, 0xa4, 0x38, 0xe6, 0x9f, 0xa9, 0xfa, 0xd1, 0x69, 0x28, 0x5a, 0xf9, 0x8b, 0xc1, 0xc9,
			0x32, 0x8f, 0x9b, 0x79, 0xf5, 0x7d, 0xe3, 0xdb, 0x16, 0xb3, 0xc0, 0x84, 0xea, 0xb6, 0xe5, 0x7e,
			0xec, 0x62, 0x51, 0xa9, 0x13, 0x54, 0x35, 0x66, 0x4b, 0x88, 0xd3, 0xf6, 0xd7, 0x1a, 0x82, 0xb9,
			0x7c, 0x36, 0xd7, 0x44, 0x74, 0xff, 0x3d, 0x8d, 0x98, 0xb4, 0xe8, 0x33, 0x7a, 0xb4, 0x04, 0x6a,
			0xeb, 0x7d, 0xfd, 0x0e, 0xd3, 0x83, 0x7f, 0xf5, 0x5a, 0x8b, 0x8a, 0x76, 0xbf, 0x7a, 0xd6, 0x48,
			0x01, 0x32, 0xa9, 0x5f, 0x04, 0x14, 0x55, 0x8c, 0xb5, 0xf4, 0x1d, 0xc1, 0xf0, 0x8e, 0x9c, 0x47,
			0x17, 0xdd, 0x26, 0xc1, 0xbe, 0x41, 0xdd, 0xd3, 0xd9, 0x54, 0x78, 0x0e, 0xab, 0x0a, 0x99, 0x99,
			0xc4, 0x9e, 0x84, 0x88, 0x90, 0x00, 0xaa, 0xb5, 0xdc, 0x44, 0xd6, 0x1f, 0xfd, 0x6c, 0x99, 0x15,
			0x26, 0x1a, 0x2e, 0x3d, 0x65, 0x9c, 0xb8, 0x53, 0xda, 0xa7, 0x51, 0xe8, 0x6c, 0x89, 0xa9, 0x31,
			0x49, 0x81, 0xfd, 0x52, 0x76, 0x84, 0xb0, 0xda, 0x91, 0xc4, 0x19, 0x75, 0x6b, 0xb9, 0xfa, 0xce,
			0xb9, 0x3c, 0x9f, 0x0e, 0xc0, 0xc1, 0x73, 0xbc, 0xc7, 0x82, 0x1a, 0x98, 0xcb, 0xcc, 0x55, 0xd6,
			0xf9, 0x3f, 0xfc, 0x09, 0xa4, 0xad, 0x42, 0x8f, 0xb4, 0x25, 0xc8, 0x5e, 0x3c, 0xcf, 0xb2, 0x0f,
			0x6d, 0xdc, 0x74, 0x00, 0xd6, 0x66, 0x8f, 0x2a, 0xcd, 0xd7, 0x88, 0xba, 0x4a, 0x93, 0x28, 0x8b,
			0xbf, 0x23, 0x9f, 0xe7, 0x02, 0xc2, 0x71, 0x79, 0x87, 0xce, 0xb3, 0x70, 0xef, 0x44, 0x3c, 0xae,
			0xe8, 0x2a, 0x8f, 0xf1, 0xd4, 0x62, 0x7f, 0x1e, 0x98, 0x22, 0xef, 0xae, 0x2b, 0x6d, 0x15, 0xba,
			0x4d, 0x32, 0x19, 0x6d, 0x55, 0x2f, 0x57, 0x81, 0xd6, 0x06, 0x72, 0x53, 0x8f, 0x1b, 0x03, 0xdd,
			0xe4, 0x5c, 0x53, 0x87, 0xa9, 0x1b, 0xc7, 0x54, 0xc6, 0x7b, 0xdc, 0x26, 0x5f, 0x96, 0xa4, 0x7e,
			0x96, 0x70, 0xb5, 0x51, 0xf3, 0xce, 0x1a, 0x79, 0xb7, 0x30, 0x7d, 0xa4, 0x9e, 0xd4, 0x0e, 0xc1,
			0xfc, 0xf0, 0xb7, 0x7d, 0x19, 0x14, 0x80, 0x76, 0xd8, 0x0c, 0x8b, 0x4a, 0x1e, 0x75, 0x29, 0x8f,
			0x93, 0xda, 0xe9, 0x36, 0x8e, 0x46, 0x52, 0x34, 0x94, 0x11, 0x99, 0x85, 0x01, 0x02, 0x1c, 0xbc,
			0xd9, 0xdc, 0xf5, 0xdc, 0xe1, 0x0a, 0xf7, 0x8e, 0x39, 0x05, 0xe8, 0x6c, 0xe2, 0xb0, 0x17, 0xf4,
			0xdc, 0xf2, 0xf6, 0x73, 0x05, 0xd4, 0xd5, 0x37, 0x1c, 0x98, 0x53, 0x13, 0xf1, 0xe9, 0x72, 0x0e,
			0xd1, 0x6e, 0x6f, 0xd2, 0x89, 0xd0, 0x8c, 0x07, 0xef, 0xf3, 0xd7, 0x54, 0x04, 0x97, 0x23, 0x74,
			0xae, 0x2c, 0x7d, 0x4d, 0x76, 0x78, 0x97, 0xb9, 0x68, 0xc5, 0xd9, 0x49, 0xc1, 0x55, 0xcd, 0x30,
			0x39, 0xdc, 0x15, 0x12, 0x9e, 0x41, 0xde, 0xee, 0xf2, 0xa1, 0xac, 0x37, 0x33, 0xff, 0xaf, 0x96,
			0x0f, 0xa5, 0xd8, 0xf8, 0x3a, 0x6b, 0x9e, 0x2c, 0x1e, 0x59, 0x27, 0xf5, 0xee, 0x63, 0x8f, 0xc7,
			0xf0, 0x20, 0x85, 0x17, 0x0a, 0x3f, 0xf0, 0x54, 0xfc, 0xde, 0xb8, 0xeb, 0x62, 0x36, 0x79, 0x96,
			0x54, 0xcc, 0xb3, 0xcc, 0x46, 0x95, 0x06, 0x78, 0xe0, 0x4f, 0xe5, 0xb4, 0xb9, 0x72, 0x8e, 0xd5,
			0xab, 0x3f, 0xd5, 0xec, 0xac, 0xb1, 0x33, 0xf8, 0x9e, 0x0c, 0x05, 0xe8, 0x83, 0x7f, 0x4a, 0xd5,
			0x88, 0x35, 0xbc, 0x59, 0xd3, 0xff, 0xbd, 0x8e, 0xff, 0x19, 0xf9, 0xd1, 0xd3, 0x0e, 0x42, 0x5a,
			0x77, 0x4a, 0xf7, 0xe3, 0x87, 0xf8, 0xa9, 0x3f, 0xf9, 0x3b, 0xc7, 0xf7, 0x64, 0x1b, 0xda, 0x5a,
			0xd9, 0x7c, 0xbc, 0x90, 0xb9, 0xcb, 0x17, 0xc5, 0x0b, 0x12, 0x0f, 0x70, 0x9e, 0xdc, 0x32, 0x3b,
			0xe6, 0x8c, 0x31, 0x0a, 0x7b, 0x31, 0xb5, 0xfe, 0x88, 0x09, 0x83, 0x8b, 0xc7, 0xd7, 0x31, 0x75,
			0x9c, 0x8b, 0x79, 0x7c, 0x6a, 0x80, 0x50, 0xa2, 0xd8, 0x68, 0x4b, 0xa5, 0xce, 0x89, 0x8d, 0x7d,
			0x14, 0x04, 0xed, 0x8a, 0x87, 0x80, 0xf8, 0x95, 0x9f, 0x03, 0x96, 0xda, 0xb3, 0xf5, 0xd5, 0x14,
			0x93, 0x7e, 0x8f, 0x16, 0xbd, 0x20, 0x4d, 0xfb, 0x8b, 0xca, 0x39, 0x1c, 0x1d, 0x85, 0x7b, 0x02,
			0xb2, 0x7e, 0xa9, 0x88, 0x37, 0xa3, 0xfb, 0xe8, 0x41, 0xeb, 0xa2, 0x68, 0x49, 0x40, 0x1e, 0x89,
			0xe5, 0xab, 0xbd, 0xc8, 0x0b, 0x6b, 0x80, 0xb8, 0x46, 0x1c, 0xe7, 0xe8, 0xf6, 0xd5, 0x9f, 0x23,
			0x41, 0x7f, 0x6a, 0x02, 0xf6, 0x27, 0x5c, 0x49, 0xa2, 0x42, 0xe1, 0xdb, 0xa9, 0xaf, 0x12, 0x68,
			0xf7, 0xf9, 0x3e, 0x01, 0xaa, 0xa4, 0xaa, 0x8d, 0x60, 0xab, 0x65, 0x36, 0xf3, 0x71, 0x24, 0xa5,
			0x6e, 0x50, 0x0b, 0xc0, 0xbc, 0x5d, 0x4e, 0xcd, 0xd7, 0xab, 0x21, 0x71, 0x4f, 0xb0, 0xda, 0xe0,
			0xee, 0x81, 0x9a, 0xa8, 0xa8, 0x4e, 0x23, 0x17, 0x3a, 0xed, 0x55, 0x54, 0x9c, 0xed, 0xc5, 0xcc,
			0xf6, 0x67, 0x3c, 0xfa, 0xc6, 0xa2, 0xc5, 0x35, 0x22, 0x19, 0x40, 0xa7, 0x4d, 0x96, 0x4b, 0x21,
			0x74, 0x27, 0x49, 0x81, 0xaf, 0x59, 0xaf, 0xbe, 0x69, 0x51, 0xf2, 0xe2, 0xe5, 0xf2, 0xdf, 0x3d,
			0x60, 0xbb, 0xc9, 0xb3, 0x15, 0x71, 0x75, 0x1a, 0x4b, 0x18, 0x0e, 0x63, 0x70, 0xc0, 0x5a, 0x72,
			0xc4, 0x44, 0xa1, 0xe6, 0x72, 0xe2, 0x58, 0xac, 0xf8, 0x34, 0xe9, 0x88, 0xac, 0x81, 0xb4, 0x61,
			0xca, 0x71, 0xcf, 0xf9, 0x50, 0xf7, 0x60, 0x95, 0x0b, 0xb1, 0x2f, 0xf8, 0xfc, 0x23, 0x3a, 0x91,
			0xdc, 0xae, 0x3f, 0x30, 0xbf, 0x20, 0xe9, 0x1d, 0x08, 0x43, 0xdb, 0xc2, 0x8d, 0xf1, 0x2d, 0xde,
			0x89, 0xf9, 0xaa, 0x37, 0xc6, 0xfa, 0x0c, 0x27, 0x41, 0x89, 0x02, 0x80, 0x47, 0xdd, 0x00, 0x4a,
			0xb8, 0x66, 0xb5, 0xdc, 0x3a, 0x15, 0x01, 0xec, 0x01, 0x15, 0x90, 0xda, 0xd1, 0xfb, 0x9e, 0x02,
			0x47, 0x3d, 0xe8, 0x81, 0xd7, 0xf4, 0x3a, 0x81, 0x71, 0x1b, 0xcb, 0xea, 0x6e, 0x5f, 0xa2, 0xd0,
			0xd5, 0x60, 0x8c, 0x71, 0x71, 0x8a, 0xa7, 0x8a, 0xf1, 0x62, 0x1d, 0xf4, 0x57, 0xa0, 0xb1, 0x5c,
			0x69, 0x07, 0x35, 0xb4, 0xeb, 0x3d, 0x7a, 0x99, 0x1b, 0x8d, 0xee, 0xcb, 0xfe, 0x00, 0xdd, 0xc3,
			0x4f, 0x43, 0x95, 0xd5, 0x0b, 0x56, 0x9c, 0x09, 0x0f, 0x05, 0x1f, 0x66, 0xa7, 0x2d, 0xae, 0xf2,
			0x39, 0x1c, 0xdd, 0xed, 0xa5, 0x92, 0x15, 0xff, 0xd8, 0x9e, 0x6c, 0x51, 0x40, 0x79, 0x6f, 0x08,
			0xdf, 0xeb, 0x18, 0xa9, 0xbe, 0xb6, 0xbd, 0xac, 0xe3, 0x54, 0xa5, 0x18, 0xaf, 0x19, 0x61, 0x09,
			0xb8, 0x30, 0xae, 0x07, 0x31, 0xee, 0x19, 0xfc, 0xc5, 0x69, 0xad, 0x0c, 0xa9, 0x0f, 0x72, 0x86,
			0xe8, 0xec, 0x5c, 0x6b, 0x9a, 0x4d, 0xa0, 0xde, 0x0c, 0x17, 0xe9, 0x6a, 0xc9, 0x4a, 0xb5, 0x3d,
			0x90, 0x20, 0x24, 0x41, 0xea, 0x51, 0x00, 0x0c, 0x86, 0xcc, 0xc9, 0xf0, 0x4d, 0x1b, 0xac, 0x99,
			0x2a, 0xbb, 0xe3, 0x0b, 0xe2, 0x11, 0xb7, 0x55, 0xd1, 0xed, 0xb9, 0x0b, 0xcd, 0x98, 0x0f, 0xd5,
			0x50, 0x96, 0xfc, 0x6e, 0x0c, 0xeb, 0xac, 0x4e, 0xb0, 0x09, 0x90, 0xdf, 0xb2, 0x49, 0xca, 0x2b,
			0x0d, 0xd4, 0x53, 0x53, 0xa6, 0x8b, 0xf3, 0x49, 0xb2, 0x90, 0x30, 0xed, 0xf3, 0xdb, 0x14, 0x40,
			0x3c, 0xef, 0x89, 0xbf, 0x17, 0x1b, 0xb8, 0x56, 0xea, 0x6a, 0x24, 0x7b, 0x72, 0xff, 0xb7, 0xe0,
			0x4a, 0xb0, 0x23, 0x96, 0x31, 0x01, 0x96, 0x16, 0x8a, 0x6c, 0x7d, 0x0d, 0x30, 0x18, 0x3c, 0xbd,
			0x71, 0x5c, 0xe6, 0xcc, 0x01, 0x85, 0x36, 0xaf, 0x33, 0x92, 0x58, 0x7d, 0xe6, 0x52, 0xf2, 0x9c,
			0xb7, 0xe0, 0x92, 0x10, 0x8d, 0x72, 0xec, 0x3e, 0xe8, 0xa2, 0xdb, 0xcf, 0x6f, 0x34, 0xcc, 0xdf,
			0x2c, 0x17, 0x91, 0x10, 0x56, 0xf5, 0xf2, 0x69, 0x1a, 0x8c, 0x20, 0x57, 0x6c, 0x92, 0x61, 0x45,
			0x0b, 0xf3, 0xf4, 0xa4, 0x01, 0x2c, 0x97, 0xaf, 0x52, 0xce, 0x83, 0x29, 0x02, 0xee, 0x6b, 0xbe,
			0x04, 0xac, 0xc9, 0x93, 0x8a, 0xaa, 0x71, 0x8e, 0x56, 0xe8, 0x68, 0x34, 0x89, 0xc0, 0x58, 0xe2,
			0x11, 0x80, 0x6e, 0xe6, 0x55, 0x12, 0x24, 0x72, 0xf6, 0xe9, 0x3a, 0x09, 0x94, 0xda, 0x28, 0xd7,
			0x10, 0xf6, 0x72, 0xe3, 0xf6, 0x0a, 0x5f, 0xdc, 0xee, 0xaf, 0xa3, 0xc3, 0x87, 0xdb, 0x80, 0x69,
		},
	},
};

WrappedMaterial rsa2048underaes256
{
	"RSA2048 wrapped under AES256 using CKM_AES_CBC_PAD",
	CKK_AES,		// wrapping key type
	CKM_AES_CBC_PAD,	// mechanism
	CKO_PRIVATE_KEY,	// wrapped object class
	CKK_RSA,		// wrapped key type
	{
		{		// key
			0xda, 0x77, 0x34, 0x87, 0x98, 0x5c, 0xd3, 0x27, 0xc6, 0x11, 0x1a, 0x83, 0x65, 0x18, 0xb9, 0xe3,
			0x55, 0x64, 0x2f, 0xe4, 0x47, 0x7e, 0xb5, 0xd8, 0x17, 0x54, 0xfa, 0xb9, 0x3b, 0x4b, 0x88, 0x2b,
		},
		{		// IV
			0x0e, 0xee, 0x81, 0x64, 0x0d, 0x6d, 0xe1, 0x6e, 0xca, 0xbb, 0x39, 0x0c, 0xda, 0xda, 0x86, 0xe6, 			
		},
		{	        // wrapped RSA key
			0x86, 0x36, 0x43, 0xcb, 0x0e, 0x85, 0xb7, 0xaa, 0x59, 0x16, 0x92, 0x70, 0x31, 0x06, 0x54, 0x22,
			0xcb, 0xf8, 0x32, 0x2f, 0xfd, 0xec, 0xb6, 0xc8, 0x19, 0x6f, 0x66, 0x50, 0x2d, 0x23, 0x26, 0xe1,
			0x3e, 0xd0, 0x45, 0x49, 0x59, 0xd4, 0x6b, 0xee, 0x6f, 0x89, 0x8f, 0xd2, 0xb6, 0x48, 0x98, 0xa6,
			0x6b, 0xf0, 0x1b, 0xb4, 0x32, 0x89, 0x20, 0x3a, 0x9b, 0xee, 0xbf, 0x5b, 0xed, 0x30, 0x97, 0x68,
			0x3e, 0x77, 0x79, 0xc2, 0xcb, 0x76, 0x41, 0xd4, 0xef, 0xdc, 0xa9, 0xfa, 0x28, 0x07, 0x9d, 0x97,
			0x4e, 0x03, 0x25, 0x90, 0x95, 0x78, 0x63, 0x3a, 0x04, 0x26, 0x48, 0xe1, 0x8d, 0x13, 0xed, 0x31,
			0x4c, 0xd0, 0x09, 0x9f, 0xef, 0xa7, 0x83, 0xe4, 0xaf, 0xaa, 0xc4, 0x0c, 0x05, 0xc4, 0xa4, 0x64,
			0x37, 0x67, 0xf0, 0xbb, 0xcc, 0x76, 0x4c, 0x74, 0x54, 0x7c, 0x9a, 0x8e, 0x8e, 0x11, 0x88, 0xd5,
			0x18, 0xc1, 0x82, 0x78, 0xc1, 0xf0, 0xd1, 0xf8, 0x02, 0xb0, 0x5a, 0x9d, 0xda, 0x15, 0xda, 0xbd,
			0x52, 0x25, 0x00, 0x12, 0xb1, 0xb9, 0x94, 0xcc, 0x16, 0x7b, 0x71, 0x37, 0x3b, 0xbf, 0x68, 0x8b,
			0x01, 0xd8, 0xd0, 0xb7, 0xaf, 0xff, 0x72, 0x78, 0x5f, 0x82, 0x3d, 0xad, 0x89, 0xca, 0x06, 0x1b,
			0xaa, 0x20, 0x73, 0xb4, 0x71, 0x3b, 0x28, 0x87, 0xe3, 0xec, 0x77, 0x9e, 0x69, 0xab, 0xea, 0x89,
			0xb9, 0xa3, 0x55, 0x9a, 0xcd, 0x80, 0x80, 0x07, 0xfb, 0x40, 0x24, 0x9d, 0xdc, 0x4a, 0xa2, 0x06,
			0xf7, 0xef, 0x42, 0x60, 0x22, 0x4b, 0x69, 0x98, 0x68, 0xf6, 0xaf, 0xad, 0x67, 0x9b, 0xba, 0xa0,
			0x26, 0x49, 0x0d, 0x0a, 0x74, 0x82, 0x51, 0xe4, 0xe3, 0x02, 0xbd, 0x04, 0x8f, 0x78, 0x91, 0xe0,
			0x46, 0x80, 0xbf, 0xa9, 0x4e, 0x9f, 0xb1, 0x44, 0xfb, 0x09, 0x14, 0x95, 0x2c, 0x6f, 0xb8, 0x3d,
			0x1d, 0x61, 0x06, 0x4e, 0xde, 0x86, 0x4a, 0x3a, 0x60, 0x1b, 0x69, 0x45, 0x37, 0x1f, 0x68, 0x37,
			0x5e, 0x36, 0x7a, 0x86, 0x8b, 0xc0, 0x06, 0x53, 0xf1, 0x9b, 0xad, 0x81, 0x34, 0xb5, 0x4b, 0xe2,
			0x56, 0x0e, 0x77, 0x26, 0x32, 0x8a, 0x0a, 0x68, 0xe0, 0xbd, 0x23, 0x6c, 0xcd, 0x8e, 0x64, 0x8d,
			0x10, 0x24, 0x3a, 0x37, 0x5b, 0xb1, 0x50, 0xb9, 0x51, 0x5c, 0x7a, 0xa9, 0x7a, 0x12, 0x0d, 0xf1,
			0xce, 0xcd, 0x1f, 0x9e, 0x6e, 0x10, 0x55, 0x16, 0xb3, 0x34, 0xba, 0xcf, 0x22, 0xf3, 0x6c, 0xbc,
			0xf8, 0x9f, 0x97, 0x2d, 0x9a, 0x43, 0xce, 0xc6, 0x9d, 0xad, 0x59, 0x12, 0xdc, 0x35, 0xc0, 0x65,
			0x52, 0x1a, 0x9e, 0x36, 0x97, 0xcb, 0x4e, 0xaf, 0xee, 0x65, 0x61, 0x42, 0x3d, 0x46, 0x74, 0x80,
			0x77, 0x32, 0x23, 0xf6, 0x15, 0xbd, 0x1c, 0x90, 0xc6, 0xc4, 0xc4, 0x15, 0x6e, 0x50, 0xbd, 0xc4,
			0x91, 0xab, 0x73, 0xcf, 0x96, 0xb4, 0x87, 0xbd, 0x55, 0x28, 0x06, 0xe9, 0xc0, 0x48, 0x66, 0x21,
			0xd8, 0x48, 0x23, 0x13, 0xa4, 0x43, 0x09, 0x58, 0x2d, 0x5e, 0x48, 0x1c, 0x45, 0xfa, 0xb1, 0xd4,
			0x4e, 0x35, 0x6c, 0xc2, 0x8d, 0x35, 0x2e, 0x1c, 0xe3, 0x9a, 0x49, 0xfa, 0x88, 0x15, 0x0e, 0xab,
			0x09, 0xb9, 0x1e, 0x50, 0xe2, 0xd9, 0xcd, 0xb4, 0x3e, 0x72, 0x2d, 0x43, 0x65, 0x73, 0x20, 0x9b,
			0x52, 0x08, 0x3f, 0x50, 0xa4, 0xdc, 0xdc, 0x4f, 0xe2, 0xe8, 0x6a, 0x26, 0xba, 0xc6, 0x22, 0x80,
			0xc1, 0xed, 0x06, 0xdf, 0xd7, 0x0a, 0x97, 0x09, 0x80, 0x6f, 0xc4, 0xcb, 0x35, 0x7a, 0xdd, 0x17,
			0xe4, 0x96, 0x85, 0xd6, 0x63, 0x4e, 0xc0, 0xb3, 0x7b, 0x58, 0xda, 0x79, 0x61, 0x38, 0x58, 0x40,
			0x4d, 0x89, 0x76, 0x7c, 0xa4, 0x6e, 0x9f, 0xeb, 0x15, 0x2b, 0x88, 0x27, 0x1b, 0x7f, 0x8b, 0xea,
			0x13, 0x27, 0x16, 0x9b, 0xac, 0xe4, 0xbe, 0x91, 0x03, 0x10, 0x33, 0xfd, 0x54, 0x59, 0xa9, 0x3d,
			0xea, 0x35, 0x8e, 0xd1, 0xf9, 0x43, 0xab, 0xe7, 0xdc, 0xe7, 0x75, 0x5d, 0x6c, 0x37, 0x18, 0x2f,
			0x22, 0x25, 0xac, 0x81, 0x26, 0x01, 0x79, 0x77, 0x2b, 0xee, 0x06, 0x10, 0xb2, 0x44, 0xdc, 0xa9,
			0x57, 0x34, 0x29, 0xce, 0xc8, 0xb3, 0x0e, 0x0c, 0x68, 0xcd, 0x10, 0x47, 0xa5, 0x83, 0x31, 0x80,
			0xb6, 0x22, 0x12, 0xaa, 0xfb, 0x67, 0x38, 0x48, 0xe0, 0xb7, 0xf7, 0x7a, 0x21, 0x3f, 0x24, 0x8e,
			0x46, 0x88, 0xb4, 0x52, 0xf2, 0xb1, 0x47, 0x13, 0xfe, 0xff, 0xca, 0x83, 0x11, 0x85, 0xad, 0x78,
			0x9a, 0xe4, 0x4c, 0x62, 0xf2, 0x5d, 0x95, 0x36, 0x37, 0xba, 0xba, 0xb1, 0xb1, 0xc6, 0xd0, 0xfa,
			0xa3, 0x52, 0xe4, 0x8b, 0xa3, 0x48, 0xc4, 0xe6, 0xd6, 0xec, 0xdf, 0xe7, 0x1a, 0x13, 0xe9, 0xdc,
			0x39, 0xde, 0x34, 0x6b, 0x24, 0x1f, 0x85, 0x1c, 0x8c, 0x32, 0x33, 0xe0, 0x2a, 0x63, 0xf7, 0x01,
			0xfa, 0xd2, 0x17, 0xe9, 0x8f, 0xb9, 0x03, 0x66, 0xed, 0xbb, 0x68, 0x2e, 0x8a, 0xe5, 0x80, 0x39,
			0x59, 0xff, 0x9d, 0xc2, 0xa9, 0xdd, 0xa0, 0x58, 0x2c, 0x23, 0x88, 0xcd, 0x22, 0xfe, 0x03, 0x35,
			0xda, 0x52, 0xc0, 0x2e, 0x39, 0x3f, 0x35, 0x74, 0xe7, 0x17, 0xfa, 0x10, 0xd2, 0xfe, 0x39, 0x79,
			0x67, 0x56, 0x13, 0x70, 0xc2, 0x57, 0x02, 0xfa, 0xf1, 0x22, 0x1a, 0x0d, 0xd8, 0x08, 0xd3, 0x6d,
			0x36, 0x16, 0xa3, 0xa9, 0x4d, 0x09, 0x21, 0x32, 0x8e, 0x85, 0x8e, 0x7e, 0x6e, 0x10, 0x06, 0x2b,
			0xb1, 0xf7, 0x64, 0x46, 0xa2, 0x33, 0x36, 0x04, 0xee, 0x17, 0x99, 0x1a, 0x23, 0x70, 0x30, 0x48,
			0x66, 0x93, 0x73, 0x50, 0xd7, 0x0b, 0xf0, 0x9e, 0x1c, 0xdd, 0x21, 0x78, 0xdc, 0x3c, 0xac, 0x34,
			0xd9, 0xf3, 0xc2, 0xb9, 0xc9, 0x6b, 0x1f, 0x13, 0x5b, 0x70, 0x7b, 0xa6, 0x97, 0xe2, 0x71, 0xdd,
			0xba, 0xf2, 0x4b, 0x2a, 0x87, 0xb5, 0x7f, 0x33, 0x5d, 0x47, 0x4b, 0x9a, 0xeb, 0xc1, 0x3f, 0xb3,
			0x0b, 0x89, 0xca, 0x1c, 0xf1, 0x63, 0xb7, 0x60, 0x1d, 0xe3, 0x25, 0x8b, 0x9f, 0xaf, 0xc1, 0x27,
			0x20, 0xc2, 0xc9, 0x57, 0x51, 0x78, 0x6d, 0x59, 0x2f, 0xf2, 0xf9, 0xf5, 0xa1, 0x1c, 0x6e, 0x83,
			0x3a, 0x3c, 0x6d, 0x2c, 0xa6, 0xc7, 0x0d, 0x1d, 0xcd, 0x24, 0xf8, 0xf8, 0x67, 0x20, 0xa0, 0xfe,
			0xea, 0xe3, 0x7c, 0x17, 0x9c, 0x60, 0x11, 0x01, 0xd0, 0x47, 0xa6, 0x49, 0x23, 0xf4, 0x84, 0x1a,
			0x5d, 0xde, 0x10, 0x0c, 0xe8, 0xcf, 0x0c, 0x04, 0x6a, 0x01, 0xd2, 0x75, 0x82, 0xf6, 0xda, 0x5f,
			0x22, 0xe8, 0xa6, 0x57, 0x86, 0x99, 0x43, 0xcb, 0x23, 0xb8, 0xb3, 0x05, 0x5f, 0x8a, 0x4a, 0x7f,
			0xbe, 0x1b, 0xed, 0x27, 0xaf, 0x9c, 0xb0, 0x84, 0xec, 0xe1, 0xe7, 0x70, 0xb5, 0xb9, 0xee, 0x9a,
			0xcd, 0xbf, 0x67, 0x0c, 0x74, 0xfa, 0x36, 0xca, 0x05, 0x3d, 0x6e, 0xe3, 0x55, 0x55, 0x2d, 0x0c,
			0x4d, 0xfb, 0xa2, 0x74, 0x33, 0x09, 0xf8, 0xd2, 0x57, 0xdf, 0x10, 0xf0, 0x14, 0x25, 0x90, 0xfb,
			0x66, 0xa7, 0x2e, 0xfb, 0x98, 0x78, 0xcf, 0x34, 0x2b, 0xba, 0x61, 0x25, 0x7e, 0x86, 0x74, 0x2a,
			0x09, 0x49, 0x68, 0x79, 0x4e, 0x25, 0xba, 0x63, 0xde, 0x27, 0xd9, 0x8c, 0x42, 0xd4, 0x48, 0xeb,
			0x15, 0x61, 0xea, 0xbb, 0xde, 0xa1, 0x34, 0xdd, 0x9b, 0xd0, 0xe7, 0x5c, 0x95, 0x20, 0x10, 0x2c,
			0x9e, 0xe9, 0x3f, 0x31, 0xfd, 0x3c, 0x40, 0x65, 0x64, 0x2c, 0xec, 0x4f, 0xf6, 0x96, 0x39, 0x35,
			0x47, 0x68, 0xd5, 0x8e, 0xf9, 0x82, 0xf1, 0xec, 0xb5, 0xdf, 0x4a, 0xfa, 0x9e, 0x2c, 0x87, 0x66,
			0x67, 0xd0, 0xd6, 0x77, 0xbf, 0x6f, 0xed, 0xa1, 0x81, 0x56, 0x77, 0x86, 0xc3, 0x51, 0x20, 0xe4,
			0x83, 0x43, 0x31, 0xaa, 0xac, 0x3f, 0x3c, 0xc5, 0xa9, 0x13, 0xcb, 0x57, 0x74, 0xde, 0xc2, 0xdb,
			0x72, 0xab, 0x61, 0x3b, 0x0f, 0x7b, 0xf6, 0x9f, 0x18, 0x13, 0xa2, 0xe7, 0xef, 0x66, 0x6e, 0x35,
			0x21, 0x0b, 0x1a, 0xbd, 0x2f, 0x0d, 0x85, 0x5d, 0xca, 0x59, 0xb5, 0xb2, 0xa1, 0xe5, 0xf3, 0x44,
			0xe3, 0xcb, 0x7d, 0x84, 0x8b, 0xbb, 0xb0, 0x8e, 0x88, 0x6e, 0x8b, 0xb0, 0x1a, 0xaa, 0xc7, 0x5b,
			0xf0, 0x6f, 0xa6, 0x8b, 0xb0, 0x17, 0x5f, 0x5b, 0x31, 0x02, 0xae, 0x16, 0xea, 0x1a, 0x97, 0x89,
			0x71, 0xc9, 0x5b, 0x67, 0x21, 0xdb, 0xdb, 0xe3, 0x10, 0x96, 0x02, 0x03, 0x3f, 0x84, 0x36, 0x6a,
			0x41, 0x8a, 0x15, 0xb4, 0xdf, 0xf1, 0xd9, 0xb5, 0xdf, 0x97, 0x4f, 0x1a, 0xfa, 0x91, 0x50, 0x16,
			0xe0, 0x3e, 0x7b, 0x66, 0x8b, 0x7d, 0x78, 0x88, 0x1d, 0xd9, 0xcd, 0xeb, 0xd8, 0x73, 0x73, 0x7b,
			0x60, 0x05, 0x35, 0xbc, 0xe3, 0xa3, 0xa8, 0x85, 0x9d, 0x33, 0xe9, 0x16, 0x46, 0xe1, 0x4b, 0x82,
			0x61, 0x43, 0x5e, 0x4c, 0x74, 0xd8, 0xf8, 0x93, 0x19, 0x0d, 0x52, 0xc2, 0x79, 0xc3, 0x40, 0x1b,
			0xd8, 0x2b, 0x12, 0x9d, 0xd1, 0x12, 0x7b, 0x0a, 0x7f, 0x37, 0x1f, 0xfb, 0x07, 0xfc, 0x31, 0x14,
			0xbf, 0xe4, 0xe2, 0xa0, 0xb2, 0x8b, 0x30, 0x97, 0x66, 0xa4, 0xf4, 0xa7, 0x13, 0x8c, 0xc1, 0x7b,
		}
	}
};

std::vector<WrappedMaterial> aesCBCWrappedKeys {
	rsa2048underaes128,
	rsa2048underaes192,
	rsa2048underaes256,
};

std::vector<WrappedMaterial> desCBCWrappedKeys {
#ifndef WITH_FIPS
	rsa2048underdes56,
	rsa2048underdes112,
#endif
	rsa2048underdes168,
};



CK_RV SymmetricAlgorithmTests::generateGenericKey(CK_SESSION_HANDLE hSession, CK_BBOOL bToken, CK_BBOOL bPrivate, CK_OBJECT_HANDLE &hKey)
{
	CK_MECHANISM mechanism = { CKM_GENERIC_SECRET_KEY_GEN, NULL_PTR, 0 };
	CK_ULONG bytes = 16;
	// CK_BBOOL bFalse = CK_FALSE;
	CK_BBOOL bTrue = CK_TRUE;
	CK_ATTRIBUTE keyAttribs[] = {
		{ CKA_TOKEN, &bToken, sizeof(bToken) },
		{ CKA_PRIVATE, &bPrivate, sizeof(bPrivate) },
		{ CKA_ENCRYPT, &bTrue, sizeof(bTrue) },
		{ CKA_DECRYPT, &bTrue, sizeof(bTrue) },
		{ CKA_WRAP, &bTrue, sizeof(bTrue) },
		{ CKA_UNWRAP, &bTrue, sizeof(bTrue) },
		{ CKA_VALUE_LEN, &bytes, sizeof(bytes) },
	};

	hKey = CK_INVALID_HANDLE;
	return CRYPTOKI_F_PTR( C_GenerateKey(hSession, &mechanism,
			     keyAttribs, sizeof(keyAttribs)/sizeof(CK_ATTRIBUTE),
			     &hKey) );
}

CK_RV SymmetricAlgorithmTests::generateAesKey(CK_SESSION_HANDLE hSession, CK_BBOOL bToken, CK_BBOOL bPrivate, CK_OBJECT_HANDLE &hKey)
{
	CK_MECHANISM mechanism = { CKM_AES_KEY_GEN, NULL_PTR, 0 };
	CK_ULONG bytes = 16;
	// CK_BBOOL bFalse = CK_FALSE;
	CK_BBOOL bTrue = CK_TRUE;
	CK_ATTRIBUTE keyAttribs[] = {
		{ CKA_TOKEN, &bToken, sizeof(bToken) },
		{ CKA_PRIVATE, &bPrivate, sizeof(bPrivate) },
		{ CKA_ENCRYPT, &bTrue, sizeof(bTrue) },
		{ CKA_DECRYPT, &bTrue, sizeof(bTrue) },
		{ CKA_WRAP, &bTrue, sizeof(bTrue) },
		{ CKA_UNWRAP, &bTrue, sizeof(bTrue) },
		{ CKA_VALUE_LEN, &bytes, sizeof(bytes) },
	};

	hKey = CK_INVALID_HANDLE;
	return CRYPTOKI_F_PTR( C_GenerateKey(hSession, &mechanism,
			     keyAttribs, sizeof(keyAttribs)/sizeof(CK_ATTRIBUTE),
			     &hKey) );
}


inline CK_RV SymmetricAlgorithmTests::importDesKey(CK_SESSION_HANDLE hSession, CK_BBOOL bToken, CK_BBOOL bPrivate, CK_OBJECT_HANDLE &hKey, const Bytes & vKeyValue )
{
	return importKey(hSession, bToken, bPrivate, hKey, CKK_DES, vKeyValue);
}

inline CK_RV SymmetricAlgorithmTests::importDes2Key(CK_SESSION_HANDLE hSession, CK_BBOOL bToken, CK_BBOOL bPrivate, CK_OBJECT_HANDLE &hKey, const Bytes & vKeyValue )
{
	return importKey(hSession, bToken, bPrivate, hKey, CKK_DES2, vKeyValue);
}

inline CK_RV SymmetricAlgorithmTests::importDes3Key(CK_SESSION_HANDLE hSession, CK_BBOOL bToken, CK_BBOOL bPrivate, CK_OBJECT_HANDLE &hKey, const Bytes & vKeyValue )
{
	return importKey(hSession, bToken, bPrivate, hKey, CKK_DES3, vKeyValue);
}

inline CK_RV SymmetricAlgorithmTests::importAesKey(CK_SESSION_HANDLE hSession, CK_BBOOL bToken, CK_BBOOL bPrivate, CK_OBJECT_HANDLE &hKey, const Bytes & vKeyValue )
{
	return importKey(hSession, bToken, bPrivate, hKey, CKK_AES, vKeyValue);
}


CK_RV SymmetricAlgorithmTests::importKey(CK_SESSION_HANDLE hSession, CK_BBOOL bToken, CK_BBOOL bPrivate, CK_OBJECT_HANDLE & hKey, const CK_KEY_TYPE keyType, const Bytes & vKeyValue )
{
	// CK_BBOOL bFalse = CK_FALSE;
	CK_BBOOL bTrue = CK_TRUE;
	CK_OBJECT_CLASS secretClass = CKO_SECRET_KEY;

	CK_ATTRIBUTE attribs[] {
		{ CKA_TOKEN, &bToken, sizeof(bToken) },
		{ CKA_PRIVATE, &bPrivate, sizeof(bPrivate) },
		{ CKA_CLASS, &secretClass, sizeof(secretClass) },
		{ CKA_KEY_TYPE, const_cast<CK_KEY_TYPE *>(&keyType), sizeof(keyType) },
		{ CKA_WRAP, &bTrue, sizeof(bTrue) },
		{ CKA_UNWRAP, &bTrue, sizeof(bTrue) },
		{ CKA_VALUE, const_cast<CK_BYTE_PTR>(vKeyValue.data()), vKeyValue.size() }
	};

	hKey = CK_INVALID_HANDLE;
	return CRYPTOKI_F_PTR( C_CreateObject(hSession, attribs, sizeof(attribs)/sizeof(CK_ATTRIBUTE), &hKey) );
}


#ifndef WITH_FIPS
CK_RV SymmetricAlgorithmTests::generateDesKey(CK_SESSION_HANDLE hSession, CK_BBOOL bToken, CK_BBOOL bPrivate, CK_OBJECT_HANDLE &hKey)
{
	CK_MECHANISM mechanism = { CKM_DES_KEY_GEN, NULL_PTR, 0 };
	// CK_BBOOL bFalse = CK_FALSE;
	CK_BBOOL bTrue = CK_TRUE;
	CK_ATTRIBUTE keyAttribs[] = {
		{ CKA_TOKEN, &bToken, sizeof(bToken) },
		{ CKA_PRIVATE, &bPrivate, sizeof(bPrivate) },
		{ CKA_ENCRYPT, &bTrue, sizeof(bTrue) },
		{ CKA_DECRYPT, &bTrue, sizeof(bTrue) },
	};

	hKey = CK_INVALID_HANDLE;
	return CRYPTOKI_F_PTR( C_GenerateKey(hSession, &mechanism,
			     keyAttribs, sizeof(keyAttribs)/sizeof(CK_ATTRIBUTE),
			     &hKey) );
}

CK_RV SymmetricAlgorithmTests::generateDes2Key(CK_SESSION_HANDLE hSession, CK_BBOOL bToken, CK_BBOOL bPrivate, CK_OBJECT_HANDLE &hKey)
{
	CK_MECHANISM mechanism = { CKM_DES2_KEY_GEN, NULL_PTR, 0 };
	// CK_BBOOL bFalse = CK_FALSE;
	CK_BBOOL bTrue = CK_TRUE;
	CK_ATTRIBUTE keyAttribs[] = {
		{ CKA_TOKEN, &bToken, sizeof(bToken) },
		{ CKA_PRIVATE, &bPrivate, sizeof(bPrivate) },
		{ CKA_ENCRYPT, &bTrue, sizeof(bTrue) },
		{ CKA_DECRYPT, &bTrue, sizeof(bTrue) },
	};

	hKey = CK_INVALID_HANDLE;
	return CRYPTOKI_F_PTR( C_GenerateKey(hSession, &mechanism,
			     keyAttribs, sizeof(keyAttribs)/sizeof(CK_ATTRIBUTE),
			     &hKey) );
}
#endif

CK_RV SymmetricAlgorithmTests::generateDes3Key(CK_SESSION_HANDLE hSession, CK_BBOOL bToken, CK_BBOOL bPrivate, CK_OBJECT_HANDLE &hKey)
{
	CK_MECHANISM mechanism = { CKM_DES3_KEY_GEN, NULL_PTR, 0 };
	// CK_BBOOL bFalse = CK_FALSE;
	CK_BBOOL bTrue = CK_TRUE;
	CK_ATTRIBUTE keyAttribs[] = {
		{ CKA_TOKEN, &bToken, sizeof(bToken) },
		{ CKA_PRIVATE, &bPrivate, sizeof(bPrivate) },
		{ CKA_ENCRYPT, &bTrue, sizeof(bTrue) },
		{ CKA_DECRYPT, &bTrue, sizeof(bTrue) },
	};

	hKey = CK_INVALID_HANDLE;
	return CRYPTOKI_F_PTR( C_GenerateKey(hSession, &mechanism,
			     keyAttribs, sizeof(keyAttribs)/sizeof(CK_ATTRIBUTE),
			     &hKey) );
}

void SymmetricAlgorithmTests::encryptDecrypt(
		const CK_MECHANISM mechanism,
		const size_t blockSize,
		const CK_SESSION_HANDLE hSession,
		const CK_OBJECT_HANDLE hKey,
		const size_t messageSize,
		const bool isSizeOK)
{
	class PartSize {// class to get random size for part
	private:        // we want to know for sure that no part length is causing any problem.
		const int blockSize;
		const unsigned* pRandom;// point to memory with random data. We are using the data to be encrypted.
		const unsigned* pBack;// point to memory where random data ends.
		int current;// the current size.
	public:
		PartSize(
				const int _blockSize,
				const std::vector<CK_BYTE>* pvData) :
					blockSize(_blockSize),
					pRandom((const unsigned*)&pvData->front()),
					pBack((const unsigned*)&pvData->back()),
					current(blockSize*4){};
		int getCurrent() {// current part size
			return current;
		}
		int getNext() {// get next part size.
			// Check if we do not have more random data
			if ((pRandom+sizeof(unsigned)-1) > pBack) {
				current = blockSize*4;
				return current;
			}
			const unsigned random(*(pRandom++));
			// Bit shift to handle 32- and 64-bit systems.
			// Just want a simple random part length,
			// not a perfect random number (bit shifting will
			// give some loss of precision).
			current = ((unsigned long)random >> 20)*blockSize*0x100/(UINT_MAX >> 20) + 1;
			//std::cout << "New random " << std::hex << random << " current " << std::hex << std::setfill('0') << std::setw(4) << current << " block size " << std::hex << blockSize << std::endl;
			return current;
		}
	};

	const std::vector<CK_BYTE> vData(messageSize);
	std::vector<CK_BYTE> vEncryptedData;
	std::vector<CK_BYTE> vEncryptedDataParted;
	PartSize partSize(blockSize, &vData);

	CPPUNIT_ASSERT_EQUAL( (CK_RV)CKR_OK, CRYPTOKI_F_PTR( C_GenerateRandom(hSession, (CK_BYTE_PTR)&vData.front(), messageSize) ) );

	CK_MECHANISM_PTR pMechanism((CK_MECHANISM_PTR)&mechanism);
	if (pMechanism->pParameter == NULL_PTR) {
		pMechanism->pParameter = (CK_VOID_PTR)&vData.front();
		pMechanism->ulParameterLen = blockSize;
	}

	// Single-part encryption
	CPPUNIT_ASSERT_EQUAL( (CK_RV)CKR_OK, CRYPTOKI_F_PTR( C_EncryptInit(hSession,pMechanism,hKey) ) );
	{
		CK_ULONG ulEncryptedDataLen;
		const CK_RV rv( CRYPTOKI_F_PTR( C_Encrypt(hSession,(CK_BYTE_PTR)&vData.front(),messageSize,NULL_PTR,&ulEncryptedDataLen) ) );
		if ( isSizeOK ) {
			CPPUNIT_ASSERT_EQUAL( (CK_RV)CKR_OK, rv );
			vEncryptedData.resize(ulEncryptedDataLen);
			CPPUNIT_ASSERT_EQUAL( (CK_RV)CKR_OK, CRYPTOKI_F_PTR( C_Encrypt(hSession,(CK_BYTE_PTR)&vData.front(),messageSize,&vEncryptedData.front(),&ulEncryptedDataLen) ) );
			vEncryptedData.resize(ulEncryptedDataLen);
		} else {
			CPPUNIT_ASSERT_EQUAL_MESSAGE("C_Encrypt should fail with C_CKR_DATA_LEN_RANGE", (CK_RV)CKR_DATA_LEN_RANGE, rv);
			vEncryptedData = vData;
		}
	}

	// Multi-part encryption
	CPPUNIT_ASSERT_EQUAL( (CK_RV)CKR_OK, CRYPTOKI_F_PTR( C_EncryptInit(hSession,pMechanism,hKey) ) );
	CK_ULONG lPartLen = 0;
	for ( std::vector<CK_BYTE>::const_iterator i(vData.begin()); i<vData.end(); i+= lPartLen) {
		lPartLen = ( i<vData.end()-partSize.getNext() ? partSize.getCurrent() : vData.end()-i );
		CK_ULONG ulEncryptedPartLen;
		CPPUNIT_ASSERT_EQUAL( (CK_RV)CKR_OK, CRYPTOKI_F_PTR( C_EncryptUpdate(hSession,(CK_BYTE_PTR)&(*i),lPartLen,NULL_PTR,&ulEncryptedPartLen) ) );
		const size_t oldSize( vEncryptedDataParted.size() );
		vEncryptedDataParted.resize(oldSize+ulEncryptedPartLen);
		CK_BYTE dummy;
		const CK_BYTE_PTR pEncryptedPart( ulEncryptedPartLen>0 ? &vEncryptedDataParted.at(oldSize) : &dummy );
		CPPUNIT_ASSERT_EQUAL( (CK_RV)CKR_OK, CRYPTOKI_F_PTR( C_EncryptUpdate(hSession,(CK_BYTE_PTR)&(*i),lPartLen,pEncryptedPart,&ulEncryptedPartLen) ) );
		vEncryptedDataParted.resize(oldSize+ulEncryptedPartLen);
	}
	{
		CK_ULONG ulLastEncryptedPartLen;
		const CK_RV rv( CRYPTOKI_F_PTR( C_EncryptFinal(hSession,NULL_PTR,&ulLastEncryptedPartLen) ) );
		if ( isSizeOK ) {
			CPPUNIT_ASSERT_EQUAL( (CK_RV)CKR_OK, rv );
			const size_t oldSize( vEncryptedDataParted.size() );
			CK_BYTE dummy;
			vEncryptedDataParted.resize(oldSize+ulLastEncryptedPartLen);
			const CK_BYTE_PTR pLastEncryptedPart( ulLastEncryptedPartLen>0 ? &vEncryptedDataParted.at(oldSize) : &dummy );
			CPPUNIT_ASSERT_EQUAL( (CK_RV)CKR_OK, CRYPTOKI_F_PTR( C_EncryptFinal(hSession,pLastEncryptedPart,&ulLastEncryptedPartLen) ) );
			vEncryptedDataParted.resize(oldSize+ulLastEncryptedPartLen);
		} else {
			CPPUNIT_ASSERT_EQUAL_MESSAGE("C_EncryptFinal should fail with C_CKR_DATA_LEN_RANGE", (CK_RV)CKR_DATA_LEN_RANGE, rv);
			vEncryptedDataParted = vData;
		}
	}

	// Single-part decryption
	CPPUNIT_ASSERT_EQUAL( (CK_RV)CKR_OK, CRYPTOKI_F_PTR( C_DecryptInit(hSession,pMechanism,hKey) ) );

	{
		CK_ULONG ulDataLen;
		const CK_RV rv( CRYPTOKI_F_PTR( C_Decrypt(hSession,&vEncryptedData.front(),vEncryptedData.size(),NULL_PTR,&ulDataLen) ) );
		if ( isSizeOK ) {
			CPPUNIT_ASSERT_EQUAL( (CK_RV)CKR_OK, rv );
			std::vector<CK_BYTE> vDecryptedData(ulDataLen);
			CPPUNIT_ASSERT_EQUAL( (CK_RV)CKR_OK, CRYPTOKI_F_PTR( C_Decrypt(hSession,&vEncryptedData.front(),vEncryptedData.size(),&vDecryptedData.front(),&ulDataLen) ) );
			vDecryptedData.resize(ulDataLen);
			CPPUNIT_ASSERT_MESSAGE("C_Encrypt C_Decrypt does not give the original", vData==vDecryptedData);
		} else {
			CPPUNIT_ASSERT_EQUAL_MESSAGE( "C_Decrypt should fail with CKR_ENCRYPTED_DATA_LEN_RANGE", (CK_RV)CKR_ENCRYPTED_DATA_LEN_RANGE, rv );
		}
	}

	// Multi-part decryption
	CPPUNIT_ASSERT_EQUAL( (CK_RV)CKR_OK, CRYPTOKI_F_PTR( C_DecryptInit(hSession,pMechanism,hKey) ) );
	{
		std::vector<CK_BYTE> vDecryptedData;
		CK_BYTE dummy;
		CK_ULONG ulPartLen = 0;
		for ( std::vector<CK_BYTE>::iterator i(vEncryptedDataParted.begin()); i<vEncryptedDataParted.end(); i+= ulPartLen) {
			ulPartLen = ( i<vEncryptedDataParted.end()- partSize.getNext() ? partSize.getCurrent() : vEncryptedDataParted.end()-i );
			CK_ULONG ulDecryptedPartLen;
			CPPUNIT_ASSERT_EQUAL( (CK_RV)CKR_OK, CRYPTOKI_F_PTR( C_DecryptUpdate(hSession,&(*i),ulPartLen,NULL_PTR,&ulDecryptedPartLen) ) );
			const size_t oldSize( vDecryptedData.size() );
			vDecryptedData.resize(oldSize+ulDecryptedPartLen);
			const CK_BYTE_PTR pDecryptedPart( ulDecryptedPartLen>0 ? &vDecryptedData.at(oldSize) : &dummy );
			CPPUNIT_ASSERT_EQUAL( (CK_RV)CKR_OK, CRYPTOKI_F_PTR( C_DecryptUpdate(hSession,&(*i),ulPartLen,pDecryptedPart,&ulDecryptedPartLen) ) );
			vDecryptedData.resize(oldSize+ulDecryptedPartLen);
		}
		{
			CK_ULONG ulLastPartLen;
			const CK_RV rv( CRYPTOKI_F_PTR( C_DecryptFinal(hSession,NULL_PTR,&ulLastPartLen) ) );
			if ( isSizeOK ) {
				CPPUNIT_ASSERT_EQUAL( (CK_RV)CKR_OK, rv );
				const size_t oldSize( vDecryptedData.size() );
				vDecryptedData.resize(oldSize+ulLastPartLen);
				const CK_BYTE_PTR pLastPart( ulLastPartLen>0 ? &vDecryptedData.at(oldSize) : &dummy );
				CPPUNIT_ASSERT_EQUAL( (CK_RV)CKR_OK, CRYPTOKI_F_PTR( C_DecryptFinal(hSession,pLastPart,&ulLastPartLen) ) );
				vDecryptedData.resize(oldSize+ulLastPartLen);
				CPPUNIT_ASSERT_MESSAGE("C_EncryptUpdate/C_EncryptFinal C_DecryptUpdate/C_DecryptFinal does not give the original", vData==vDecryptedData);
			} else {
				CPPUNIT_ASSERT_EQUAL_MESSAGE( "C_EncryptFinal should fail with CKR_ENCRYPTED_DATA_LEN_RANGE", (CK_RV)CKR_ENCRYPTED_DATA_LEN_RANGE, rv );
			}
		}
	}
}

CK_RV SymmetricAlgorithmTests::generateRsaPrivateKey(CK_SESSION_HANDLE hSession, CK_BBOOL bToken, CK_BBOOL bPrivate, CK_OBJECT_HANDLE &hKey)
{
	CK_MECHANISM mechanism = { CKM_RSA_PKCS_KEY_PAIR_GEN, NULL_PTR, 0 };
	CK_ULONG bits = 1536;
	CK_BYTE pubExp[] = {0x01, 0x00, 0x01};
	CK_BYTE subject[] = { 0x12, 0x34 }; // dummy
	CK_BYTE id[] = { 123 } ; // dummy
	CK_BBOOL bFalse = CK_FALSE;
	CK_BBOOL bTrue = CK_TRUE;
	CK_ATTRIBUTE pubAttribs[] = {
		{ CKA_TOKEN, &bToken, sizeof(bToken) },
		{ CKA_PRIVATE, &bPrivate, sizeof(bPrivate) },
		{ CKA_ENCRYPT, &bFalse, sizeof(bFalse) },
		{ CKA_VERIFY, &bTrue, sizeof(bTrue) },
		{ CKA_WRAP, &bFalse, sizeof(bFalse) },
		{ CKA_MODULUS_BITS, &bits, sizeof(bits) },
		{ CKA_PUBLIC_EXPONENT, &pubExp[0], sizeof(pubExp) }
	};
	CK_ATTRIBUTE privAttribs[] = {
		{ CKA_TOKEN, &bToken, sizeof(bToken) },
		{ CKA_PRIVATE, &bPrivate, sizeof(bPrivate) },
		{ CKA_SUBJECT, &subject[0], sizeof(subject) },
		{ CKA_ID, &id[0], sizeof(id) },
		{ CKA_SENSITIVE, &bTrue, sizeof(bTrue) },
		{ CKA_DECRYPT, &bFalse, sizeof(bFalse) },
		{ CKA_SIGN, &bTrue, sizeof(bTrue) },
		{ CKA_UNWRAP, &bFalse, sizeof(bFalse) },
		{ CKA_SENSITIVE, &bFalse, sizeof(bFalse) },
		{ CKA_EXTRACTABLE, &bTrue, sizeof(bTrue) }
	};

	CK_OBJECT_HANDLE hPub = CK_INVALID_HANDLE;
	hKey = CK_INVALID_HANDLE;
	CK_RV rv;
	rv = CRYPTOKI_F_PTR( C_GenerateKeyPair(hSession, &mechanism,
			       pubAttribs, sizeof(pubAttribs)/sizeof(CK_ATTRIBUTE),
			       privAttribs, sizeof(privAttribs)/sizeof(CK_ATTRIBUTE),
			       &hPub, &hKey) );
	if (hPub != CK_INVALID_HANDLE)
	{
		CRYPTOKI_F_PTR( C_DestroyObject(hSession, hPub) );
	}
	return rv;
}

#ifdef WITH_EDDSA
CK_RV SymmetricAlgorithmTests::generateEDPrivateKey(CK_SESSION_HANDLE hSession, CK_BBOOL bToken, CK_BBOOL bPrivate, CK_OBJECT_HANDLE &hKey, EDCurveParam &curveparam)
{
	CK_MECHANISM mechanism = { CKM_EC_EDWARDS_KEY_PAIR_GEN, NULL_PTR, 0 };
	CK_BYTE subject[] = { 0x12, 0x34 }; // dummy
	CK_BYTE id[] = { 123 } ; // dummy
	CK_BBOOL bTrue = CK_TRUE;
	CK_BBOOL bFalse = CK_FALSE;
	CK_ATTRIBUTE pubAttribs[] = {
		{ CKA_TOKEN, &bToken, sizeof(bToken) },
		{ CKA_PRIVATE, &bPrivate, sizeof(bPrivate) },
		{ CKA_SUBJECT, &subject[0], sizeof(subject) },
		{ CKA_ID, &id[0], sizeof(id) },
		{ CKA_VERIFY, &bTrue, sizeof(bTrue) },
		{ CKA_EC_PARAMS, const_cast<CK_BYTE *>(curveparam.data()), curveparam.size() }
	};
	CK_ATTRIBUTE privAttribs[] = {
		{ CKA_TOKEN, &bToken, sizeof(bToken) },
		{ CKA_PRIVATE, &bPrivate, sizeof(bPrivate) },
		{ CKA_SUBJECT, &subject[0], sizeof(subject) },
		{ CKA_ID, &id[0], sizeof(id) },
		{ CKA_SENSITIVE, &bFalse, sizeof(bFalse) },
		{ CKA_EXTRACTABLE, &bTrue, sizeof(bTrue) },
		{ CKA_SIGN, &bTrue, sizeof(bTrue) },
	};

	CK_OBJECT_HANDLE hPub = CK_INVALID_HANDLE;
	hKey = CK_INVALID_HANDLE;
	CK_RV rv;
	rv = CRYPTOKI_F_PTR( C_GenerateKeyPair(hSession, &mechanism,
			       pubAttribs, sizeof(pubAttribs)/sizeof(CK_ATTRIBUTE),
			       privAttribs, sizeof(privAttribs)/sizeof(CK_ATTRIBUTE),
			       &hPub, &hKey) );
	if (hPub != CK_INVALID_HANDLE)
	{
		CRYPTOKI_F_PTR( C_DestroyObject(hSession, hPub) );
	}
	return rv;
}

#endif

#ifdef WITH_GOST
CK_RV SymmetricAlgorithmTests::generateGostPrivateKey(CK_SESSION_HANDLE hSession, CK_BBOOL bToken, CK_BBOOL bPrivate, CK_OBJECT_HANDLE &hKey)
{
	CK_MECHANISM mechanism = { CKM_GOSTR3410_KEY_PAIR_GEN, NULL_PTR, 0 };
	CK_BYTE param_a[] = { 0x06, 0x07, 0x2a, 0x85, 0x03, 0x02, 0x02, 0x23, 0x01 };
	CK_BYTE param_b[] = { 0x06, 0x07, 0x2a, 0x85, 0x03, 0x02, 0x02, 0x1e, 0x01 };
	CK_BYTE subject[] = { 0x12, 0x34 }; // dummy
	CK_BYTE id[] = { 123 } ; // dummy
	CK_BBOOL bFalse = CK_FALSE;
	CK_BBOOL bTrue = CK_TRUE;
	CK_ATTRIBUTE pubAttribs[] = {
		{ CKA_TOKEN, &bToken, sizeof(bToken) },
		{ CKA_PRIVATE, &bPrivate, sizeof(bPrivate) },
		{ CKA_ENCRYPT, &bFalse, sizeof(bFalse) },
		{ CKA_VERIFY, &bTrue, sizeof(bTrue) },
		{ CKA_WRAP, &bFalse, sizeof(bFalse) },
		{ CKA_GOSTR3410_PARAMS, &param_a[0], sizeof(param_a) },
		{ CKA_GOSTR3411_PARAMS, &param_b[0], sizeof(param_b) }
	};
	CK_ATTRIBUTE privAttribs[] = {
		{ CKA_TOKEN, &bToken, sizeof(bToken) },
		{ CKA_PRIVATE, &bPrivate, sizeof(bPrivate) },
		{ CKA_SUBJECT, &subject[0], sizeof(subject) },
		{ CKA_ID, &id[0], sizeof(id) },
		{ CKA_SENSITIVE, &bTrue, sizeof(bTrue) },
		{ CKA_DECRYPT, &bFalse, sizeof(bFalse) },
		{ CKA_SIGN, &bTrue, sizeof(bTrue) },
		{ CKA_UNWRAP, &bFalse, sizeof(bFalse) },
		{ CKA_SENSITIVE, &bFalse, sizeof(bFalse) },
		{ CKA_EXTRACTABLE, &bTrue, sizeof(bTrue) }
	};

	CK_OBJECT_HANDLE hPub = CK_INVALID_HANDLE;
	hKey = CK_INVALID_HANDLE;
	CK_RV rv;
	rv = CRYPTOKI_F_PTR( C_GenerateKeyPair(hSession, &mechanism,
			       pubAttribs, sizeof(pubAttribs)/sizeof(CK_ATTRIBUTE),
			       privAttribs, sizeof(privAttribs)/sizeof(CK_ATTRIBUTE),
			       &hPub, &hKey) );
	if (hPub != CK_INVALID_HANDLE)
	{
		CRYPTOKI_F_PTR( C_DestroyObject(hSession, hPub) );
	}
	return rv;
}
#endif

void SymmetricAlgorithmTests::aesWrapUnwrapGeneric(CK_MECHANISM_TYPE mechanismType, CK_SESSION_HANDLE hSession, CK_OBJECT_HANDLE hKey)
{
	CK_MECHANISM mechanism = { mechanismType, NULL_PTR, 0 };
	CK_BBOOL bFalse = CK_FALSE;
	CK_BBOOL bTrue = CK_TRUE;
	CK_OBJECT_CLASS secretClass = CKO_SECRET_KEY;
	CK_KEY_TYPE genKeyType = CKK_GENERIC_SECRET;
	CK_BYTE keyPtr[128];
	CK_ULONG keyLen =
		mechanismType == CKM_AES_KEY_WRAP_PAD ? 125UL : 128UL;

	CK_RV rv;
	CK_BYTE ivPtr[16];
	if( mechanismType == CKM_AES_CBC_PAD ) {
		rv = CRYPTOKI_F_PTR( C_GenerateRandom(hSession, ivPtr, sizeof ivPtr) );
		CPPUNIT_ASSERT(rv == CKR_OK);
		mechanism.pParameter = ivPtr;
		mechanism.ulParameterLen = sizeof ivPtr;
	}

	CK_ATTRIBUTE attribs[] = {
		{ CKA_EXTRACTABLE, &bFalse, sizeof(bFalse) },
		{ CKA_CLASS, &secretClass, sizeof(secretClass) },
		{ CKA_KEY_TYPE, &genKeyType, sizeof(genKeyType) },
		{ CKA_TOKEN, &bFalse, sizeof(bFalse) },
		{ CKA_PRIVATE, &bTrue, sizeof(bTrue) },
		{ CKA_SENSITIVE, &bTrue, sizeof(bTrue) }, // Wrapping is allowed even on sensitive objects
		{ CKA_VALUE, keyPtr, keyLen }
	};
	CK_OBJECT_HANDLE hSecret;

	rv = CRYPTOKI_F_PTR( C_GenerateRandom(hSession, keyPtr, keyLen) );
	CPPUNIT_ASSERT(rv == CKR_OK);

	hSecret = CK_INVALID_HANDLE;
	rv = CRYPTOKI_F_PTR( C_CreateObject(hSession, attribs, sizeof(attribs)/sizeof(CK_ATTRIBUTE), &hSecret) );
	CPPUNIT_ASSERT(rv == CKR_OK);
	CPPUNIT_ASSERT(hSecret != CK_INVALID_HANDLE);

	CK_BYTE_PTR wrappedPtr = NULL_PTR;
	CK_ULONG wrappedLen = 0UL;
	CK_ULONG zero = 0UL;
	CK_ULONG rndKeyLen = keyLen;
	if (mechanismType == CKM_AES_KEY_WRAP_PAD)
		rndKeyLen =  (keyLen + 7) & ~7;
	rv = CRYPTOKI_F_PTR( C_WrapKey(hSession, &mechanism, hKey, hSecret, wrappedPtr, &wrappedLen) );
	CPPUNIT_ASSERT(rv == CKR_KEY_UNEXTRACTABLE);
	rv = CRYPTOKI_F_PTR( C_DestroyObject(hSession, hSecret) );
	CPPUNIT_ASSERT(rv == CKR_OK);

	attribs[0].pValue = &bTrue;

	hSecret = CK_INVALID_HANDLE;
	rv = CRYPTOKI_F_PTR( C_CreateObject(hSession, attribs, sizeof(attribs)/sizeof(CK_ATTRIBUTE), &hSecret) );
	CPPUNIT_ASSERT(rv == CKR_OK);
	CPPUNIT_ASSERT(hSecret != CK_INVALID_HANDLE);

	// Estimate wrapped length
	rv = CRYPTOKI_F_PTR( C_WrapKey(hSession, &mechanism, hKey, hSecret, wrappedPtr, &wrappedLen) );
	CPPUNIT_ASSERT(rv == CKR_OK);

	auto wrapOverhead = [mechanismType]() {
		return (mechanismType == CKM_AES_KEY_WRAP || mechanismType == CKM_AES_KEY_WRAP_PAD) ? 8 : 16;
	};
	CPPUNIT_ASSERT(wrappedLen == rndKeyLen + wrapOverhead() );

	wrappedPtr = (CK_BYTE_PTR) malloc(wrappedLen);
	CPPUNIT_ASSERT(wrappedPtr != NULL_PTR);
	rv = CRYPTOKI_F_PTR( C_WrapKey(hSession, &mechanism, hKey, hSecret, wrappedPtr, &wrappedLen) );
	CPPUNIT_ASSERT(rv == CKR_OK);
	CPPUNIT_ASSERT(wrappedLen == rndKeyLen + wrapOverhead() );

	// This should always fail because wrapped data have to be longer than 0 bytes
	zero = 0;
	rv = CRYPTOKI_F_PTR( C_WrapKey(hSession, &mechanism, hKey, hSecret, wrappedPtr, &zero) );
	CPPUNIT_ASSERT(rv == CKR_BUFFER_TOO_SMALL);

	CK_ATTRIBUTE nattribs[] = {
		{ CKA_CLASS, &secretClass, sizeof(secretClass) },
		{ CKA_KEY_TYPE, &genKeyType, sizeof(genKeyType) },
		{ CKA_TOKEN, &bFalse, sizeof(bFalse) },
		{ CKA_PRIVATE, &bTrue, sizeof(bTrue) },
		{ CKA_ENCRYPT, &bFalse, sizeof(bFalse) },
		{ CKA_DECRYPT, &bTrue, sizeof(bTrue) },
		{ CKA_SIGN, &bFalse,sizeof(bFalse) },
		{ CKA_VERIFY, &bTrue, sizeof(bTrue) }
	};
	CK_OBJECT_HANDLE hNew;

	hNew = CK_INVALID_HANDLE;
	rv = CRYPTOKI_F_PTR( C_UnwrapKey(hSession, &mechanism, hKey, wrappedPtr, wrappedLen, nattribs, sizeof(nattribs)/sizeof(CK_ATTRIBUTE), &hNew) );
	CPPUNIT_ASSERT(rv == CKR_OK);
	CPPUNIT_ASSERT(hNew != CK_INVALID_HANDLE);

	free(wrappedPtr);
	wrappedPtr = NULL_PTR;
	rv = CRYPTOKI_F_PTR( C_DestroyObject(hSession, hSecret) );
	CPPUNIT_ASSERT(rv == CKR_OK);
}

void SymmetricAlgorithmTests::aesWrapUnwrapNonModifiableGeneric(CK_MECHANISM_TYPE mechanismType, CK_SESSION_HANDLE hSession, CK_OBJECT_HANDLE hKey)
{
	CK_MECHANISM mechanism = { mechanismType, NULL_PTR, 0 };
	CK_BBOOL bFalse = CK_FALSE;
	CK_BBOOL bTrue = CK_TRUE;
	CK_OBJECT_CLASS secretClass = CKO_SECRET_KEY;
	CK_KEY_TYPE genKeyType = CKK_GENERIC_SECRET;
	CK_BYTE keyPtr[128];
	CK_ULONG keyLen =
		mechanismType == CKM_AES_KEY_WRAP_PAD ? 125UL : 128UL;

	CK_RV rv;
	CK_BYTE ivPtr[16];
	if( mechanismType == CKM_AES_CBC_PAD ) {
		rv = CRYPTOKI_F_PTR( C_GenerateRandom(hSession, ivPtr, sizeof ivPtr) );
		CPPUNIT_ASSERT(rv == CKR_OK);
		mechanism.pParameter = ivPtr;
		mechanism.ulParameterLen = sizeof ivPtr;
	}

	CK_ATTRIBUTE attribs[] = {
		{ CKA_EXTRACTABLE, &bFalse, sizeof(bFalse) },
		{ CKA_CLASS, &secretClass, sizeof(secretClass) },
		{ CKA_KEY_TYPE, &genKeyType, sizeof(genKeyType) },
		{ CKA_TOKEN, &bFalse, sizeof(bFalse) },
		{ CKA_PRIVATE, &bTrue, sizeof(bTrue) },
		{ CKA_SENSITIVE, &bTrue, sizeof(bTrue) }, // Wrapping is allowed even on sensitive objects
		{ CKA_VALUE, keyPtr, keyLen }
	};
	CK_OBJECT_HANDLE hSecret;

	rv = CRYPTOKI_F_PTR( C_GenerateRandom(hSession, keyPtr, keyLen) );
	CPPUNIT_ASSERT(rv == CKR_OK);

	hSecret = CK_INVALID_HANDLE;
	rv = CRYPTOKI_F_PTR( C_CreateObject(hSession, attribs, sizeof(attribs)/sizeof(CK_ATTRIBUTE), &hSecret) );
	CPPUNIT_ASSERT(rv == CKR_OK);
	CPPUNIT_ASSERT(hSecret != CK_INVALID_HANDLE);

	CK_BYTE_PTR wrappedPtr = NULL_PTR;
	CK_ULONG wrappedLen = 0UL;
	CK_ULONG zero = 0UL;
	CK_ULONG rndKeyLen = keyLen;
	if (mechanismType == CKM_AES_KEY_WRAP_PAD)
		rndKeyLen =  (keyLen + 7) & ~7;
	rv = CRYPTOKI_F_PTR( C_WrapKey(hSession, &mechanism, hKey, hSecret, wrappedPtr, &wrappedLen) );
	CPPUNIT_ASSERT(rv == CKR_KEY_UNEXTRACTABLE);
	rv = CRYPTOKI_F_PTR( C_DestroyObject(hSession, hSecret) );
	CPPUNIT_ASSERT(rv == CKR_OK);

	attribs[0].pValue = &bTrue;

	hSecret = CK_INVALID_HANDLE;
	rv = CRYPTOKI_F_PTR( C_CreateObject(hSession, attribs, sizeof(attribs)/sizeof(CK_ATTRIBUTE), &hSecret) );
	CPPUNIT_ASSERT(rv == CKR_OK);
	CPPUNIT_ASSERT(hSecret != CK_INVALID_HANDLE);

	// Estimate wrapped length
	rv = CRYPTOKI_F_PTR( C_WrapKey(hSession, &mechanism, hKey, hSecret, wrappedPtr, &wrappedLen) );
	CPPUNIT_ASSERT(rv == CKR_OK);

	auto wrapOverhead = [mechanismType]() {
		return (mechanismType == CKM_AES_KEY_WRAP || mechanismType == CKM_AES_KEY_WRAP_PAD) ? 8 : 16;
	};
	CPPUNIT_ASSERT(wrappedLen == rndKeyLen + wrapOverhead() );

	wrappedPtr = (CK_BYTE_PTR) malloc(wrappedLen);
	CPPUNIT_ASSERT(wrappedPtr != NULL_PTR);
	rv = CRYPTOKI_F_PTR( C_WrapKey(hSession, &mechanism, hKey, hSecret, wrappedPtr, &wrappedLen) );
	CPPUNIT_ASSERT(rv == CKR_OK);
	CPPUNIT_ASSERT(wrappedLen == rndKeyLen + wrapOverhead() );

	// This should always fail because wrapped data have to be longer than 0 bytes
	zero = 0;
	rv = CRYPTOKI_F_PTR( C_WrapKey(hSession, &mechanism, hKey, hSecret, wrappedPtr, &zero) );
	CPPUNIT_ASSERT(rv == CKR_BUFFER_TOO_SMALL);

	CK_ATTRIBUTE nattribs[] = {
		{ CKA_CLASS, &secretClass, sizeof(secretClass) },
		{ CKA_KEY_TYPE, &genKeyType, sizeof(genKeyType) },
		{ CKA_TOKEN, &bFalse, sizeof(bFalse) },
		{ CKA_PRIVATE, &bTrue, sizeof(bTrue) },
		{ CKA_ENCRYPT, &bFalse, sizeof(bFalse) },
		{ CKA_DECRYPT, &bTrue, sizeof(bTrue) },
		{ CKA_MODIFIABLE, &bFalse, sizeof(bFalse) },
		{ CKA_SIGN, &bFalse,sizeof(bFalse) },
		{ CKA_VERIFY, &bTrue, sizeof(bTrue) }
	};
	CK_OBJECT_HANDLE hNew;

	hNew = CK_INVALID_HANDLE;
	rv = CRYPTOKI_F_PTR( C_UnwrapKey(hSession, &mechanism, hKey, wrappedPtr, wrappedLen, nattribs, sizeof(nattribs)/sizeof(CK_ATTRIBUTE), &hNew) );
	CPPUNIT_ASSERT(rv == CKR_OK);
	CPPUNIT_ASSERT(hNew != CK_INVALID_HANDLE);

	CK_ATTRIBUTE modifiableAttribs[] = {
		{ CKA_MODIFIABLE, &bFalse, sizeof(bFalse) }
	};

	rv = CRYPTOKI_F_PTR( C_GetAttributeValue(hSession, hNew, modifiableAttribs, sizeof(modifiableAttribs)/sizeof(CK_ATTRIBUTE)) );
	CPPUNIT_ASSERT(rv == CKR_OK);

	CPPUNIT_ASSERT(modifiableAttribs[0].ulValueLen == sizeof(bFalse));
	CPPUNIT_ASSERT(*(CK_BBOOL*)modifiableAttribs[0].pValue == bFalse);

	free(wrappedPtr);
	wrappedPtr = NULL_PTR;
	rv = CRYPTOKI_F_PTR( C_DestroyObject(hSession, hSecret) );
	CPPUNIT_ASSERT(rv == CKR_OK);
	rv = CRYPTOKI_F_PTR( C_DestroyObject(hSession, hNew) );
	CPPUNIT_ASSERT(rv == CKR_OK);
}

inline void SymmetricAlgorithmTests::aesWrapUnwrapRsa(CK_MECHANISM_TYPE mechanismType, CK_SESSION_HANDLE hSession, CK_OBJECT_HANDLE hKey)
{
	wrapUnwrapRsa(mechanismType, hSession, hKey);
}

inline void SymmetricAlgorithmTests::desWrapUnwrapRsa(CK_MECHANISM_TYPE mechanismType, CK_SESSION_HANDLE hSession, CK_OBJECT_HANDLE hKey)
{
	wrapUnwrapRsa(mechanismType, hSession, hKey);
}


void SymmetricAlgorithmTests::wrapUnwrapRsa(CK_MECHANISM_TYPE mechanismType, CK_SESSION_HANDLE hSession, CK_OBJECT_HANDLE hKey)
{
	CK_MECHANISM mechanism = { mechanismType, NULL_PTR, 0 };
	CK_BBOOL bFalse = CK_FALSE;
	CK_BBOOL bTrue = CK_TRUE;
	CK_OBJECT_HANDLE hPrk = CK_INVALID_HANDLE;
	CK_RV rv = generateRsaPrivateKey(hSession, CK_TRUE, CK_TRUE, hPrk);
	CPPUNIT_ASSERT(rv == CKR_OK);
	CPPUNIT_ASSERT(hPrk != CK_INVALID_HANDLE);

	CK_BYTE ivPtr[16];
	switch(mechanismType) {
		case CKM_AES_CBC_PAD:
		case CKM_DES_CBC_PAD:
		case CKM_DES3_CBC_PAD:
			rv = CRYPTOKI_F_PTR( C_GenerateRandom(hSession, ivPtr, sizeof ivPtr) );
			CPPUNIT_ASSERT(rv == CKR_OK);
			mechanism.pParameter = ivPtr;
			mechanism.ulParameterLen = mechanismType == CKM_AES_CBC_PAD ? 16 : 8;
			// falls through
	}
		
	CK_OBJECT_CLASS privateClass = CKO_PRIVATE_KEY;
	CK_KEY_TYPE keyType = CKK_RSA;
	CK_BYTE_PTR prkAttrPtr = NULL_PTR;
	CK_ULONG prkAttrLen = 0UL;
	CK_ATTRIBUTE prkAttribs[] = {
		{ CKA_CLASS, &privateClass, sizeof(privateClass) },
		{ CKA_KEY_TYPE, &keyType, sizeof(keyType) },
		{ CKA_PRIME_2, NULL_PTR, 0UL }
	};

	rv = CRYPTOKI_F_PTR( C_GetAttributeValue(hSession, hPrk, prkAttribs, sizeof(prkAttribs)/sizeof(CK_ATTRIBUTE)) );
	CPPUNIT_ASSERT(rv == CKR_OK);

	CPPUNIT_ASSERT(prkAttribs[0].ulValueLen == sizeof(CK_OBJECT_CLASS));
	CPPUNIT_ASSERT(*(CK_OBJECT_CLASS*)prkAttribs[0].pValue == CKO_PRIVATE_KEY);
	CPPUNIT_ASSERT(prkAttribs[1].ulValueLen == sizeof(CK_KEY_TYPE));
	CPPUNIT_ASSERT(*(CK_KEY_TYPE*)prkAttribs[1].pValue == CKK_RSA);

	prkAttrLen = prkAttribs[2].ulValueLen;
	prkAttrPtr = (CK_BYTE_PTR) malloc(2 * prkAttrLen);
	CPPUNIT_ASSERT(prkAttrPtr != NULL_PTR);
	prkAttribs[2].pValue = prkAttrPtr;
	prkAttribs[2].ulValueLen = prkAttrLen;

	rv = CRYPTOKI_F_PTR( C_GetAttributeValue(hSession, hPrk, prkAttribs, sizeof(prkAttribs)/sizeof(CK_ATTRIBUTE)) );
	CPPUNIT_ASSERT(rv == CKR_OK);
	CPPUNIT_ASSERT(prkAttribs[2].ulValueLen == prkAttrLen);

	CK_BYTE_PTR wrappedPtr = NULL_PTR;
	CK_ULONG wrappedLen = 0UL;
	rv = CRYPTOKI_F_PTR( C_WrapKey(hSession, &mechanism, hKey, hPrk, wrappedPtr, &wrappedLen) );
	CPPUNIT_ASSERT(rv == CKR_OK);
	wrappedPtr = (CK_BYTE_PTR) malloc(wrappedLen);
	CPPUNIT_ASSERT(wrappedPtr != NULL_PTR);
	rv = CRYPTOKI_F_PTR( C_WrapKey(hSession, &mechanism, hKey, hPrk, wrappedPtr, &wrappedLen) );
	CPPUNIT_ASSERT(rv == CKR_OK);

	rv = CRYPTOKI_F_PTR( C_DestroyObject(hSession, hPrk) );
	CPPUNIT_ASSERT(rv == CKR_OK);

	CK_ATTRIBUTE nPrkAttribs[] = {
		{ CKA_CLASS, &privateClass, sizeof(privateClass) },
		{ CKA_KEY_TYPE, &keyType, sizeof(keyType) },
		{ CKA_TOKEN, &bFalse, sizeof(bFalse) },
		{ CKA_PRIVATE, &bTrue, sizeof(bTrue) },
		{ CKA_DECRYPT, &bTrue, sizeof(bTrue) },
		{ CKA_SIGN, &bFalse,sizeof(bFalse) },
		{ CKA_UNWRAP, &bTrue, sizeof(bTrue) },
		{ CKA_SENSITIVE, &bFalse, sizeof(bFalse) },
		{ CKA_EXTRACTABLE, &bTrue, sizeof(bTrue) }
	};

	hPrk = CK_INVALID_HANDLE;
	rv = CRYPTOKI_F_PTR( C_UnwrapKey(hSession, &mechanism, hKey, wrappedPtr, wrappedLen, nPrkAttribs, sizeof(nPrkAttribs)/sizeof(CK_ATTRIBUTE), &hPrk) );
	CPPUNIT_ASSERT(rv == CKR_OK);
	CPPUNIT_ASSERT(hPrk != CK_INVALID_HANDLE);

	prkAttribs[2].pValue = prkAttrPtr + prkAttrLen;
	rv = CRYPTOKI_F_PTR( C_GetAttributeValue(hSession, hPrk, prkAttribs, sizeof(prkAttribs)/sizeof(CK_ATTRIBUTE)) );
	CPPUNIT_ASSERT(rv == CKR_OK);

	CPPUNIT_ASSERT(prkAttribs[0].ulValueLen == sizeof(CK_OBJECT_CLASS));
	CPPUNIT_ASSERT(*(CK_OBJECT_CLASS*)prkAttribs[0].pValue == CKO_PRIVATE_KEY);
	CPPUNIT_ASSERT(prkAttribs[1].ulValueLen == sizeof(CK_KEY_TYPE));
	CPPUNIT_ASSERT(*(CK_KEY_TYPE*)prkAttribs[1].pValue == CKK_RSA);
	CPPUNIT_ASSERT(prkAttribs[2].ulValueLen == prkAttrLen);
	CPPUNIT_ASSERT(memcmp(prkAttrPtr, prkAttrPtr + prkAttrLen, prkAttrLen) == 0);

	free(wrappedPtr);
	free(prkAttrPtr);
	rv = CRYPTOKI_F_PTR( C_DestroyObject(hSession, hPrk) );
	CPPUNIT_ASSERT(rv == CKR_OK);
}


void SymmetricAlgorithmTests::unwrapKnownKey(const CK_SESSION_HANDLE hSession, WrappedMaterial &sWrapped)
{
	CK_BBOOL bFalse = CK_FALSE;
	CK_BBOOL bTrue = CK_TRUE;
	CK_OBJECT_HANDLE hKey = CK_INVALID_HANDLE, hPrk = CK_INVALID_HANDLE;

	// first regerenate our key
	CK_RV rv = importKey(hSession, CK_FALSE, CK_TRUE, hKey, sWrapped.wrappingKeyType(), sWrapped.wrappingKeyBytes());
	CPPUNIT_ASSERT_MESSAGE(sWrapped.description(), rv == CKR_OK);
	CPPUNIT_ASSERT_MESSAGE(sWrapped.description(), hKey != CK_INVALID_HANDLE);

	CK_ATTRIBUTE nPrkAttribs[] = {
		{ CKA_CLASS, &sWrapped.wrappedObjectClass(), sizeof(CK_OBJECT_CLASS) },
		{ CKA_KEY_TYPE, &sWrapped.wrappedKeyType(), sizeof(CK_KEY_TYPE) },
		{ CKA_TOKEN, &bFalse, sizeof(bFalse) },
		{ CKA_PRIVATE, &bTrue, sizeof(bTrue) },
		{ CKA_SENSITIVE, &bFalse, sizeof(bFalse) },
		{ CKA_EXTRACTABLE, &bTrue, sizeof(bTrue) },
	};
	
	hPrk = CK_INVALID_HANDLE;
	rv = CRYPTOKI_F_PTR( C_UnwrapKey( hSession,
					  &sWrapped.mechanism(),
					  hKey,
					  sWrapped.wrappedKey().data(),
					  sWrapped.wrappedKey().size(),
					  nPrkAttribs,
					  sizeof(nPrkAttribs)/sizeof(CK_ATTRIBUTE), &hPrk) );
	CPPUNIT_ASSERT_MESSAGE(sWrapped.description(), rv == CKR_OK);
	CPPUNIT_ASSERT_MESSAGE(sWrapped.description(), hPrk != CK_INVALID_HANDLE);
	// no further test: if the key could be recreated as a PKCS8 structure, it means it was successully recovered
	
	rv = CRYPTOKI_F_PTR( C_DestroyObject(hSession, hPrk) );
	CPPUNIT_ASSERT_MESSAGE(sWrapped.description(), rv == CKR_OK);
}


#ifdef WITH_GOST
void SymmetricAlgorithmTests::aesWrapUnwrapGost(CK_MECHANISM_TYPE mechanismType, CK_SESSION_HANDLE hSession, CK_OBJECT_HANDLE hKey)
{
	CK_MECHANISM mechanism = { mechanismType, NULL_PTR, 0 };
	CK_BBOOL bFalse = CK_FALSE;
	CK_BBOOL bTrue = CK_TRUE;
	CK_OBJECT_HANDLE hPrk = CK_INVALID_HANDLE;
	CK_RV rv = generateGostPrivateKey(hSession, CK_TRUE, CK_TRUE, hPrk);
	CPPUNIT_ASSERT(rv == CKR_OK);
	CPPUNIT_ASSERT(hPrk != CK_INVALID_HANDLE);

	CK_OBJECT_CLASS privateClass = CKO_PRIVATE_KEY;
	CK_KEY_TYPE keyType = CKK_GOSTR3410;
	CK_BYTE_PTR prkAttrPtr = NULL_PTR;
	CK_ULONG prkAttrLen = 0UL;
	CK_ATTRIBUTE prkAttribs[] = {
		{ CKA_CLASS, &privateClass, sizeof(privateClass) },
		{ CKA_KEY_TYPE, &keyType, sizeof(keyType) },
		{ CKA_VALUE, NULL_PTR, 0UL }
	};

	rv = CRYPTOKI_F_PTR( C_GetAttributeValue(hSession, hPrk, prkAttribs, sizeof(prkAttribs)/sizeof(CK_ATTRIBUTE)) );
	CPPUNIT_ASSERT(rv == CKR_OK);

	CPPUNIT_ASSERT(prkAttribs[0].ulValueLen == sizeof(CK_OBJECT_CLASS));
	CPPUNIT_ASSERT(*(CK_OBJECT_CLASS*)prkAttribs[0].pValue == CKO_PRIVATE_KEY);
	CPPUNIT_ASSERT(prkAttribs[1].ulValueLen == sizeof(CK_KEY_TYPE));
	CPPUNIT_ASSERT(*(CK_KEY_TYPE*)prkAttribs[1].pValue == CKK_GOSTR3410);

	prkAttrLen = prkAttribs[2].ulValueLen;
	prkAttrPtr = (CK_BYTE_PTR) malloc(2 * prkAttrLen);
	CPPUNIT_ASSERT(prkAttrPtr != NULL_PTR);
	prkAttribs[2].pValue = prkAttrPtr;
	prkAttribs[2].ulValueLen = prkAttrLen;

	rv = CRYPTOKI_F_PTR( C_GetAttributeValue(hSession, hPrk, prkAttribs, sizeof(prkAttribs)/sizeof(CK_ATTRIBUTE)) );
	CPPUNIT_ASSERT(rv == CKR_OK);
	CPPUNIT_ASSERT(prkAttribs[2].ulValueLen == prkAttrLen);

	CK_BYTE_PTR wrappedPtr = NULL_PTR;
	CK_ULONG wrappedLen = 0UL;
	rv = CRYPTOKI_F_PTR( C_WrapKey(hSession, &mechanism, hKey, hPrk, wrappedPtr, &wrappedLen) );
	CPPUNIT_ASSERT(rv == CKR_OK);
	wrappedPtr = (CK_BYTE_PTR) malloc(wrappedLen);
	CPPUNIT_ASSERT(wrappedPtr != NULL_PTR);
	rv = CRYPTOKI_F_PTR( C_WrapKey(hSession, &mechanism, hKey, hPrk, wrappedPtr, &wrappedLen) );
	CPPUNIT_ASSERT(rv == CKR_OK);

	rv = CRYPTOKI_F_PTR( C_DestroyObject(hSession, hPrk) );
	CPPUNIT_ASSERT(rv == CKR_OK);

	CK_ATTRIBUTE nPrkAttribs[] = {
		{ CKA_CLASS, &privateClass, sizeof(privateClass) },
		{ CKA_KEY_TYPE, &keyType, sizeof(keyType) },
		{ CKA_TOKEN, &bFalse, sizeof(bFalse) },
		{ CKA_PRIVATE, &bTrue, sizeof(bTrue) },
		{ CKA_DECRYPT, &bTrue, sizeof(bTrue) },
		{ CKA_SIGN, &bFalse,sizeof(bFalse) },
		{ CKA_UNWRAP, &bTrue, sizeof(bTrue) },
		{ CKA_SENSITIVE, &bFalse, sizeof(bFalse) },
		{ CKA_EXTRACTABLE, &bTrue, sizeof(bTrue) }
	};

	hPrk = CK_INVALID_HANDLE;
	rv = CRYPTOKI_F_PTR( C_UnwrapKey(hSession, &mechanism, hKey, wrappedPtr, wrappedLen, nPrkAttribs, sizeof(nPrkAttribs)/sizeof(CK_ATTRIBUTE), &hPrk) );
	CPPUNIT_ASSERT(rv == CKR_OK);
	CPPUNIT_ASSERT(hPrk != CK_INVALID_HANDLE);

	prkAttribs[2].pValue = prkAttrPtr + prkAttrLen;
	rv = CRYPTOKI_F_PTR( C_GetAttributeValue(hSession, hPrk, prkAttribs, sizeof(prkAttribs)/sizeof(CK_ATTRIBUTE)) );
	CPPUNIT_ASSERT(rv == CKR_OK);

	CPPUNIT_ASSERT(prkAttribs[0].ulValueLen == sizeof(CK_OBJECT_CLASS));
	CPPUNIT_ASSERT(*(CK_OBJECT_CLASS*)prkAttribs[0].pValue == CKO_PRIVATE_KEY);
	CPPUNIT_ASSERT(prkAttribs[1].ulValueLen == sizeof(CK_KEY_TYPE));
	CPPUNIT_ASSERT(*(CK_KEY_TYPE*)prkAttribs[1].pValue == CKK_GOSTR3410);
	CPPUNIT_ASSERT(prkAttribs[2].ulValueLen == prkAttrLen);
	CPPUNIT_ASSERT(memcmp(prkAttrPtr, prkAttrPtr + prkAttrLen, prkAttrLen) == 0);

	free(wrappedPtr);
	free(prkAttrPtr);
	rv = CRYPTOKI_F_PTR( C_DestroyObject(hSession, hPrk) );
	CPPUNIT_ASSERT(rv == CKR_OK);
}
#endif

#ifdef WITH_EDDSA
void SymmetricAlgorithmTests::aesWrapUnwrapED(CK_MECHANISM_TYPE mechanismType, CK_SESSION_HANDLE hSession, CK_OBJECT_HANDLE hKey)
{
	CK_MECHANISM mechanism = { mechanismType, NULL_PTR, 0 };
	CK_BBOOL bFalse = CK_FALSE;
	CK_BBOOL bTrue = CK_TRUE;

	std::map<std::string, EDCurveParam > curves {
		{ "ED25519", {0x06, 0x03, 0x2b, 0x65, 0x70} }
#ifndef WITH_BOTAN
		, { "ED448", {0x06, 0x03, 0x2b, 0x65, 0x71} }
#endif
	};

	for(auto &curve : curves) {
	        const auto curvename = "Curve name: " + curve.first;

		CK_OBJECT_HANDLE hPrk = CK_INVALID_HANDLE;
		CK_RV rv = generateEDPrivateKey(hSession, CK_TRUE, CK_TRUE, hPrk, curve.second);
		CPPUNIT_ASSERT_MESSAGE(curvename, rv == CKR_OK);
		CPPUNIT_ASSERT_MESSAGE(curvename, hPrk != CK_INVALID_HANDLE);

		CK_OBJECT_CLASS privateClass = CKO_PRIVATE_KEY;
		CK_KEY_TYPE keyType = CKK_EC_EDWARDS;
		CK_BYTE_PTR prkAttrPtr = NULL_PTR;
		CK_ULONG prkAttrLen = 0UL;
		CK_ATTRIBUTE prkAttribs[] = {
			{ CKA_CLASS, &privateClass, sizeof(privateClass) },
			{ CKA_KEY_TYPE, &keyType, sizeof(keyType) },
			{ CKA_VALUE, NULL_PTR, 0UL }
		};

		rv = CRYPTOKI_F_PTR( C_GetAttributeValue(hSession, hPrk, prkAttribs, sizeof(prkAttribs)/sizeof(CK_ATTRIBUTE)) );
		CPPUNIT_ASSERT_MESSAGE(curvename, rv == CKR_OK);

		CPPUNIT_ASSERT_MESSAGE(curvename, prkAttribs[0].ulValueLen == sizeof(CK_OBJECT_CLASS));
		CPPUNIT_ASSERT_MESSAGE(curvename, *(CK_OBJECT_CLASS*)prkAttribs[0].pValue == CKO_PRIVATE_KEY);
		CPPUNIT_ASSERT_MESSAGE(curvename, prkAttribs[1].ulValueLen == sizeof(CK_KEY_TYPE));
		CPPUNIT_ASSERT_MESSAGE(curvename, *(CK_KEY_TYPE*)prkAttribs[1].pValue == CKK_EC_EDWARDS);

		prkAttrLen = prkAttribs[2].ulValueLen;
		prkAttrPtr = (CK_BYTE_PTR) malloc(2 * prkAttrLen);
		CPPUNIT_ASSERT_MESSAGE(curvename, prkAttrPtr != NULL_PTR);
		prkAttribs[2].pValue = prkAttrPtr;
		prkAttribs[2].ulValueLen = prkAttrLen;

		rv = CRYPTOKI_F_PTR( C_GetAttributeValue(hSession, hPrk, prkAttribs, sizeof(prkAttribs)/sizeof(CK_ATTRIBUTE)) );
		CPPUNIT_ASSERT_MESSAGE(curvename, rv == CKR_OK);
		CPPUNIT_ASSERT_MESSAGE(curvename, prkAttribs[2].ulValueLen == prkAttrLen);

		CK_BYTE_PTR wrappedPtr = NULL_PTR;
		CK_ULONG wrappedLen = 0UL;
		rv = CRYPTOKI_F_PTR( C_WrapKey(hSession, &mechanism, hKey, hPrk, wrappedPtr, &wrappedLen) );
		CPPUNIT_ASSERT_MESSAGE(curvename, rv == CKR_OK);
		wrappedPtr = (CK_BYTE_PTR) malloc(wrappedLen);
		CPPUNIT_ASSERT_MESSAGE(curvename, wrappedPtr != NULL_PTR);
		rv = CRYPTOKI_F_PTR( C_WrapKey(hSession, &mechanism, hKey, hPrk, wrappedPtr, &wrappedLen) );
		CPPUNIT_ASSERT_MESSAGE(curvename, rv == CKR_OK);

		rv = CRYPTOKI_F_PTR( C_DestroyObject(hSession, hPrk) );
		CPPUNIT_ASSERT_MESSAGE(curvename, rv == CKR_OK);

		CK_ATTRIBUTE nPrkAttribs[] = {
			{ CKA_CLASS, &privateClass, sizeof(privateClass) },
			{ CKA_KEY_TYPE, &keyType, sizeof(keyType) },
			{ CKA_TOKEN, &bFalse, sizeof(bFalse) },
			{ CKA_PRIVATE, &bTrue, sizeof(bTrue) },
			{ CKA_DECRYPT, &bTrue, sizeof(bTrue) },
			{ CKA_SIGN, &bFalse,sizeof(bFalse) },
			{ CKA_UNWRAP, &bTrue, sizeof(bTrue) },
			{ CKA_SENSITIVE, &bFalse, sizeof(bFalse) },
			{ CKA_EXTRACTABLE, &bTrue, sizeof(bTrue) }
		};

		hPrk = CK_INVALID_HANDLE;
		rv = CRYPTOKI_F_PTR( C_UnwrapKey(hSession, &mechanism, hKey, wrappedPtr, wrappedLen, nPrkAttribs, sizeof(nPrkAttribs)/sizeof(CK_ATTRIBUTE), &hPrk) );
		CPPUNIT_ASSERT_MESSAGE(curvename, rv == CKR_OK);
		CPPUNIT_ASSERT_MESSAGE(curvename, hPrk != CK_INVALID_HANDLE);

		prkAttribs[2].pValue = prkAttrPtr + prkAttrLen;
		rv = CRYPTOKI_F_PTR( C_GetAttributeValue(hSession, hPrk, prkAttribs, sizeof(prkAttribs)/sizeof(CK_ATTRIBUTE)) );
		CPPUNIT_ASSERT_MESSAGE(curvename, rv == CKR_OK);

		CPPUNIT_ASSERT_MESSAGE(curvename, prkAttribs[0].ulValueLen == sizeof(CK_OBJECT_CLASS));
		CPPUNIT_ASSERT_MESSAGE(curvename, *(CK_OBJECT_CLASS*)prkAttribs[0].pValue == CKO_PRIVATE_KEY);
		CPPUNIT_ASSERT_MESSAGE(curvename, prkAttribs[1].ulValueLen == sizeof(CK_KEY_TYPE));
		CPPUNIT_ASSERT_MESSAGE(curvename, *(CK_KEY_TYPE*)prkAttribs[1].pValue == CKK_EC_EDWARDS);
		CPPUNIT_ASSERT_MESSAGE(curvename, prkAttribs[2].ulValueLen == prkAttrLen);
		CPPUNIT_ASSERT_MESSAGE(curvename, memcmp(prkAttrPtr, prkAttrPtr + prkAttrLen, prkAttrLen) == 0);

		free(wrappedPtr);
		free(prkAttrPtr);
		rv = CRYPTOKI_F_PTR( C_DestroyObject(hSession, hPrk) );
		CPPUNIT_ASSERT_MESSAGE(curvename, rv == CKR_OK);
	}
}
#endif

void SymmetricAlgorithmTests::testAesEncryptDecrypt()
{
	CK_RV rv;
	// CK_UTF8CHAR sopin[] = SLOT_0_SO1_PIN;
	// CK_ULONG sopinLength = sizeof(sopin) - 1;
	CK_SESSION_HANDLE hSessionRO;
	CK_SESSION_HANDLE hSessionRW;

	CK_AES_CTR_PARAMS ctrParams =
	{
		32,
		{
			0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01
		}
	};
	CK_BYTE gcmIV[] = {
		0xCA, 0xFE, 0xBA, 0xBE, 0xFA, 0xCE,
		0xDB, 0xAD, 0xDE, 0xCA, 0xF8, 0x88
	};
	CK_BYTE gcmAAD[] = {
		0xFE, 0xED, 0xFA, 0xCE, 0xDE, 0xAD, 0xBE, 0xEF,
		0xFE, 0xED, 0xFA, 0xCE, 0xDE, 0xAD, 0xBE, 0xEF,
		0xAB, 0xAD, 0xDA, 0xD2
	};
	CK_GCM_PARAMS gcmParamsWithAAD =
	{
		&gcmIV[0],
		sizeof(gcmIV),
		sizeof(gcmIV)*8,
		&gcmAAD[0],
		sizeof(gcmAAD),
		16*8
	};
	CK_GCM_PARAMS gcmParamsWithoutAAD =
	{
		&gcmIV[0],
		sizeof(gcmIV),
		sizeof(gcmIV)*8,
		NULL_PTR,
		0,
		16*8
	};


	// Just make sure that we finalize any previous tests
	CRYPTOKI_F_PTR( C_Finalize(NULL_PTR) );

	// Open read-only session on when the token is not initialized should fail
	rv = CRYPTOKI_F_PTR( C_OpenSession(m_initializedTokenSlotID, CKF_SERIAL_SESSION, NULL_PTR, NULL_PTR, &hSessionRO) );
	CPPUNIT_ASSERT(rv == CKR_CRYPTOKI_NOT_INITIALIZED);

	// Initialize the library and start the test.
	rv = CRYPTOKI_F_PTR( C_Initialize(NULL_PTR) );
	CPPUNIT_ASSERT(rv == CKR_OK);

	// Open read-only session
	rv = CRYPTOKI_F_PTR( C_OpenSession(m_initializedTokenSlotID, CKF_SERIAL_SESSION, NULL_PTR, NULL_PTR, &hSessionRO) );
	CPPUNIT_ASSERT(rv == CKR_OK);

	// Open read-write session
	rv = CRYPTOKI_F_PTR( C_OpenSession(m_initializedTokenSlotID, CKF_SERIAL_SESSION | CKF_RW_SESSION, NULL_PTR, NULL_PTR, &hSessionRW) );
	CPPUNIT_ASSERT(rv == CKR_OK);

	// Login USER into the sessions so we can create a private objects
	rv = CRYPTOKI_F_PTR( C_Login(hSessionRO,CKU_USER,m_userPin1,m_userPin1Length) );
	CPPUNIT_ASSERT(rv==CKR_OK);

	CK_OBJECT_HANDLE hKey = CK_INVALID_HANDLE;

	// Generate all combinations of session/token keys.
	rv = generateAesKey(hSessionRW,IN_SESSION,IS_PUBLIC,hKey);
	CPPUNIT_ASSERT(rv == CKR_OK);

	// AES allways have the block size of 128 bits (0x80 bits 0x10 bytes).
	// with padding all message sizes could be encrypted-decrypted.
	// without padding the message size must be a multiple of the block size.
	const int blockSize(0x10);

	encryptDecrypt({CKM_AES_CBC_PAD,NULL_PTR,0},blockSize,hSessionRO,hKey,blockSize*NR_OF_BLOCKS_IN_TEST-1);
	encryptDecrypt({CKM_AES_CBC_PAD,NULL_PTR,0},blockSize,hSessionRO,hKey,blockSize*NR_OF_BLOCKS_IN_TEST+1);
	encryptDecrypt({CKM_AES_CBC_PAD,NULL_PTR,0},blockSize,hSessionRO,hKey,blockSize*NR_OF_BLOCKS_IN_TEST);
	encryptDecrypt({CKM_AES_CBC,NULL_PTR,0},blockSize,hSessionRO,hKey,blockSize*NR_OF_BLOCKS_IN_TEST);
	encryptDecrypt({CKM_AES_CBC,NULL_PTR,0},blockSize,hSessionRO,hKey,blockSize*NR_OF_BLOCKS_IN_TEST+1, false);
	encryptDecrypt({CKM_AES_ECB,NULL_PTR,0},blockSize,hSessionRO,hKey,blockSize*NR_OF_BLOCKS_IN_TEST);
	encryptDecrypt({CKM_AES_ECB,NULL_PTR,0},blockSize,hSessionRO,hKey,blockSize*NR_OF_BLOCKS_IN_TEST+1, false);
	encryptDecrypt({CKM_AES_CTR,&ctrParams,sizeof(ctrParams)},blockSize,hSessionRO,hKey,blockSize*NR_OF_BLOCKS_IN_TEST-1);
	encryptDecrypt({CKM_AES_CTR,&ctrParams,sizeof(ctrParams)},blockSize,hSessionRO,hKey,blockSize*NR_OF_BLOCKS_IN_TEST+1);
	encryptDecrypt({CKM_AES_CTR,&ctrParams,sizeof(ctrParams)},blockSize,hSessionRO,hKey,blockSize*NR_OF_BLOCKS_IN_TEST);
	encryptDecrypt({CKM_AES_GCM,&gcmParamsWithAAD,sizeof(gcmParamsWithAAD)},blockSize,hSessionRO,hKey,blockSize*NR_OF_BLOCKS_IN_TEST-1);
	encryptDecrypt({CKM_AES_GCM,&gcmParamsWithAAD,sizeof(gcmParamsWithAAD)},blockSize,hSessionRO,hKey,blockSize*NR_OF_BLOCKS_IN_TEST+1);
	encryptDecrypt({CKM_AES_GCM,&gcmParamsWithAAD,sizeof(gcmParamsWithAAD)},blockSize,hSessionRO,hKey,blockSize*NR_OF_BLOCKS_IN_TEST);
	encryptDecrypt({CKM_AES_GCM,&gcmParamsWithoutAAD,sizeof(gcmParamsWithoutAAD)},blockSize,hSessionRO,hKey,blockSize*NR_OF_BLOCKS_IN_TEST-1);
	encryptDecrypt({CKM_AES_GCM,&gcmParamsWithoutAAD,sizeof(gcmParamsWithoutAAD)},blockSize,hSessionRO,hKey,blockSize*NR_OF_BLOCKS_IN_TEST+1);
	encryptDecrypt({CKM_AES_GCM,&gcmParamsWithoutAAD,sizeof(gcmParamsWithoutAAD)},blockSize,hSessionRO,hKey,blockSize*NR_OF_BLOCKS_IN_TEST);
}


void SymmetricAlgorithmTests::testAesWrapUnwrap()
{
	CK_RV rv;
	// CK_UTF8CHAR sopin[] = SLOT_0_SO1_PIN;
	// CK_ULONG sopinLength = sizeof(sopin) - 1;
	CK_SESSION_HANDLE hSession;

	// Just make sure that we finalize any previous tests
	CRYPTOKI_F_PTR( C_Finalize(NULL_PTR) );

	// Initialize the library and start the test.
	rv = CRYPTOKI_F_PTR( C_Initialize(NULL_PTR) );
	CPPUNIT_ASSERT(rv == CKR_OK);

	// Open session
	rv = CRYPTOKI_F_PTR( C_OpenSession(m_initializedTokenSlotID, CKF_SERIAL_SESSION | CKF_RW_SESSION, NULL_PTR, NULL_PTR, &hSession) );
	CPPUNIT_ASSERT(rv == CKR_OK);

	// Login USER into the session so we can create a private object
	rv = CRYPTOKI_F_PTR( C_Login(hSession,CKU_USER,m_userPin1,m_userPin1Length) );
	CPPUNIT_ASSERT(rv==CKR_OK);

	CK_OBJECT_HANDLE hKey = CK_INVALID_HANDLE;

	// Generate a wrapping session public key
	rv = generateAesKey(hSession,IN_SESSION,IS_PUBLIC,hKey);
	CPPUNIT_ASSERT(rv == CKR_OK);

	aesWrapUnwrapGeneric(CKM_AES_KEY_WRAP, hSession, hKey);
	aesWrapUnwrapGeneric(CKM_AES_CBC_PAD, hSession, hKey);
	aesWrapUnwrapNonModifiableGeneric(CKM_AES_KEY_WRAP, hSession, hKey);
    aesWrapUnwrapNonModifiableGeneric(CKM_AES_CBC_PAD, hSession, hKey);
	aesWrapUnwrapRsa(CKM_AES_KEY_WRAP, hSession, hKey);
	aesWrapUnwrapRsa(CKM_AES_CBC_PAD, hSession, hKey);

#ifdef WITH_GOST
	aesWrapUnwrapGost(CKM_AES_KEY_WRAP, hSession, hKey);
#endif

#ifdef HAVE_AES_KEY_WRAP_PAD
	aesWrapUnwrapGeneric(CKM_AES_KEY_WRAP_PAD, hSession, hKey);
	aesWrapUnwrapRsa(CKM_AES_KEY_WRAP_PAD, hSession, hKey);
#ifdef WITH_GOST
	aesWrapUnwrapGost(CKM_AES_KEY_WRAP_PAD, hSession, hKey);
#endif
#ifdef WITH_EDDSA
	aesWrapUnwrapED(CKM_AES_KEY_WRAP_PAD, hSession, hKey);
#endif
#endif
	for ( auto &wrapped : aesCBCWrappedKeys )
	{
		unwrapKnownKey(hSession, wrapped);
	}
}

void SymmetricAlgorithmTests::testDesEncryptDecrypt()
{
	CK_RV rv;
	// CK_UTF8CHAR sopin[] = SLOT_0_SO1_PIN;
	// CK_ULONG sopinLength = sizeof(sopin) - 1;
	CK_SESSION_HANDLE hSessionRO;
	CK_SESSION_HANDLE hSessionRW;

	// Just make sure that we finalize any previous tests
	CRYPTOKI_F_PTR( C_Finalize(NULL_PTR) );

	// Open read-only session on when the token is not initialized should fail
	rv = CRYPTOKI_F_PTR( C_OpenSession(m_initializedTokenSlotID, CKF_SERIAL_SESSION, NULL_PTR, NULL_PTR, &hSessionRO) );
	CPPUNIT_ASSERT(rv == CKR_CRYPTOKI_NOT_INITIALIZED);

	// Initialize the library and start the test.
	rv = CRYPTOKI_F_PTR( C_Initialize(NULL_PTR) );
	CPPUNIT_ASSERT(rv == CKR_OK);

	// Open read-only session
	rv = CRYPTOKI_F_PTR( C_OpenSession(m_initializedTokenSlotID, CKF_SERIAL_SESSION, NULL_PTR, NULL_PTR, &hSessionRO) );
	CPPUNIT_ASSERT(rv == CKR_OK);

	// Open read-write session
	rv = CRYPTOKI_F_PTR( C_OpenSession(m_initializedTokenSlotID, CKF_SERIAL_SESSION | CKF_RW_SESSION, NULL_PTR, NULL_PTR, &hSessionRW) );
	CPPUNIT_ASSERT(rv == CKR_OK);

	// Login USER into the sessions so we can create a private objects
	rv = CRYPTOKI_F_PTR( C_Login(hSessionRO,CKU_USER,m_userPin1,m_userPin1Length) );
	CPPUNIT_ASSERT(rv==CKR_OK);

	// 3DES and DES always have the block size of 64 bits (0x40 bits 0x8 bytes).
	// with padding all message sizes could be encrypted-decrypted.
	// without padding the message size must be a multiple of the block size.
	const int blockSize(0x8);

#ifndef WITH_FIPS
	CK_OBJECT_HANDLE hKey = CK_INVALID_HANDLE;

	// Generate all combinations of session/token keys.
	rv = generateDesKey(hSessionRW,IN_SESSION,IS_PUBLIC,hKey);
	CPPUNIT_ASSERT(rv == CKR_OK);

	encryptDecrypt({CKM_DES_CBC_PAD,NULL_PTR,0},blockSize,hSessionRO,hKey,blockSize*NR_OF_BLOCKS_IN_TEST-1);
	encryptDecrypt({CKM_DES_CBC_PAD,NULL_PTR,0},blockSize,hSessionRO,hKey,blockSize*NR_OF_BLOCKS_IN_TEST+1);
	encryptDecrypt({CKM_DES_CBC_PAD,NULL_PTR,0},blockSize,hSessionRO,hKey,blockSize*NR_OF_BLOCKS_IN_TEST);
	encryptDecrypt({CKM_DES_CBC,NULL_PTR,0},blockSize,hSessionRO,hKey,blockSize*NR_OF_BLOCKS_IN_TEST);
	encryptDecrypt({CKM_DES_CBC,NULL_PTR,0},blockSize,hSessionRO,hKey,blockSize*NR_OF_BLOCKS_IN_TEST+1, false);
	encryptDecrypt({CKM_DES_ECB,NULL_PTR,0},blockSize,hSessionRO,hKey,blockSize*NR_OF_BLOCKS_IN_TEST);
	encryptDecrypt({CKM_DES_ECB,NULL_PTR,0},blockSize,hSessionRO,hKey,blockSize*NR_OF_BLOCKS_IN_TEST+1, false);

	CK_OBJECT_HANDLE hKey2 = CK_INVALID_HANDLE;

	// Generate all combinations of session/token keys.
	rv = generateDes2Key(hSessionRW,IN_SESSION,IS_PUBLIC,hKey2);
	CPPUNIT_ASSERT(rv == CKR_OK);

	encryptDecrypt({CKM_DES3_CBC_PAD,NULL_PTR,0},blockSize,hSessionRO,hKey2,blockSize*NR_OF_BLOCKS_IN_TEST-1);
	encryptDecrypt({CKM_DES3_CBC_PAD,NULL_PTR,0},blockSize,hSessionRO,hKey2,blockSize*NR_OF_BLOCKS_IN_TEST+1);
	encryptDecrypt({CKM_DES3_CBC_PAD,NULL_PTR,0},blockSize,hSessionRO,hKey2,blockSize*NR_OF_BLOCKS_IN_TEST);
	encryptDecrypt({CKM_DES3_CBC,NULL_PTR,0},blockSize,hSessionRO,hKey2,blockSize*NR_OF_BLOCKS_IN_TEST);
	encryptDecrypt({CKM_DES3_CBC,NULL_PTR,0},blockSize,hSessionRO,hKey2,blockSize*NR_OF_BLOCKS_IN_TEST+1, false);
	encryptDecrypt({CKM_DES3_ECB,NULL_PTR,0},blockSize,hSessionRO,hKey2,blockSize*NR_OF_BLOCKS_IN_TEST);
	encryptDecrypt({CKM_DES3_ECB,NULL_PTR,0},blockSize,hSessionRO,hKey2,blockSize*NR_OF_BLOCKS_IN_TEST+1, false);
#endif

	CK_OBJECT_HANDLE hKey3 = CK_INVALID_HANDLE;

	// Generate all combinations of session/token keys.
	rv = generateDes3Key(hSessionRW,IN_SESSION,IS_PUBLIC,hKey3);
	CPPUNIT_ASSERT(rv == CKR_OK);

	encryptDecrypt({CKM_DES3_CBC_PAD,NULL_PTR,0},blockSize,hSessionRO,hKey3,blockSize*NR_OF_BLOCKS_IN_TEST-1);
	encryptDecrypt({CKM_DES3_CBC_PAD,NULL_PTR,0},blockSize,hSessionRO,hKey3,blockSize*NR_OF_BLOCKS_IN_TEST+1);
	encryptDecrypt({CKM_DES3_CBC_PAD,NULL_PTR,0},blockSize,hSessionRO,hKey3,blockSize*NR_OF_BLOCKS_IN_TEST);
	encryptDecrypt({CKM_DES3_CBC,NULL_PTR,0},blockSize,hSessionRO,hKey3,blockSize*NR_OF_BLOCKS_IN_TEST);
	encryptDecrypt({CKM_DES3_CBC,NULL_PTR,0},blockSize,hSessionRO,hKey3,blockSize*NR_OF_BLOCKS_IN_TEST+1, false);
	encryptDecrypt({CKM_DES3_ECB,NULL_PTR,0},blockSize,hSessionRO,hKey3,blockSize*NR_OF_BLOCKS_IN_TEST);
	encryptDecrypt({CKM_DES3_ECB,NULL_PTR,0},blockSize,hSessionRO,hKey3,blockSize*NR_OF_BLOCKS_IN_TEST+1, false);
}

void SymmetricAlgorithmTests::testDesWrapUnwrap()
{
	CK_RV rv;
	// CK_UTF8CHAR sopin[] = SLOT_0_SO1_PIN;
	// CK_ULONG sopinLength = sizeof(sopin) - 1;
	CK_SESSION_HANDLE hSession;

	// Just make sure that we finalize any previous tests
	CRYPTOKI_F_PTR( C_Finalize(NULL_PTR) );

	// Initialize the library and start the test.
	rv = CRYPTOKI_F_PTR( C_Initialize(NULL_PTR) );
	CPPUNIT_ASSERT(rv == CKR_OK);

	// Open session
	rv = CRYPTOKI_F_PTR( C_OpenSession(m_initializedTokenSlotID, CKF_SERIAL_SESSION | CKF_RW_SESSION, NULL_PTR, NULL_PTR, &hSession) );
	CPPUNIT_ASSERT(rv == CKR_OK);

	// Login USER into the session so we can create a private object
	rv = CRYPTOKI_F_PTR( C_Login(hSession,CKU_USER,m_userPin1,m_userPin1Length) );
	CPPUNIT_ASSERT(rv==CKR_OK);

	std::array<CK_OBJECT_HANDLE,3> hKey { CK_INVALID_HANDLE, CK_INVALID_HANDLE, CK_INVALID_HANDLE };

#ifndef WITH_FIPS
	rv = generateDesKey(hSession,IN_SESSION,IS_PUBLIC,hKey[0]);
	CPPUNIT_ASSERT(rv == CKR_OK);
	rv = generateDes2Key(hSession,IN_SESSION,IS_PUBLIC,hKey[1]);
	CPPUNIT_ASSERT(rv == CKR_OK);
#endif
	rv = generateDes3Key(hSession,IN_SESSION,IS_PUBLIC,hKey[2]);
	CPPUNIT_ASSERT(rv == CKR_OK);

#ifndef WITH_FIPS
	desWrapUnwrapRsa(CKM_DES_CBC_PAD, hSession, hKey[0]);
	desWrapUnwrapRsa(CKM_DES3_CBC_PAD, hSession, hKey[1]);
#endif
	desWrapUnwrapRsa(CKM_DES3_CBC_PAD, hSession, hKey[2]);

	for ( auto &wrapped : desCBCWrappedKeys )
	{
		unwrapKnownKey(hSession, wrapped);
	}	
}

void SymmetricAlgorithmTests::testNullTemplate()
{
	CK_RV rv;
	CK_SESSION_HANDLE hSession;
	CK_MECHANISM mechanism1 = { CKM_DES3_KEY_GEN, NULL_PTR, 0 };
	CK_MECHANISM mechanism2 = { CKM_AES_KEY_GEN, NULL_PTR, 0 };
	CK_OBJECT_HANDLE hKey = CK_INVALID_HANDLE;

	// Just make sure that we finalize any previous tests
	CRYPTOKI_F_PTR( C_Finalize(NULL_PTR) );

	// Initialize the library and start the test.
	rv = CRYPTOKI_F_PTR( C_Initialize(NULL_PTR) );
	CPPUNIT_ASSERT(rv == CKR_OK);

	// Open read-write session
	rv = CRYPTOKI_F_PTR( C_OpenSession(m_initializedTokenSlotID, CKF_SERIAL_SESSION | CKF_RW_SESSION, NULL_PTR, NULL_PTR, &hSession) );
	CPPUNIT_ASSERT(rv == CKR_OK);

	// Login USER into the sessions so we can create a private objects
	rv = CRYPTOKI_F_PTR( C_Login(hSession, CKU_USER, m_userPin1, m_userPin1Length) );
	CPPUNIT_ASSERT(rv==CKR_OK);

	rv = CRYPTOKI_F_PTR( C_GenerateKey(hSession, &mechanism1, NULL_PTR, 0, &hKey) );
	CPPUNIT_ASSERT(rv == CKR_OK);

	rv = CRYPTOKI_F_PTR( C_GenerateKey(hSession, &mechanism1, NULL_PTR, 1, &hKey) );
	CPPUNIT_ASSERT(rv == CKR_ARGUMENTS_BAD);

	rv = CRYPTOKI_F_PTR( C_DestroyObject(hSession, hKey) );
	CPPUNIT_ASSERT(rv == CKR_OK);

	rv = CRYPTOKI_F_PTR( C_GenerateKey(hSession, &mechanism2, NULL_PTR, 0, &hKey) );
	CPPUNIT_ASSERT(rv == CKR_TEMPLATE_INCOMPLETE);
}

void SymmetricAlgorithmTests::testNonModifiableDesKeyGeneration()
{
	CK_RV rv;
	CK_SESSION_HANDLE hSession;
	CK_MECHANISM mechanism = { CKM_DES3_KEY_GEN, NULL_PTR, 0 };
	CK_OBJECT_HANDLE hKey = CK_INVALID_HANDLE;
	CK_BBOOL bFalse = CK_FALSE;
	CK_BBOOL bTrue = CK_TRUE;
	CK_BBOOL bToken = IN_SESSION;

	CK_ATTRIBUTE keyAttribs[] =
		{
		{ CKA_TOKEN, &bToken, sizeof(bToken) },
		{ CKA_PRIVATE, &bTrue, sizeof(bTrue) },
		{ CKA_MODIFIABLE, &bTrue, sizeof(bTrue) },
		{ CKA_ENCRYPT, &bTrue, sizeof(bTrue) },
		{ CKA_DECRYPT, &bTrue, sizeof(bTrue) },
		{ CKA_WRAP, &bTrue, sizeof(bTrue) }
	};

	// Just make sure that we finalize any previous tests
	CRYPTOKI_F_PTR( C_Finalize(NULL_PTR) );

	// Initialize the library and start the test.
	rv = CRYPTOKI_F_PTR( C_Initialize(NULL_PTR) );
	CPPUNIT_ASSERT(rv == CKR_OK);

	// Open read-write session
	rv = CRYPTOKI_F_PTR( C_OpenSession(m_initializedTokenSlotID, CKF_SERIAL_SESSION | CKF_RW_SESSION, NULL_PTR, NULL_PTR, &hSession) );
	CPPUNIT_ASSERT(rv == CKR_OK);

	// Login USER into the sessions so we can create a private objects
	rv = CRYPTOKI_F_PTR( C_Login(hSession, CKU_USER, m_userPin1, m_userPin1Length) );
	CPPUNIT_ASSERT(rv==CKR_OK);

	rv = CRYPTOKI_F_PTR( C_GenerateKey(hSession, &mechanism,
		keyAttribs, sizeof(keyAttribs)/sizeof(CK_ATTRIBUTE),
		&hKey) );
	CPPUNIT_ASSERT(rv == CKR_OK);

	rv = CRYPTOKI_F_PTR( C_DestroyObject(hSession, hKey) );
	CPPUNIT_ASSERT(rv == CKR_OK);

	// The C_GenerateKey call failed if CKA_MODIFIABLE was bFalse
	// This was a bug in the SoftHSM implementation
	keyAttribs[2].pValue = &bFalse;
	keyAttribs[2].ulValueLen = sizeof(bFalse);

	rv = CRYPTOKI_F_PTR( C_GenerateKey(hSession, &mechanism,
		keyAttribs, sizeof(keyAttribs) / sizeof(CK_ATTRIBUTE),
		&hKey) );
	// The call would fail with CKR_ATTRIBUTE_READ_ONLY
	CPPUNIT_ASSERT(rv == CKR_OK);

	// Now create a template where the CKA_MODIFIABLE attribute is last in the list
	CK_ATTRIBUTE keyAttribs1[] =
	{
		{ CKA_TOKEN, &bToken, sizeof(bToken) },
		{ CKA_PRIVATE, &bTrue, sizeof(bTrue) },
		{ CKA_ENCRYPT, &bTrue, sizeof(bTrue) },
		{ CKA_DECRYPT, &bTrue, sizeof(bTrue) },
		{ CKA_WRAP, &bTrue, sizeof(bTrue) },
		{ CKA_MODIFIABLE, &bTrue, sizeof(bTrue) }
	};

	rv = CRYPTOKI_F_PTR( C_GenerateKey(hSession, &mechanism,
		keyAttribs1, sizeof(keyAttribs1) / sizeof(CK_ATTRIBUTE),
		&hKey) );
	CPPUNIT_ASSERT(rv == CKR_OK);

	// Now when CKA_MODIFIABLE is bFalse the key generation succeeds
	keyAttribs1[2].pValue = &bFalse;
	keyAttribs1[2].ulValueLen = sizeof(bFalse);

	rv = CRYPTOKI_F_PTR( C_GenerateKey(hSession, &mechanism,
		keyAttribs1, sizeof(keyAttribs1) / sizeof(CK_ATTRIBUTE),
		&hKey) );
	CPPUNIT_ASSERT(rv == CKR_OK);
}

void SymmetricAlgorithmTests::testCheckValue()
{
	CK_RV rv;
	CK_SESSION_HANDLE hSession;
	CK_MECHANISM mechanism = { CKM_AES_KEY_GEN, NULL_PTR, 0 };
	CK_OBJECT_HANDLE hKey = CK_INVALID_HANDLE;

	// Just make sure that we finalize any previous tests
	CRYPTOKI_F_PTR( C_Finalize(NULL_PTR) );

	// Initialize the library and start the test.
	rv = CRYPTOKI_F_PTR( C_Initialize(NULL_PTR) );
	CPPUNIT_ASSERT(rv == CKR_OK);

	// Open read-write session
	rv = CRYPTOKI_F_PTR( C_OpenSession(m_initializedTokenSlotID, CKF_SERIAL_SESSION | CKF_RW_SESSION, NULL_PTR, NULL_PTR, &hSession) );
	CPPUNIT_ASSERT(rv == CKR_OK);

	// Login USER into the sessions so we can create a private objects
	rv = CRYPTOKI_F_PTR( C_Login(hSession, CKU_USER, m_userPin1, m_userPin1Length) );
	CPPUNIT_ASSERT(rv==CKR_OK);

	CK_ULONG bytes = 16;
	CK_BYTE pCheckValue[] = { 0x2b, 0x84, 0xf6 };
	CK_BBOOL bFalse = CK_FALSE;
	CK_BBOOL bTrue = CK_TRUE;
	CK_ATTRIBUTE keyAttribs[] = {
		{ CKA_TOKEN, &bFalse, sizeof(bFalse) },
		{ CKA_PRIVATE, &bTrue, sizeof(bTrue) },
		{ CKA_ENCRYPT, &bTrue, sizeof(bTrue) },
		{ CKA_DECRYPT, &bTrue, sizeof(bTrue) },
		{ CKA_WRAP, &bTrue, sizeof(bTrue) },
		{ CKA_UNWRAP, &bTrue, sizeof(bTrue) },
		{ CKA_VALUE_LEN, &bytes, sizeof(bytes) },
		{ CKA_CHECK_VALUE, &pCheckValue, sizeof(pCheckValue) }
	};

	rv = CRYPTOKI_F_PTR( C_GenerateKey(hSession, &mechanism,
			   keyAttribs, 8,
			   &hKey) );
	CPPUNIT_ASSERT(rv == CKR_ATTRIBUTE_VALUE_INVALID);

	keyAttribs[7].ulValueLen = 0;
	rv = CRYPTOKI_F_PTR( C_GenerateKey(hSession, &mechanism,
			   keyAttribs, 8,
			   &hKey) );
	CPPUNIT_ASSERT(rv == CKR_OK);

	CK_ATTRIBUTE checkAttrib[] = {
		{ CKA_CHECK_VALUE, &pCheckValue, sizeof(pCheckValue) }
	};

	rv = CRYPTOKI_F_PTR( C_GetAttributeValue(hSession, hKey, checkAttrib, 1) );
	CPPUNIT_ASSERT(rv == CKR_OK);
	CPPUNIT_ASSERT(checkAttrib[0].ulValueLen == 0);

	rv = CRYPTOKI_F_PTR( C_DestroyObject(hSession, hKey) );
	CPPUNIT_ASSERT(rv == CKR_OK);

	rv = CRYPTOKI_F_PTR( C_GenerateKey(hSession, &mechanism,
			   keyAttribs, 7,
			   &hKey) );
	CPPUNIT_ASSERT(rv == CKR_OK);

	checkAttrib[0].ulValueLen = sizeof(pCheckValue);
	rv = CRYPTOKI_F_PTR( C_GetAttributeValue(hSession, hKey, checkAttrib, 1) );
	CPPUNIT_ASSERT(rv == CKR_OK);
	CPPUNIT_ASSERT(checkAttrib[0].ulValueLen == 3);

	rv = CRYPTOKI_F_PTR( C_DestroyObject(hSession, hKey) );
	CPPUNIT_ASSERT(rv == CKR_OK);
}

void SymmetricAlgorithmTests::testAesCtrOverflow()
{
	CK_RV rv;
	CK_SESSION_HANDLE hSession;

	// Just make sure that we finalize any previous tests
	CRYPTOKI_F_PTR( C_Finalize(NULL_PTR) );

	// Initialize the library and start the test.
	rv = CRYPTOKI_F_PTR( C_Initialize(NULL_PTR) );
	CPPUNIT_ASSERT(rv == CKR_OK);

	// Open read-write session
	rv = CRYPTOKI_F_PTR( C_OpenSession(m_initializedTokenSlotID, CKF_SERIAL_SESSION | CKF_RW_SESSION, NULL_PTR, NULL_PTR, &hSession) );
	CPPUNIT_ASSERT(rv == CKR_OK);

	// Login USER into the session so we can create a private objects
	rv = CRYPTOKI_F_PTR( C_Login(hSession,CKU_USER,m_userPin1,m_userPin1Length) );
	CPPUNIT_ASSERT(rv==CKR_OK);

	CK_OBJECT_HANDLE hKey = CK_INVALID_HANDLE;

	// Generate a session keys.
	rv = generateAesKey(hSession,IN_SESSION,IS_PUBLIC,hKey);
	CPPUNIT_ASSERT(rv == CKR_OK);

	CK_MECHANISM mechanism = { CKM_AES_CTR, NULL_PTR, 0 };
	CK_AES_CTR_PARAMS ctrParams =
	{
		2,
		{
			0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01
		}
	};
	mechanism.pParameter = &ctrParams;
	mechanism.ulParameterLen = sizeof(ctrParams);

	CK_BYTE plainText[] = { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
				0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
				0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
				0x00 };
	std::vector<CK_BYTE> vEncryptedData;
	std::vector<CK_BYTE> vEncryptedDataParted;
	std::vector<CK_BYTE> vDecryptedData;
	std::vector<CK_BYTE> vDecryptedDataParted;
	CK_ULONG ulEncryptedDataLen;
	CK_ULONG ulEncryptedPartLen;
	CK_ULONG ulDataLen;
	CK_ULONG ulDataPartLen;

	// Single-part encryption
	rv = CRYPTOKI_F_PTR( C_EncryptInit(hSession,&mechanism,hKey) );
	CPPUNIT_ASSERT_EQUAL( (CK_RV)CKR_OK, rv );
	rv = CRYPTOKI_F_PTR( C_Encrypt(hSession,plainText,sizeof(plainText),NULL_PTR,&ulEncryptedDataLen) );
	CPPUNIT_ASSERT_EQUAL( (CK_RV)CKR_DATA_LEN_RANGE, rv );
	rv = CRYPTOKI_F_PTR( C_EncryptInit(hSession,&mechanism,hKey) );
	CPPUNIT_ASSERT_EQUAL( (CK_RV)CKR_OK, rv );
	rv = CRYPTOKI_F_PTR( C_Encrypt(hSession,plainText,sizeof(plainText)-1,NULL_PTR,&ulEncryptedDataLen) );
	CPPUNIT_ASSERT_EQUAL( (CK_RV)CKR_OK, rv );
	vEncryptedData.resize(ulEncryptedDataLen);
	rv = CRYPTOKI_F_PTR( C_Encrypt(hSession,plainText,sizeof(plainText)-1,&vEncryptedData.front(),&ulEncryptedDataLen) );
	CPPUNIT_ASSERT_EQUAL( (CK_RV)CKR_OK, rv );
	vEncryptedData.resize(ulEncryptedDataLen);

	// Multi-part encryption
	rv = CRYPTOKI_F_PTR( C_EncryptInit(hSession,&mechanism,hKey) );
	CPPUNIT_ASSERT_EQUAL( (CK_RV)CKR_OK, rv );
	rv = CRYPTOKI_F_PTR( C_EncryptUpdate(hSession,plainText,sizeof(plainText)-1,NULL_PTR,&ulEncryptedPartLen) );
	CPPUNIT_ASSERT_EQUAL( (CK_RV)CKR_OK, rv );
	vEncryptedDataParted.resize(ulEncryptedPartLen);
	rv = CRYPTOKI_F_PTR( C_EncryptUpdate(hSession,plainText,sizeof(plainText)-1,&vEncryptedDataParted.front(),&ulEncryptedPartLen) );
	CPPUNIT_ASSERT_EQUAL( (CK_RV)CKR_OK, rv );
	vEncryptedDataParted.resize(ulEncryptedPartLen);
	rv = CRYPTOKI_F_PTR( C_EncryptUpdate(hSession,plainText,1,NULL_PTR,&ulEncryptedPartLen) );
	CPPUNIT_ASSERT_EQUAL( (CK_RV)CKR_DATA_LEN_RANGE, rv );

	// Single-part decryption
	rv = CRYPTOKI_F_PTR( C_DecryptInit(hSession,&mechanism,hKey) );
	CPPUNIT_ASSERT_EQUAL( (CK_RV)CKR_OK, rv );
	rv = CRYPTOKI_F_PTR( C_Decrypt(hSession,&vEncryptedData.front(),vEncryptedData.size()+1,NULL_PTR,&ulDataLen) );
	CPPUNIT_ASSERT_EQUAL( (CK_RV)CKR_ENCRYPTED_DATA_LEN_RANGE, rv );
	rv = CRYPTOKI_F_PTR( C_DecryptInit(hSession,&mechanism,hKey) );
	CPPUNIT_ASSERT_EQUAL( (CK_RV)CKR_OK, rv );
	rv = CRYPTOKI_F_PTR( C_Decrypt(hSession,&vEncryptedData.front(),vEncryptedData.size(),NULL_PTR,&ulDataLen) );
	CPPUNIT_ASSERT_EQUAL( (CK_RV)CKR_OK, rv );
	vDecryptedData.resize(ulDataLen);
	rv = CRYPTOKI_F_PTR( C_Decrypt(hSession,&vEncryptedData.front(),vEncryptedData.size(),&vDecryptedData.front(),&ulDataLen) );
	CPPUNIT_ASSERT_EQUAL( (CK_RV)CKR_OK, rv );
	vDecryptedData.resize(ulDataLen);

	// Multi-part decryption
	rv = CRYPTOKI_F_PTR( C_DecryptInit(hSession,&mechanism,hKey) );
	CPPUNIT_ASSERT_EQUAL( (CK_RV)CKR_OK, rv );
	rv = CRYPTOKI_F_PTR( C_DecryptUpdate(hSession,&vEncryptedData.front(),vEncryptedData.size(),NULL_PTR,&ulDataPartLen) );
	CPPUNIT_ASSERT_EQUAL( (CK_RV)CKR_OK, rv );
	vDecryptedDataParted.resize(ulDataPartLen);
	rv = CRYPTOKI_F_PTR( C_DecryptUpdate(hSession,&vEncryptedData.front(),vEncryptedData.size(),&vDecryptedDataParted.front(),&ulDataPartLen) );
	CPPUNIT_ASSERT_EQUAL( (CK_RV)CKR_OK, rv );
	vDecryptedDataParted.resize(ulDataPartLen);
	rv = CRYPTOKI_F_PTR( C_DecryptUpdate(hSession,&vEncryptedData.front(),1,NULL_PTR,&ulDataPartLen) );
	CPPUNIT_ASSERT_EQUAL( (CK_RV)CKR_ENCRYPTED_DATA_LEN_RANGE, rv );
}

void SymmetricAlgorithmTests::testGenericKey()
{
	CK_RV rv;
	CK_SESSION_HANDLE hSession;

	// Just make sure that we finalize any previous tests
	CRYPTOKI_F_PTR( C_Finalize(NULL_PTR) );

	// Initialize the library and start the test.
	rv = CRYPTOKI_F_PTR( C_Initialize(NULL_PTR) );
	CPPUNIT_ASSERT(rv == CKR_OK);

	// Open read-write session
	rv = CRYPTOKI_F_PTR( C_OpenSession(m_initializedTokenSlotID, CKF_SERIAL_SESSION | CKF_RW_SESSION, NULL_PTR, NULL_PTR, &hSession) );
	CPPUNIT_ASSERT(rv == CKR_OK);

	// Login USER into the session so we can create a private objects
	rv = CRYPTOKI_F_PTR( C_Login(hSession,CKU_USER,m_userPin1,m_userPin1Length) );
	CPPUNIT_ASSERT(rv==CKR_OK);

	CK_OBJECT_HANDLE hKey = CK_INVALID_HANDLE;

	// Generate a session key.
	rv = generateGenericKey(hSession,IN_SESSION,IS_PUBLIC,hKey);
	CPPUNIT_ASSERT(rv == CKR_OK);
}

void SymmetricAlgorithmTests::testEncDecFinalNULLValidation()
{
	CK_RV rv;
	CK_SESSION_HANDLE hSession;

	// Just make sure that we finalize any previous tests
	CRYPTOKI_F_PTR( C_Finalize(NULL_PTR) );

	// Initialize the library and start the test.
	rv = CRYPTOKI_F_PTR( C_Initialize(NULL_PTR) );
	CPPUNIT_ASSERT(rv == CKR_OK);

	// Open read-write session
	rv = CRYPTOKI_F_PTR( C_OpenSession(m_initializedTokenSlotID, CKF_SERIAL_SESSION | CKF_RW_SESSION, NULL_PTR, NULL_PTR, &hSession) );
	CPPUNIT_ASSERT(rv == CKR_OK);

	// Login USER into the sessions so we can create a private objects
	rv = CRYPTOKI_F_PTR( C_Login(hSession,CKU_USER,m_userPin1,m_userPin1Length) );
	CPPUNIT_ASSERT(rv==CKR_OK);

	CK_OBJECT_HANDLE hKey = CK_INVALID_HANDLE;

	// Generate all combinations of session/token keys.
	rv = generateAesKey(hSession,IN_SESSION,IS_PUBLIC,hKey);

	CPPUNIT_ASSERT(rv == CKR_OK);
	
	CK_MECHANISM mechanism = { CKM_AES_CTR, NULL_PTR, 0 };
	CK_AES_CTR_PARAMS ctrParams =
	{
		2,
		{
			0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01
		}
	};
	mechanism.pParameter = &ctrParams;
	mechanism.ulParameterLen = sizeof(ctrParams);
	CK_BYTE plainText[] = { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
				0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
				0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
				0x00 };
	std::vector<CK_BYTE> vEncryptedData;
	std::vector<CK_BYTE> vEncryptedDataParted;
	std::vector<CK_BYTE> vDecryptedData;
	std::vector<CK_BYTE> vDecryptedDataParted;
	CK_ULONG ulEncryptedDataLen;
	CK_ULONG ulEncryptedPartLen;
	CK_ULONG ulDecryptedPartLen;
	CK_ULONG ulDataPartLen;

	// Single-part encryption
	rv = CRYPTOKI_F_PTR( C_EncryptInit(hSession,&mechanism,hKey) );
	CPPUNIT_ASSERT_EQUAL( (CK_RV)CKR_OK, rv );
	rv = CRYPTOKI_F_PTR( C_Encrypt(hSession,plainText,sizeof(plainText)-1,NULL_PTR,&ulEncryptedDataLen) );
	CPPUNIT_ASSERT_EQUAL( (CK_RV)CKR_OK, rv );
	vEncryptedData.resize(ulEncryptedDataLen);
	rv = CRYPTOKI_F_PTR( C_Encrypt(hSession,plainText,sizeof(plainText)-1,&vEncryptedData.front(),&ulEncryptedDataLen) );
	CPPUNIT_ASSERT_EQUAL( (CK_RV)CKR_OK, rv );
	vEncryptedData.resize(ulEncryptedDataLen);

	// Multi-part encryption
	rv = CRYPTOKI_F_PTR( C_EncryptInit(hSession,&mechanism,hKey) );
	CPPUNIT_ASSERT_EQUAL( (CK_RV)CKR_OK, rv );
	rv = CRYPTOKI_F_PTR( C_EncryptUpdate(hSession,plainText,sizeof(plainText)-1,NULL_PTR,&ulEncryptedPartLen) );
	CPPUNIT_ASSERT_EQUAL( (CK_RV)CKR_OK, rv );
	vEncryptedDataParted.resize(ulEncryptedPartLen);
	rv = CRYPTOKI_F_PTR( C_EncryptUpdate(hSession,plainText,sizeof(plainText)-1,&vEncryptedDataParted.front(),&ulEncryptedPartLen) );
	CPPUNIT_ASSERT_EQUAL( (CK_RV)CKR_OK, rv );

	// Test input validation
	rv = CRYPTOKI_F_PTR( C_EncryptFinal(hSession, NULL_PTR, NULL_PTR) );
	CPPUNIT_ASSERT_EQUAL( (CK_RV)CKR_ARGUMENTS_BAD, rv );
	ulEncryptedPartLen = 0;
	rv = CRYPTOKI_F_PTR( C_EncryptFinal(hSession, NULL_PTR, &ulEncryptedPartLen) );
	CPPUNIT_ASSERT_EQUAL( (CK_RV)CKR_OPERATION_NOT_INITIALIZED, rv );

	// Multi-part decryption
	rv = CRYPTOKI_F_PTR( C_DecryptInit(hSession,&mechanism,hKey) );
	CPPUNIT_ASSERT_EQUAL( (CK_RV)CKR_OK, rv );
	rv = CRYPTOKI_F_PTR( C_DecryptUpdate(hSession,&vEncryptedData.front(),vEncryptedData.size(),NULL_PTR,&ulDataPartLen) );
	CPPUNIT_ASSERT_EQUAL( (CK_RV)CKR_OK, rv );
	vDecryptedDataParted.resize(ulDataPartLen);
	rv = CRYPTOKI_F_PTR( C_DecryptUpdate(hSession,&vEncryptedData.front(),vEncryptedData.size(),&vDecryptedDataParted.front(),&ulDataPartLen) );
	CPPUNIT_ASSERT_EQUAL( (CK_RV)CKR_OK, rv );
	
	// Test input validation
	rv = CRYPTOKI_F_PTR( C_DecryptFinal(hSession, NULL_PTR, NULL_PTR) );
	CPPUNIT_ASSERT_EQUAL( (CK_RV)CKR_ARGUMENTS_BAD, rv );
	ulDecryptedPartLen = 0;
	rv = CRYPTOKI_F_PTR( C_DecryptFinal(hSession, NULL_PTR, &ulDecryptedPartLen) );
	CPPUNIT_ASSERT_EQUAL( (CK_RV)CKR_OPERATION_NOT_INITIALIZED, rv );

	CRYPTOKI_F_PTR( C_Finalize(NULL_PTR) );
}

void SymmetricAlgorithmTests::testOpTermIssue585()
{
	CK_RV rv;
	CK_SESSION_HANDLE hSessionRO;
	CK_SESSION_HANDLE hSessionRW;

	// Just make sure that we finalize any previous tests
	CRYPTOKI_F_PTR( C_Finalize(NULL_PTR) );

	// Open read-only session on when the token is not initialized should fail
	rv = CRYPTOKI_F_PTR( C_OpenSession(m_initializedTokenSlotID, CKF_SERIAL_SESSION, NULL_PTR, NULL_PTR, &hSessionRO) );
	CPPUNIT_ASSERT(rv == CKR_CRYPTOKI_NOT_INITIALIZED);

	// Initialize the library and start the test.
	rv = CRYPTOKI_F_PTR( C_Initialize(NULL_PTR) );
	CPPUNIT_ASSERT(rv == CKR_OK);

	// Open read-only session
	rv = CRYPTOKI_F_PTR( C_OpenSession(m_initializedTokenSlotID, CKF_SERIAL_SESSION, NULL_PTR, NULL_PTR, &hSessionRO) );
	CPPUNIT_ASSERT(rv == CKR_OK);

	// Open read-write session
	rv = CRYPTOKI_F_PTR( C_OpenSession(m_initializedTokenSlotID, CKF_SERIAL_SESSION | CKF_RW_SESSION, NULL_PTR, NULL_PTR, &hSessionRW) );
	CPPUNIT_ASSERT(rv == CKR_OK);

	// Login USER into the sessions so we can create a private objects
	rv = CRYPTOKI_F_PTR( C_Login(hSessionRO,CKU_USER,m_userPin1,m_userPin1Length) );
	CPPUNIT_ASSERT(rv==CKR_OK);

	CK_OBJECT_HANDLE hKey = CK_INVALID_HANDLE;

	// Generate test key
	rv = generateAesKey(hSessionRW,IN_SESSION,IS_PUBLIC,hKey);
	CPPUNIT_ASSERT(rv == CKR_OK);

	CK_BYTE iv[16] = { 0 };
	CK_MECHANISM mechanism = { CKM_AES_CBC, iv, 16 };

	// Initialise encryption
	rv = CRYPTOKI_F_PTR( C_EncryptInit(hSessionRW, &mechanism, hKey) );
	CPPUNIT_ASSERT(rv == CKR_OK);

	CK_BYTE data[16] = { 0 };
	CK_ULONG ulEncryptedDataLen = 0;

	// Attempt call to C_Encrypt with NULL for pData
	rv = CRYPTOKI_F_PTR( C_Encrypt(hSessionRW, NULL, 16, NULL, &ulEncryptedDataLen) );
	CPPUNIT_ASSERT(rv == CKR_ARGUMENTS_BAD);

	// Verify operation was terminated
	rv = CRYPTOKI_F_PTR( C_Encrypt(hSessionRW, data, 16, NULL, &ulEncryptedDataLen) );
	CPPUNIT_ASSERT(rv == CKR_OPERATION_NOT_INITIALIZED);

	// Initialise encryption
	rv = CRYPTOKI_F_PTR( C_EncryptInit(hSessionRW, &mechanism, hKey) );
	CPPUNIT_ASSERT(rv == CKR_OK);

	// Attempt call to C_Encrypt with NULL for pulEncryptedDataLen
	rv = CRYPTOKI_F_PTR( C_Encrypt(hSessionRW, data, 16, NULL, NULL) );
	CPPUNIT_ASSERT(rv == CKR_ARGUMENTS_BAD);

	// Verify operation was terminated
	rv = CRYPTOKI_F_PTR( C_Encrypt(hSessionRW, data, 16, NULL, &ulEncryptedDataLen) );
	CPPUNIT_ASSERT(rv == CKR_OPERATION_NOT_INITIALIZED);

	// Attempt call to C_EncryptUpdate with NULL for pData
	rv = CRYPTOKI_F_PTR( C_EncryptUpdate(hSessionRW, NULL, 16, NULL, &ulEncryptedDataLen) );
	CPPUNIT_ASSERT(rv == CKR_ARGUMENTS_BAD);

	// Verify operation was terminated
	rv = CRYPTOKI_F_PTR( C_EncryptUpdate(hSessionRW, data, 16, NULL, &ulEncryptedDataLen) );
	CPPUNIT_ASSERT(rv == CKR_OPERATION_NOT_INITIALIZED);

	// Initialise encryption
	rv = CRYPTOKI_F_PTR( C_EncryptInit(hSessionRW, &mechanism, hKey) );
	CPPUNIT_ASSERT(rv == CKR_OK);

	// Attempt call to C_EncryptUpdate with NULL for pulEncryptedDataLen
	rv = CRYPTOKI_F_PTR( C_EncryptUpdate(hSessionRW, data, 16, NULL, NULL) );
	CPPUNIT_ASSERT(rv == CKR_ARGUMENTS_BAD);

	// Verify operation was terminated
	rv = CRYPTOKI_F_PTR( C_EncryptUpdate(hSessionRW, data, 16, NULL, &ulEncryptedDataLen) );
	CPPUNIT_ASSERT(rv == CKR_OPERATION_NOT_INITIALIZED);

	// Initialise decryption
	rv = CRYPTOKI_F_PTR( C_DecryptInit(hSessionRW, &mechanism, hKey) );
	CPPUNIT_ASSERT(rv == CKR_OK);

	CK_ULONG ulDecryptedDataLen = 0;

	// Attempt call to C_Decrypt with NULL for pData
	rv = CRYPTOKI_F_PTR( C_Decrypt(hSessionRW, NULL, 16, NULL, &ulDecryptedDataLen) );
	CPPUNIT_ASSERT(rv == CKR_ARGUMENTS_BAD);

	// Verify operation was terminated
	rv = CRYPTOKI_F_PTR( C_Decrypt(hSessionRW, data, 16, NULL, &ulDecryptedDataLen) );
	CPPUNIT_ASSERT(rv == CKR_OPERATION_NOT_INITIALIZED);

	// Initialise decryption
	rv = CRYPTOKI_F_PTR( C_DecryptInit(hSessionRW, &mechanism, hKey) );
	CPPUNIT_ASSERT(rv == CKR_OK);

	// Attempt call to C_Decrypt with NULL for pulDecryptedDataLen
	rv = CRYPTOKI_F_PTR( C_Decrypt(hSessionRW, data, 16, NULL, NULL) );
	CPPUNIT_ASSERT(rv == CKR_ARGUMENTS_BAD);

	// Verify operation was terminated
	rv = CRYPTOKI_F_PTR( C_Decrypt(hSessionRW, data, 16, NULL, &ulDecryptedDataLen) );
	CPPUNIT_ASSERT(rv == CKR_OPERATION_NOT_INITIALIZED);

	// Attempt call to C_DecryptUpdate with NULL for pData
	rv = CRYPTOKI_F_PTR( C_DecryptUpdate(hSessionRW, NULL, 16, NULL, &ulDecryptedDataLen) );
	CPPUNIT_ASSERT(rv == CKR_ARGUMENTS_BAD);

	// Verify operation was terminated
	rv = CRYPTOKI_F_PTR( C_DecryptUpdate(hSessionRW, data, 16, NULL, &ulDecryptedDataLen) );
	CPPUNIT_ASSERT(rv == CKR_OPERATION_NOT_INITIALIZED);

	// Initialise decryption
	rv = CRYPTOKI_F_PTR( C_DecryptInit(hSessionRW, &mechanism, hKey) );
	CPPUNIT_ASSERT(rv == CKR_OK);

	// Attempt call to C_DecryptUpdate with NULL for pulDecryptedDataLen
	rv = CRYPTOKI_F_PTR( C_DecryptUpdate(hSessionRW, data, 16, NULL, NULL) );
	CPPUNIT_ASSERT(rv == CKR_ARGUMENTS_BAD);

	// Verify operation was terminated
	rv = CRYPTOKI_F_PTR( C_DecryptUpdate(hSessionRW, data, 16, NULL, &ulDecryptedDataLen) );
	CPPUNIT_ASSERT(rv == CKR_OPERATION_NOT_INITIALIZED);
}

