/*
 * Copyright (c) 2002-2005 The TenDRA Project <http://www.tendra.org/>.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 3. Neither the name of The TenDRA Project nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific, prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: types.c 1228 2007-05-31 23:40:52Z kate $
 */
/*
    		 Crown Copyright (c) 1997

    This TenDRA(r) Computer Program is subject to Copyright
    owned by the United Kingdom Secretary of State for Defence
    acting through the Defence Evaluation and Research Agency
    (DERA).  It is made available to Recipients with a
    royalty-free licence for its use, reproduction, transfer
    to other parties and amendment for any purpose not excluding
    product development provided that any such use et cetera
    shall be deemed to be acceptance of the following conditions:-

        (1) Its Recipients shall ensure that this Notice is
        reproduced upon any copies or amended versions of it;

        (2) Any amended version of it shall be clearly marked to
        show both the nature of and the organisation responsible
        for the relevant amendment or amendments;

        (3) Its onward transfer from a recipient to another
        party shall be deemed to be that party's acceptance of
        these conditions;

        (4) DERA gives no warranty or assurance as to its
        quality or suitability for any purpose and DERA accepts
        no liability whatsoever in relation to any use to which
        it may be put.
*/

/*
 * types.c - Type ADTs.
 *
 * This file implements the type manipulation routines specified.
 */

#include <assert.h>
#include <stddef.h>
#include <limits.h>

#include "types.h"
#include "../eds/dalloc.h"
#include "../gen-errors.h"
#include "name.h"
#include "rstack.h"
#include "../rules/rule.h"
#include "table.h"

/*
 * Type tuple handling fuctions
 */

static void
types_add_name_and_type_1(TypeTupleT * to, EntryT * name, EntryT * type,
			  BoolT reference, BoolT assign)
{
    TypeTupleEntryT * link = ALLOCATE(TypeTupleEntryT);

    link->next      = NULL;
    link->type      = type;
    link->name      = name;
    link->reference = reference;
    link->mutated   = FALSE;
    link->assign    = assign;
    *(to->tail)    = link;
    to->tail        = &(link->next);
}

static void
types_iter_alt_item_type_names(AltT * alts, void(*proc)(NameT *))
{
    AltT *            alt;
    TypeTupleEntryT * type;

    for (alt = alts; alt; alt = alt_next(alt)) {
	ItemT * item;

	for (item = alt_item_head(alt); item; item = item_next(item)) {
	    TypeTupleT * param  = item_param(item);
	    TypeTupleT * result = item_result(item);

	    for (type = param->head; type; type = type->next) {
		(*proc)(entry_get_name(type->name));
	    }
	    for (type = result->head; type; type = type->next) {
		(*proc)(entry_get_name(type->name));
	    }
	}
    }
}

void
types_init(TypeTupleT * tuple)
{
    tuple->head = NULL;
    tuple->tail = &(tuple->head);
}

void
types_copy(TypeTupleT * to, TypeTupleT * from)
{
    TypeTupleEntryT * from_ptr;

    to->head = NULL;
    to->tail = &(to->head);
    for (from_ptr = from->head; from_ptr; from_ptr = from_ptr->next) {
	types_add_name_and_type_1(to, from_ptr->name, from_ptr->type,
				   from_ptr->reference, from_ptr->assign);
    }
}

void
types_copy_and_translate(TypeTupleT * to, TypeTupleT * from, TypeTransT * translator,
			 TableT * table)
{
    TypeTupleEntryT * from_ptr;

    to->head = NULL;
    to->tail = &(to->head);
    for (from_ptr = from->head; from_ptr; from_ptr = from_ptr->next) {
	EntryT * new_name;

	new_name = trans_get_translation(translator, from_ptr->name);
	if (new_name == NULL) {
	    new_name = table_add_generated_name(table);
	    trans_add_translation(translator, from_ptr->name, new_name);
	}
	types_add_name_and_type_1(to, new_name, from_ptr->type,
				   from_ptr->reference, from_ptr->assign);
    }
}

void
types_append_copy(TypeTupleT * to, TypeTupleT * from)
{
    TypeTupleEntryT * from_ptr;

    for (from_ptr = from->head; from_ptr; from_ptr = from_ptr->next) {
	types_add_name_and_type_1(to, from_ptr->name, from_ptr->type,
				   from_ptr->reference, from_ptr->assign);
    }
}

void
types_translate(TypeTupleT * tuple, TypeBTransT * translator)
{
    TypeTupleEntryT * tuple_ptr;

    for (tuple_ptr = tuple->head; tuple_ptr; tuple_ptr = tuple_ptr->next) {
	EntryT * new_name;

	new_name = btrans_get_translation(translator, tuple_ptr->name);
	if (new_name != NULL) {
	    tuple_ptr->name = new_name;
	}
    }
}

void
types_renumber(TypeTupleT * tuple, TypeNTransT * translator)
{
    TypeTupleEntryT * tuple_ptr;

    for (tuple_ptr = tuple->head; tuple_ptr; tuple_ptr = tuple_ptr->next) {
	if (!entry_is_non_local(tuple_ptr->name)) {
	    tuple_ptr->number = ntrans_get_translation(translator,
							tuple_ptr->name);
	}
    }
}

void
types_assign(TypeTupleT * to, TypeTupleT * from)
{
    if ((to->head = from->head) != NULL) {
	to->tail = from->tail;
    } else {
	to->tail = &(to->head);
    }
}

EntryT *
types_find_name_type(TypeTupleT * tuple, EntryT * name, BoolT *reference_ref)
{
    TypeTupleEntryT * type;

    for (type = tuple->head; type; type = type->next) {
	if (type->name == name) {
	    *reference_ref = type->reference;
	    return(type->type);
	}
    }
    return(NULL);
}

BoolT
types_mutated(TypeTupleT * tuple, EntryT * name)
{
    TypeTupleEntryT * type;

    for (type = tuple->head; type; type = type->next) {
	if (type->name == name) {
	    type->mutated = TRUE;
	    return(TRUE);
	}
    }
    return(FALSE);
}

BoolT
types_compute_mutations(TypeTupleT * rule, TypeTupleT * item, TypeTupleT * action)
{
    BoolT           propogate  = FALSE;
    TypeTupleEntryT * item_ptr   = item->head;
    TypeTupleEntryT * action_ptr = action->head;

    while (action_ptr) {
	assert(item_ptr);
	if (action_ptr->mutated) {
	    TypeTupleEntryT * rule_ptr = rule->head;

	    while (rule_ptr) {
		if ((rule_ptr->name == item_ptr->name) &&
		   (!(rule_ptr->mutated))) {
		    rule_ptr->mutated = TRUE;
		    if (rule_ptr->reference) {
			propogate = TRUE;
		    }
		    break;
		}
		rule_ptr = rule_ptr->next;
	    }
	}
	item_ptr   = item_ptr->next;
	action_ptr = action_ptr->next;
    }
    assert(item_ptr == NULL);
    return(propogate);
}

BoolT
types_compute_assign_mutations(TypeTupleT * rule, TypeTupleT * item)
{
    BoolT           propogate  = FALSE;
    TypeTupleEntryT * item_ptr   = item->head;

    while (item_ptr) {
	if (item_ptr->assign) {
	    TypeTupleEntryT * rule_ptr = rule->head;

	    while (rule_ptr) {
		if ((rule_ptr->name == item_ptr->name) &&
		   (!(rule_ptr->mutated))) {
		    rule_ptr->mutated = TRUE;
		    if (rule_ptr->reference) {
			propogate = TRUE;
		    }
		    break;
		}
		rule_ptr = rule_ptr->next;
	    }
	}
	item_ptr = item_ptr->next;
    }
    return(propogate);
}

void
types_propogate_mutations(TypeTupleT * to, TypeTupleT * from)
{
    TypeTupleEntryT * to_ptr   = to->head;
    TypeTupleEntryT * from_ptr = from->head;

    while (to_ptr) {
	assert(from_ptr);
	to_ptr->mutated = from_ptr->mutated;
	to_ptr          = to_ptr->next;
	from_ptr        = from_ptr->next;
    }
    assert(from_ptr == NULL);
}

BoolT
types_contains(TypeTupleT * tuple, EntryT * name)
{
    TypeTupleEntryT * type;

    for (type = tuple->head; type; type = type->next) {
	if (type->name == name) {
	    return(TRUE);
	}
    }
    return(FALSE);
}

BoolT
types_contains_names(TypeTupleT * tuple)
{
    TypeTupleEntryT * type;

    for (type = tuple->head; type; type = type->next) {
	if (type->name) {
	    return(TRUE);
	}
    }
    return(FALSE);
}

BoolT
types_contains_references(TypeTupleT * tuple)
{
    TypeTupleEntryT * type;

    for (type = tuple->head; type; type = type->next) {
	if (type->reference) {
	    return(TRUE);
	}
    }
    return(FALSE);
}

void
types_make_references(TypeTupleT * param, TypeTupleT * args)
{
    TypeTupleEntryT * ptr;

    for (ptr = param->head; ptr; ptr = ptr->next) {
	ptr->reference = TRUE;
    }
    for (ptr = args->head; ptr; ptr = ptr->next) {
	ptr->reference = TRUE;
    }
}

BoolT
types_intersect(TypeTupleT * tuple1, TypeTupleT * tuple2)
{
    TypeTupleEntryT * type;

    for (type = tuple1->head; type; type = type->next) {
	if (types_contains(tuple2, type->name)) {
	    return(TRUE);
	}
    }
    return(FALSE);
}

void
types_inplace_intersection(TypeTupleT * to, TypeTupleT * from)
{
    TypeTupleEntryT * type;

    to->tail = &(to->head);
    while ((type = *(to->tail)) != NULL) {
	if (!types_contains(from, type->name)) {
	    *(to->tail) = type->next;
	    DEALLOCATE(type);
	} else {
	    to->tail = &(type->next);
	}
    }
}

void
types_compute_intersection(TypeTupleT * to, TypeTupleT * tuple1, TypeTupleT * tuple2)
{
    TypeTupleEntryT * type;

    for (type = tuple1->head; type; type = type->next) {
	if ((types_contains(tuple2, type->name)) &&
	    (!types_contains(to, type->name))) {
	    types_add_name_and_type_1(to, type->name, type->type,
				       type->reference, type->assign);
	}
    }
}

CmpT
types_compare(TypeTupleT * tuple1, TypeTupleT * tuple2)
{
    TypeTupleEntryT * tuple1_ptr = (tuple1->head);
    TypeTupleEntryT * tuple2_ptr = (tuple2->head);

    while (tuple1_ptr && tuple2_ptr) {
	if (tuple1_ptr->number < tuple2_ptr->number) {
	    return(CMP_LT);
	} else if (tuple1_ptr->number > tuple2_ptr->number) {
	    return(CMP_GT);
	}
	switch (key_compare(entry_key(tuple1_ptr->type),
			     entry_key(tuple2_ptr->type)))EXHAUSTIVE {
	  case CMP_LT:
	    return(CMP_LT);
	  case CMP_GT:
	    return(CMP_GT);
	  case CMP_EQ:
	    break;
	}
	if (tuple1_ptr->reference != tuple2_ptr->reference) {
	    return((CmpT)((tuple1_ptr->reference)? CMP_GT : CMP_LT));
	} else if (tuple1_ptr->assign != tuple2_ptr->assign) {
	    return((CmpT)((tuple1_ptr->assign)? CMP_GT : CMP_LT));
	}
	tuple1_ptr = tuple1_ptr->next;
	tuple2_ptr = tuple2_ptr->next;
    }
    if (tuple1_ptr != NULL) {
	return(CMP_GT);
    } else if (tuple2_ptr != NULL) {
	return(CMP_LT);
    } else {
	return(CMP_EQ);
    }
}

BoolT
types_equal(TypeTupleT * tuple1, TypeTupleT * tuple2)
{
    TypeTupleEntryT * tuple1_ptr = (tuple1->head);
    TypeTupleEntryT * tuple2_ptr = (tuple2->head);

    while ((tuple1_ptr) && (tuple2_ptr)) {
	if (((tuple1_ptr->type) != (tuple2_ptr->type)) ||
	    ((tuple1_ptr->reference) != (tuple2_ptr->reference)) ||
	    ((tuple1_ptr->assign) != (tuple2_ptr->assign))) {
	    return(FALSE);
	}
	tuple1_ptr = (tuple1_ptr->next);
	tuple2_ptr = (tuple2_ptr->next);
    }
    return((tuple1_ptr == NULL) &&
	   (tuple2_ptr == NULL));
}

BoolT
types_equal_zero_tuple(TypeTupleT * tuple)
{
    return(tuple->head == NULL);
}

BoolT
types_equal_names(TypeTupleT * tuple1,			   TypeTupleT * tuple2)
{
    TypeTupleEntryT * tuple1_ptr = (tuple1->head);
    TypeTupleEntryT * tuple2_ptr = (tuple2->head);

    while ((tuple1_ptr) && (tuple2_ptr)) {
	if (((tuple1_ptr->type) != (tuple2_ptr->type)) ||
	    ((tuple1_ptr->reference) != (tuple2_ptr->reference)) ||
	    ((tuple1_ptr->assign) != (tuple2_ptr->assign)) ||
	    ((tuple1_ptr->name) != (tuple2_ptr->name))) {
	    return(FALSE);
	}
	tuple1_ptr = (tuple1_ptr->next);
	tuple2_ptr = (tuple2_ptr->next);
    }
    return((tuple1_ptr == NULL) &&
	   (tuple2_ptr == NULL));
}

BoolT
types_equal_numbers(TypeTupleT * tuple1, TypeTupleT * tuple2)
{
    TypeTupleEntryT * tuple1_ptr = (tuple1->head);
    TypeTupleEntryT * tuple2_ptr = (tuple2->head);

    while ((tuple1_ptr) && (tuple2_ptr)) {
	if ((tuple1_ptr->type != tuple2_ptr->type) ||
	    (tuple1_ptr->reference != tuple2_ptr->reference) ||
	    (tuple1_ptr->assign != tuple2_ptr->assign)) {
	    return(FALSE);
	} else if (entry_is_non_local(tuple1_ptr->name) ||
		   entry_is_non_local(tuple2_ptr->name)) {
	    if (tuple1_ptr->name != tuple2_ptr->name) {
		return(FALSE);
	    }
	} else if (tuple1_ptr->number != tuple2_ptr->number) {
	    return(FALSE);
	}
	tuple1_ptr = (tuple1_ptr->next);
	tuple2_ptr = (tuple2_ptr->next);
    }
    return((tuple1_ptr == NULL) &&
	   (tuple2_ptr == NULL));
}

void
types_add_name_and_type(TypeTupleT * to, EntryT * name, EntryT * type,
			BoolT reference)
{
    types_add_name_and_type_1(to, name, type, reference, FALSE);
}

void
types_add_name_and_type_var(TypeTupleT * to, EntryT * name, EntryT * type)
{
    types_add_name_and_type_1(to, name, type, FALSE, TRUE);
}

BoolT
types_add_type(TypeTupleT * tuple, TableT * table, NStringT * name, BoolT reference)
{
    EntryT * entry = table_get_type(table, name);

    if (entry) {
	types_add_name_and_type(tuple, NULL, entry, reference);
	return(TRUE);
    }
    return(FALSE);
}

void
types_add_name(TypeTupleT * tuple, TableT * table, NStringT * name, BoolT reference)
{
    EntryT * entry = table_add_name(table, name);

    types_add_name_and_type(tuple, entry, NULL, reference);
}

BoolT
types_add_typed_name(TypeTupleT * tuple, TableT * table, NStringT * name,
		     NStringT * type, BoolT reference)
{
    EntryT * type_entry = table_get_type(table, type);
    EntryT * name_entry = table_add_name(table, name);

    if (type_entry) {
	types_add_name_and_type(tuple, name_entry, type_entry, reference);
	return(TRUE);
    }
    return(FALSE);
}

void
types_add_name_entry(TypeTupleT * tuple, EntryT * entry)
{
    types_add_name_and_type(tuple, entry, NULL, FALSE);
}

void
types_add_type_entry(TypeTupleT * tuple, EntryT * entry, BoolT reference)
{
    types_add_name_and_type(tuple, NULL, entry, reference);
}

void
types_add_new_names(TypeTupleT * to, TypeTupleT * from, EntryT * exclude)
{
    TypeTupleEntryT * from_ptr;

    for (from_ptr = from->head; from_ptr; from_ptr = from_ptr->next) {
	if ((from_ptr->name != exclude) &&
	    (!(from_ptr->assign)) &&
	    (!types_contains(to, from_ptr->name))) {
	    types_add_name_and_type(to, from_ptr->name, from_ptr->type,
				     from_ptr->reference);
	}
    }
}

BoolT
types_disjoint_names(TypeTupleT * tuple)
{
    BoolT           disjoint = TRUE;
    TypeTupleEntryT * ptr;

    for (ptr = tuple->head; ptr; ptr = ptr->next) {
	if (ptr->name) {
	    if (name_test_and_set_clash(entry_get_name(ptr->name))) {
		disjoint = FALSE;
		goto done;
	    }
	} else {
	    disjoint = FALSE;
	    goto done;
	}
    }
  done:
    for (ptr = tuple->head; ptr; ptr = ptr->next) {
	if (ptr->name) {
	    name_reset_clash(entry_get_name(ptr->name));
	}
    }
    return(disjoint);
}

BoolT
types_resolve(TypeTupleT * to, TypeTupleT * args, TypeTupleT * locals,
	      void (*unknown_proc)(KeyT *, KeyT *, unsigned), KeyT * rule,
	      unsigned alt)
{
    BoolT           ok = TRUE;
    TypeTupleEntryT * name;

    for (name = to->head; name; name = name->next) {
	BoolT reference;

	if (entry_is_non_local(name->name)) {
	    name->type = entry_get_non_local(name->name);
	} else if (((name->type = types_find_name_type(args, name->name,
						       &reference)) ==
		    NULL) &&
		   ((name->type = types_find_name_type(locals, name->name,
						       &reference)) ==
		    NULL)) {
	   (*unknown_proc)(entry_key(name->name), rule, alt);
	    ok = FALSE;
	}
    }
    return(ok);
}

BoolT
types_check_undefined(TypeTupleT * to, TypeTupleT * args, TypeTupleT * locals,
		      void (*error_proc)(KeyT *, KeyT *, unsigned), KeyT * rule,
		      unsigned alt)
{
    BoolT           ok = TRUE;
    TypeTupleEntryT * name;

    for (name = to->head; name; name = name->next) {
	if ((!(name->assign)) &&
	   (entry_is_non_local(name->name) ||
	    types_contains(args, name->name) ||
	    types_contains(locals, name->name))) {
	   (*error_proc)(entry_key(name->name), rule, alt);
	    ok = FALSE;
	}
    }
    return(ok);
}

BoolT
types_fillin_types(TypeTupleT * to, TypeTupleT * from)
{
    TypeTupleEntryT * to_ptr   = to->head;
    TypeTupleEntryT * from_ptr = from->head;

    while ((to_ptr) && (from_ptr)) {
	if (to_ptr->type == NULL) {
	    to_ptr->type      = from_ptr->type;
	    to_ptr->reference = from_ptr->reference;
	} else if ((to_ptr->type != from_ptr->type) ||
		   (to_ptr->reference != from_ptr->reference)) {
	    return(FALSE);
	}
	to_ptr   = to_ptr->next;
	from_ptr = from_ptr->next;
    }
    return((to_ptr == NULL) &&
	   (from_ptr == NULL));
}

BoolT
types_fillin_names(TypeTupleT * to, TypeTupleT * from)
{
    TypeTupleEntryT * to_ptr   = to->head;
    TypeTupleEntryT * from_ptr = from->head;

    while ((to_ptr) && (from_ptr)) {
	assert(to_ptr->name == NULL);
	to_ptr->name = from_ptr->name;
	if ((from_ptr->type) &&
	    ((to_ptr->type != from_ptr->type) ||
	     (to_ptr->reference != from_ptr->reference))) {
	    return(FALSE);
	}
	to_ptr   = to_ptr->next;
	from_ptr = from_ptr->next;
    }
    return((to_ptr == NULL) &&
	   (from_ptr == NULL));
}

BoolT
types_check_names(TypeTupleT * to, TypeTupleT * from)
{
    TypeTupleEntryT * to_ptr;

    for (to_ptr = to->head; to_ptr; to_ptr = to_ptr->next) {
	BoolT reference;

	if ((types_find_name_type(from, to_ptr->name, &reference) !=
	     to_ptr->type) || (reference != to_ptr->reference)) {
	    return(FALSE);
	}
    }
    return(TRUE);
}

void
types_check_used(TypeTupleT * tuple, void (*error_proc)(void *, EntryT *),
		 void * gclosure)
{
    TypeTupleEntryT * ptr;

    for (ptr = tuple->head; ptr; ptr = ptr->next) {
	assert(!entry_is_non_local(ptr->name));
	if (!name_is_used(entry_get_name(ptr->name))) {
	   (*error_proc)(gclosure, ptr->name);
	}
    }
}

void
types_unlink_used(TypeTupleT * to, TypeTupleT * from)
{
    TypeTupleEntryT * type;

    to->tail = &(to->head);
    while ((type = *(to->tail)) != NULL) {
	if (types_contains(from, type->name)) {
	    *(to->tail) = type->next;
	    DEALLOCATE(type);
	} else {
	    to->tail = &(type->next);
	}
    }
}

void
types_unlink_unused(TypeTupleT * tuple, AltT * alts)
{
    TypeTupleEntryT * type;

    types_iter_alt_item_type_names(alts, name_used);
    tuple->tail = &(tuple->head);
    while ((type = *(tuple->tail)) != NULL) {
	assert(!entry_is_non_local(type->name));
	if (name_is_used(entry_get_name(type->name))) {
	    tuple->tail = &(type->next);
	} else {
	    *(tuple->tail) = type->next;
	    DEALLOCATE(type);
	}
    }
    types_iter_alt_item_type_names(alts, name_not_used);
}

void
types_compute_formal_renaming(TypeTupleT * names, TypeRTransT * translator)
{
    TypeTupleEntryT * ptr;

    for (ptr = names->head; ptr; ptr = ptr->next) {
	rtrans_add_translation(translator, ptr->name, ptr->name, ptr->type,
				ptr->reference);
    }
}

void
types_compute_formal_inlining(TypeTupleT * names, TypeTupleT * renames,
			      TypeRTransT * translator, SaveRStackT * state)
{
    TypeTupleEntryT * ptr   = names->head;
    TypeTupleEntryT * reptr = renames->head;

    while (ptr) {
	EntryT * entry;
	EntryT * type;
	BoolT  reference;

	assert(reptr);
	entry = rstack_get_translation(state, reptr->name, &type, &reference);
	assert(entry);
	rtrans_add_translation(translator, ptr->name, entry, type, reference);
	ptr   = ptr->next;
	reptr = reptr->next;
    }
    assert(reptr == NULL);
}

void
types_compute_local_renaming(TypeTupleT * names, TypeTupleT * exclude,
			     TypeRTransT * translator, SaveRStackT * state,
			     TableT * table)
{
    TypeTupleEntryT * ptr;

    for (ptr = names->head; ptr; ptr = ptr->next) {
	if (!types_contains(exclude, ptr->name)) {
	    EntryT * type;
	    BoolT  reference;

	    if (rstack_get_translation(state, ptr->name, &type,
				       &reference) != NULL) {
		EntryT * entry = table_add_generated_name(table);

		rtrans_add_translation(translator, ptr->name, entry,
				       ptr->type, ptr->reference);
	    } else {
		rtrans_add_translation(translator, ptr->name, ptr->name,
				       ptr->type, ptr->reference);
	    }
	}
    }
}

void
types_compute_param_from_trans(TypeTupleT * new_param,
			       TypeNTransT * from_translator,
			       TypeNTransT * to_translator, TypeTupleT * old_param)
{
    TypeTupleEntryT * ptr;

    types_init(new_param);
    for (ptr = old_param->head; ptr; ptr = ptr->next) {
	EntryT * entry = ntrans_get_indirect_translation(from_translator,
						       to_translator,
						       ptr->name);

	if (entry) {
	    types_add_name_and_type(new_param, entry, ptr->type,
				    ptr->reference);
	}
    }
}

BoolT
types_check_shadowing(TypeTupleT * tuple, ScopeStackT * stack, RuleT * rule)
{
    BoolT           errored = FALSE;
    TypeTupleEntryT * ptr;

    for (ptr = tuple->head; ptr; ptr = ptr->next) {
	if (scope_stack_check_shadowing(stack, ptr->name, rule)) {
	    errored = TRUE;
	}
    }
    return(errored);
}

void
types_iter_for_table(TypeTupleT * tuple, void (*proc)(EntryT *, void *),
		     void * closure)
{
    TypeTupleEntryT * ptr;

    for (ptr = tuple->head; ptr; ptr = ptr->next) {
	if (ptr->type) {
	    entry_iter(ptr->type, TRUE, proc, closure);
	}
	if (ptr->name) {
	    entry_iter(ptr->name, TRUE, proc, closure);
	}
    }
}

void
types_destroy(TypeTupleT * tuple)
{
    TypeTupleEntryT * tuple_ptr = (tuple->head);

    while (tuple_ptr) {
	TypeTupleEntryT * tmp_ptr = (tuple_ptr->next);

	DEALLOCATE(tuple_ptr);
	tuple_ptr = tmp_ptr;
    }
}

void
write_type_types(OStreamT * ostream, TypeTupleT * tuple)
{
    TypeTupleEntryT * type;

    write_char(ostream, '(');
    for (type = tuple->head; type; type = type->next) {
	if (type->type) {
	    write_cstring(ostream, ": ");
	    write_key(ostream, entry_key(type->type));
	    if (type->reference) {
		write_cstring(ostream, " &");
	    }
	} else {
	    write_cstring(ostream, ": <unknown>");
	}
	if (type->next) {
	    write_cstring(ostream, ", ");
	}
    }
    write_char(ostream, ')');
}

void
write_type_names(OStreamT * ostream, TypeTupleT * tuple, BoolT call)
{
    TypeTupleEntryT * type;

    write_char(ostream, '(');
    for (type = tuple->head; type; type = type->next) {
	if (type->name) {
	    if ((call && type->reference) || (type->assign)) {
		write_char(ostream, '&');
	    }
	    write_key(ostream, entry_key(type->name));
	}
	if (type->type) {
	    write_cstring(ostream, ": ");
	    write_key(ostream, entry_key(type->type));
	    if (type->reference) {
		write_cstring(ostream, " &");
	    }
	}
	if (type->next) {
	    write_cstring(ostream, ", ");
	}
    }
    write_char(ostream, ')');
}


/*
 * Basic name translator handling functions
 */

void
btrans_init(TypeBTransT * translator)
{
    translator->head = NULL;
    translator->tail = &(translator->head);
}

void
btrans_add_translations(TypeBTransT * translator, TypeTupleT * from, TypeTupleT * to)
{
    TypeTupleEntryT * from_ptr = from->head;
    TypeTupleEntryT * to_ptr   = to->head;

    while (from_ptr) {
	assert(to_ptr != NULL);
	btrans_add_translation(translator, from_ptr->name, to_ptr->name);
	from_ptr = from_ptr->next;
	to_ptr   = to_ptr->next;
    }
    assert(to_ptr == NULL);
}

void
btrans_add_translation(TypeBTransT * translator, EntryT * from, EntryT * to)
{
    TransT * link = ALLOCATE(TransT);

    link->to            = to;
    link->from          = from;
    link->next          = NULL;
    *(translator->tail) = link;
    translator->tail    = &(link->next);
}

void
btrans_generate_names(TypeBTransT * translator, TypeTupleT * tuple, TableT * table)
{
    TypeTupleEntryT * tuple_ptr = tuple->head;

    while (tuple_ptr) {
	btrans_add_translation(translator, tuple_ptr->name,
				table_add_generated_name(table));
	tuple_ptr = tuple_ptr->next;
    }
}

void
btrans_regenerate_names(TypeBTransT * translator, TypeTupleT * tuple)
{
    TypeTupleEntryT *  tuple_ptr = tuple->head;
    TransT *           trans_ptr = translator->head;

    while (tuple_ptr) {
	assert(trans_ptr != NULL);
	trans_ptr->from = tuple_ptr->name;
	trans_ptr       = trans_ptr->next;
	tuple_ptr       = tuple_ptr->next;
    }
    assert(trans_ptr == NULL);
}

ItemT *
btrans_generate_non_pred_names(TypeBTransT * translator, TypeTupleT * tuple,
			       TypeTupleT * result, EntryT * predicate_id,
			       TableT * table)
{
    TypeTupleEntryT * ptr = tuple->head;
    TypeTupleT      from;
    TypeTupleT      to;

    types_init(&from);
    types_init(&to);
    while (ptr) {
	if (ptr->name == predicate_id) {
	    btrans_add_translation(translator, predicate_id, predicate_id);
	} else {
	    EntryT * entry = table_add_generated_name(table);

	    btrans_add_translation(translator, ptr->name, entry);
	    if (types_contains(result, ptr->name)) {
		types_add_name_and_type(&from, entry, ptr->type,
					ptr->reference);
		types_add_name_and_type(&to, ptr->name, ptr->type,
					ptr->reference);
	    }
	}
	ptr = ptr->next;
    }
    if (types_equal_zero_tuple(&from)) {
	types_destroy(&from);
	types_destroy(&to);
	return(NULL);
    } else {
	ItemT * item = item_create(table_add_rename(table));

	types_assign(item_param(item), &from);
	types_assign(item_result(item), &to);
	return(item);
    }
}

ItemT *
btrans_regen_non_pred_names(TypeBTransT * translator, TypeTupleT * tuple,
			    TypeTupleT * result, TableT * table)
{
    TypeTupleEntryT * tuple_ptr = tuple->head;
    TransT *          trans_ptr = translator->head;
    TypeTupleT      from;
    TypeTupleT      to;

    types_init(&from);
    types_init(&to);
    while (tuple_ptr) {
	assert(trans_ptr != NULL);
	trans_ptr->from = tuple_ptr->name;
	if (types_contains(result, tuple_ptr->name)) {
	    types_add_name_and_type(&from, trans_ptr->to, tuple_ptr->type,
				    tuple_ptr->reference);
	    types_add_name_and_type(&to, trans_ptr->from, tuple_ptr->type,
				    tuple_ptr->reference);
	}
	trans_ptr       = trans_ptr->next;
	tuple_ptr       = tuple_ptr->next;
    }
    assert(trans_ptr == NULL);
    if (types_equal_zero_tuple(&from)) {
	types_destroy(&from);
	types_destroy(&to);
	return(NULL);
    } else {
	ItemT * item = item_create(table_add_rename(table));

	types_assign(item_param(item), &from);
	types_assign(item_result(item), &to);
	return(item);
    }
}

EntryT *
btrans_get_translation(TypeBTransT * translator, EntryT * entry)
{
    EntryT * translation = NULL;
    TransT * ptr;

    for (ptr = translator->head; ptr; ptr = ptr->next) {
	if (ptr->from == entry) {
	    translation = ptr->to;
	}
    }
    return(translation);
}

void
btrans_destroy(TypeBTransT * translator)
{
    TransT * ptr = (translator->head);
    TransT * tmp;

    while ((tmp = ptr) != NULL) {
	ptr = ptr->next;
	DEALLOCATE(tmp);
    }
}


/*
 * Rename stack name translator handling functions
 */

void
rtrans_init(TypeRTransT * translator)
{
    translator->head = NULL;
    translator->tail = &(translator->head);
}

void
rtrans_add_translation(TypeRTransT * translator, EntryT * from, EntryT * to,
		       EntryT * type, BoolT reference)
{
    RTransT * link = ALLOCATE(RTransT);

    link->next          = NULL;
    link->to            = to;
    link->from          = from;
    link->type          = type;
    link->reference     = reference;
    *(translator->tail) = link;
    translator->tail    = &(link->next);
}

EntryT *
rtrans_get_translation(TypeRTransT * translator, EntryT * entry, EntryT * *type_ref,
		       BoolT *reference_ref)
{
    RTransT * ptr;

    for (ptr = translator->head; ptr; ptr = ptr->next) {
	if (ptr->from == entry) {
	    *type_ref      = ptr->type;
	    *reference_ref = ptr->reference;
	    return(ptr->to);
	}
    }
    return(NULL);
}

void
rtrans_apply_for_non_locals(TypeRTransT * translator,
			    void (*proc)(EntryT *, EntryT *, void *),
			    void * closure)
{
    RTransT * ptr;

    for (ptr = translator->head; ptr; ptr = ptr->next) {
	(*proc)(ptr->from, ptr->to, closure);
    }
}

void
rtrans_destroy(TypeRTransT * translator)
{
    RTransT * ptr = (translator->head);
    RTransT * tmp;

    while ((tmp = ptr) != NULL) {
	ptr = ptr->next;
	DEALLOCATE(tmp);
    }
}


/*
 * Name translator handling functions
 */

void
trans_init(TypeTransT * translator, TypeTupleT * param, TypeTupleT * result,
	   AltT * alt)
{
    TypeTupleEntryT * ptr;
    ItemT *           item;

    translator->head = NULL;
    translator->tail = &(translator->head);
    entry_list_init(&(translator->used_names));
    for (ptr = param->head; ptr; ptr = ptr->next) {
        entry_list_add_if_missing(&(translator->used_names), ptr->name);
    }
    for (ptr = result->head; ptr; ptr = ptr->next) {
	entry_list_add_if_missing(&(translator->used_names), ptr->name);
    }
    for (item = alt_item_head(alt); item; item = item_next(item)) {
        TypeTupleT * type = item_result(item);

	for (ptr = type->head; ptr; ptr = ptr->next) {
	    entry_list_add_if_missing(&(translator->used_names), ptr->name);
	}
    }
}

void
trans_add_translation(TypeTransT * translator, EntryT * from, EntryT * to)
{
    TransT * link = ALLOCATE(TransT);

    link->to            = to;
    link->from          = from;
    link->next          = NULL;
    *(translator->tail) = link;
    translator->tail    = &(link->next);
}

void
trans_add_translations(TypeTransT * translator, TypeTupleT * from, TypeTupleT * to)
{
    TypeTupleEntryT * from_ptr = from->head;
    TypeTupleEntryT * to_ptr   = to->head;

    while (from_ptr) {
	assert(to_ptr != NULL);
	trans_add_translation(translator, from_ptr->name, to_ptr->name);
	from_ptr = from_ptr->next;
	to_ptr   = to_ptr->next;
    }
    assert(to_ptr == NULL);
}

void
trans_save_state(TypeTransT * translator, SaveTransT * state)
{
    state->last_ref = translator->tail;
}

EntryT *
trans_get_translation(TypeTransT * translator, EntryT * entry)
{
    EntryT * translation = NULL;
    TransT *      ptr;

    for (ptr = translator->head; ptr; ptr = ptr->next) {
	if (ptr->from == entry) {
	    translation = ptr->to;
	}
    }
    if (translation) {
	return(translation);
    }
    if (!entry_list_contains(&(translator->used_names), entry)) {
        return(entry);
    }
    return(NULL);
}

void
trans_restore_state(TypeTransT * translator, SaveTransT * state)
{
    TransT * ptr = (*(state->last_ref));
    TransT * tmp;

    *(state->last_ref) = NULL;
    while ((tmp = ptr) != NULL) {
	ptr = ptr->next;
	DEALLOCATE(tmp);
    }
    translator->tail = state->last_ref;
}

void
trans_destroy(TypeTransT * translator)
{
    TransT * ptr = (translator->head);
    TransT * tmp;

    while ((tmp = ptr) != NULL) {
	ptr = ptr->next;
	DEALLOCATE(tmp);
    }
    entry_list_destroy(&(translator->used_names));
}

/*
 * Numeric translator handling functions
 */

static unsigned
ntrans_add_translation(TypeNTransT * translator, EntryT * from)
{
    NTransT * link = ALLOCATE(NTransT);

    if (translator->count == UINT_MAX) {
	E_too_many_generated_names();
	UNREACHED;
    }
    link->to            = (translator->count)++;
    link->from          = from;
    link->next          = NULL;
    *(translator->tail) = link;
    translator->tail    = &(link->next);
    return(link->to);
}

void
ntrans_init(TypeNTransT * translator)
{
    translator->count      = 0;
    translator->head       = NULL;
    translator->tail       = &(translator->head);
}

void
ntrans_save_state(TypeNTransT * translator, SaveNTransT * state)
{
    state->last_count = translator->count;
    state->last_ref   = translator->tail;
}

unsigned
ntrans_get_translation(TypeNTransT * translator, EntryT * entry)
{
    NTransT * ptr;

    for (ptr = translator->head; ptr; ptr = ptr->next) {
	if (ptr->from == entry) {
	    return(ptr->to);
	}
    }
    return(ntrans_add_translation(translator, entry));
}

EntryT *
ntrans_get_indirect_translation(TypeNTransT * from_translator,
				TypeNTransT * to_translator, EntryT * entry)
{
    NTransT *  ptr;
    unsigned name;

    for (ptr = from_translator->head; ptr; ptr = ptr->next) {
	if (ptr->from == entry) {
	    name = ptr->to;
	    goto found;
	}
    }
    return(NULL);
  found:
    for (ptr = to_translator->head; ptr; ptr = ptr->next) {
	if (ptr->to == name) {
	    return(ptr->from);
	}
    }
    UNREACHED;
}

void
ntrans_restore_state(TypeNTransT * translator, SaveNTransT * state)
{
    NTransT * ptr = (*(state->last_ref));
    NTransT * tmp;

    *(state->last_ref) = NULL;
    translator->count  = state->last_count;
    while ((tmp = ptr) != NULL) {
	ptr = ptr->next;
	DEALLOCATE(tmp);
    }
    translator->tail = state->last_ref;
}

void
ntrans_destroy(TypeNTransT * translator)
{
    NTransT * ptr = (translator->head);
    NTransT * tmp;

    while ((tmp = ptr) != NULL) {
	ptr = ptr->next;
	DEALLOCATE(tmp);
    }
}
