#include "config.h"

#include <iostream>
#include <vector>
#include <map>
#include <string>

#include "asserts.h"
#include "error.h"
#include "rmath.h"
#include "fs.h"
#include "rconfig.h"
#include "logger.h"
#include "timer.h"
#include "estat.h"
#include "reporter.h"
#include "strfmt.h"

#include "vaulter.h"

/** C'tor */
vault_manager::vault_manager()
{
	if (this != &vaulter)
		throw(INTERNAL_ERROR(0,"Attempt to allocate multiple vault managers"));
	clear();
}

/** Clear the vault manager */
void vault_manager::clear(void)
{
	m_lock.clear();
	m_selected_vault.erase();
	m_deleted_archives.clear();
	m_da_err = false;
	m_initialized = false;
}

/** Initialize the vault manager */
void vault_manager::init(void)
{
	clear();
	m_initialized = true;
}

/** Return the initialized status of the vault manager */
const bool vault_manager::initialized(void) const
{
	return(m_initialized);
}

/** Select a vault

	If an archive of the same timestamp for the given timestamp resolution
	already exists on any vault, then that vault is selected.  Otherwise select
	a vault according to vault-selection-behavior.

 */
void vault_manager::select(void)
{
	std::string es;
	std::string ts;
	configuration_manager::vaults_type::const_iterator vi;
	std::string lockfile;
	std::string lstr;

	if (!initialized())
		throw(INTERNAL_ERROR(0,"Vault manager not initialized"));

	TRY_nomem(es = "Could not select a vault");
	try {
		// If an archive exists in any vault by the same timestamp as conf.stamp(),
		// then use that vault regardless of the selection behavior.
		for (
			vi = config.vaults().begin(); 
			vi != config.vaults().end(); 
			vi++
			)
		{
			subdirectory subdir;
			subdirectory::iterator sdi;
	
			subdir.path(*vi);
			for (sdi = subdir.begin(); sdi != subdir.end(); sdi++) {
				TRY_nomem(ts = config.timestamp().str());

				if (*sdi == ts) {
					TRY_nomem(lstr = "Existing archive directory found in vault: \"");
					TRY_nomem(lstr += *vi);
					TRY_nomem(lstr += "\"\n");
					logger.write(lstr);

					if (config.vault_locking()) {
						TRY_nomem(lockfile = *vi);
						TRY_nomem(lockfile += "/.rvm_lock");
						m_lock.clear();
						m_lock.lockfile(lockfile);
						if (!m_lock.lock()) {
							error e(ENOLCK);
							e.push_back(ERROR_INSTANCE(es));
							TRY_nomem(es = "Could not lock vault: \"");
							TRY_nomem(es += *vi);
							TRY_nomem(es += "\"");
							e.push_back(ERROR_INSTANCE(es));
							if (m_lock.is_locked()) {
								TRY_nomem(es = "Vault is locked by PID: ");
								TRY_nomem(es += estring(m_lock.locked_by()));
								e.push_back(ERROR_INSTANCE(es));
							}
							throw(e);
						}
					}
					TRY_nomem(m_selected_vault = *vi);
					return;
				}

				TRY_nomem(ts += ".incomplete");

				if (*sdi == ts) {
					TRY_nomem(lstr = "Existing archive directory found in vault: \"");
					TRY_nomem(lstr += *vi);
					TRY_nomem(lstr += "\"\n");
					logger.write(lstr);

					if (config.vault_locking()) {
						TRY_nomem(lockfile = *vi);
						TRY_nomem(lockfile += "/.rvm_lock");
						m_lock.clear();
						m_lock.lockfile(lockfile);
						if (!m_lock.lock()) {
							error e(ENOLCK);
							e.push_back(ERROR_INSTANCE(es));
							TRY_nomem(es = "Could not lock vault: \"");
							TRY_nomem(es += *vi);
							TRY_nomem(es += "\"");
							e.push_back(ERROR_INSTANCE(es));
							if (m_lock.is_locked()) {
								TRY_nomem(es = "Vault is locked by PID: ");
								TRY_nomem(es += estring(m_lock.locked_by()));
								e.push_back(ERROR_INSTANCE(es));
							}
							throw(e);
						}
					}
					TRY_nomem(m_selected_vault = *vi);
					return;
				}
			}
		}
	
		// If an archive by the same timestamp does not already exist, then select
		// a vault.
		if (config.vault_selection_behavior() 
			== configuration_manager::selection_round_robin)
		{
			std::pair<std::string,std::string> youngest;
	
			for (
				vi = config.vaults().begin();
				vi != config.vaults().end();
				vi++
				)
			{
				subdirectory subdir;
				subdirectory::iterator sdi;
	
				if (config.vault_locking()) {
					TRY_nomem(lockfile = *vi);
					TRY_nomem(lockfile += "/.rvm_lock");
					m_lock.clear();
					m_lock.lockfile(lockfile);
					if (m_lock.is_locked()) {
						std::string lstr;

						TRY_nomem(lstr = "Skipping locked vault: \"");
						TRY_nomem(lstr += *vi);
						TRY_nomem(lstr += "\"\n");
						logger.write(lstr);

						continue;
					}
				}

				subdir.path(*vi);
				for (sdi = subdir.begin(); sdi != subdir.end(); sdi++) {
					if ((youngest.first < *sdi) || (youngest.first.size() == 0)) {
						TRY_nomem(youngest.first = *sdi);
						TRY_nomem(youngest.second = *vi);
					}
				}
			}
			
			TRY_nomem(m_selected_vault = "");
			if (youngest.second.size() == 0) {
				TRY_nomem(m_selected_vault = config.vaults()[0]);
			}
			else {
				for (
					vi = config.vaults().begin(); 
					vi != config.vaults().end(); 
					vi++
					)
				{
					if (*vi == youngest.second) {
						if ((vi+1) == config.vaults().end()) {
							TRY_nomem(m_selected_vault = config.vaults()[0]);
						}
						else {
							TRY_nomem(m_selected_vault = *(vi+1));
						}
					}
				}
			}
		}
		else {
			std::pair<std::string,filesystem::size_type> most_space;
			filesystem fsys;
	
			TRY_nomem(most_space.first = config.vaults()[0]);
			fsys.path(config.vaults()[0]);
			most_space.second = fsys.free_blocks();
			for (
				vi = (config.vaults().begin()+1); 
				vi != config.vaults().end();
				vi++
				)
			{
				fsys.path(*vi);
				if (most_space.second < fsys.free_blocks()) {
					TRY_nomem(most_space.first = *vi);
					most_space.second = fsys.free_blocks();
				}
			}
			TRY_nomem(m_selected_vault = most_space.first);
		}
	}
	catch(error e) {
		e.push_back(es);
		throw(e);
	}
	catch(...) {
		error e(0);
		if (errno == ENOMEM) {
			e = err_nomem;
			e.push_back(es);
			throw(e);
		}
		e = err_unknown;
		e.push_back(es);
		throw(e);
	}

	if (m_selected_vault.size() == 0)
		throw(ERROR(0,es));
}

/** Return a list of archive directories in the selected vault

	Return a list of archive directories in the selected vault, including
	incomplete archives.  Ignore all other directories.
 */
const subdirectory vault_manager::get_archive_list(void)
{
	subdirectory subdir;
	subdirectory::iterator sdi;

	if (!initialized())
		throw(INTERNAL_ERROR(0,"Vault manager not initialized"));

	if (!selected())
		throw(INTERNAL_ERROR(0,"No vault selected"));

	subdir.path(vault());

	sdi = subdir.begin();
	while (sdi != subdir.end()) {
		if (!is_timestamp((*sdi).substr(0,(*sdi).find(".incomplete"))))
		{
			subdir.erase(sdi);
			sdi = subdir.begin();
		}
		else {
			sdi++;
		}
	}

	return(subdir);
}

/** Return the path to the selected vault */
const std::string vault_manager::vault(void) const
{
	if (!initialized())
		throw(INTERNAL_ERROR(0,"Vault manager not initialized"));

	if (!selected())
		throw(INTERNAL_ERROR(0,"No vault selected"));

	return(m_selected_vault);
}

/** Return whether or not a vault has been selected yet */
const bool vault_manager::selected(void) const
{
	bool value = true;

	if (!initialized())
		throw(INTERNAL_ERROR(0,"Vault manager not initialized"));

	if (m_selected_vault.size() == 0)
		value = false;
	
	return(value);
}

/** Return the percent of used blocks and used inodes in the selected vault */
void vault_manager::usage(uint16 &a_blocks, uint16 &a_inodes) const
{
	filesystem fsys;
	safe_num<uint64> blocks, inodes;

	if (!initialized())
		throw(INTERNAL_ERROR(0,"Vault manager not initialized"));

	if (!selected())
		throw(INTERNAL_ERROR(0,"No vault selected"));

	fsys.path(vault());

	if (fsys.total_blocks() == 0)
		blocks = 0;
	else {
		try {
			blocks = fsys.free_blocks();
			// blocks *= 100;
			// blocks /= fsys.total_blocks();
			blocks /= fsys.total_blocks() / 100;
		}
		catch(error e) {
			logger.write("*** ERROR: Overflow error detected in vault_manager::usage() while calculating percent blocks used\n");
			logger.write(e.str());
			blocks = 0;
		}
		catch(...) {
			logger.write("*** ERROR: Unknown error detected in vault_manager::usage() while calculating percent blocks used\n");
			blocks = 0;
		}
	}

	if (fsys.free_inodes() == 0)
		inodes = 0;
	else {
		try {
			inodes = fsys.free_inodes();
			// inodes *= 100;
			// inodes /= fsys.total_inodes();
			inodes /= fsys.total_inodes() / 100;
		}
		catch(error e) {
			logger.write("*** ERROR: Overflow error detected in vault_manager::usage() while calculating percent inodes used\n");
			logger.write(e.str());
			blocks = 0;
		}
		catch(...) {
			logger.write("*** ERROR: Unknown error detected in vault_manager::usage() while calculating percent inodes used\n");
			blocks = 0;
		}
	}

	ASSERT(blocks <= max_limit(blocks));
	ASSERT(inodes <= max_limit(inodes));

	a_blocks = blocks.value();
	a_inodes = inodes.value();
}

/** Test to see if a vault has exceeded it's overflow threshold */
const bool vault_manager::overflow(bool a_report)
{
	uint16 free_blocks;
	uint16 free_inodes;
	bool value = false;
	std::string lstr;

	if (!initialized())
		throw(INTERNAL_ERROR(0,"Vault manager not initialized"));

	if (!selected())
		throw(INTERNAL_ERROR(0,"No vault selected"));

	usage(free_blocks, free_inodes);
	if (free_blocks < config.vault_overflow_blocks())
		value = true;
	if (free_inodes < config.vault_overflow_inodes())
		value = true;
	
	if (value && a_report) {
		TRY_nomem(lstr = "Vault overflow detected: ");
		TRY_nomem(lstr += vault());
		TRY_nomem(lstr += "\n");
		logger.write(lstr);

		TRY_nomem(lstr = "     Threshold: ");
		TRY_nomem(lstr += 
			percent_string(config.vault_overflow_blocks(),
				static_cast<uint16>(100)));
		TRY_nomem(lstr += " free blocks, ");
		TRY_nomem(lstr += 
			percent_string(config.vault_overflow_inodes(),
				static_cast<uint16>(100)));
		TRY_nomem(lstr += " free inodes");
		TRY_nomem(lstr += "\n");
		logger.write(lstr);

		TRY_nomem(lstr = "Vault capacity: ");
		TRY_nomem(lstr += percent_string(free_blocks,static_cast<uint16>(100)));
		TRY_nomem(lstr += " free blocks, ");
		TRY_nomem(lstr += percent_string(free_inodes,static_cast<uint16>(100)));
		TRY_nomem(lstr += " free inodes");
		TRY_nomem(lstr += "\n");
		logger.write(lstr);

		estring __e = estring("Overflow Detected:");
		reporter.vault().add_report(
			vault_stats_report(estring("Overflow Detected:"),filesystem(vault()))
			);
	}
	
	return(value);
}

/** Find the oldest archive in the vault and delete it */
void vault_manager::delete_oldest_archive(void)
{
	std::string es;
	estring ts;
	estring tsi;
	std::string lstr;
	subdirectory archive_list; 
	std::string oldest;
	std::string dir;
	std::string delete_dir;
	estring estr;
	timer t;
	uint16 free_blocks, free_inodes;
	subdirectory logfile_list;
	subdirectory::const_iterator ilf;

	if (!initialized())
		throw(INTERNAL_ERROR(0,"Vault manager not initialized"));
	
	if (!selected())
		throw(INTERNAL_ERROR(0,"No vault selected"));

	TRY_nomem(ts = config.timestamp().str());
	TRY_nomem(tsi = ts + ".incomplete");

	archive_list = get_archive_list();
	if (
		(archive_list.size() == 0)
		|| (
			(archive_list.size() == 1)
			&& (archive_list[0] == ts || archive_list[0] == tsi)
			)
		)
	{
		TRY_nomem(es = "Vault has insufficient space: \"");
		TRY_nomem(es += vault());
		TRY_nomem(es += "\"");
		exit_manager.assign(exitstat::vault_full);
		throw(ERROR(0,es));
	}

	TRY_nomem(oldest = archive_list[0]);
	if (oldest == ts || oldest == tsi) {
		TRY_nomem(lstr = "Oldest is actually archive in use: \"");
		TRY_nomem(lstr += oldest);
		TRY_nomem(lstr += "\"  Skipping to next archive...\n");
		logger.write(lstr);
		TRY_nomem(oldest = archive_list[1]);
	}

	if (oldest.find(".incomplete") != std::string::npos) {
		TRY_nomem(lstr = "WARNING: Oldest archive found is incomplete.\n");
		logger.write(lstr);
	}

	TRY_nomem(lstr = "Deleting oldest archive: \"");
	TRY_nomem(lstr += oldest);
	TRY_nomem(lstr += "\"\n");
	logger.write(lstr);

	TRY_nomem(dir = vault());
	TRY_nomem(dir += "/");
	TRY_nomem(dir += oldest);
	t.start();
	try {
		m_deleted_archives.push_back(oldest);
	}
	catch(...) {
		m_da_err = true;
	}
	
	delete_dir = dir;
	if (delete_dir.find(".deleting") == std::string::npos)
		delete_dir += ".deleting";
	rename_file(dir, delete_dir);
	if (config.delete_command_path().size() == 0)
		rm_recursive(delete_dir);
	else {
		std::string es;
		std::string cmdline;

		TRY_nomem(es = "Delete command returned non-zero exit value: \"");
		TRY_nomem(es += oldest);
		TRY_nomem(es += "\"");
		cmdline = config.delete_command_path();
		cmdline += " \"";
		cmdline += delete_dir;
		cmdline += "\"";
		if (system(cmdline.c_str()) != 0)
			throw(ERROR(1,es));
	}
	t.stop();

	TRY_nomem(lstr = "Deletion complete, duration: ");
	TRY_nomem(lstr += t.duration());
	TRY_nomem(lstr += "\n");
	logger.write(lstr);

	usage(free_blocks, free_inodes);
	TRY_nomem(lstr = "Vault capacity: ");
	TRY_nomem(lstr += percent_string(free_blocks,static_cast<uint16>(100)));
	TRY_nomem(lstr += " free blocks, ");
	TRY_nomem(lstr += percent_string(free_inodes,static_cast<uint16>(100)));
	TRY_nomem(lstr += " free inodes");
	TRY_nomem(lstr += "\n");
	logger.write(lstr);

	estr = "Deleted ";
	estr += oldest;
	estr += ":";
	reporter.vault().add_report(vault_stats_report(estr,filesystem(vault())));

	// This is a grey area for me: Should log/report file removal be managed by
	// log_manager/report_manager, or is it OK to do it here?  For simplicity
	// sake, I do it here.
	//
	if (config.delete_old_log_files()) {
		estring wildcard;

		logger.write("Searching for old log files to delete...\n");
		wildcard = oldest;
		wildcard += ".log*";
		logfile_list.path(config.log_dir(), wildcard);
		for (ilf = logfile_list.begin(); ilf != logfile_list.end(); ilf++) {
			estring file;

			file = config.log_dir();
			file += "/";
			file += *ilf;
			
			try {
				rm_file(file);
			}
			catch(error e) {
				estring es;

				es = "*** ERROR: Could not remove log file: ";
				es += *ilf;

				logger.write(es);
				logger.write(e.str());
				
				// Should I throw an error here, or should deletion of old log files
				// not be considered important enough to interrupt archiving?
			}
			catch(...) {
				estring es;

				es = "*** ERROR: Unknown error detected in vault_manager::delete_oldest_archive() while deleting old log file: ";
				es += *ilf;
				es += '\n';

				logger.write(es);
				
				// Should I throw an error here, or should deletion of old log files
				// not be considered important enough to interrupt archiving?
			}
		}

		wildcard = oldest;
		wildcard += ".relink*";
		logfile_list.path(config.log_dir(), wildcard);
		for (ilf = logfile_list.begin(); ilf != logfile_list.end(); ilf++) {
			estring file;

			file = config.log_dir();
			file += "/";
			file += *ilf;
			
			try {
				rm_file(file);
			}
			catch(error e) {
				estring es;

				es = "*** ERROR: Could not remove relink file: ";
				es += *ilf;

				logger.write(es);
				logger.write(e.str());
				
				// Should I throw an error here, or should deletion of old log files
				// not be considered important enough to interrupt archiving?
			}
			catch(...) {
				estring es;

				es = "*** ERROR: Unknown error detected in vault_manager::delete_oldest_archive() while deleting old relink file: ";
				es += *ilf;
				es += '\n';

				logger.write(es);
				
				// Should I throw an error here, or should deletion of old log files
				// not be considered important enough to interrupt archiving?
			}
		}
	}

	if (config.delete_old_report_files()) {
		estring wildcard;

		logger.write("Searching for old report files to delete...\n");
		wildcard = oldest;
		wildcard += ".report*";
		logfile_list.path(config.log_dir(), wildcard);
		for (ilf = logfile_list.begin(); ilf != logfile_list.end(); ilf++) {
			estring file;

			file = config.log_dir();
			file += "/";
			file += *ilf;
			
			try {
				rm_file(file);
			}
			catch(error e) {
				estring es;

				es = "*** ERROR: Could not remove report file: ";
				es += *ilf;

				logger.write(es);
				logger.write(e.str());
				
				// Should I throw an error here, or should deletion of old log files
				// not be considered important enough to interrupt archiving?
			}
			catch(...) {
				estring es;

				es = "*** ERROR: Unknown error detected in vault_manager::delete_oldest_archive() while deleting old log file: ";
				es += *ilf;
				es += '\n';

				logger.write(es);
				
				// Should I throw an error here, or should deletion of old log files
				// not be considered important enough to interrupt archiving?
			}
		}
	}
}

/** Prepare the selected vault

	Check the selected vault for overflow.  If the overflow threshold has been
	exceeded and the vault-overflow-behavior setting is not quit, then delete
	the oldest archive in the vault.  Depending on vault-overflow-behavior,
	possibly repeat this action until either there is space free or there are no
	archives left in the vault.

*/
void vault_manager::prepare(bool a_assume_overflow)
{
	std::string es;
	std::string lstr;

	/*
	estring debug_estr;
	
	debug_estr = "[DEBUG]: vault_manager::prepare() - a_assume_overflow = ";
	debug_estr += estring(a_assume_overflow);
	debug_estr += "\n";
	logger.write(debug_estr);
	*/

	if (!initialized())
		throw(INTERNAL_ERROR(0,"Vault manager not initialized"));

	if (!selected())
		throw(INTERNAL_ERROR(0,"No vault selected"));
	
	if (!overflow() && !a_assume_overflow)
		return;
	
	// logger.write("Vault has insufficient space\n");
	// logger.write("[DEBUG]: vault_manager::prepare() - Cleaning vault...\n");

	if (config.vault_overflow_behavior() 
		== configuration_manager::overflow_quit)
	{
		TRY_nomem(es = "Vault has insufficient space: \"");
		TRY_nomem(es += vault());
		TRY_nomem(es += "\"");
		throw(ERROR(0,es));
	}

	if (config.vault_overflow_behavior() 
			== configuration_manager::overflow_delete_oldest)
	{
		if (m_deleted_archives.size() == 0) {
			TRY(delete_oldest_archive(),"Failure preparing vault");
			a_assume_overflow = false;
		}
		else {
			logger.write("Vault has insufficient space\n");
			throw(ERROR(0,"Vault has insufficient space"));
		}
	}

	if (!overflow() && !a_assume_overflow)
		return;

	if (config.vault_overflow_behavior() 
		== configuration_manager::overflow_delete_until_free)
	{
		while (overflow() || a_assume_overflow) {
			// logger.write("Vault has insufficient space\n");
			TRY(delete_oldest_archive(),"Failure preparing vault");
			a_assume_overflow = false;
		}
	}
	else {
		// logger.write("Vault has insufficient space\n");
		exit_manager.assign(exitstat::vault_full);
		// throw(ERROR(0,"Vault has insufficient space"));
	}
}

/** Return a list of deleted archives */
const std::vector<std::string>& vault_manager::deleted_archives(void) const
{
	if (!initialized())
		throw(INTERNAL_ERROR(0,"Vault manager not initialized"));

	return(m_deleted_archives);
}

/** Return whether or not there was an error deleting archives */
const bool vault_manager::err_deleted_archives(void) const
{
	if (!initialized())
		throw(INTERNAL_ERROR(0,"Vault manager not initialized"));

	return(m_da_err);
}

/** The global vault manager */
vault_manager vaulter;

