#ifndef __types_h__
#define __types_h__

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

/** \file types.h Basic types definitions and templates. */

// ----------------------------------------------------------------------------

/**
	\typedef
	Define a 8-bit unsigned integer type.

	The following code, along with tests performed by the GNU Autoconf
	scripts, define a system-independent 8-bit unsigned integer.
 */
#if SIZEOF_UNSIGNED_CHAR == 1
	typedef unsigned char uint8;
#elif SIZEOF_UNSIGNED_SHORT == 1
	typedef unsigned short uint8;
#elif SIZEOF_UNSIGNED_INT == 1
	typedef unsigned int uint8;
#elif SIZEOF_UNSIGNED_LONG == 1
	typedef unsigned long uint8;
#elif SIZEOF_UNSIGNED_LONG_LONG == 1
	typedef unsigned long long uint8;
#else
#	error *** ERROR: No 8-bit type found to use as uint8
#endif

/**
	\typedef
	Define a 16-bit unsigned integer type.

	The following code, along with tests performed by the GNU Autoconf
	scripts, define a system-independent 16-bit unsigned integer.
 */
#if SIZEOF_UNSIGNED_CHAR == 2
	typedef unsigned char uint16;
#elif SIZEOF_UNSIGNED_SHORT == 2
	typedef unsigned short uint16;
#elif SIZEOF_UNSIGNED_INT == 2
	typedef unsigned int uint16;
#elif SIZEOF_UNSIGNED_LONG == 2
	typedef unsigned long uint16;
#elif SIZEOF_UNSIGNED_LONG_LONG == 2
	typedef unsigned long long uint16;
#else
#	error *** ERROR: No 16-bit type found to use as uint16
#endif

/**
	\typedef
	Define a 32-bit unsigned integer type.

	The following code, along with tests performed by the GNU Autoconf
	scripts, define a system-independent 32-bit unsigned integer.
 */
#if SIZEOF_UNSIGNED_CHAR == 4
	typedef unsigned char uint32;
#elif SIZEOF_UNSIGNED_SHORT == 4
	typedef unsigned short uint32;
#elif SIZEOF_UNSIGNED_INT == 4
	typedef unsigned int uint32;
#elif SIZEOF_UNSIGNED_LONG == 4
	typedef unsigned long uint32;
#elif SIZEOF_UNSIGNED_LONG_LONG == 4
	typedef unsigned long long uint32;
#else
#	error *** ERROR: No 32-bit type found to use as uint32
#endif

/**
	\typedef
	Define a 64-bit unsigned integer type.

	The following code, along with tests performed by the GNU Autoconf
	scripts, define a system-independent 64-bit unsigned integer.
 */
#if SIZEOF_UNSIGNED_CHAR == 8
	typedef unsigned char uint64;
#elif SIZEOF_UNSIGNED_SHORT == 8
	typedef unsigned short uint64;
#elif SIZEOF_UNSIGNED_INT == 8
	typedef unsigned int uint64;
#elif SIZEOF_UNSIGNED_LONG == 8
	typedef unsigned long uint64;
#elif SIZEOF_UNSIGNED_LONG_LONG == 8
	typedef unsigned long long uint64;
#else
#	error *** ERROR: No 64-bit type found to use as uint64
#endif

// ---------------------------------------------------------------------------

/** Return false for any variable that is not a char */
template<typename T>
static bool is_signed_char(const T& a_arg)
{ return(false); }

/** Return true for any variable that is a char */
// template<>
bool is_signed_char(const char& a_arg);
// { return(true); }

/** Return false for any variable that is not an unsigned char */
template<typename T>
static bool is_unsigned_char(const T& a_arg) 
{ return(false); }

/** Return true for any variable that is an unsigned char */
// template<>
bool is_unsigned_char(const unsigned char& a_arg);
// { return(true); }

/** Return true for any variable that is either a signed or unsigned char */
template<typename T>
static bool is_char(const T& a_arg) 
{ return(is_signed_char(a_arg) || is_unsigned_char(a_arg)); }

// -----

/** Return false for any variable that is not a short */
template<typename T>
static bool is_signed_short(const T& a_arg)
{ return(false); }

/** Return true for any variable that is a short */
// template<>
bool is_signed_short(const short& a_arg);
// { return(true); }

/** Return false for any variable that is not an unsigned short */
template<typename T>
static bool is_unsigned_short(const T& a_arg) 
{ return(false); }

/** Return true for any variable that is an unsigned short */
// template<>
bool is_unsigned_short(const unsigned short& a_arg);
// { return(true); }

/** Return true for any variable that is either a signed or unsigned short */
template<typename T>
static bool is_short(const T& a_arg) 
{ return(is_signed_short(a_arg) || is_unsigned_short(a_arg)); }

// -----

/** Return false for any variable that is not a int */
template<typename T>
static bool is_signed_int(const T& a_arg)
{ return(false); }

/** Return true for any variable that is a int */
// template<>
bool is_signed_int(const int& a_arg);
// { return(true); }

/** Return false for any variable that is not an unsigned int */
template<typename T>
static bool is_unsigned_int(const T& a_arg) 
{ return(false); }

/** Return true for any variable that is an unsigned int */
// template<>
bool is_unsigned_int(const unsigned int& a_arg);
// { return(true); }

/** Return true for any variable that is either a signed or unsigned int */
template<typename T>
static bool is_int(const T& a_arg) 
{ return(is_signed_int(a_arg) || is_unsigned_int(a_arg)); }

// -----

/** Return false for any variable that is not a long */
template<typename T>
static bool is_signed_long(const T& a_arg)
{ return(false); }

/** Return true for any variable that is a long */
// template<>
bool is_signed_long(const long& a_arg);
// { return(true); }

/** Return false for any variable that is not an unsigned long */
template<typename T>
static bool is_unsigned_long(const T& a_arg) 
{ return(false); }

/** Return true for any variable that is an unsigned long */
// template<>
bool is_unsigned_long(const unsigned long& a_arg);
// { return(true); }

/** Return true for any variable that is either a signed or unsigned long */
template<typename T>
static bool is_long(const T& a_arg) 
{ return(is_signed_long(a_arg) || is_unsigned_long(a_arg)); }

// -----

/** Return false for any variable that is not a long long */
template<typename T>
static bool is_signed_long_long(const T& a_arg)
{ return(false); }

/** Return true for any variable that is a long long */
// template<>
bool is_signed_long_long(const long long& a_arg);
// { return(true); }

/** Return false for any variable that is not an unsigned long long */
template<typename T>
static bool is_unsigned_long_long(const T& a_arg) 
{ return(false); }

/** Return true for any variable that is an unsigned long long */
// template<>
bool is_unsigned_long_long(const unsigned long long& a_arg);
// { return(true); }

/** Return true for any variable that is either a signed or unsigned long long */
template<typename T>
static bool is_long_long(const T& a_arg) 
{ return(is_signed_long_long(a_arg) || is_unsigned_long_long(a_arg)); }

// -----

/** Return false for any variable that is not a float */
template<typename T>
static bool is_float(const T& a_arg)
{ return(false); }

/** Return true for any variable that is a float */
// template<>
bool is_float(const float& a_arg);
// { return(true); }

// -----

/** Return false for any variable that is not a double */
template<typename T>
static bool is_double(const T& a_arg)
{ return(false); }

/** Return true for any variable that is a double */
// template<>
bool is_double(const double& a_arg);
// { return(true); }

// -----

/** Return false for any variable that is not a bool */
template<typename T>
static bool is_bool(const T& a_arg)
{ return(false); }

/** Return true for any variable that is a bool */
// template<>
bool is_bool(const bool& a_arg);
// { return(true); }

// -----

/** Return true for any variable that is a signed type */
template<typename T>
static bool is_signed(const T& a_arg)
{
	return(
		is_signed_char(a_arg)
		|| is_signed_short(a_arg)
		|| is_signed_int(a_arg)
		|| is_signed_long(a_arg)
		|| is_signed_long_long(a_arg)
		|| is_float(a_arg)
		|| is_double(a_arg)
		);
}

/** Return true for any variable that is an unsigned type */
template<typename T>
static bool is_unsigned(const T& a_arg)
{
	return(
		is_unsigned_char(a_arg)
		|| is_unsigned_short(a_arg)
		|| is_unsigned_int(a_arg)
		|| is_unsigned_long(a_arg)
		|| is_unsigned_long_long(a_arg)
		);
}

// ----------------------------------------------------------------------------

template<typename T>
static const char * type_name(const T & a_arg)
{ return("unknown"); }

// template<>
const char * type_name(const unsigned char & a_arg);
// { return("unsigned char"); }

// template<>
const char * type_name(const char & a_arg);
// { return("char"); }

// template<>
const char * type_name(const unsigned short & a_arg);
// { return("unsigned short"); }

// template<>
const char * type_name(const short & a_arg);
// { return("short"); }

// template<>
const char * type_name(const unsigned int & a_arg);
// { return("unsigned int"); }

// template<>
const char * type_name(const int & a_arg);
// { return("int"); }

// template<>
const char * type_name(const unsigned long & a_arg);
// { return("unsigned long"); }

// template<>
const char * type_name(const long & a_arg);
// { return("long"); }

// template<>
const char * type_name(const unsigned long long & a_arg);
// { return("unsigned long long"); }

// template<>
const char * type_name(const long long & a_arg);
// { return("long long"); }

// template<>
const char * type_name(const float & a_arg);
// { return("float"); }

// template<>
const char * type_name(const double & a_arg);
// { return("double"); }

// template<>
const char * type_name(const bool & a_arg);
// { return("bool"); }

// ----------------------------------------------------------------------------

#endif
