#ifndef __table_h__
#define __table_h__

#include <iostream>
#include <vector>
#include <list>

#include "asserts.h"
#include "estring.h"

class table;

class table_cell_base 
{
public:
	typedef estring::size_type size_type;
	static const size_type npos;

	table_cell_base();
	table_cell_base(const table_cell_base& a_class);
	virtual ~table_cell_base();

	virtual void assign(const table_cell_base& a_class);

	virtual size_type height(void) const;
	virtual size_type width(void) const;
	virtual void write(
		std::ostream& out,
		size_type a_line,
		size_type a_width
		) const;
	
	table_cell_base& operator=(const table_cell_base& a_class);

	size_type x;
	size_type y;
	bool repeat;

private:
};

bool operator==(const table_cell_base& a_1, const table_cell_base& a_2);

class table_cell_estring : public table_cell_base
{
public:
	table_cell_estring();
	table_cell_estring(const table_cell_estring& a_class);
	table_cell_estring(const estring& a_class);
	virtual ~table_cell_estring();

	virtual void assign(const table_cell_estring& a_class);
	virtual void assign(const estring& a_class);

	virtual size_type height(void) const;
	virtual size_type width(void) const;
	virtual void write(
		std::ostream& out,
		size_type a_line,
		size_type a_width
		) const;

	table_cell_estring& operator=(const table_cell_estring& a_class);
	table_cell_estring& operator=(const estring& a_class);

private:
	estring m_value;
};

class table_cell_table : public table_cell_base
{
public:
	table_cell_table();
	table_cell_table(const table_cell_table& a_class);
	table_cell_table(const table& a_class);
	virtual ~table_cell_table();

	virtual void assign(const table_cell_table& a_class);
	virtual void assign(const table& a_class);

	virtual size_type height(void) const;
	virtual size_type width(void) const;
	virtual void write(
		std::ostream& out,
		size_type a_line,
		size_type a_width
		) const;

	table_cell_table& operator=(const table_cell_table& a_class);
	table_cell_table& operator=(const table& a_class);

private:
	table * m_value;
};

class table
{
public:
	typedef std::vector<table_cell_base *> table_row_type;
	typedef std::vector<table_row_type> table_type;
	typedef table_cell_base::size_type size_type;

	table();
	table(const size_type a_ncols, const size_type a_nrows);
	table(const table& a_class);
	virtual ~table();

	virtual void resize(const size_type a_x, const size_type a_y);
	virtual void insert_row(const size_type a_y);
	virtual void insert_col(const size_type a_x);

	virtual void assign(const table& a_class);
	virtual void assign(
		const size_type a_x, 
		const size_type a_y, 
		const estring& a_class
		);
	virtual void assign(
		const size_type a_x, 
		const size_type a_y, 
		const table& a_class
		);

	size_type col_width(const size_type a_x) const;
	size_type row_width(void) const;
	size_type col_height(void) const;
	size_type row_height(const size_type a_y) const;

	virtual size_type height(void) const;
	virtual size_type width(void) const;
	virtual void write(
		std::ostream& out,
		size_type a_line,
		size_type a_width
		) const;
	
	const size_type ncols(void) const;
	const size_type nrows(void) const;

	table& operator=(const table& a_class);

	size_type cursor_x(void) const;
	size_type cursor_y(void) const;
	void set_cursor(size_type a_x, size_type a_y);
	bool eot(void) const;

	void repeat(const bool a_bool);

private:
	table_type m_table;
	std::list<table_cell_estring> m_list_estring;
	std::list<table_cell_table> m_list_table;
	size_type m_cursor_x;
	size_type m_cursor_y;
	bool m_repeat;

	void mf_init(void);
	void mf_add_elt(const table_cell_estring& a_class);
	void mf_add_elt(const table_cell_table& a_class);
	void mf_remove_elt(const table_cell_estring& a_class);
	void mf_remove_elt(const table_cell_table& a_class);
	void mf_remove_elt(
		size_type a_x,
		size_type a_y
		);
	void mf_remove_row(size_type a_y);
};

std::ostream& operator<<(std::ostream& a_out, table& a_class);

table& table_write(table& a_table, const estring& a_estr);
table& table_write(table& a_table, const table& a_subtable);
table& table_endl(table& a_table);
table& table_rewind(table& a_table);
table& table_skip(table& a_table);
table& table_repeat(table& a_table);
table& operator<<(table& a_table, const estring& a_estr);
table& operator<<(table& a_table, const table& a_subtable);
table& operator<<(table& a_table, table& (*a_arg)(table&));

#endif
