#include "config.h"

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include <iostream>
#include <string>
#include <fstream>

#include "asserts.h"
#include "error.h"
#include "timer.h"
#include "rconfig.h"
#include "estring.h"

#include "logger.h"

//-----------------------------------------------------------------------------

/** C'tor */
log_manager::log_manager()
{
	if (this != &logger)
		throw(
			INTERNAL_ERROR(0,"Attempt to allocate multiple log managers")
			);
	clear();
}

/** Clear the log manager */
void log_manager::clear(void)
{
	m_new_line = true;
	if (m_out.is_open())
		m_out.close();
	m_use_error_logging_level = false;
	m_initialized = false;
}

/** Initialize the log manager */
void log_manager::init(void)
{
	std::string es;
	std::string filename;
	bool done = false;
	int count = 0;
	estring str;


	done = false;
	while (!done) {
		TRY_nomem(filename = config.log_dir());
		TRY_nomem(filename += "/");
		TRY_nomem(filename += config.timestamp().str());
		if (config.action() == configuration_manager::action_archive) {
			TRY_nomem(filename += ".log");
		}
		else if (config.action() == configuration_manager::action_relink) {
			TRY_nomem(filename += ".relink");
		}
		if (count != 0) {
			TRY_nomem(filename += ".");
			TRY_nomem(filename += estring(count));
		}
		if (exists(filename))
			++count;
		else
			done = true;
	}
	m_out.open(filename.c_str());
	if (!m_out.is_open()) {
		TRY_nomem(es = "Could not open log file: \"");
		TRY_nomem(es += filename);
		TRY_nomem(es += "\"");
		throw(ERROR(errno,es));
	}

	m_use_error_logging_level = false;

	m_initialized = true;

	str = "Rsync Vault Manager - ";
	str += VERSION;
	str += '\n';
	logger.write(str);
}

/** The initialized state of the log manager */
const bool log_manager::initialized(void) const
{
	return(m_initialized);
}

/** Write a string to the log file */
void log_manager::write(
	const std::string& a_str,
	const uint16 a_indention,
	const configuration_manager::logging_type a_logging_level,
	const pid_t a_pid
	)
{
	std::string::const_iterator csi;

	if (!initialized())
		throw(INTERNAL_ERROR(0,"Log manager is not initialized"));

	if (m_use_error_logging_level) {
		if (a_logging_level > config.error_logging_level())
			return;
	}
	else {
		if (a_logging_level > config.logging_level())
			return;
	}
	
	for (csi = a_str.begin(); csi != a_str.end(); ++csi) {
		if (m_new_line) {
			m_out << stamp(a_pid, a_indention);
			m_new_line = false;
		}
		m_out << *csi;
		if (*csi == '\n')
			m_new_line = true;
	}
	m_out.flush();
}

/** Use error-logging-level instead of logging-level */
void log_manager::set_error_logging(bool a_b)
{
	m_use_error_logging_level = a_b;
}

//-----------------------------------------------------------------------------

/** The global log manager */
log_manager logger;

//-----------------------------------------------------------------------------

