#include "config.h"

#include <iostream>
#include <string>
#include <cassert>

#include "asserts.h"
#include "types.h"
#include "error.h"
#include "estring.h"
#include "strfmt.h"

// #define ERR_OUT(e) std::cerr << e
#define ERR_OUT(e)

#define TEST(code) \
	try { \
		code; \
	} \
	catch(error e) { \
		ERR_OUT(e); \
		thrown = true; \
	} \
	catch(...) { \
		assert(0); \
	}


void test1(void)
{
	std::string str;
	bool thrown;
	uint64 ui64;

	thrown = false;
	TEST(str = size_to_string(static_cast<signed char>(-127)));
	assert(!thrown);
	ERR_OUT(str << std::endl);
	assert(str == "  -127 b");

	thrown = false;
	ui64 = 1023;
	TEST(str = size_to_string(ui64));
	assert(!thrown);
	ERR_OUT(str << std::endl);
	assert(str == "  1023 b");

	thrown = false;
	ui64 = 1024;
	TEST(str = size_to_string(ui64));
	assert(!thrown);
	ERR_OUT(str << std::endl);
	assert(str == "   1.0Kb");

	thrown = false;
	ui64 = 1224;
	TEST(str = size_to_string(ui64));
	assert(!thrown);
	ERR_OUT(str << std::endl);
	assert(str == "   1.1Kb");

	thrown = false;
	ui64 = 156789;
	ui64 *= 1000;
	ui64 *= 1000;
	ui64 *= 1000;
	ui64 *= 1000;
	TEST(str = size_to_string(ui64,"b",8,2,1000));
	assert(!thrown);
	ERR_OUT(str << std::endl);
	assert(str == "156.78Pb");

	thrown = false;
	ui64 = 156;
	ui64 *= 1000;
	ui64 *= 1000;
	ui64 *= 1000;
	ui64 *= 1000;
	ui64 *= 1000;
	TEST(str = size_to_string(ui64,"b",8,1,1000));
	assert(!thrown);
	ERR_OUT(str << std::endl);
	assert(str == " 156.0Pb");

	thrown = false;
	ui64 = 156;
	ui64 *= 1000;
	ui64 *= 1000;
	ui64 *= 1000;
	ui64 *= 1000;
	ui64 *= 1000;
	ui64 *= 1000;
	TEST(str = size_to_string(ui64,"b",8,1,1000));
	assert(thrown);
	ERR_OUT(str << std::endl);
	assert(str == " 156.0Pb");
}

void test2(void)
{
	std::string str;
	bool thrown = false;

	thrown = false;
	TEST(str = num_to_string(0));
	assert(!thrown);
	ERR_OUT(str << std::endl);
	assert(str == "0");

	thrown = false;
	TEST(str = num_to_string(10));
	assert(!thrown);
	ERR_OUT(str << std::endl);
	assert(str == "10");

	thrown = false;
	TEST(str = num_to_string(100));
	assert(!thrown);
	ERR_OUT(str << std::endl);
	assert(str == "100");

	thrown = false;
	TEST(str = num_to_string(1000));
	assert(!thrown);
	ERR_OUT(str << std::endl);
	assert(str == "1,000");

	thrown = false;
	TEST(str = num_to_string(10000));
	assert(!thrown);
	ERR_OUT(str << std::endl);
	assert(str == "10,000");

	thrown = false;
	TEST(str = num_to_string(100000));
	assert(!thrown);
	ERR_OUT(str << std::endl);
	assert(str == "100,000");

	thrown = false;
	TEST(str = num_to_string(1000000));
	assert(!thrown);
	ERR_OUT(str << std::endl);
	assert(str == "1,000,000");

	thrown = false;
	TEST(str = num_to_string(0,9));
	assert(!thrown);
	ERR_OUT(str << std::endl);
	assert(str == "        0");

	thrown = false;
	TEST(str = num_to_string(10,9));
	assert(!thrown);
	ERR_OUT(str << std::endl);
	assert(str == "       10");

	thrown = false;
	TEST(str = num_to_string(100,9));
	assert(!thrown);
	ERR_OUT(str << std::endl);
	assert(str == "      100");

	thrown = false;
	TEST(str = num_to_string(1000,9));
	assert(!thrown);
	ERR_OUT(str << std::endl);
	assert(str == "    1,000");

	thrown = false;
	TEST(str = num_to_string(10000,9));
	assert(!thrown);
	ERR_OUT(str << std::endl);
	assert(str == "   10,000");

	thrown = false;
	TEST(str = num_to_string(100000,9));
	assert(!thrown);
	ERR_OUT(str << std::endl);
	assert(str == "  100,000");

	thrown = false;
	TEST(str = num_to_string(1000000,9));
	assert(!thrown);
	ERR_OUT(str << std::endl);
	assert(str == "1,000,000");
}

void test3(void)
{
	std::string str;
	bool thrown;
	uint64 ui64;

	thrown = false;
	TEST(str = throughput_to_string(static_cast<signed char>(-127)));
	assert(!thrown);
	ERR_OUT(str << std::endl);
	assert(str == "  -127 b/s");

	thrown = false;
	ui64 = 1023;
	TEST(str = throughput_to_string(ui64));
	assert(!thrown);
	ERR_OUT(str << std::endl);
	assert(str == "  1023 b/s");

	thrown = false;
	ui64 = 1024;
	TEST(str = throughput_to_string(ui64));
	assert(!thrown);
	ERR_OUT(str << std::endl);
	assert(str == "   1.0Kb/s");

	thrown = false;
	ui64 = 1224;
	TEST(str = throughput_to_string(ui64));
	assert(!thrown);
	ERR_OUT(str << std::endl);
	assert(str == "   1.1Kb/s");

	thrown = false;
	ui64 = 156789;
	ui64 *= 1000;
	ui64 *= 1000;
	ui64 *= 1000;
	ui64 *= 1000;
	TEST(str = throughput_to_string(ui64,"b","s",10,2,1000));
	assert(!thrown);
	ERR_OUT(str << std::endl);
	assert(str == "156.78Pb/s");

	thrown = false;
	ui64 = 156;
	ui64 *= 1000;
	ui64 *= 1000;
	ui64 *= 1000;
	ui64 *= 1000;
	ui64 *= 1000;
	TEST(str = throughput_to_string(ui64,"b","s",10,1,1000));
	assert(!thrown);
	ERR_OUT(str << std::endl);
	assert(str == " 156.0Pb/s");

	thrown = false;
	ui64 = 156;
	ui64 *= 1000;
	ui64 *= 1000;
	ui64 *= 1000;
	ui64 *= 1000;
	ui64 *= 1000;
	ui64 *= 1000;
	TEST(str = throughput_to_string(ui64,"b","s",10,1,1000));
	assert(thrown);
	ERR_OUT(str << std::endl);
	assert(str == " 156.0Pb/s");
}

void test4(void)
{
	const uint8 max = max_limit<uint8>();
	uint8 x;
	std::string s;
	std::string::size_type s_len;
	std::string::size_type idx;

	for (x = 0; x != max; ++x) {
		s = percent_string(x,max);
		idx = s.find(".");
		s_len = s.size();
		/**/
		if (s_len - idx > 3)
			std::cerr 
				<< "percent_string(" 
				<< static_cast<unsigned int>(x)
				<< "," 
				<< static_cast<unsigned int>(max)
				<< ") = " 
				<< s
				<< std::endl;
		/**/
		assert(s_len - idx < 4);
	}

}

int main(int argc, char const * argv[])
{
	try {
		test1();
		test2();
		test3();
		test4();
	}
	catch(error e) {
		std::cerr << e;
		assert(0);
	}
	catch(...) {
		std::cerr << err_unknown;
		assert(0);
	}
	return(0);
}

