#include "config.h"

#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include <iostream>
#include <string>
#include <fstream>
#include <iomanip>
#include <string>
#include <cassert>
#include <algorithm>

#include "asserts.h"
#include "error.h"
#include "fs.h"
#include "rconfig.h"
#include "logger.h"
#include "vaulter.h"
#include "archiver.h"
#include "cataloger.h"
#include "reporter.h"
#include "test-fs-cwd.h"

#define ERR_OUT(e) std::cerr << e
// #define ERR_OUT(e)

bool make_dir(const std::string& path)
{
	bool thrown;
	bool value;

	thrown = false;
	try {
		mk_dir(path);
	}
	catch(...) {
		thrown = true;
	}

	value = (thrown == false);

	return(value);
}

void setup(void)
{
	std::ofstream out;

	assert(make_dir("./test-rvm.dir"));
	assert(make_dir("./test-rvm.dir/vault-1"));
	assert(make_dir("./test-rvm.dir/vault-1/2006-06-16.112632"));
	assert(make_dir("./test-rvm.dir/vault-1/2006-06-16.112632/test-job"));
	assert(make_dir("./test-rvm.dir/log.dir"));
	assert(make_dir("./test-rvm.dir/catalog.dir"));
	assert(chmod("./test-rvm.dir/vault-1/2006-06-16.112632",00500) == 0);
	assert(chmod("./test-rvm.dir/vault-1/2006-06-16.112632/test-job",00500) == 0);

	out.open("./test-rvm.dir/file-1.conf");
	assert(out.is_open());
	out << "link-catalog-dir ./test-rvm.dir/catalog.dir" << std::endl;
	out << "log-dir ./test-rvm.dir/log.dir" << std::endl;
	out << "logging-level rsync" << std::endl;
	out << "error-logging-level rsync" << std::endl;
	out << "rsync-local-path " << LOCAL_RSYNC << std::endl;
	out << "rsync-parallel 1" << std::endl;
	out << "timestamp-resolution second" << std::endl;
	out << "vault ./test-rvm.dir/vault-*" << std::endl;
	out << "vault-overflow-behavior quit" << std::endl;
	out << "vault-overflow-blocks 0" << std::endl;
	out << "vault-overflow-inodes 0" << std::endl;
	out << "vault-selection-behavior round-robin" << std::endl;
	out << "<job>" << std::endl;
	out << "	jobname test-job" << std::endl;
	out << "	archive-path jobname/permutation" << std::endl;
	out << "	path " << check_cwd << "/" << std::endl;
	out << "	rsync-connection-type local" << std::endl;
	out << "	rsync-hardlink true" << std::endl;
	out << "	<rsync-options>" << std::endl;
	out << "		-a -v --progress --stats" << std::endl;
	out << "		--exclude '/.svn/'" << std::endl;
	out << "		--exclude '/.deps/'" << std::endl;
	out << "		--exclude '/autom4te.cache/'" << std::endl;
	out << "		--exclude '/test-rvm.dir/'" << std::endl;
	out << "	</rsync-options>" << std::endl;
	out << "	rsync-retry-count 3" << std::endl;
	out << "</job>" << std::endl;
	out.close();
}

void cleanup(void)
{
	int r;

	r = system("chmod -R u+w ./test-rvm.dir 2> /dev/null");
	assert(system("rm -fr ./test-rvm.dir") == 0);
}

bool test_inodes(
	const std::string& path1,
	const std::string& path2,
	const std::string& file
	)
{
	estring p1, p2;
	filestatus f1, f2;
	bool r;

	p1 = path1;
	p1 += "/";
	p1 += file;
	p1 = reform_path(p1);
	f1.path(p1);

	p2 = path2;
	p2 += "/";
	p2 += file;
	p2 = reform_path(p2);
	f2.path(p2);

	r = (f1.inode() == f2.inode());

	return(r);
}

void test_rvm(void)
{
	std::string local_rsync;
	estring adir[4];
	int adirc = 0;

	local_rsync = LOCAL_RSYNC;
	if (local_rsync.size() == 0) {
		char const * text[] = {
			"",
			"NOTICE: Skipping this test",
			"",
			"This test cannot be run because no local rsync binary was found during",
			"configuration.  This also means that RVM will be compiled with no",
			"default value for the rsync-local-path command.",
			"",
			"To change this, run the configure script with --with-rsync=<path>,",
			"where <path> is the absolute path to the rsync binary.",
			"",
			0
		};
		int c;

		for (c = 0; text[c] != 0; ++c) {
			std::cerr << text[c] << std::endl;
		}
		return;
	}

	{
		char const * argv[256] = { 0 };
		int argc = 0;
		bool thrown = false;
		subdirectory subdir;
		estring dir;

		argv[argc++] = "<program>";
		argv[argc++] = "--timestamp";
		argv[argc++] = "2006-06-16.112632";
		argv[argc++] = "--archive";

		cataloger.clear();
		archiver.clear();
		reporter.clear();
		vaulter.clear();
		logger.clear();
		config.clear();

		config.default_file("./test-rvm.dir/file-1.conf");
		try {
			timer t;

			t.start();

			config.init(argc, argv);
			logger.init();
			vaulter.init();
			reporter.init();
			archiver.init();
			archiver.archive();
			cataloger.init();
			cataloger.catalog();

			t.stop();
			reporter.set_total_time(t);

			// reporter.print_report();
			reporter.file_report();
			logger.clear();

			// std::cerr << "vaulter.vault() == " << vaulter.vault() << std::endl;
			assert(vaulter.vault() == "./test-rvm.dir/vault-1");

			dir = vaulter.vault();
			dir += "/";
			dir += config.timestamp().str();
			dir += "/";
			dir += config.jobs()[0].generate_archive_path(check_cwd);
			adir[adirc++] = dir;

			// std::cout << "dir = " << dir << std::endl;

			assert(
				exists(
					config.log_dir()
					+ static_cast<std::string>("/")
					+ config.timestamp().str()
					+ static_cast<std::string>(".log")
					)
				);
			assert(
				exists(
					config.log_dir()
					+ static_cast<std::string>("/")
					+ config.timestamp().str()
					+ static_cast<std::string>(".report")
					)
				);

			subdir.path(dir);
		}
		catch(error e) {
			ERR_OUT(e);
			thrown = true;
		}
		catch(...) {
			ERR_OUT(err_unknown);
			assert(0);
		}
		assert(!thrown);
		assert(find(subdir.begin(), subdir.end(), "AUTHORS") == subdir.end());
	}

}

int main(int argc, char const * argv[])
{
	if (geteuid() != 0) {
		cleanup();
		setup();
		try {
			test_rvm();
		}
		catch(error e) {
			std::cerr << e;
			assert(0);
		}
		catch(...) {
			std::cerr << err_unknown;
			assert(0);
		}
		cleanup();
	}
	else {
		std::cerr 
			<< "NOTE: Skipping this test because I'm running as root" 
			<< std::endl;
	}
	return(0);
}

