/********************************************************************************
*                                                                               *
*                       FXVTable                                                *
*                                                                               *
*********************************************************************************
* Copyright (C) 2002 by Nikitinskiy Dmitriy.   All Rights Reserved.             *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
********************************************************************************/
#include "FXVTable.h"
#include <fxkeys.h>

#include <stdio.h>

// TODO
// - Rows/Cols resizing. Add options to allow/deny resizing.
// - Moving cell focus by keyboard (Home, End, Top, Bottom, PgUp, PgDown)
// - convFromInt add processing r==-1 and/or c==-1
// - Errors on selecting Join cells, Leading/Trailing cells.
// - Errors on moving to Join cell when them partialy visible
// - When FXVTable.*CellStyle(...) not allow Style operations on joined cells
// - setStyle() not searching prev style, just adding. Better add option.
// - ScrollBars options (V/H/ON_DEMAND)
// - On TAB FocusOut
// - Speedup onMouse on searching mouseCell. Need search from previous mouseCell.
// - Speedup getCellRect(FXint,FXint,FXint&,FXint&) searching from nearest cover
// - On sendCellEvent do autoscroll if need
// - Double buffering
// - DnD

//
// FXVTableStyle
//
FXVTableStyle& FXVTableStyle::operator=(const FXVTableStyle& f){
 row=f.row;
 col=f.col;
 rowHeight=f.rowHeight; 
 colWidth=f.colWidth;
 background=f.background;
 text=f.text;
 border=f.border;
 parameters=f.parameters;
 cellType=f.cellType; 
 return *this;
};
FXVTableStyle::FXVTableStyle(const FXVTableStyle& f) {
 row=f.row;
 col=f.col;
 rowHeight=f.rowHeight; 
 colWidth=f.colWidth;
 background=f.background;
 text=f.text;
 border=f.border;
 parameters=f.parameters;
 cellType=f.cellType; 
 next=NULL;
};


//
// FXVTableStyles
//
FXVTableStyle* FXVTableStyles::lookup(FXVTableStyle* s, FXint r, FXint c) const {
    FXVTableStyle* res=s;
    if(r<0)
      while(res && res->col!=c) 
        res=res->next;
    else
      if(c<0)
        while(res && res->row!=r) 
	  res=res->next;
      else
        while(res && (res->row!=r || res->col!=c)) 
	  res=res->next;
    return res;
};
FXint FXVTableStyles::delElem(FXVTableStyle** s, FXVTableStyle* a){
    FXVTableStyle* res=*s, *prev=NULL;
    while(res && res!=a) { prev=res; res=res->next; };
    
    if(!res)
      return 0;

    if(prev) prev->next=res->next;
    else *s=res->next;
    
    delete res;
    return 1;
};

void FXVTableStyles::delElems(FXVTableStyle** s, FXint r, FXint c){
    FXVTableStyle* s1=*s, *s2;
    while(s1=lookup(s1,r,c)){
        s2=s1->next;
        delElem(s,s1);
        s1=s2;
    };
};

void FXVTableStyles::delList(FXVTableStyle* l){
    FXVTableStyle* r=l,*r1;
    while(r){
        r1=r->next;
        delete r;
        r=r1;
    };
};

void FXVTableStyles::changeAttr(FXint FXVTableStyle::*a, FXint before, FXint d){
   FXVTableStyle *res=cells;
   while(res) {
       if(res->*a>=before) res->*a+=d;
       res=res->next;
   };
   res=cols;
   while(res) {
       if(res->*a>=before) res->*a+=d;
       res=res->next;
   };
   res=rows;
   while(res) {
       if(res->*a>=before) res->*a+=d;
       res=res->next;
   };
};

const FXVTableStyle& FXVTableStyles::getStyle(const FXVTableCellPos& cell) const {// First look in selection.
  const FXVTableStyle *res=NULL;
  if(selection && cell.row>=0 && cell.col>=0)
    res=lookup(selection,cell.row,cell.col);

  if(!res){

    if(cell.row<0 && cell.col<0)
      return global;

    if(cell.row>=0 && cell.col>=0)
        res=lookup(cells,cell.row,cell.col);
    if(cell.row<0 || !res && cell.col>=0)
      res=lookup(cols,-1,cell.col);
    if(cell.col<0 || !res && cell.row>=0)
      res=lookup(rows,cell.row,-1);
    
    if(!res)
       res=&global;
  };

  return *res;
};

void FXVTableStyles::setStyle(const FXVTableCellPos& cell,const FXVTableStyle& s) {
  FXVTableStyle* res;
  if(cell.row<0 && cell.col<0){
     global=s;
     return;
  };
  if(cell.col<0){
    res=lookup(rows,cell.row,-1);
    if(res)
      *res=s;
    else {
      FXVTableStyle* s1=new FXVTableStyle(s);
      s1->row=cell.row;
      s1->col=-1;
      addElem(&rows,s1); 
    };  
    return;
  };
  if(cell.row<0){
    res=lookup(cols,-1,cell.col);
    if(res)
      *res=s;
    else {
      FXVTableStyle* s1=new FXVTableStyle(s);
      s1->col=cell.col;
      s1->row=-1;
      addElem(&cols,s1); 
    };  
    return;
  };
  res=lookup(cells,cell.row,cell.col);
  if(res)
    *res=s;
  else {
    FXVTableStyle* s1=new FXVTableStyle(s);
    s1->row=cell.row;
    s1->col=cell.col;
    addElem(&cells,s1);
  };  
};

void FXVTableStyles::delStyle(const FXVTableCellPos& cell) {
   FXVTableStyle* res;
   if(cell.row<0 && cell.col<0) return;
   
   
   if(cell.col<0){
       res=lookup(rows,cell.row,-1);
       if(res) delElem(&rows,res);
   };
   if(cell.row<0){
       res=lookup(rows,-1,cell.col);
       if(res) delElem(&cols,res);
   };
   res=lookup(cells,cell.row,cell.col);
   if(res) delElem(&cells,res);   
};

FXbool FXVTableStyles::cellJoined(const FXVTableCellPos& cell){
   FXVTableStyle* res;
   if(cell.row<0 || cell.col<0) return FALSE;
   res=lookup(cells,cell.row,cell.col);
   if(!res) return FALSE;
   return res->isJoined();
};


void FXVTableStyles::addSelection(const FXVTableCellPos& cell) {
   FXVTableStyle* s=new FXVTableStyle(getStyle(cell));
   s->parameters|=FXVTableStyle::SELECTED;
   s->background=select.background;
   s->text=select.text;
   s->border=select.border;
   s->row=cell.row;
   s->col=cell.col;
   FXVTableStyle *l=selection, **l1=&selection;
   /*
   while(l && l->next && (l->row<=r && l->col<=c)){
      l1=&(l->next);
      l=l->next;
   };
   */
   addElem(l1,s);
};

void FXVTableStyles::delSelection(const FXVTableCellPos& cell) {
   FXVTableStyle* s=lookup(selection,cell.row,cell.col);
   if(s) delElem(&selection,s);
};

void FXVTableStyles::clearSelection(){
   delList(selection);
   selection=NULL;
};

void FXVTableStyles::insertColumn(FXint before) {
   changeAttr(&FXVTableStyle::col,before,1);
};
void FXVTableStyles::insertRow(FXint before) {
   changeAttr(&FXVTableStyle::row,before,1);
};
void FXVTableStyles::deleteColumn(FXint col) {
   delElems(&cells,-1,col);
   delElems(&cols,-1,col);
   changeAttr(&FXVTableStyle::col,col,-1);
};
void FXVTableStyles::deleteRow(FXint row){
   delElems(&cells,row,-1);
   delElems(&rows,row,-1);
   changeAttr(&FXVTableStyle::row,row,-1);
};

FXVTableStyles::~FXVTableStyles(){
  delList(selection);
  delList(cells);
  delList(cols);
  delList(rows);
};

//
// FXVTableItem
//
FXIMPLEMENT_ABSTRACT(FXVTableItem,FXObject,NULL,0)

FXString FXVTableItem::getData(const FXVTableCellPos& cell){
    return table->getDataSource()->getData(table->convFromInt(cell));
};

void FXVTableItem::setData(const FXVTableCellPos& cell, const FXString& s){
    table->getDataSource()->setData(table->convFromInt(cell),s);
};
void FXVTableItem::drawFrame(FXVTableCellEvent* ev) {
    FXRectangle rect(ev->x,ev->y,ev->w,ev->h);
    if(ev->style.parameters & FXVTableStyle::FOCUSED){
      ev->dc->setForeground(FXRGB(0,0,0));
      ev->dc->setLineWidth(2);
      rect.shrink(1);
      ev->dc->drawRectangles(&rect,1);
    }
    else{
      ev->dc->setLineWidth(1);
      ev->dc->setForeground(ev->style.border);
      ev->dc->drawRectangles(&rect,1);
    };
};


//
// FXVTableLabel
//
FXDEFMAP(FXVTableLabel) FXVTableLabelMap[]={
    FXMAPFUNC(SEL_PAINT,0,FXVTableLabel::onPaint),
    FXMAPFUNC(SEL_FOCUSIN,0,FXVTableLabel::onFocusIn),
    FXMAPFUNC(SEL_FOCUSOUT,0,FXVTableLabel::onFocusOut),
    FXMAPFUNC(SEL_CLICKED,0,FXVTableLabel::onClicked),
    FXMAPFUNC(SEL_KEYPRESS,0,FXVTableLabel::onKeyPress),
    FXMAPFUNC(SEL_CHANGED,FXVTableLabel::ID_EDITOR,FXVTableLabel::onEditorChanged),
    FXMAPFUNC(SEL_COMMAND,FXVTableLabel::ID_EDITOR,FXVTableLabel::onEditorCommand),
//    FXMAPFUNC(SEL_KEYRELEASE,0,FXVTableLabel::onKeyRelease),
};
FXIMPLEMENT(FXVTableLabel,FXVTableItem,FXVTableLabelMap,ARRAYNUMBER(FXVTableLabelMap))

FXVTableLabel::FXVTableLabel(FXVTable* t) : FXVTableItem(t), editorChanged(FALSE) {
  editor=new FXTextField(t,10,this,ID_EDITOR,FRAME_SUNKEN);
};

FXVTableLabel::~FXVTableLabel(){
  if(editor)
    delete editor;
};

void FXVTableLabel::drawText(FXVTableCellEvent* ev){
    ev->dc->setForeground(ev->style.text);
    FXString t=getData(ev->pos);
    ev->dc->drawText(ev->x,ev->y+ev->h,t.text(),t.size());

};

void FXVTableLabel::drawLabel(FXVTableCellEvent* ev){
    ev->dc->setForeground(ev->style.background);
    ev->dc->fillRectangle(ev->x,ev->y,ev->w,ev->h);
    drawText(ev);
    drawFrame(ev);
};

long FXVTableLabel::onPaint(FXObject* obj,FXSelector sel,void* ptr){
    FXVTableCellEvent* ev=(FXVTableCellEvent*)ptr;
    drawLabel(ev);
    return 1;
};
long FXVTableLabel::onFocusIn(FXObject* obj,FXSelector sel,void* ptr) {
    FXVTableCellEvent* ev=(FXVTableCellEvent*)ptr;
    drawLabel(ev);
    return 1;
};
long FXVTableLabel::onFocusOut(FXObject* obj,FXSelector sel,void* ptr) {
    FXVTableCellEvent* ev=(FXVTableCellEvent*)ptr;
    if(editor->shown()) {
      if(editorChanged){
         setData(editorPos,editor->getText());
         editorChanged=FALSE;
      };
      editor->hide();
    };
    drawLabel(ev);
    return 1;
};
long FXVTableLabel::onClicked(FXObject* obj,FXSelector sel,void* ptr) {
    return 1;
};
long FXVTableLabel::onKeyPress(FXObject*,FXSelector sel,void* ptr) {
    FXVTableCellEvent* ev=(FXVTableCellEvent*)ptr;
    editor->position(ev->x,ev->y,ev->w,ev->h); 
    editor->setText(getData(ev->pos));
    editor->show();
    editor->setFocus();
    editorPos=ev->pos;
    editor->handle(table,sel,ev->event);
    return 1;
};

long FXVTableLabel::onEditorChanged(FXObject*,FXSelector,void*) {
    editorChanged=TRUE;
    FXVTableEvent cell(editorPos,0);
    table->handle(this,MKUINT(FXVTable::ID_CHANGE_CELL,SEL_CHANGED),&cell);
    return 1;
};

long FXVTableLabel::onEditorCommand(FXObject*,FXSelector,void*) {

    if(editorChanged){
       setData(editorPos,editor->getText());
       editorChanged=FALSE;
       FXVTableEvent cell(editorPos,0);
       table->handle(this,MKUINT(FXVTable::ID_CHANGE_CELL,SEL_COMMAND),&cell);
    };
    return 1;
};

//
// FXVTableButton
//
 FXDEFMAP(FXVTableButton) FXVTableButtonMap[]={
    FXMAPFUNC(SEL_PAINT,0,FXVTableButton::onPaint),
    FXMAPFUNC(SEL_FOCUSIN,0,FXVTableButton::onFocusIn),
    FXMAPFUNC(SEL_FOCUSOUT,0,FXVTableButton::onFocusOut),
    FXMAPFUNC(SEL_CLICKED,0,FXVTableButton::onClicked),
    FXMAPFUNC(SEL_KEYPRESS,0,FXVTableButton::onKeyPress),
};
FXIMPLEMENT(FXVTableButton,FXVTableItem,FXVTableButtonMap,ARRAYNUMBER(FXVTableButtonMap))

FXVTableButton::FXVTableButton(FXVTable* t) : FXVTableLabel(t) {
  hilite=t->getApp()->getHiliteColor();
  shadow=t->getApp()->getShadowColor();
  base=t->getApp()->getBaseColor();
};

void FXVTableButton::drawButton(FXVTableCellEvent* ev){
    FXDC& dc=*(ev->dc);
    FXint x=ev->x, y=ev->y, w=ev->w, h=ev->h;
    
    dc.setForeground(base);
    dc.fillRectangle(x,y,w,h);

    drawText(ev);

    dc.setForeground(shadow);
    dc.fillRectangle(x,y+h-1,w,1);
    dc.fillRectangle(x+w-1,y,1,h);
    dc.setForeground(hilite);
    dc.fillRectangle(x,y,w,1);
    dc.fillRectangle(x,y,1,h);
};

long FXVTableButton::onPaint(FXObject*,FXSelector,void* ptr) {
    FXVTableCellEvent* ev=(FXVTableCellEvent*)ptr;
    drawButton(ev);
    return 1;
};
long FXVTableButton::onFocusIn(FXObject*,FXSelector,void* ptr) {
    FXVTableCellEvent* ev=(FXVTableCellEvent*)ptr;
    drawButton(ev);
    return 1;
};
long FXVTableButton::onFocusOut(FXObject*,FXSelector,void* ptr) {
    FXVTableCellEvent* ev=(FXVTableCellEvent*)ptr;
    drawButton(ev);
    return 1;
};
long FXVTableButton::onClicked(FXObject*,FXSelector,void* ptr) {
    FXVTableCellEvent* ev=(FXVTableCellEvent*)ptr;
    drawButton(ev);
    return 1;
};
long FXVTableButton::onKeyPress(FXObject*,FXSelector,void* ptr) {
    FXVTableCellEvent* ev=(FXVTableCellEvent*)ptr;
    drawButton(ev);
    return 1;
};
 

//
// FXVTable
//

FXDEFMAP(FXVTable) FXVTableMap[]={
  FXMAPFUNC(SEL_PAINT,0,FXVTable::onPaint),
//  FXMAPFUNC(SEL_TIMEOUT,FXVTable::ID_BLINK,FXVTable::onBlink),
  FXMAPFUNC(SEL_MOTION,0,FXVTable::onMotion),
//  FXMAPFUNC(SEL_TIMEOUT,FXWindow::ID_AUTOSCROLL,FXVTable::onAutoScroll),
  FXMAPFUNC(SEL_UNGRABBED,0,FXVTable::onUngrabbed),
  FXMAPFUNC(SEL_LEFTBUTTONPRESS,0,FXVTable::onLeftBtnPress),
  FXMAPFUNC(SEL_LEFTBUTTONRELEASE,0,FXVTable::onLeftBtnRelease),
  FXMAPFUNC(SEL_RIGHTBUTTONPRESS,0,FXVTable::onRightBtnPress),
  FXMAPFUNC(SEL_RIGHTBUTTONRELEASE,0,FXVTable::onRightBtnRelease),
  FXMAPFUNC(SEL_KEYPRESS,0,FXVTable::onKeyPress),
  FXMAPFUNC(SEL_KEYRELEASE,0,FXVTable::onKeyRelease),
  FXMAPFUNC(SEL_FOCUSIN,0,FXVTable::onFocusIn),
  FXMAPFUNC(SEL_FOCUSOUT,0,FXVTable::onFocusOut),
  FXMAPFUNC(SEL_SELECTION_LOST,0,FXVTable::onSelectionLost),
  FXMAPFUNC(SEL_SELECTION_GAINED,0,FXVTable::onSelectionGained),
  FXMAPFUNC(SEL_CLICKED,0,FXVTable::onClicked),
  FXMAPFUNC(SEL_DOUBLECLICKED,0,FXVTable::onDoubleClicked),
//  FXMAPFUNC(SEL_TRIPLECLICKED,0,FXVTable::onTripleClicked),
  FXMAPFUNC(SEL_COMMAND,0,FXVTable::onCommand),

//  FXMAPFUNC(SEL_COMMAND,FXVTable::ID_EXTEND,FXVTable::onCmdExtend),
  FXMAPFUNC(SEL_COMMAND,FXVTable::ID_HSCROLL,FXVTable::onHScroll),
  FXMAPFUNC(SEL_COMMAND,FXVTable::ID_VSCROLL,FXVTable::onVScroll),
  
  FXMAPFUNC(SEL_COMMAND,FXVTable::ID_MOVE_LEFT,FXVTable::onCmdMoveLeft),
  FXMAPFUNC(SEL_COMMAND,FXVTable::ID_MOVE_RIGHT,FXVTable::onCmdMoveRight),
  FXMAPFUNC(SEL_COMMAND,FXVTable::ID_MOVE_UP,FXVTable::onCmdMoveUp),
  FXMAPFUNC(SEL_COMMAND,FXVTable::ID_MOVE_DOWN,FXVTable::onCmdMoveDown),
  FXMAPFUNC(SEL_COMMAND,FXVTable::ID_MOVE_HOME,FXVTable::onCmdMoveHome),
  FXMAPFUNC(SEL_COMMAND,FXVTable::ID_MOVE_END,FXVTable::onCmdMoveEnd),
  FXMAPFUNC(SEL_COMMAND,FXVTable::ID_MOVE_TOP,FXVTable::onCmdMoveTop),
  FXMAPFUNC(SEL_COMMAND,FXVTable::ID_MOVE_BOTTOM,FXVTable::onCmdMoveBottom),
  FXMAPFUNC(SEL_COMMAND,FXVTable::ID_MOVE_PAGEDOWN,FXVTable::onCmdMovePageDown),
  FXMAPFUNC(SEL_COMMAND,FXVTable::ID_MOVE_PAGEUP,FXVTable::onCmdMovePageUp),
 
};

// Object implementation
FXIMPLEMENT(FXVTable,FXComposite,FXVTableMap,ARRAYNUMBER(FXVTableMap))


FXVTable::FXVTable(FXComposite *p,FXVTableSource* src,FXObject* tgt=NULL,FXSelector sel=0,FXuint opts=0,FXint x=0,FXint y=0,FXint w=0,FXint h=0):
    FXComposite(p,opts,x,y,w,h), dataSource(src), rowsLeading(0), rowsTrailing(0), colsLeading(0),colsTrailing(0), 
    firstRow(0), firstCol(0), lastRow(0), lastCol(0), xPos(0), yPos(0), cursorRow(-1), cursorCol(-1),
    selBeginRow(0), selBeginCol(0), selEndRow(-1), selEndCol(-1)
{
    rows=src->getRows();
    cols=src->getCols();
    flags|=FLAG_SHOWN | FLAG_ENABLED;
    target=tgt;
    message=sel;

    hScroll=new FXScrollbar(this,this,ID_HSCROLL,SCROLLBAR_HORIZONTAL,0,0,0,0);
    vScroll=new FXScrollbar(this,this,ID_VSCROLL,SCROLLBAR_VERTICAL,0,0,0,0);

    tableItemsCount=2;
    FXCALLOC(&tableItems,FXVTableItem*,2);
    tableItems[0]=new FXVTableLabel(this);
    tableItems[1]=new FXVTableButton(this);
   
    FXVTableStyle g(styles.getSelectStyle());
    FXApp* app=p->getApp();
    g.background=app->getSelbackColor();
    g.text=app->getSelforeColor();
    g.parameters=FXVTableStyle::SELECTED;
    styles.setSelectStyle(g);

    mouseRow=0;
    mouseCol=0;
    const FXVTableStyle& s=styles.getStyle(0,0);
    mouseCell.x=0;
    mouseCell.y=0;
    mouseCell.w=s.colWidth;
    mouseCell.h=s.rowHeight;
};

FXVTable::~FXVTable() {
    delete hScroll;
    delete vScroll;
    for(int i=0;i<tableItemsCount;i++)
        delete tableItems[i];
    FXFREE(tableItems);
};

void FXVTable::detach() {
   hScroll->detach();
   vScroll->detach();
   FXComposite::detach();
};

void FXVTable::create() {
    FXComposite::create();
};

FXbool FXVTable::canFocus() const { 
    return TRUE; 
};
void FXVTable::setFocus(){
    FXComposite::setFocus();
    setDefault(TRUE);
}

void FXVTable::killFocus(){
    FXComposite::killFocus();
    setDefault(MAYBE);
}
  
FXVTableCellPos FXVTable::convToInt(const FXVTableCellPos& from) const {
  FXint r=0,c=0;
  if(from.rowSw==0)
    r=rowsLeading;
  if(from.rowSw<0)
    r=rows-rowsTrailing;
  if(from.colSw==0)
    c=colsLeading;
  if(from.colSw<0)
    c=cols-colsTrailing;
  if(from.row<0)
    r=from.row;
  else
    r=from.row+r;
  if(from.col<0)
    c=from.col;
  else
    c=from.col+c;
    
  return FXVTableCellPos(r,c);
};
FXVTableCellPos FXVTable::convFromInt(const FXVTableCellPos& from) const {
  FXint r=-rowsLeading, c=-colsLeading, rs=0, cs=0;
  if(from.row<rowsLeading) {
    rs=1;
    r=0;
  };  
  if(from.row>=rows-rowsTrailing){
    rs=-1;
    r=rowsTrailing-rows;
  };
  if(from.col<colsLeading) {
    cs=1;
    c=0;
  };  
  if(from.col>=cols-colsTrailing){
    cs=-1;
    c=colsTrailing-cols;
  };
  return FXVTableCellPos(from.row+r,from.col+c, rs, cs);
};


void FXVTable::layout(){
    FXint sw=16;
    FXint sh=16;
    vScroll->position(width-sw,0,sw,height-sh);
    hScroll->position(0,height-sh,width-sw,sh);
    dataWidth=width-sw;
    dataHeight=height-sh;

    cellsRect.x=getColsWidth(0,colsLeading-1);
    cellsRect.y=getRowsHeight(0,rowsLeading-1);
    cellsRect.w=dataWidth-getColsWidth(cols-colsTrailing,cols-1)-cellsRect.x;
    cellsRect.h=dataHeight-getRowsHeight(rows-rowsTrailing,rows-1)-cellsRect.y;
    
    recalcView();
}

void FXVTable::recalcView(){
    FXint a=cellsRect.w+xPos;
    FXint b=firstCol;
    while(a>0 && b<cols-1) {
        a-=styles.getStyle(-1,b).colWidth;
        b++;
    };
    lastCol=b;

    a=cellsRect.h+yPos;
    b=firstRow;
    while(a>0 && b<rows-1) {
        a-=styles.getStyle(b,-1).rowHeight;
        b++;
    };
    lastRow=b;

    hScroll->setPage(lastCol-firstCol);
    vScroll->setPage(lastRow-firstRow);

    hScroll->setRange(cols-colsLeading-rowsTrailing);
    vScroll->setRange(rows-rowsLeading-rowsTrailing);
};

FXint FXVTable::getColsWidth(FXint from,FXint to) const {
    FXint w=0;
    for(FXint i=from;i<=to;i++)
        w+=styles.getStyle(-1,i).colWidth;
    return w;
};
FXint FXVTable::getRowsHeight(FXint from,FXint to) const {
    FXint h=0;
    for(FXint i=from;i<=to;i++)
        h+=styles.getStyle(i,-1).rowHeight;
    return h;
};

FXbool FXVTable::cellVisible(FXint row, FXint col) const {
    FXbool x=FALSE, y=FALSE;
    if(row<rowsLeading)
        x=TRUE;
    if(row>=rows-rowsTrailing)
        x=TRUE;
    if(col<colsLeading)
        y=TRUE;
    if(col>=cols-colsTrailing)
        y=TRUE;
    if(! (x && y) ) {
        if(!x && row>=firstRow && row<=lastRow)
            x=TRUE;
        if(!y && col>=firstCol && col<=lastCol)
            y=TRUE;
    };
    return (x && y);
};

FXSize FXVTable::getCellSize(FXint row, FXint col){
        FXint w,h;
	const FXVTableStyle& s=styles.getStyle(row,col);
        if(s.isJoining()){
            // This first cell of join
            w=0;
            h=0;
            for(FXint i=0;i<s.colWidth;i++)
                w+=styles.getStyle(-1,col+i).colWidth;
            for(FXint i=0;i<s.rowHeight;i++)
                h+=styles.getStyle(row+i,-1).rowHeight;
        }
        else{
           w=styles.getStyle(-1,col).colWidth;
           h=styles.getStyle(row,-1).rowHeight;
        };
	return FXSize(w,h);
};

FXRectangle FXVTable::getCellRect(FXint row, FXint col) {
    FXint x=0, y=0, w=0, h=0;
    const FXVTableStyle& s=styles.getStyle(row,col);
    if( cellVisible(row,col) && !s.isJoined() ){
        if(row<rowsLeading){
            y=getRowsHeight(0,row-1);
            h=1;
        };
        if(row>=rows-rowsTrailing){
            y=dataHeight-getRowsHeight(row,rows-1);
            h=1;
        };
        if(col<colsLeading){
            x=getColsWidth(0,col-1);
            w=1;
        };
        if(col>=cols-colsTrailing){
            x=dataWidth-getColsWidth(col,cols-1);
            w=1;
        };

        if(!w){
            x=cellsRect.x+getColsWidth(firstCol,col-1);
        };
        if(!h){
            y=cellsRect.y+getRowsHeight(firstRow,row-1);
        };
        FXSize cs(getCellSize(row,col));
	w=cs.w;
	h=cs.h;
    };
    return FXRectangle(x,y,w,h);
};

FXRectangle FXVTable::getCellRect(FXint cx, FXint cy, FXint& row, FXint& col) {
    FXint x=0, y=0, w=0, h=0;
    FXint r,c, lr=-1, lc=-1;

    if(cx<cellsRect.x){
      c=0;
      x=0;
      lc=colsLeading-1;
    }
    else
        if(cx>cellsRect.x+cellsRect.w){
          c=cols-colsTrailing;
          x=dataWidth-getColsWidth(cols-colsTrailing,cols-1);
          lc=cols-1;
        }
        else{
          c=firstCol;
          x=cellsRect.x;
          lc=lastCol;
        };
    if(cy<cellsRect.y){
      r=0;
      y=0;
      lr=rowsLeading-1;
    }
    else
        if(cy>cellsRect.y+cellsRect.h){
          r=rows-rowsTrailing;
          y=dataHeight-getRowsHeight(rows-rowsTrailing,rows-1);
          lr=rows-1;
        }
        else{
          r=firstRow;
          y=cellsRect.y;
          lr=lastRow;
        };

    while(r<=lr && y+(h=styles.getStyle(r,-1).rowHeight)<cy){
          r++;
          y+=h;
    };
    while(c<=lc && x+(w=styles.getStyle(-1,c).colWidth)<cx){
          c++;
          x+=w;
    };
    
    const FXVTableStyle& s=styles.getStyle(r,c);
    FXRectangle rect(x,y,w,h);
    row=r;
    col=c;
     if(s.row!=-1 && s.col!=-1){
	if(s.isJoining()){
        //Join cell
        for(FXint i=1;i<s.colWidth;i++)
            rect.w+=styles.getStyle(-1,c+i).colWidth;
        for(FXint i=1;i<s.rowHeight;i++)
            rect.h+=styles.getStyle(r+i,-1).rowHeight;
	}
	else 
	   if(s.isJoined()){
    	     row=r+s.rowHeight;
             col=c+s.colWidth;
 	     rect=getCellRect(row,col); 
	   };
	  
    };

   return rect;
};


FXint FXVTable::sendCellEvent(FXuint event, FXVTableCellEvent* ev){
    FXRectangle rv(cellsRect.x,0,cellsRect.w,dataHeight), rh(0,cellsRect.y,dataWidth,cellsRect.h);
    FXRectangle rc(ev->x,ev->y,ev->w,ev->h);
    if(overlap(rv,rc)){
       rc*=rv;
    };
    if(overlap(rh,rc)){
       rc*=rh;
    };
    ev->dc->setClipRectangle(rc);
    if(ev->style.cellType>=0){
      if(mode&SELECTING && ev->pos.row>=FXMIN(selBeginRow,selEndRow) && ev->pos.col>=FXMIN(selBeginCol,selEndCol) &&
         ev->pos.row<=FXMAX(selBeginRow,selEndRow) && ev->pos.col<=FXMAX(selBeginCol,selEndCol)) {
           FXint tc=ev->style.cellType;
           ev->style=styles.getSelectStyle();
	   ev->style.cellType=tc;
      };
      if(ev->pos.row==cursorRow && ev->pos.col==cursorCol && !(mode&SELECTING)){
           ev->style.parameters|=FXVTableStyle::FOCUSED;
	   ev->x=rc.x;
	   ev->y=rc.y;
	   ev->w=rc.w;
	   ev->h=rc.h;
      }
      return tableItems[ev->style.cellType]->handle(this,event,(void*)ev);
    };
};


void FXVTable::drawRange(FXDC& dc,FXRectangle& rect,FXint fr,FXint fc,FXint lr,FXint lc,FXint x, FXint y){
  FXint w,h;
  FXint yc=y;
  for(int i=fr;i<lr;i++) {
      FXint xc=x;
      for(int j=fc;j<lc;j++){
          const FXVTableStyle& s=styles.getStyle(i,j);
          w=styles.getStyle(-1,j).colWidth;
          h=styles.getStyle(i,-1).rowHeight;
          if(!s.isJoined() && overlap(rect,FXRectangle(xc,yc,w,h))){
            FXRectangle rect(xc,yc,w,h);
            if(s.isJoining()){
                rect=getCellRect(i,j);
//                w=styles.getStyle(-1,j).colWidth;
//                h=styles.getStyle(i,-1).rowHeight;
            };
            FXVTableCellEvent ev(&dc,s,FXVTableCellPos(i,j),rect);
            sendCellEvent(MKUINT(0,SEL_PAINT),&ev);
          } else{
              if(s.row!=-1 && s.col!=-1){
//                w=styles.getStyle(-1,j).colWidth;
//                h=styles.getStyle(i,-1).rowHeight;
              };
          };
          xc+=w;
          if(j==lc-1)
            yc+=h;
      };
  };

};

void FXVTable::drawCells(FXint rb, FXint cb, FXint re, FXint ce){
      FXint srb=FXMIN(rb,re), scb=FXMIN(cb,ce);
      FXint sre=FXMAX(rb,re), sce=FXMAX(cb,ce);

      FXRectangle fcell=getCellRect(srb,scb);
      FXDCWindow dc(this);
      drawRange(dc,cellsRect,srb,scb,sre+1,sce+1,fcell.x,fcell.y);
};

void FXVTable::drawCell(FXint row, FXint col){
  FXRectangle cell=getCellRect(row,col);
  FXDCWindow dc(this);
  drawRange(dc,cell,row,col,row+1,col+1,cell.x,cell.y);
};
  
FXbool FXVTable::joinCells(const FXVTableCellPos& cell, FXint right, FXint down){
//
// Need check before make join
//
    FXVTableCellPos c(convToInt(cell));
    FXVTableStyle s(styles.getStyle(c.row,c.col));
    s.parameters|=FXVTableStyle::JOIN;
    for(FXint i=c.row;i<c.row+down;i++){
        for(FXint j=c.col;j<c.col+right;j++){
            s.row=i;
            s.col=j;
            if(i==c.row && j==c.col){
                s.rowHeight=down;
                s.colWidth=right;
            }
            else{
                s.rowHeight=c.row-i;
                s.colWidth=c.col-j;
            };
            styles.setStyle(FXVTableCellPos(i,j),s);
        };
    };
    return TRUE;
};
FXbool FXVTable::joinDelete(const FXVTableCellPos& cell){
//
// Need check all cells before remove join
//
    FXVTableCellPos c(convToInt(cell));
    FXint &fromRow=c.row, &fromCol=c.col;
    FXVTableStyle s=styles.getStyle(fromRow,fromCol);
    if((s.row!=fromRow || s.col!=fromCol) && !s.isJoining())
        return FALSE;
    for(FXint i=fromRow;i<fromRow+s.rowHeight;i++)
        for(FXint j=fromCol;j<fromCol+s.colWidth;j++)
            styles.delStyle(i,j);
    return TRUE;
};

void FXVTable::clearSelection() {
    styles.clearSelection();
};
void FXVTable::addSelectionRange(FXint rowB, FXint colB, FXint rowE, FXint colE){
    FXint rb=FXMIN(rowB,rowE), re=FXMAX(rowB,rowE);
    FXint cb=FXMIN(colB,colE), ce=FXMAX(colB,colE);
    for(FXint i=rb;i<=re;i++)
      for(FXint j=cb;j<=ce;j++)
        styles.addSelection(i,j);
};

void FXVTable::moveCursorPos(FXint row, FXint col, FXRectangle* r){
    FXRectangle rect;
    FXDCWindow dc(this);
    FXVTableStyle s=styles.getStyle(cursorRow,cursorCol);
    s.parameters&=~(FXVTableStyle::FOCUSED | FXVTableStyle::SELECTED);
    
//    FXbool lcRow=cursorRow, lcCol=cursorCol;
    
    if(!cellVisible(cursorRow,cursorCol)){
       cursorRect=FXRectangle(0,0,0,0);
    };
//    if(lcRow==selRow && lcCol==selCol){
      FXVTableCellEvent eventCurs(&dc,s,FXVTableCellPos(cursorRow,cursorCol),cursorRect);
      cursorRow=-1;
      sendCellEvent(MKUINT(0,SEL_FOCUSOUT),&eventCurs);
//    };
    if(r) rect=*r;
    else rect=getCellRect(row,col);
    cursorRow=row;
    cursorCol=col;
    s=styles.getStyle(cursorRow,cursorCol);
    if(s.isJoined()){
      s=styles.getStyle(s.row+s.rowHeight,s.col+s.colWidth);
      cursorRow=s.row;
      cursorCol=s.col;
    };

    if(mode & SELECTING){
      FXint orow=selEndRow, ocol=selEndCol;
      selEndRow=cursorRow;
      selEndCol=cursorCol;
      if(selEndRow<selBeginRow)
         orow=FXMIN(selEndRow,orow);
      else
         if(selEndRow!=selBeginRow)
           orow=FXMAX(selEndRow,orow);
      if(selEndCol<selBeginCol)
         ocol=FXMIN(selEndCol,ocol);
      else
         if(selEndCol!=selBeginCol)
           ocol=FXMAX(selEndCol,ocol);
     drawCells(selBeginRow,selBeginCol,orow,ocol);
    }
    else{
     //No selection
      FXVTableCellEvent event(&dc,s,FXVTableCellPos(cursorRow,cursorCol),rect);
      cursorRect=rect;
      sendCellEvent(MKUINT(0,SEL_FOCUSIN),&event);
      clearSelection();
      FXint rb=selBeginRow, cb=selBeginCol, re=selEndRow, ce=selEndCol; 
      selBeginRow=cursorRow;
      selBeginCol=cursorCol;
      selEndRow=cursorRow;
      selEndCol=cursorCol;
      drawCells(rb,cb,re,ce);
    };
};

void FXVTable::childPosition(FXWindow* o, FXint x, FXint y, FXint w, FXint h){
  noPaint=FXRectangle(o->getX(),o->getY(),o->getWidth(),o->getHeight());
  o->position(x,y,w,h);
};

void FXVTable::scrollContext(FXint x, FXint y, FXint w, FXint h, FXint dx, FXint dy){
//   for(FXint i=0; i<tableItemsCount; i++){
//     tableItems[i]->scrollContext(x,y,w,h,dx,dy);
//   };
   scroll(x,y,w,h,dx,dy);
};

//
//  Event Handlers
//

long FXVTable::onPaint(FXObject*,FXSelector,void* ptr) {
  FXEvent* ev=(FXEvent*)ptr;

  if(/*ev->rect!=noPaint*/ !noPaint.w){
  FXDCWindow dc(this,ev);
  
  dc.setForeground(FXRGB(255,255,255));
  dc.fillRectangle(ev->rect.x,ev->rect.y,ev->rect.w,ev->rect.h);
  FXint x0=cellsRect.x+cellsRect.w, y0=cellsRect.y+cellsRect.h;

  if(rowsLeading) {
    drawRange(dc,ev->rect,0,0,rowsLeading,colsLeading,0,0);
    drawRange(dc,ev->rect,0,firstCol,rowsLeading,lastCol,cellsRect.x,0);
    drawRange(dc,ev->rect,0,cols-colsLeading,rowsLeading,cols,x0,0);
  };

  if(colsLeading) {
    drawRange(dc,ev->rect,firstRow,0,lastRow,colsLeading,0,cellsRect.y);
  };
  if(colsTrailing) {
    drawRange(dc,ev->rect,firstRow,cols-colsTrailing,lastRow,cols,x0,cellsRect.y);
  };
  if(rowsTrailing) {
    drawRange(dc,ev->rect,rows-rowsTrailing,0,rows,colsLeading,0,y0);
    drawRange(dc,ev->rect,rows-rowsTrailing,firstCol,rows,lastCol,cellsRect.x,y0);
    drawRange(dc,ev->rect,rows-rowsTrailing,cols-colsTrailing,rows,cols,x0,y0);
  };

  dc.setClipRectangle(cellsRect);
  drawRange(dc,ev->rect,firstRow,firstCol,lastRow,lastCol,cellsRect.x,cellsRect.y);
  dc.clearClipRectangle();
  
  }
  else {
    noPaint.w=0;
    noPaint.h=0;
  };
  
  return 1;
};    

long FXVTable::onFocusIn(FXObject*,FXSelector,void* ptr) {
    if(cursorRow==-1 || cursorCol==-1){
        cursorRow=firstRow;
        cursorCol=firstCol;
        cursorRect=getCellRect(firstRow,firstCol);
    };
    FXDCWindow dc(this);
    FXVTableCellEvent event(&dc,styles.getStyle(cursorRow,cursorCol),FXVTableCellPos(cursorRow,cursorCol),cursorRect);
    sendCellEvent(MKUINT(0,SEL_FOCUSIN),&event);
    return 1;
};
long FXVTable::onFocusOut(FXObject*,FXSelector,void* ptr) {
    if(cursorRow!=-1 && cursorCol!=-1 ){
      FXEvent *ev=(FXEvent*)ptr;
      FXDCWindow dc(this);
      FXVTableCellEvent event(&dc,styles.getStyle(cursorRow,cursorCol),FXVTableCellPos(cursorRow,cursorCol),cursorRect);
      sendCellEvent(MKUINT(0,SEL_FOCUSOUT),&event);
    }
    return 1;
};
long FXVTable::onMotion(FXObject*,FXSelector,void* ptr) {
    FXEvent* ev=(FXEvent*) ptr;
    FXint r=-1,c=-1;
    FXRectangle cell;
    FXCursor* curs;
    
    if(mode & (RESIZING_ROW | RESIZING_COL)){
      FXint mr, mc;
      if(mode & RESIZING_ROW) { mr=mouseRow; mc=-1; }
      else { mr=-1; mc=mouseCol; };
      FXVTableStyle s=styles.getStyle(mr,mc);
      if(mode & RESIZING_ROW){
        s.rowHeight=abs(ev->win_y-mouseCell.y);
	mouseCell.h=s.rowHeight;
      }
      else{
        s.colWidth=abs(ev->win_x-mouseCell.x);
	mouseCell.w=s.colWidth;
      };
      s.row=mr;
      s.col=mc;
      styles.setStyle(mr,mc,s);
      update();
      return 1;
    };

    //Correcting mouseCell
    if(!mouseCell.contains(ev->win_x,ev->win_y)){
      mouseCell=getCellRect(ev->win_x,ev->win_y,mouseRow,mouseCol);
      mouseCell.x++;
      mouseCell.y++;
      mouseCell.w--;
      mouseCell.h--;
    };

    if(/*ev->win_x==mouseCell.x ||*/ ev->win_x==mouseCell.x+mouseCell.w){
      curs=getApp()->getDefaultCursor(DEF_HSPLIT_CURSOR);
      setDefaultCursor(curs);
      return 1;
    };
    if(/*ev->win_y==mouseCell.y ||*/ ev->win_y==mouseCell.y+mouseCell.h) {
      curs=getApp()->getDefaultCursor(DEF_VSPLIT_CURSOR);
      setDefaultCursor(curs);
      return 1;
    };
    
    setDefaultCursor(getApp()->getDefaultCursor(DEF_ARROW_CURSOR));
    
    
    if(mode & SELECTING && !(mode&KEY_SELECT)){
//      cell=getCellRect(ev->win_x,ev->win_y,r,c);
      if(mouseRow!=cursorRow || mouseCol!=cursorCol){
         moveCursorPos(mouseRow,mouseCol,&mouseCell);
      };
    };
    return 1;
};
long FXVTable::onKeyPress(FXObject* obj,FXSelector sel,void* ptr) {
    FXEvent* ev=(FXEvent*) ptr;
    if(!isEnabled()) return 0;
    
    if(ev->state & (KEY_Shift_L | KEY_Shift_R))
      mode|=SHIFT_KEY;
    if(ev->state & (KEY_Control_L | KEY_Control_R))
      mode|=CTRL_KEY;
   
    switch(ev->code){
       case KEY_Up:
       case KEY_KP_Up:
           handle(this,MKUINT(ID_MOVE_UP,SEL_COMMAND),NULL);
	   return 1;
       case KEY_Down:
       case KEY_KP_Down:
           handle(this,MKUINT(ID_MOVE_DOWN,SEL_COMMAND),NULL);
	   return 1;
       case KEY_Left:
       case KEY_KP_Left:
           handle(this,MKUINT(ID_MOVE_LEFT,SEL_COMMAND),NULL);
	   return 1;
       case KEY_Right:
       case KEY_KP_Right:
           handle(this,MKUINT(ID_MOVE_RIGHT,SEL_COMMAND),NULL);
	   return 1;
       default:
           if(getFocus())
	     getFocus()->handle(obj,sel,ptr);
	   else {
	     FXDCWindow dc(this);
	     FXVTableCellEvent e(&dc, styles.getStyle(cursorRow,cursorCol), FXVTableCellPos(cursorRow, cursorCol), cursorRect, ev); 
	     sendCellEvent(sel, &e);
	   };
   };
    return 1;
};
long FXVTable::onKeyRelease(FXObject*,FXSelector,void* ptr) {
    FXEvent* ev=(FXEvent*) ptr;
    
    if(! (ev->state & (KEY_Shift_L | KEY_Shift_R))){
      mode&=~SHIFT_KEY;
      if(mode&KEY_SELECT){
        mode&=~(SELECTING|KEY_SELECT);
      };
    };
    if(!(ev->state & (KEY_Control_L | KEY_Control_R)))
      mode&=~CTRL_KEY;
     return 1;
};
long FXVTable::onLeftBtnPress(FXObject*,FXSelector,void* ptr) {
    FXEvent *ev=(FXEvent*)ptr;
    FXint row, col;
    if(/*ev->win_x==mouseCell.x ||*/ ev->win_x==mouseCell.x+mouseCell.w){
      mode|=RESIZING_COL;
      return 1;
    };
    if(/*ev->win_y==mouseCell.y ||*/ ev->win_y==mouseCell.y+mouseCell.h) {
      mode|=RESIZING_ROW;
      return 1;
    };
//    FXRectangle rect=getCellRect(ev->win_x,ev->win_y,row,col);
    moveCursorPos(mouseRow,mouseCol,&mouseCell);
    if(!mode&SELECTING){
      mode|=SELECTING;
      mode&=~KEY_SELECT;
    };
    return 1;
};
long FXVTable::onLeftBtnRelease(FXObject*,FXSelector,void*) {
    if(mode & (RESIZING_ROW|RESIZING_COL)){
      mode&=~(RESIZING_ROW|RESIZING_COL);
    };
    if(mode&SELECTING && !(mode&KEY_SELECT)){
      addSelectionRange(selBeginRow,selBeginCol,selEndRow,selEndCol);
      mode&=~SELECTING;
    };
    return 1;
};
long FXVTable::onRightBtnPress(FXObject*,FXSelector,void*) {
    return 1;
};
long FXVTable::onRightBtnRelease(FXObject*,FXSelector,void*) {
    return 1;
};
long FXVTable::onUngrabbed(FXObject*,FXSelector,void*) {
    return 1;
};
long FXVTable::onClicked(FXObject*,FXSelector,void*) {
    return 1;
};
long FXVTable::onDoubleClicked(FXObject*,FXSelector,void*) {
    return 1;
};

long FXVTable::onCommand(FXObject*,FXSelector,void*) {    
    return 1;
};

long FXVTable::onHScroll(FXObject*,FXSelector,void*) {
    FXint c=hScroll->getPosition()+colsLeading, f, l;
    FXint dx=0;
    if(c>firstCol){
        f=FXMIN(c-firstCol,cols-colsTrailing-lastCol);
        if(f){
          dx=getColsWidth(firstCol,firstCol+f-1);
          firstCol+=f;
          lastCol+=f;
          scrollContext(cellsRect.x,0,cellsRect.w,dataHeight,-dx,0);
        };
    }
    else{
        f=firstCol-c;
        if(f){
          dx=getColsWidth(c,firstCol-1);
          firstCol=c;
          lastCol-=f;
          scrollContext(cellsRect.x,0,cellsRect.w,dataHeight,dx,0);
        };
    };
    return 1;
};
long FXVTable::onVScroll(FXObject*,FXSelector,void*) {
    FXint c=vScroll->getPosition()+rowsLeading, f, l;
    FXint dx=0;
    if(c>firstRow){
        f=FXMIN(c-firstRow,rows-rowsTrailing-lastRow);
        if(f){
          dx=getRowsHeight(firstRow,firstRow+f-1);
          firstRow+=f;
          lastRow+=f;
          scrollContext(0,cellsRect.y,dataWidth,cellsRect.h,0,-dx);
        };
    }
    else{
        f=firstRow-c;
        if(f){
          dx=getRowsHeight(c,firstRow-1);
          firstRow=c;
          lastRow-=f;
          scrollContext(0,cellsRect.y,dataWidth,cellsRect.h,0,dx);
        };
    };
    return 1;
};

long FXVTable::onCmdMoveLeft(FXObject*,FXSelector,void*) {
    if(!cursorCol) return 1;
    if(mode&SELECTING && !(mode&KEY_SELECT)) return 1;
    FXint nr=cursorRow, nc=cursorCol-1;
    const FXVTableStyle& s=styles.getStyle(nr,nc);
    if(s.isJoined()){
      nc+=s.colWidth;
      nr+=s.rowHeight;
    };
    if(cursorCol==firstCol && firstCol>colsLeading){
      //Check scroll needs
      FXSize d=getCellSize(nr,nc);
      scrollContext(cellsRect.x,0,cellsRect.w,dataHeight,d.w,0);
      firstCol--;
      lastCol--;
      if(nc<firstCol){
        lastCol-=firstCol-nc;
	firstCol=nc;
      };
      cursorRect.x+=d.w;
    };
    if(mode&SHIFT_KEY){
      if(!(mode&SELECTING))
        mode|=SELECTING|KEY_SELECT;
    }
    moveCursorPos(nr,nc);
};
long FXVTable::onCmdMoveRight(FXObject*,FXSelector,void*) {
    if(cursorCol==cols-1) return 1;
    if(mode&SELECTING && !(mode&KEY_SELECT)) return 1;
    FXint nr=cursorRow, nc=cursorCol+1;
    const FXVTableStyle& s=styles.getStyle(nr,nc);
    if(s.isJoined()){
      nc+=s.colWidth;
      nr+=s.rowHeight;
      const FXVTableStyle& s1=styles.getStyle(cursorRow,cursorCol);
      if(s1.isJoining()){
        nc+=s1.colWidth;
      };
    };
    if(cursorCol==lastCol-1 && lastCol<cols-colsTrailing){
      //Check scroll needs
      FXSize d=getCellSize(nr,nc);
      scrollContext(cellsRect.x,0,cellsRect.w,dataHeight,-d.w,0);
      firstCol++;
      lastCol++;
      cursorRect.x-=d.w;
    };
    if(mode&SHIFT_KEY){
      if(!(mode&SELECTING))
        mode|=SELECTING|KEY_SELECT;
    }
    moveCursorPos(nr,nc);
};

long FXVTable::onCmdMoveUp(FXObject*,FXSelector,void*) {
    if(!cursorRow) return 1;
    if(mode&SELECTING && !(mode&KEY_SELECT)) return 1;
    FXint nr=cursorRow-1, nc=cursorCol;
    const FXVTableStyle& s=styles.getStyle(nr,nc);
    if(s.isJoined()){
      nr+=s.rowHeight;
      nc+=s.colWidth;
    };
    if(cursorRow==firstRow && firstRow>rowsLeading){
      //Check scroll needs
      FXSize d=getCellSize(nr,nc);
      scrollContext(0,cellsRect.y,dataWidth,cellsRect.h,0,d.h);
      firstRow--;
      lastRow--;
      if(nr<firstRow){
        lastRow-=firstRow-nr;
	firstRow=nr;
      };
      cursorRect.y+=d.h;
    };
    if(mode&SHIFT_KEY){
      if(!(mode&SELECTING))
        mode|=SELECTING|KEY_SELECT;
    }
    moveCursorPos(nr,nc);
   return 1;
};

long FXVTable::onCmdMoveDown(FXObject*,FXSelector,void*) {
    if(cursorRow==rows-1) return 1;
    if(mode&SELECTING && !(mode&KEY_SELECT)) return 1;
    FXint nr=cursorRow+1, nc=cursorCol;
    const FXVTableStyle& s=styles.getStyle(nr,nc);
    if(s.isJoined()){
     nc+=s.colWidth;
     nr+=s.rowHeight;
      const FXVTableStyle& s1=styles.getStyle(cursorRow,cursorCol);
      if(s1.isJoining()){
        nr+=s1.rowHeight;
      };
    };
    if(cursorRow==lastRow-1 && lastRow<rows-rowsTrailing){
      //Check scroll needs
      FXSize d=getCellSize(nr,nc);
      scrollContext(0,cellsRect.y,dataWidth,cellsRect.h,0,-d.h);
      firstRow++;
      lastRow++;
      cursorRect.y-=d.h;
    };
    if(mode&SHIFT_KEY){
      if(!(mode&SELECTING))
        mode|=SELECTING|KEY_SELECT;
    }
    moveCursorPos(nr,nc);
   return 1;

};
long FXVTable::onCmdMoveTop(FXObject*,FXSelector,void*) {
   return 1;

};
long FXVTable::onCmdMoveBottom(FXObject*,FXSelector,void*) {
   return 1;

};

long FXVTable::onCmdMoveHome(FXObject*,FXSelector,void*) {
   return 1;

};
long FXVTable::onCmdMoveEnd(FXObject*,FXSelector,void*) {
   return 1;

};
long FXVTable::onCmdMovePageUp(FXObject*,FXSelector,void*) {
   return 1;

};
long FXVTable::onCmdMovePageDown(FXObject*,FXSelector,void*) {
   return 1;

};
