/********************************************************************************
*                                                                               *
*                  Child process object/interface                               *
*                                                                               *
*********************************************************************************
* Copyright (C) 2002 by Mathew Robertson.   All Rights Reserved.                *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
*********************************************************************************/
#ifdef WIN32
#include "FXProcess.h"
using namespace FXEX;
namespace FXEX {

FXDEFMAP(FXProcess) FXProcessMap[]={
  FXMAPFUNC(SEL_IO_READ,FXProcess::ID_PROCESS,FXProcess::onTerminate),
  FXMAPFUNC(SEL_IO_READ,FXProcess::ID_OUTPUT,FXProcess::onOutput),
  FXMAPFUNC(SEL_IO_READ,FXProcess::ID_ERROR,FXProcess::onError)
  };
FXIMPLEMENT(FXProcess,FXBaseObject,FXProcessMap,ARRAYNUMBER(FXProcessMap));

static CRITICAL_SECTION  process_input_critical_section;
static int process_is_initialized = 0;

static unsigned long WINAPI InputThreadProc(void* param)
{ return ((FXProcess*)param)->inputThreadHandler(); }

static unsigned long WINAPI OutputThreadProc(void* param)
{ return ((FXProcess*)param)->outputThreadHandler(); }

static unsigned long WINAPI ErrorThreadProc(void* param)
{ return ((FXProcess*)param)->errorThreadHandler(); }

static int popenCreateProcess(const char *cmdstring, HANDLE hStdin, HANDLE hStdout, HANDLE hStderr, HANDLE *hProcess) {
  PROCESS_INFORMATION piProcInfo;
  STARTUPINFO siStartInfo;
  char *s1,*s2, *s3 = " /c ";
  const char *szConsoleSpawn = "w9xpopen.exe \"";
  int i;
  int x;
  
  i = GetEnvironmentVariable("COMSPEC",NULL,0);
  if (i == 0) {
    // Could try cmd.exe / command.com in the path but we'll just error out..
    return -1;
    }
  s1 = (char *)_alloca(i);
  x = GetEnvironmentVariable("COMSPEC", s1, i);
  if (x == 0) return x;  // Better safe than sorry

  if (GetVersion() < 0x80000000) {
    // NT/2000
    x = i + strlen(s3) + strlen(cmdstring) + 1;
    s2 = (char *)_alloca(x);
    ZeroMemory(s2, x);
    sprintf(s2, "%s%s%s", s1, s3, cmdstring);
    }
  else {
    // Oh gag, we're on Win9x. Use the workaround listed in KB: Q150956
    char modulepath[256];
    GetModuleFileName(NULL, modulepath, sizeof(modulepath));
    for (i = x = 0; modulepath[i]; i++) {
      if (modulepath[i] == '\\') x = i+1;
    }
    modulepath[x] = '\0';
    x = i + strlen(s3) + strlen(cmdstring) + 1 + strlen(modulepath) + strlen(szConsoleSpawn) + 1;
    s2 = (char *)_alloca(x);
    ZeroMemory(s2, x);
    sprintf(s2,"%s%s%s%s%s\"",modulepath,szConsoleSpawn,s1,s3,cmdstring);
    }

  ZeroMemory(&siStartInfo, sizeof(STARTUPINFO));
  siStartInfo.cb = sizeof(STARTUPINFO);
  siStartInfo.dwFlags = STARTF_USESTDHANDLES | STARTF_USESHOWWINDOW;
  siStartInfo.hStdInput = hStdin;
  siStartInfo.hStdOutput = hStdout;
  siStartInfo.hStdError = hStderr;
  siStartInfo.wShowWindow = SW_HIDE;

  if (CreateProcess(NULL,s2,NULL,NULL,TRUE,CREATE_NEW_CONSOLE,NULL,NULL,&siStartInfo,&piProcInfo) ) {
    /* Close the handles now so anyone waiting is woken. */
    CloseHandle(piProcInfo.hThread);
    /* Return process handle */
    *hProcess = piProcInfo.hProcess;
    return TRUE;
    }
  return FALSE;
  }


static int win32_error(const char* /*string*/)
{ return -1; }


FXProcess::FXProcess(FXApp* a, FXObject* tgt,FXSelector sel) {
  application = a;
  target = tgt;
  message = sel;
  captureMode = CAPTURE_NONE;
  running = FALSE;
  outAvailEvent = CreateEvent(NULL, TRUE, FALSE, NULL);
  errAvailEvent = CreateEvent(NULL, TRUE, FALSE, NULL);
  outEatenEvent = CreateEvent(NULL, TRUE, FALSE, NULL);
  errEatenEvent = CreateEvent(NULL, TRUE, FALSE, NULL);
  inAvailEvent  = CreateEvent(NULL, TRUE, FALSE, NULL);
  input_list_first = NULL;
  input_list_last = NULL;
  if (process_is_initialized == 0) {
    InitializeCriticalSection(&process_input_critical_section);
    process_is_initialized = 1;
  }
}
 
 
FXProcess::~FXProcess() {
  free_all_input_data();
  CloseHandle(inAvailEvent);
  CloseHandle(outAvailEvent);
  CloseHandle(errAvailEvent);
  CloseHandle(outEatenEvent);
  CloseHandle(errEatenEvent);
  }


int FXProcess::outputThreadHandler() {
  char buf[2050];
  DWORD avail, count = 0;
  output.buffer = buf;
  for(;;) {
    if ((ReadFile(ohnd,&buf[0],1,&count,NULL) == 0) || (count == 0)) break;
    if (PeekNamedPipe(ohnd, NULL, 0, NULL, &avail, NULL) == 0) break;
    if (avail > 2048) avail = 2048;
    if (avail > 0) {
      ReadFile(ohnd,&buf[1],avail,&count,NULL);
      buf[count+1]=0;
      output.buffer_length = count + 1;
      }
    else {
      buf[1]=0;
      output.buffer_length = 1;
      }
    SetEvent(outAvailEvent);
    WaitForSingleObject(outEatenEvent,INFINITE);
    output.buffer_length = 0;
    ResetEvent(outEatenEvent);
    }
  return 0;
  }

int FXProcess::errorThreadHandler() {
  char buf[2050];
  DWORD avail, count = 0;
  error.buffer = buf;
  for(;;) {
    if ((ReadFile(ehnd,&buf[0],1,&count,NULL) == 0) || (count == 0)) break;
    if (PeekNamedPipe(ehnd, NULL, 0, NULL, &avail, NULL) == 0) break;
    if (avail > 2048) avail = 2048;
    if (avail > 0) {
      ReadFile(ehnd,&buf[1],avail,&count,NULL);
      buf[count+1]=0;
      error.buffer_length = count + 1;
      }
    else {
      buf[1]=0;
      error.buffer_length = 1;
      }
    SetEvent(errAvailEvent);
    WaitForSingleObject(errEatenEvent,INFINITE);
    ResetEvent(errEatenEvent);
    }
  return 0;
  }


int FXProcess::inputThreadHandler() {
  DWORD written_bytes;
  InputData* idata;
  for(;;) {
    // wait until some input data is available
    EnterCriticalSection(&process_input_critical_section);
    if (input_list_first == NULL) {
      idata = NULL;
      }
    else {
      idata = input_list_first;
      input_list_first = idata->next;
      if (input_list_first == NULL) input_list_last = NULL;
      }
    LeaveCriticalSection(&process_input_critical_section);
    if (idata == NULL) WaitForSingleObject(inAvailEvent,INFINITE);
    if (phnd == 0) return 0; // process is terminated
    if (idata != NULL) {
      if (WriteFile(ihnd, &idata->buffer[0], idata->buffer_length, &written_bytes, NULL) == 0) break;
      free(idata);
      }
    ResetEvent(inAvailEvent);
    }
  return 0;
  }


/* The following code is based off of KB: Q190351 */
int FXProcess::popen(const char *cmdstring) {
  HANDLE hChildStdinRd,
    hChildStdinWr,
    hChildStdoutRd, 
    hChildStdoutWr,
    hChildStderrRd, 
    hChildStderrWr, 
    hChildStdinWrDup, 
    hChildStdoutRdDup,
    hChildStderrRdDup, 
    hProcess; /* hChildStdoutWrDup; */
	  
  SECURITY_ATTRIBUTES saAttr;
  BOOL fSuccess;
  
  saAttr.nLength = sizeof(SECURITY_ATTRIBUTES);
  saAttr.bInheritHandle = TRUE;
  saAttr.lpSecurityDescriptor = NULL;

  if (!CreatePipe(&hChildStdinRd, &hChildStdinWr, &saAttr, 0)) {
    return win32_error("CreatePipe");
    }
  fSuccess = DuplicateHandle(GetCurrentProcess(), hChildStdinWr,
			     GetCurrentProcess(), &hChildStdinWrDup, 0,
			     FALSE, DUPLICATE_SAME_ACCESS);
  if (!fSuccess) return win32_error("DuplicateHandle");
  CloseHandle(hChildStdinWr);
  
  //------  Output
  if (!CreatePipe(&hChildStdoutRd, &hChildStdoutWr, &saAttr, 0)) {
    return win32_error("CreatePipe");
    }
  fSuccess = DuplicateHandle(GetCurrentProcess(), hChildStdoutRd,
			     GetCurrentProcess(), &hChildStdoutRdDup, 0,
			     FALSE, DUPLICATE_SAME_ACCESS);
  if (!fSuccess) return win32_error("DuplicateHandle");
  CloseHandle(hChildStdoutRd);

  //------  Error
  if (captureMode != CAPTURE_INPUT_OUTPUTERROR) {
    if (!CreatePipe(&hChildStderrRd, &hChildStderrWr, &saAttr, 0)) {
      return win32_error("CreatePipe");
      }
    fSuccess = DuplicateHandle(GetCurrentProcess(), hChildStderrRd,
			       GetCurrentProcess(), &hChildStderrRdDup, 0,
			       FALSE, DUPLICATE_SAME_ACCESS);
    if (!fSuccess) return win32_error("DuplicateHandle");
    CloseHandle(hChildStderrRd);
    }
  
  switch (captureMode) {
  case CAPTURE_INPUT:
    /* Case for writing to child Stdin in text mode. */
    /* We don't care about these pipes anymore, so close them. */
    if (hChildStdoutRdDup) CloseHandle(hChildStdoutRdDup);
    if (hChildStderrRdDup) CloseHandle(hChildStderrRdDup);
    ihnd = hChildStdinWrDup;
    break;

  case CAPTURE_OUTPUT:
    /* Case for readinig from child Stdout in binary mode. */
    /* We don't care about these pipes anymore, so close them. */
    if (hChildStdinWrDup) CloseHandle(hChildStdinWrDup);
    if (hChildStderrRdDup) CloseHandle(hChildStderrRdDup);
    ohnd = hChildStdoutRdDup;
    break;
	
  case CAPTURE_INPUT_OUTPUTERROR:
  case CAPTURE_INPUT_OUTPUT:
    if (captureMode != CAPTURE_INPUT_OUTPUTERROR) {
      if (hChildStderrRdDup) CloseHandle(hChildStderrRdDup);
      }
    ihnd = hChildStdinWrDup;
    ohnd = hChildStdoutRdDup;
    break;
    
  case CAPTURE_INPUT_OUTPUT_ERROR:
    ihnd = hChildStdinWrDup;
    ohnd = hChildStdoutRdDup;
    ehnd = hChildStderrRdDup;
    break;
    }

  if (!popenCreateProcess(cmdstring,hChildStdinRd,hChildStdoutWr,hChildStdoutWr,&hProcess)) {
    return win32_error("CreateProcess");
    }

  /* Child is launched. Close the parents copy of those pipe
   * handles that only the child should have open.  You need to
   * make sure that no handles to the write end of the output pipe
   * are maintained in this process or else the pipe will not close
   * when the child process exits and the ReadFile will hang. */
  running = TRUE;

  phnd = hProcess;
  application->addInput(phnd,INPUT_READ,this,ID_PROCESS);

  if (ohnd) application->addInput(outAvailEvent,INPUT_READ,this,ID_OUTPUT);
  if (ehnd) application->addInput(errAvailEvent,INPUT_READ,this,ID_ERROR);

  if (!CloseHandle(hChildStdinRd)) {
    return win32_error("CloseHandle");
  }

  if (!CloseHandle(hChildStdoutWr)) {
    return win32_error("CloseHandle");
  }
	  
  if (CAPTURE_INPUT_OUTPUTERROR != captureMode) {
    if (!CloseHandle(hChildStderrWr)) {
      return win32_error("CloseHandle");
    }
  }

  DWORD thread_id;
  if (ihnd) {
    ithread = CreateThread(NULL,4096,InputThreadProc,this,0,&thread_id);
    }
  if (ohnd) {
    othread = CreateThread(NULL,4096,OutputThreadProc,this,0,&thread_id);
    application->addInput(outAvailEvent,INPUT_READ,this,ID_OUTPUT);
    }
  if (ehnd) {
    ethread = CreateThread(NULL,4096,ErrorThreadProc,this,0,&thread_id);
    application->addInput(errAvailEvent,INPUT_READ,this,ID_ERROR);
    }
  return 0;
  }


long FXProcess::onTerminate(FXObject*,FXSelector,void*) {
  application->removeInput(phnd,INPUT_READ);
  running = false;
  DWORD exit_code;
  if (0==GetExitCodeProcess(phnd,&exit_code)) exit_code=-1;
  CloseHandle(phnd);
  phnd=0;

  if (othread) {
    application->removeInput(outAvailEvent,INPUT_READ);
    if (outAvailEvent) onOutput(NULL,0,NULL);
    CloseHandle(ohnd);
    WaitForSingleObject(othread,INFINITE);
    CloseHandle(othread);
    ohnd = 0;
    othread = 0;
    }
  if (ethread) {
    application->removeInput(errAvailEvent,INPUT_READ);
    if (errAvailEvent) onError(NULL,0,NULL);
    CloseHandle(ehnd);
    WaitForSingleObject(ethread,INFINITE);
    CloseHandle(ethread);
    ehnd = 0;
    ethread = 0;
    }
  if (ithread) {
    SetEvent(inAvailEvent);
    WaitForSingleObject(ithread,INFINITE);
    CloseHandle(ithread);
    CloseHandle(ihnd);
    ihnd = 0;
    ithread = 0;
    }
  return target ? target->handle(this,MKUINT(message,SEL_END),(void*)exit_code) : 0;
  }


FXbool FXProcess::execute(FXString& cmdstring) {
  if (running) return FALSE;
  ihnd = 0;
  ohnd = 0;
  ehnd = 0;
  phnd = 0;
  ithread = 0;
  othread = 0;
  ethread = 0;
  free_all_input_data();
  ResetEvent(inAvailEvent);
  ResetEvent(outAvailEvent);
  ResetEvent(errAvailEvent);
  ResetEvent(outEatenEvent);
  ResetEvent(errEatenEvent);
  popen(cmdstring.text()); //TODO result
  return TRUE;
  }


FXbool FXProcess::terminate(FXint code) {
  return (TerminateProcess(phnd,code)!=0);
  }


// remove all input data that is currently queued
void FXProcess::free_all_input_data() {
  EnterCriticalSection(&process_input_critical_section);
  while(input_list_first != NULL) {
    InputData* temp = input_list_first->next;
    free(input_list_first);
    input_list_first = temp;
    }
  input_list_first = NULL;
  input_list_last = NULL;
  LeaveCriticalSection(&process_input_critical_section);
  }


// adds the data to the input queue
void FXProcess::feedInput(const char* str, int len) {
  if (len == 0) return;
  InputData* d = (InputData*)malloc(len + sizeof(InputData) - 1);
  memcpy(&d->buffer[0],str,len);
  d->buffer_length = len;
  d->next = NULL;
  EnterCriticalSection(&process_input_critical_section);
  if (input_list_first == NULL) {
    input_list_first = d;
    }
  else {
    input_list_last->next = d;
    }
  input_list_last = d;
  SetEvent(inAvailEvent);
  LeaveCriticalSection(&process_input_critical_section);
  }


// Called when the 'outAvailEvent' Event-Object is signaled.
// FIXME this are just returning zero - why?
long FXProcess::onOutput(FXObject*,FXSelector,void*) {
  long res = 0;
  ResetEvent(outAvailEvent);
  if (target) target->handle(this,MKUINT(message,SEL_OUTPUT),(void*)&output);
  SetEvent(outEatenEvent);
  return res;
  }


// Called when the 'errAvailEvent' Event-Object is signaled.
// FIXME this are just returning zero - why?
long FXProcess::onError(FXObject*,FXSelector,void*) {
  long res = 0;
  ResetEvent(errAvailEvent);
  if (target) target->handle(this,MKUINT(message,SEL_ERROR),(void*)&error);
  SetEvent(errEatenEvent);
  return res;
  }

}
#endif
