/********************************************************************************
*                                                                               *
*                  Utility functions                                            *
*                                                                               *
*********************************************************************************
* Copyright (C) 2003 by Mathew Robertson.   All Rights Reserved.                *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
*********************************************************************************/
#include <config.h>
#include <fox/fxver.h>
#include <fox/xincs.h>
#include <fox/fxdefs.h>
#include <fox/FXStream.h>
#include <fox/FXString.h>
#include <fox/FXFont.h>
#include <fox/FXSize.h>
#include <fox/FXPoint.h>
#include <fox/FXRectangle.h>
#include <fox/FXRegistry.h>
#include <fox/FXApp.h>
#include <fox/FXDC.h>
#include <fox/FXList.h>
#include <fox/FXFrame.h>
#include <fox/FXComboBox.h>
#include <fox/FXGroupBox.h>
#include <fox/FXHeader.h>
#include <fox/FXIconList.h>
#include <fox/FXLabel.h>
#include <fox/FXListBox.h>
#include <fox/FXIcon.h>
#include <fox/FXMenuPane.h>
#include <fox/FXMDIChild.h>
#include <fox/FXMenuCaption.h>
#include <fox/FXProgressBar.h>
#include <fox/FXRuler.h>
#include <fox/FXSpinner.h>
#include <fox/FXStatusLine.h>
#include <fox/FXTable.h>
#include <fox/FXTextField.h>
#include <fox/FXCharset.h>
#include <fox/FXText.h>
#include <fox/FXToolTip.h>
#include <fox/FXTreeList.h>
#include <fox/FXTreeListBox.h>
#include <fox/FXFile.h>
#include <fox/FXImage.h>
#include <fox/FXGIFImage.h>
#include <fox/FXBMPImage.h>
#include <fox/FXXPMImage.h>
#include <fox/FXPCXImage.h>
#include <fox/FXICOImage.h>
#include <fox/FXTGAImage.h>
#include <fox/FXRGBImage.h>
#include <fox/FXPNGImage.h>
#include <fox/FXJPGImage.h>
#include <fox/FXTIFImage.h>
#include <fox/FXIcon.h>
#include <fox/FXGIFIcon.h>
#include <fox/FXBMPIcon.h>
#include <fox/FXXPMIcon.h>
#include <fox/FXPCXIcon.h>
#include <fox/FXICOIcon.h>
#include <fox/FXTGAIcon.h>
#include <fox/FXRGBIcon.h>
#include <fox/FXPNGIcon.h>
#include <fox/FXJPGIcon.h>
#include <fox/FXTIFIcon.h>
using namespace FX;
#include "fxexdefs.h"
#include "fxexutils.h"
#include "FXDateTime.h"
#include "FXCheckList.h"
#include "FXCheckTreeList.h"
#include "FXComboBoxEx.h"
#include "FXTableBox.h"
#include "FXTicker.h"
using namespace FXEX;

namespace FXEX {

// save the machine dependant endianness value
void FXUtils::saveEndianness(FXStream& store) {
  store << FXStream::isLittleEndian();
  }

// load the machine dependant endianness value,
// and setup stream to swap the bytes if the stream is non-native
void FXUtils::loadEndianness(FXStream &store){
  FXbool endianness;
  store >> endianness;
  store.swapBytes(endianness!=FXStream::isLittleEndian());
  }

// return a seed for use with fxrandom
FXuint FXUtils::fxrandomseed(){
  return (FXuint) FXDateTime::now();
  }

// return a random integer between a low value and a high value
FXint FXUtils::fxrand(FXint upperBound, FXint lowerBound){
  static FXuint seed=FXUtils::fxrandomseed();
  FXint value=lowerBound-1;
  while (value < lowerBound || value > (upperBound-1)){
    value = (FXint) lrint(((fxrandom(seed) / (RAND_MAX * 2.0)) - 0.05) * upperBound) + lowerBound;
    }
  return value;
  }

// return a random float between a low value and a high value
FXfloat FXUtils::fxrand(FXfloat upperBound, FXfloat lowerBound){
  static FXuint seed=FXUtils::fxrandomseed();
  FXfloat value=lowerBound-1.0;
  while (value < lowerBound || value > (upperBound-1.0)){
    value = (FXfloat) rint(((fxrandom(seed) / (RAND_MAX * 2.0)) - 0.05) * upperBound) + lowerBound;
    }
  return value;
  }

// set the font of all children of this window
void FXUtils::setFontOfChildren(FXComposite* w,FXFont* f) {
  if(w) {
    for(FXint t=0;t<w->numChildren();t++) {
      FXObject *c=w->childAtIndex(t);

      // recurse into children
      if(dynamic_cast<FXComposite*>(c)) setFontOfChildren(static_cast<FXComposite*>(c),f);

      // FOX widgets
      else if(dynamic_cast<FXComboBox*>(c)   ) static_cast<FXComboBox*>(c)->setFont(f);
      else if(dynamic_cast<FXGroupBox*>(c)   ) static_cast<FXGroupBox*>(c)->setFont(f);
      else if(dynamic_cast<FXHeader*>(c)     ) static_cast<FXHeader*>(c)->setFont(f);
      else if(dynamic_cast<FXIconList*>(c)   ) static_cast<FXIconList*>(c)->setFont(f);
      else if(dynamic_cast<FXLabel*>(c)      ) static_cast<FXLabel*>(c)->setFont(f);
      else if(dynamic_cast<FXListBox*>(c)    ) static_cast<FXListBox*>(c)->setFont(f);
      else if(dynamic_cast<FXList*>(c)       ) static_cast<FXList*>(c)->setFont(f);
      else if(dynamic_cast<FXMDIChild*>(c)   ) static_cast<FXMDIChild*>(c)->setFont(f);
      else if(dynamic_cast<FXMenuCaption*>(c)) static_cast<FXMenuCaption*>(c)->setFont(f);
      else if(dynamic_cast<FXProgressBar*>(c)) static_cast<FXProgressBar*>(c)->setFont(f);
      else if(dynamic_cast<FXRuler*>(c)      ) static_cast<FXRuler*>(c)->setFont(f);
      else if(dynamic_cast<FXSpinner*>(c)    ) static_cast<FXSpinner*>(c)->setFont(f);
      else if(dynamic_cast<FXStatusLine*>(c) ) static_cast<FXStatusLine*>(c)->setFont(f);
      else if(dynamic_cast<FXTable*>(c)      ) static_cast<FXTable*>(c)->setFont(f);
      else if(dynamic_cast<FXTextField*>(c)  ) static_cast<FXTextField*>(c)->setFont(f);
      else if(dynamic_cast<FXText*>(c)       ) static_cast<FXText*>(c)->setFont(f);
      else if(dynamic_cast<FXToolTip*>(c)    ) static_cast<FXToolTip*>(c)->setFont(f);
      else if(dynamic_cast<FXTreeListBox*>(c)) static_cast<FXTreeListBox*>(c)->setFont(f);
      else if(dynamic_cast<FXTreeList*>(c)   ) static_cast<FXTreeList*>(c)->setFont(f);

      // FXEX widgets
      else if(dynamic_cast<FXCheckList*>(c)    ) static_cast<FXCheckList*>(c)->setFont(f);
      else if(dynamic_cast<FXCheckTreeList*>(c)) static_cast<FXCheckTreeList*>(c)->setFont(f);
      else if(dynamic_cast<FXComboBoxEx*>(c)   ) static_cast<FXComboBoxEx*>(c)->setFont(f);
      else if(dynamic_cast<FXTableBox*>(c)     ) static_cast<FXTableBox*>(c)->setFont(f);
      else if(dynamic_cast<FXTicker*>(c)       ) static_cast<FXTicker*>(c)->setFont(f);
      }
    }
  }

// enable all widgets in a window
void FXUtils::enableWindow(FXWindow *w){
  if (w) {
    w->enable();
    for (w=w->getFirst();w;w=w->getNext()) disableWindow(w);
    }
  }

// disable all widgets in a window - from Thomas Fridrichsmeier
void FXUtils::disableWindow(FXWindow *w){
  if (w) {
    w->disable();
    for (w=w->getFirst();w;w=w->getNext()) disableWindow(w);
    }
  }

// sets the flags as appropriate for the platform
FXuint FXUtils::fxfilemode(FXFilePermission permissions){
  FXuint mode=0;
#ifndef WIN32
  if (permissions&FILEPERM_USER_READ  ) mode|=S_IRUSR;
  if (permissions&FILEPERM_USER_WRITE ) mode|=S_IWUSR;
  if (permissions&FILEPERM_USER_EXEC  ) mode|=S_IXUSR;
  if (permissions&FILEPERM_GROUP_READ ) mode|=S_IRGRP;
  if (permissions&FILEPERM_GROUP_WRITE) mode|=S_IWGRP;
  if (permissions&FILEPERM_GROUP_EXEC ) mode|=S_IXGRP;
  if (permissions&FILEPERM_OTHER_READ ) mode|=S_IROTH;
  if (permissions&FILEPERM_OTHER_WRITE) mode|=S_IWOTH;
  if (permissions&FILEPERM_OTHER_EXEC ) mode|=S_IXOTH;
  if (permissions&FILEPERM_SET_UID    ) mode|=S_ISUID;
  if (permissions&FILEPERM_SET_GID    ) mode|=S_ISGID;
  if (permissions&FILEPERM_STICKY     ) mode|=S_ISVTX;
#endif
  return mode;
  }

// load an image from a file
FXbool FXUtils::loadFileImage(const FXString& file,FXImage *img) {
  FXFileStream stream;
  if(FXFile::exists(file) && stream.open(file,FXStreamLoad)){
    FXApp::instance()->beginWaitCursor();
    img->loadPixels(stream);
    stream.close();
    img->create();
    FXApp::instance()->endWaitCursor();
    return TRUE;
    }
  return FALSE;
  }

// load an icon from a file
FXbool FXUtils::loadFileIcon(const FXString& file,FXIcon *ico) {
  FXFileStream stream;
  if(FXFile::exists(file) && stream.open(file,FXStreamLoad)){
    FXApp::instance()->beginWaitCursor();
    ico->loadPixels(stream);
    stream.close();
    ico->create();
    FXApp::instance()->endWaitCursor();
    return TRUE;
    }
  return FALSE;
  }

// load an image from a file; returns an FXImage
FXImage* FXUtils::loadImage(const FXString& file){
  FXString ext=FXFile::extension(file);
  FXImage *img=NULL;

  // try matching image type, by file name extension
  if(comparecase(ext,"gif")==0)
    img=new FXGIFImage(FXApp::instance(),NULL,IMAGE_KEEP|IMAGE_SHMI|IMAGE_SHMP);
  else if(comparecase(ext,"bmp")==0)
    img=new FXBMPImage(FXApp::instance(),NULL,IMAGE_KEEP|IMAGE_SHMI|IMAGE_SHMP);
  else if(comparecase(ext,"xpm")==0)
    img=new FXXPMImage(FXApp::instance(),NULL,IMAGE_KEEP|IMAGE_SHMI|IMAGE_SHMP);
  else if(comparecase(ext,"pcx")==0)
    img=new FXPCXImage(FXApp::instance(),NULL,IMAGE_KEEP|IMAGE_SHMI|IMAGE_SHMP);
  else if(comparecase(ext,"ico")==0)
    img=new FXICOImage(FXApp::instance(),NULL,IMAGE_KEEP|IMAGE_SHMI|IMAGE_SHMP);
  else if(comparecase(ext,"tga")==0)
    img=new FXTGAImage(FXApp::instance(),NULL,IMAGE_KEEP|IMAGE_SHMI|IMAGE_SHMP);
  else if(comparecase(ext,"rgb")==0)
    img=new FXRGBImage(FXApp::instance(),NULL,IMAGE_KEEP|IMAGE_SHMI|IMAGE_SHMP);
  else if(comparecase(ext,"png")==0)
    img=new FXPNGImage(FXApp::instance(),NULL,IMAGE_KEEP|IMAGE_SHMI|IMAGE_SHMP);
  else if(comparecase(ext,"jpg")==0 || comparecase(ext,"jpeg")==0)
    img=new FXJPGImage(FXApp::instance(),NULL,IMAGE_KEEP|IMAGE_SHMI|IMAGE_SHMP);
  else if(comparecase(ext,"tif")==0 || comparecase(ext,"tiff")==0)
    img=new FXTIFImage(FXApp::instance(),NULL,IMAGE_KEEP|IMAGE_SHMI|IMAGE_SHMP);

  // Load image if possible...
  if(img && FXUtils::loadFileImage(file,img)) return img;
  delete img;
  return NULL;
  }

// load an icon from a file; returns an FXIcon
FXIcon* FXUtils::loadIcon(const FXString& file){
  FXString ext=FXFile::extension(file);
  FXIcon *ico=NULL;

  // try matching icon type, by file name extension
  if(comparecase(ext,"gif")==0)
    ico=new FXGIFIcon(FXApp::instance(),NULL,IMAGE_KEEP|IMAGE_SHMI|IMAGE_SHMP);
  else if(comparecase(ext,"bmp")==0)
    ico=new FXBMPIcon(FXApp::instance(),NULL,IMAGE_KEEP|IMAGE_SHMI|IMAGE_SHMP);
  else if(comparecase(ext,"xpm")==0)
    ico=new FXXPMIcon(FXApp::instance(),NULL,IMAGE_KEEP|IMAGE_SHMI|IMAGE_SHMP);
  else if(comparecase(ext,"pcx")==0)
    ico=new FXPCXIcon(FXApp::instance(),NULL,IMAGE_KEEP|IMAGE_SHMI|IMAGE_SHMP);
  else if(comparecase(ext,"ico")==0)
    ico=new FXICOIcon(FXApp::instance(),NULL,IMAGE_KEEP|IMAGE_SHMI|IMAGE_SHMP);
  else if(comparecase(ext,"tga")==0)
    ico=new FXTGAIcon(FXApp::instance(),NULL,IMAGE_KEEP|IMAGE_SHMI|IMAGE_SHMP);
  else if(comparecase(ext,"rgb")==0)
    ico=new FXRGBIcon(FXApp::instance(),NULL,IMAGE_KEEP|IMAGE_SHMI|IMAGE_SHMP);
  else if(comparecase(ext,"png")==0)
    ico=new FXPNGIcon(FXApp::instance(),NULL,IMAGE_KEEP|IMAGE_SHMI|IMAGE_SHMP);
  else if(comparecase(ext,"jpg")==0 || comparecase(ext,"jpeg")==0)
    ico=new FXJPGIcon(FXApp::instance(),NULL,IMAGE_KEEP|IMAGE_SHMI|IMAGE_SHMP);
  else if(comparecase(ext,"tif")==0 || comparecase(ext,"tiff")==0)
    ico=new FXTIFIcon(FXApp::instance(),NULL,IMAGE_KEEP|IMAGE_SHMI|IMAGE_SHMP);

  // Load icon if possible...
  if(ico && FXUtils::loadFileIcon(file,ico)) return ico;
  delete ico;
  return NULL;
  }

// alpha blend an icon, againast some base color...
// loop though each pixel in icon, blending its color against the
// specified base color.
FXIcon* blendIcon(FXIcon *icon, FXColor baseColor) {
  FXuchar nr, ng, nb;
  FXuchar sr, sg, sb, sa;
  FXuchar br, bg, bb;
  FXint w, h;
  br = FXREDVAL(baseColor);
  bg = FXGREENVAL(baseColor);
  bb = FXBLUEVAL(baseColor);
  w = icon->getWidth();
  h = icon->getHeight();
  for (FXint x = 0; x < w; x++) {
    for (FXint y = 0; y < h; y++) {
      FXColor color = icon->getPixel(x, y);
      sr = FXREDVAL(color);
      sg = FXGREENVAL(color);
      sb = FXBLUEVAL(color);
      sa = FXALPHAVAL(color);
      nr = ((sr*(sa*256/255)) + (br*(256-(sa*256/255)))) / 256;
      ng = ((sg*(sa*256/255)) + (bg*(256-(sa*256/255)))) / 256;
      nb = ((sb*(sa*256/255)) + (bb*(256-(sa*256/255)))) / 256;
      icon->setPixel(x, y, FXRGB(nr, ng, nb));
      }
    }
  return icon;
  }

}
