/********************************************************************************
*                                                                               *
*                   Scrolling text widget                                       *
*                                                                               *
*********************************************************************************
* Copyright (C) 2002 by Jereon van der Zijp.   All Rights Reserved.             *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
*********************************************************************************/
#include <config.h>
#include <fox/fxver.h>
#include <fox/xincs.h>
#include <fox/fxdefs.h>
#include <fox/FXStream.h>
#include <fox/FXString.h>
#include <fox/FXSize.h>
#include <fox/FXPoint.h>
#include <fox/FXRectangle.h>
#include <fox/FXRegistry.h>
#include <fox/FXApp.h>
#include <fox/FXFont.h>
#include <fox/FXDCWindow.h>
using namespace FX;
#include "FXTicker.h"
using namespace FXEX;
namespace FXEX {

FXDEFMAP(FXTicker) FXTickerMap[]={
  FXMAPFUNC(SEL_PAINT,0,FXTicker::onPaint),
  FXMAPFUNC(SEL_TIMEOUT,FXTicker::ID_TIMER,FXTicker::onTimer),
  };
FXIMPLEMENT(FXTicker,FXFrame,FXTickerMap,ARRAYNUMBER(FXTickerMap))

// Construct and initialize the ticker
FXTicker::FXTicker(FXComposite* p,const FXString& txt,FXuint opts,FXint x,FXint y,FXint w,FXint h,FXint pl,FXint pr,FXint pt,FXint pb) : FXFrame(p,opts,x,y,w,h,pl,pr,pt,pb) {
  label=txt;
  font=getApp()->getNormalFont();
  offset=0;
  speed=0;
  rate=0;
  }

// Report the desired width, as determined by the text and the borders and padding
FXint FXTicker::getDefaultWidth(){
  FXint len=label.length()<5?label.length():5;
  return padleft+font->getTextWidth(label.text(),len)+padleft+padright+(border<<1);
  }

// Report the desired height; this is determined by the font height, and the borders and padding
FXint FXTicker::getDefaultHeight(){
  return font->getFontHeight()+padtop+padbottom+(border<<1);
  }

// Realize the widget and the font
void FXTicker::create(){
  FXFrame::create();
  font->create();
  }

// Detach resources for widget and font
void FXTicker::detach(){
  FXFrame::detach();
  font->detach();
  }

// serialise resources
void FXTicker::save(FXStream& store) const {
  FXFrame::save(store);
  store << label;
  store << font;
  store << speed;
  store << rate;
  store << offset;
  }

// deserialise resources
void FXTicker::load(FXStream& store) {
  FXFrame::load(store);
  store >> label;
  store >> font;
  store >> speed;
  store >> rate;
  store >> offset;
  }


// Change the text; recalculate the layout because we may
// be given a new size based on the new text inside the ticker
void FXTicker::setText(const FXString& txt){
  if (label!=txt){
    label=txt;
    recalc();
    update();
    }
  }

// Change the font; because the font changed, we need to
// enforce a repaint and of course recalculate the layout
void FXTicker::setFont(FXFont *fnt){
  if (font!=fnt){
    font=fnt;
    recalc();
    update();
    }
  }

// Change the ticker speed; a speed of 0 disables the ticker
// We reset the offset so we see the text normally, if the ticker speed is disabled
void FXTicker::setSpeed(FXint s){
  if (speed!=s){
    speed=s;
    getApp()->removeTimeout(this,ID_TIMER);
    if(s>0) getApp()->addTimeout(this,ID_TIMER,s);
    else offset=0;
    recalc();
    update();
    }
  }

// set the ticker rate
void FXTicker::setRate(FXint r){
  if (rate!=r) rate=r;
  }

// Paint the interior of the ticker
long FXTicker::onPaint(FXObject*,FXSelector,void* ptr){
  FXDCWindow dc(this,(FXEvent*)ptr);
  drawFrame(dc,0,0,width,height);
  dc.setClipRectangle(border,border,width-(border<<1),height-(border<<1));
  dc.setForeground(FXRGB(255,255,255));
  dc.fillRectangle(border,border,width-(border<<1),height-(border<<1));
  dc.setForeground(FXRGB(0,0,0));
  dc.setFont(font);
  dc.drawText(border+padleft+offset,height/2+border+padtop,label.text(),label.length());
  return 1;
  }

// When the timer fires, we scroll the ticker and reset the timer for next time.
long FXTicker::onTimer(FXObject*,FXSelector,void*){
  if (rate) {
    FXint textwidth=font->getTextWidth(label.text(),label.length());
    offset+=rate;
    if(offset < -textwidth) offset=width-(border<<1)-padleft-padright;
    else if(offset > width-(border<<1)-padleft-padright) offset=-textwidth;
    scroll(border,border,width-(border<<1),height-(border<<1),rate,0);
    }
  getApp()->addTimeout(this,ID_TIMER,speed);
  return 1;
  }

// The ticker widget is destroyed; remove the timer
FXTicker::~FXTicker(){
  getApp()->removeTimeout(this,ID_TIMER);
  }

}
