/********************************************************************************
*                                                                               *
*                Region helper routines                                         *
*                                                                               *
*********************************************************************************
* Copyright (C) 2003 by Mathew Robertson.   All Rights Reserved.                *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
********************************************************************************/
#include <config.h>
#include <fox/fxver.h>
#include <fox/xincs.h>
#include <fox/fxdefs.h>
#include <fox/FXSize.h>
#include <fox/FXPoint.h>
#include <fox/FXRectangle.h>
#include <fox/FXRegion.h>
using namespace FX;
#include "FXRegionHelper.h"
using namespace FXEX;
namespace FXEX {

// Create helper region
FXRegionHelper::FXRegionHelper(){
  }

// build from existing region helper
FXRegionHelper::FXRegionHelper(const FXRegionHelper& r){
  region=r.region;
  }

// Create a region, based on existing region
FXRegionHelper::FXRegionHelper(const FXRegion& r){
  region=r;
  }

// construct from existing region helper
FXRegionHelper& FXRegionHelper::operator=(const FXRegionHelper& r){
  region=r.region;
  return *this;
  }

// assign from existing region
FXRegionHelper& FXRegionHelper::operator=(const FXRegion& r){
  region=r;
  return *this;
  }

// dtor
FXRegionHelper::~FXRegionHelper(){
  clear();
  }

// empty/clear/reste the region
void FXRegionHelper::clear(){
  region=FXRegion();
  }

// create a polygon region
void FXRegionHelper::createPolygon(const FXPoint* points,FXuint npoints,FXbool winding){
  FXRegion r(points,npoints,winding);
  region+=r;
  }

// make an arc
void FXRegionHelper::makeArc(FXint x,FXint y,FXint w,FXint h,FXuint npoints,FXPoint *points) {
  register FXdouble a = w / 2, b = h / 2;
  register FXint point = 0;
  register FXdouble dx, dy, step=a/(npoints/4), aa=a*a;

  // do first point
  points[point].x = (FXshort)(x-a);
  points[point].y = (FXshort)y;

  // loop around radii/arc
  for (dx = -a; dx < 0; dx += step){
    dy = b * sqrt(1.0 - dx * dx / aa);

    // II quarter
    points[point].x = (FXshort)(x + dx - 0.5);
    points[point].y = (FXshort)(y - dy);

    // I quarter
    points[(npoints/2) - point].x = (FXshort)(x - dx + 0.5);
    points[(npoints/2) - point].y = points[point].y;

    // IV quarter
    points[(npoints/2) + point].x = points[(npoints/2) - point].x;
    points[(npoints/2) + point].y = (FXshort)(y + dy + 0.5);

    // III quarter
    points[npoints - point].x = points[point].x;
    points[npoints - point].y = points[(npoints/2) + point].y;

    point++;
    }

  // point at 90 degrees
  points[point].x = (FXshort)x;
  points[point].y = (FXshort)(y - b);

  // point at 0 degrees
  points[point*2].x = (FXshort)(x + a);
  points[point*2].y = (FXshort)y;

  // point at 270 degrees
  points[point*3].x = (FXshort)x;
  points[point*3].y = (FXshort)(y + b);

  // create ellipse as an elliptic polygon
  FXRegion r(points, point * 4,TRUE);
  region+=r;
  }

// create elliptical region
// - x,y is the center point
// - npoints must be divisable by 4 (eg 180 deg, 360 deg)
void FXRegionHelper::createEllipse(FXint x,FXint y,FXint w,FXint h,FXuint npoints) {
  if (npoints%4) fxerror("FXRegionHelper::createEllipse: npoints must be divisable by 4\n");
  FXPoint points[npoints + 2]; // points of elliptic polygon (eg 180 + 2)
  makeArc(x,y,w,h,npoints,points);
  FXRegion r(points,npoints,TRUE);
  region+=r;
  }

// create circular region
void FXRegionHelper::createCircle(FXint x,FXint y,FXint w,FXuint npoints) {
  if (npoints%4) fxerror("FXRegionHelper::createCircle: npoints must be divisable by 4\n");
  FXPoint points[npoints + 2]; // points of circular polygon (eg 180 + 2)
  makeArc(x,y,w,w,npoints,points);
  FXRegion r(points,npoints,TRUE);
  region+=r;
  }

// create a rectangular region
void FXRegionHelper::createRectangle(FXint x,FXint y,FXint w,FXint h){
  FXRegion r(x,y,w,h);
  region+=r;
  }

// create a square region
void FXRegionHelper::createSquare(FXint x,FXint y,FXint w){
  FXRegion r(x,y,w,w);
  region+=r;
  }

// create a triangle'd region
void FXRegionHelper::createTriangle(FXint x1,FXint y1,FXint x2,FXint y2,FXint x3,FXint y3){
  FXPoint points[3];
  points[0].x=x1;  points[0].y=y1;
  points[1].x=x2;  points[1].y=y2;
  points[2].x=x3;  points[2].y=y3;
  FXRegion r(points,3);
  region+=r;
  }

} // namespace FXEX

