/********************************************************************************
*                                                                               *
*                  Generic history list traversal                               *
*                                                                               *
*********************************************************************************
* Copyright (C) 2003 by Mathew Robertson.   All Rights Reserved.                *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
*********************************************************************************/
#include <config.h>
#include <fox/fxver.h>
#include <fox/xincs.h>
#include <fox/fxdefs.h>
#include <fox/FXStream.h>
#include <fox/FXString.h>
#include <fox/FXSize.h>
#include <fox/FXPoint.h>
#include <fox/FXRectangle.h>
#include <fox/FXRegistry.h>
#include <fox/FXApp.h>
#include <fox/FXWindow.h>
using namespace FX;
#include "FXHistoryItems.h"
using namespace FXEX;
namespace FXEX {


/* This HistoryItems works by creating an new object (in a linked list fashion) for
 * each new item.  That item is appended to the front of the list (as in a LIFO).
 * The app can move up and down the list.
 *
 * Decendants:
 * 1 - Provides a branch mechanism which if you have moved back in the list,
 *     then a new item causes the tree to branch.
 *     As such you then start moving along the new branch. eg the classic browser URL history
 * 2 - Provides a recent item list by which if the entry that is added is already in
 *     the list, that entry moves to the top of the list.
 *     eg: a window list as used by the MDI parent window
 *
 * Todo:
 * - change the 'data' object to use (void*) so that any object can be attached to the list
 *   --> theres a catch, how do you write a void* to the registry?
 */

// Message map
FXDEFMAP(FXHistoryItems) FXHistoryItemsMap[] = {
  FXMAPFUNC(SEL_UPDATE,FXHistoryItems::ID_ANY_ITEMS,FXHistoryItems::onUpdAnyItems),
  FXMAPFUNC(SEL_UPDATE,FXHistoryItems::ID_NEXT_ITEMS,FXHistoryItems::onUpdNextItems),
  FXMAPFUNC(SEL_UPDATE,FXHistoryItems::ID_PREVIOUS_ITEMS,FXHistoryItems::onUpdPreviousItems),
  FXMAPFUNC(SEL_UPDATE,FXHistoryItems::ID_CLEAR,FXHistoryItems::onUpdAnyItems),
  FXMAPFUNC(SEL_COMMAND,FXHistoryItems::ID_CLEAR,FXHistoryItems::onCmdClear),
  FXMAPFUNC(SEL_COMMAND,FXHistoryItems::ID_ITEM_NEXT,FXHistoryItems::onCmdNext),
  FXMAPFUNC(SEL_COMMAND,FXHistoryItems::ID_ITEM_PREVIOUS,FXHistoryItems::onCmdPrevious),
  FXMAPFUNC(SEL_UPDATE,FXHistoryItems::ID_ITEM_NEXT,FXHistoryItems::onUpdNext),
  FXMAPFUNC(SEL_UPDATE,FXHistoryItems::ID_ITEM_PREVIOUS,FXHistoryItems::onUpdPrevious),
  FXMAPFUNCS(SEL_COMMAND,FXHistoryItems::ID_ITEM_0,FXHistoryItems::ID_ITEM_9,FXHistoryItems::onCmdItem),
  FXMAPFUNCS(SEL_UPDATE,FXHistoryItems::ID_ITEM_0,FXHistoryItems::ID_ITEM_9,FXHistoryItems::onUpdItem),
  };
FXIMPLEMENT(FXHistoryItems,FXBaseObject,FXHistoryItemsMap,ARRAYNUMBER(FXHistoryItemsMap))

// Make new History Items group
FXHistoryItems::FXHistoryItems(FXApp *a,const FXString& gp,FXObject* tgt,FXSelector sel,FXint size):FXBaseObject(a,tgt,sel){
  group = gp;
  maxitems = size;
  key = ID_ITEM_NOT_FOUND;
  amount = 0;
  listStart = NULL;
  currentItem = NULL;
  destroySave = FALSE;
  }


// Reset the keys for each item
void FXHistoryItems::renumber(){
  if ( amount == 0 ) return;
  ItemDict* item = listStart;
  register FXint i=0;
  while ( i < amount ){
    item->key = i;
    item = item->previous;
    i++;
    }
  FXASSERT ( item == NULL );
  }


// remove any extra items if necessary
void FXHistoryItems::removeExtras(){
  if ( maxitems != -1 ){
    removeOlder(maxitems-1);
    }
  }


// Change number of items we're tracking
FXint FXHistoryItems::size(const FXint mx){
  FXint max=mx;
  if (max<-1) max=-1;
  if (max==0) clear();
  else{
    maxitems=max;
    removeExtras();
    }
  return maxitems;
  }


// Remove all items from the list
void FXHistoryItems::clear(){
  while ( amount > 0 ){ remove(); }
  key=ID_ITEM_NOT_FOUND;
  currentItem = NULL;
  listStart = NULL;
  }


// Set current key
FXint FXHistoryItems::index(const FXint item){
  if (item<0 || item>=amount) return ID_ITEM_NOT_FOUND;
  currentItem = listStart;
  register FXint i=0;
  while ( i < item ){
    i++;
    currentItem = currentItem->previous;
    }
  key = i;
  return i;
  }


// Get previous/older item
FXint FXHistoryItems::previous(){
  return ( key < (amount-1) ) ? (key + 1) : ID_ITEM_NOT_FOUND;
  }


// Get next/newer item
FXint FXHistoryItems::next(){
  return (key > 0) ? (key - 1) : ID_ITEM_NOT_FOUND;
  }


// Find an item
FXint FXHistoryItems::find(const FXString& name){
  ItemDict* item = listStart;
  register FXint i=0;
  while ( i < amount && item->name != name ){
    i++;
    item = item->previous;
    }
  if ( i == amount ) return ID_ITEM_NOT_FOUND;
  return i;
  }


// Remove current item
FXint FXHistoryItems::remove(){
  if ( key == ID_ITEM_NOT_FOUND || amount == 0 ) return ID_ITEM_NOT_FOUND;
  if ( amount == 1 ){
    listStart = NULL;
    delete currentItem;
    currentItem = NULL;
    key--;
    }
  else{
    if ( key == 0 ){
      listStart = listStart->previous;
      listStart->next = NULL;
      delete currentItem;
      currentItem = listStart;
      }
    else{
      if ( key == (amount-1) ){
        ItemDict* nextItem = currentItem->next;
        nextItem->previous = NULL;
        delete currentItem;
        currentItem = nextItem;
        key--;
        }
      else{
        ItemDict* previousItem = currentItem->previous;
        ItemDict* nextItem     = currentItem->next;
        previousItem->next = nextItem;
        nextItem->previous = previousItem;
        delete currentItem;
        currentItem = nextItem;
        key--;
        }
      }
    }
  amount--;
  renumber();
  FXASSERT ( key < amount );
  return key;
  }


// Remove an item by name, returns the current item
FXint FXHistoryItems::remove(const FXString& name){
  if ( amount == 0 ) return ID_ITEM_NOT_FOUND;
  FXint newKey = find(name);
  if ( newKey == ID_ITEM_NOT_FOUND ) return ID_ITEM_NOT_FOUND;
  FXint currentKey = key;
  if ( currentKey == newKey && currentKey > 0 ) currentKey --;
  index(newKey);
  remove();
  return index(currentKey);
  }


// Remove item by key, returns the current item
FXint FXHistoryItems::remove(const FXint item){
  if ( item < 0 || item >= amount ) return ID_ITEM_NOT_FOUND;
  FXint currentKey = key;
  if ( currentKey == item && currentKey > 0 ) currentKey --;
  index(item);
  remove();
  return index(currentKey);
  }


// Add an item; its added to the top of the list, and everything else
// is moved down the list one notch; the last one is dropped from the list.
// -> Remove any extras
FXint FXHistoryItems::add(const FXString& name,void* data){
  if ( listStart != NULL && listStart->name == name ) return index(0);
  ItemDict* newItem = new ItemDict;
  newItem->name = name;
  newItem->data = data;
  newItem->next = NULL;
  newItem->previous = listStart;
  if (listStart != NULL) listStart->next = newItem;
  listStart = newItem;
  currentItem = newItem;
  key = 0;
  amount++;
  removeExtras();
  renumber();
  return key;
  }


// Remove items newer than specified item
FXint FXHistoryItems::removeNewer(const FXint item){
  if ( item < 1 || item >= amount ) return ID_ITEM_NOT_FOUND;
  FXint i=0;
  while ( i < item ){
    i++;
    remove(0);
    }
  return key;
  }


// Remove items older than specified item
FXint FXHistoryItems::removeOlder(const FXint item){
  if ( item < 0 || item > (amount-2) ) return ID_ITEM_NOT_FOUND;
  while( item < (amount-1) ){
    remove(item+1);
    }
  return key;
  }


// Move named item to top of list
FXint FXHistoryItems::moveFirst(const FXString& name){
  if ( listStart != NULL && listStart->name == name ) return index(0);
  return moveFirst(find(name));
  }


// Move item to top of list
FXint FXHistoryItems::moveFirst(const FXint item){
  if ( item < 0 || item >= amount ) return ID_ITEM_NOT_FOUND;
  if ( index(item) == 0 ) return 0;
  ItemDict* previousItem = currentItem->previous;
  ItemDict* nextItem     = currentItem->next;
  if ( previousItem != NULL ) previousItem->next = nextItem;
  nextItem->previous = previousItem;
  listStart->next = currentItem;
  currentItem->previous = listStart;
  currentItem->next = NULL;
  listStart = currentItem;
  key = 0;
  return key;
  }


// Remove registry instance
void FXHistoryItems::clearReg(){
  getApp()->reg().deleteSection(group.text());
  }


// Save the current history to the registry
// Returns the current item
FXint FXHistoryItems::writeReg(){
  clearReg();
  getApp()->reg().writeIntEntry(group.text(),"Amount",amount);
  FXchar entry[20];
  FXint currentKey = key;
  currentItem = listStart;
  register FXint i=0;
  while ( i < amount ){
    sprintf(entry,"NAME%d",i);
    getApp()->reg().writeStringEntry(group.text(),entry,currentItem->name.text());
    if(stringData){
      sprintf(entry,"DATA%d",i);
//FIXME      getApp()->reg().writeStringEntry(group.text(),entry,currentItem->data.text());
      }
    currentItem = currentItem->next;
    i++;
    }
  getApp()->reg().write();
  return index(currentKey);
  }


// Load the history from the registory
// Returns the most recent item
FXint FXHistoryItems::readReg(){
  clear();
  amount = getApp()->reg().readIntEntry(group.text(),"Amount",0);
  if (amount == 0) return ID_ITEM_NOT_FOUND;
  FXchar entry[20];
  FXString name, data = NULL;
  register FXint i=amount-1;
  while ( i >= 0 ){
    sprintf(entry,"NAME%d",i);
    name = getApp()->reg().readStringEntry(group.text(),entry,NULL);
    FXASSERT ( name.text() == "" );
    if(stringData){
      sprintf(entry,"DATA%d",i);
//FIXME      data = getApp()->reg().readStringEntry(group.text(),entry,NULL);
//FIXME      FXASSERT ( data.text() == "" );
      }
    add ( name, (void*) data.text() );   //FIXME
    i--;
    }
  getApp()->reg().read();
  return index (0);
  }


// Save to the registry on destroy
void FXHistoryItems::saveOnDestroy(FXbool state){
  destroySave = state;
  }


// Save to the registry on destroy
void FXHistoryItems::dataIsString(FXbool state){
  stringData = state;
  }


// Clear the items list
long FXHistoryItems::onCmdClear(FXObject*,FXSelector,void*){
  clear();
  return 1;
  }

 
// Move to an older item in the list
long FXHistoryItems::onCmdPrevious(FXObject*,FXSelector,void*){
  if ( index(previous()) != ID_ITEM_NOT_FOUND){
    if (target) target->handle(this,FXSEL(SEL_COMMAND,message),(void*)currentItem->name.text());
    }
  return 1;
  }

 
// Move to a newer item in the list
long FXHistoryItems::onCmdNext(FXObject*,FXSelector,void*){
  if (index(next()) != ID_ITEM_NOT_FOUND){
    if (target) target->handle(this,FXSEL(SEL_COMMAND,message),(void*)currentItem->name.text());
    }
  return 1;
  }


// Update any 'previous' buttons
long FXHistoryItems::onUpdPrevious(FXObject *sender,FXSelector,void*){
  if ( index() >= (amount-1) )
    sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_DISABLE),NULL);
  else
    sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_ENABLE),NULL);
  return 1;
  }

 
// Update any 'next' buttons
long FXHistoryItems::onUpdNext(FXObject *sender,FXSelector,void*){
  if ( index() <= 0 )
    sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_DISABLE),NULL);
  else
    sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_ENABLE),NULL);
  return 1;
  }

 
// User clicks on one of the item names
long FXHistoryItems::onCmdItem(FXObject*,FXSelector sel,void*){
  FXint which=FXSELID(sel);
  index(which);
  if (target) target->handle(this,FXSEL(SEL_COMMAND,message),(void*)currentItem->name.text());
  return 1;
  }

  
// Update handler for same
long FXHistoryItems::onUpdItem(FXObject *sender,FXSelector sel,void*){
  FXint which=FXSELID(sel);
  if ( which >= amount || which == key ){
    sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_HIDE),NULL);
    return 1;
    }
  ItemDict* item = listStart;
  register FXint i=0;
  while ( i < which ){
    i++;
    item = item->previous;
    }
  FXString string;
  string.format("&%d %s",which,item->name.text());
  sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_SETSTRINGVALUE),(void*)&string);
  sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_SHOW),NULL);
  return 1;
  }


// Show or hide depending on whether there are any items
long FXHistoryItems::onUpdAnyItems(FXObject *sender,FXSelector,void*){
  if (amount > 1)
    sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_SHOW),NULL);
  else
    sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_HIDE),NULL);
  return 1;
  }


// Show or hide depending on whether there are any items next
long FXHistoryItems::onUpdNextItems(FXObject *sender,FXSelector,void*){
  if (amount > 0 && key > 0 )
    sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_SHOW),NULL);
  else
    sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_HIDE),NULL);
  return 1;
  }


// Show or hide depending on whether there are any items previous
long FXHistoryItems::onUpdPreviousItems(FXObject *sender,FXSelector,void*){
  if (amount < 1 || key == (amount-1) )
    sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_HIDE),NULL);
  else
    sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_SHOW),NULL);
  return 1;
  }


// Save data
void FXHistoryItems::save(FXStream& store) const{
  FXBaseObject::save(store);
  store << group;
  store << maxitems;
  store << key;
  store << amount;
  store << destroySave;
  store << stringData;
  ItemDict* item = listStart;
  register FXint i = 0;
  while ( i<amount){
    store << item->name;
// FIXME    store << item->data;
    item = item->previous;
    i++;
    }
  }


// Load data
void FXHistoryItems::load(FXStream& store){ 
  FXBaseObject::load(store);
  store >> group;
  store >> maxitems;
  store >> key;
  store >> amount;
  store >> destroySave;
  store >> stringData;
  // FIXME: need to add stream load of struct
  }


// Destructor
FXHistoryItems::~FXHistoryItems(){
  if (destroySave) writeReg();
  clear();
  }

}

