/***********************************-*- mode: c++; tab-width: 2 -*-*\
 *
 * NAME:     
 *   FXFileDialogEx - File dialog using native dialog
 *
 * AUTHOR:
 *   Daniel Gehriger (gehriger@linkcad.com)
 *
 * Copyright (c) 1999 by Daniel Gehriger.  All Rights Reserved.
 *
 * PUPROSE:
 *   File dialog using native Common Controls dialog
 *
 * NOTE
 *
 * This library is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU Library General Public   
 * License as published by the Free Software Foundation; either  
 * version 2 of the License, or (at your option) any later version.
 *                                                                 
 * This library is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of  
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.                 
 *                                                                  
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free       
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 * $Id: FXFileDialogEx.cpp,v 1.6 1999/12/15 09:39:31 dgehrige Exp $
 *
 * HISTORY:
 *        dgehrige - Dec, 10 1999: Created.
 *
\*******************************************************************/
#include <config.h>

#if defined(WIN32) && defined (WIN32_COMDLG)
#include <fox/fxver.h>
#include <fox/xincs.h>
#include <fox/fxdefs.h>
#include <fox/FXStream.h>
#include <fox/FXString.h>
#include <fox/FXSize.h>
#include <fox/FXPoint.h>
#include <fox/FXRectangle.h>
#include <fox/FXRegistry.h>
#include <fox/FXApp.h>
using namespace FX;
#include "FXFileDialogEx.h"
using namespace FXEX;
namespace FXEX {

// maps
FXIMPLEMENT(FXFileDialogEx, FXObject, NULL, 0);

// de-serialisation
FXFileDialogEx::FXFileDialogEx() : FXObject() {}

// ctor
FXFileDialogEx::FXFileDialogEx(FXWindow* owner, const FXString& title, FXuint opts, FXint x, FXint y, FXint w, FXint h) : FXObject(),m_opts(opts),m_pszFoxPats(NULL),m_pszFoxCustomPat(NULL),m_nFoxPats(0) {
  /*
  ** Initialize OPENFILENAME
  */
  memset(&m_ofn, 0, sizeof(m_ofn));
  m_ofn.lStructSize = sizeof(OPENFILENAME);
  m_ofn.hwndOwner = (HWND)owner->id();

  m_ofn.lpstrFile = new FXchar[_MAX_PATH + 1];
  memset(m_ofn.lpstrFile, 0, _MAX_PATH + 1);
  m_ofn.nMaxFile = _MAX_PATH + 1;

  setTitle(title);
  const FXchar* pPat[] = { NULL };
  setPatternList(pPat); // Init with "All Files"

  if (opts & DLGEX_CREATEPROMPT) m_ofn.Flags |= OFN_CREATEPROMPT;
  if (opts & DLGEX_READONLY) m_ofn.Flags |= OFN_READONLY;
  if (opts & DLGEX_PATHMUSTEXIST) m_ofn.Flags |= OFN_PATHMUSTEXIST;
  if (opts & DLGEX_FILEMUSTEXIST) m_ofn.Flags |= OFN_FILEMUSTEXIST;
  if (opts & DLGEX_HIDEREADONLY) m_ofn.Flags |= OFN_HIDEREADONLY;
  if (opts & DLGEX_OVERWRITEPROMPT) m_ofn.Flags |= OFN_OVERWRITEPROMPT; 
  }

FXFileDialogEx::~FXFileDialogEx() {
  delete [] (FXchar*)m_ofn.lpstrInitialDir;
  delete [] (FXchar*)m_ofn.lpstrFilter;
  delete [] (FXchar*)m_ofn.lpstrCustomFilter;
  delete [] (FXchar*)m_ofn.lpstrFile;
  delete [] (FXchar*)m_ofn.lpstrTitle;
  for (FXint i = 0; i < m_nFoxPats; i++) {
    delete [] m_pszFoxPats[i];
    }
  delete [] m_pszFoxPats;
  delete [] m_pszFoxCustomPat;
  }

/*-----------------------------------------------------------------*\
 * 
 * Set / get the dialog title
 *
\*-----------------------------------------------------------------*/
void FXFileDialogEx::setTitle(const FXString& name) {
  delete [] (FXchar*)m_ofn.lpstrTitle;
  m_ofn.lpstrTitle = NULL;

  if (!name.empty()) {
    m_ofn.lpstrTitle = new FXchar[strlen(name.text()) + 1];
    strcpy((FXchar*)m_ofn.lpstrTitle, name.text());
    }
  }

FXString FXFileDialogEx::getTitle() const {
  return FXString(m_ofn.lpstrTitle);
  }

/*-----------------------------------------------------------------*\
 * 
 * Set / get the file name
 *
\*-----------------------------------------------------------------*/
void FXFileDialogEx::setFilename(const FXString& path) {
  if (path.size() > _MAX_PATH + 1) {
    fxerror("File path exceeds maxmimum path length: %d\n", path.size());
    return;
    }
  strcpy(m_ofn.lpstrFile, path.text()); 
  }

FXString FXFileDialogEx::getFilename() const {
  return FXString(m_ofn.lpstrFile);
  }

/*-----------------------------------------------------------------*\
 * 
 * Set the initial directory
 *
\*-----------------------------------------------------------------*/
void FXFileDialogEx::setDirectory(const FXString& path) {
  delete [] (FXchar*)m_ofn.lpstrInitialDir;
  m_ofn.lpstrInitialDir = NULL;

  if (!path.empty()) {
    m_ofn.lpstrInitialDir = new FXchar[strlen(path.text()) + 1];
    strcpy((FXchar*)m_ofn.lpstrInitialDir, path.text());
    }
  }

/*-----------------------------------------------------------------*\
 * 
 * Get the selected directory
 *
\*-----------------------------------------------------------------*/
FXString FXFileDialogEx::getDirectory() const {
  FXchar  pathBuf[_MAX_PATH];
  fxdirpart(pathBuf, m_ofn.lpstrFile);
  return FXString(pathBuf);
  }

/*-----------------------------------------------------------------*\
 * 
 * Select the file pattern
 *
\*-----------------------------------------------------------------*/
void FXFileDialogEx::setPattern(const FXString& ptrn) {
  FXint i, len;
  FXchar* pszWinPat;
  /*
  ** Pattern in the pattern list?
  */
  for (i = 0; i < m_nFoxPats; i++) {
    if (ptrn == m_pszFoxPats[i]) {
      m_ofn.nFilterIndex = i + 1;
      return;
      }
    }

  // translate pattern
  delete [] m_ofn.lpstrCustomFilter;  // delete old filter
  getWinPattern(pszWinPat, ptrn.text());
  len = FXMAX(ptrn.size() + 6 + strlen(pszWinPat) + 1 + 1, 40 /* magic value from MS doc*/);
  m_ofn.nMaxCustFilter = len;
  m_ofn.lpstrCustomFilter = new FXchar[len];
  memset(m_ofn.lpstrCustomFilter, 0, len);
  sprintf(m_ofn.lpstrCustomFilter, "%s Files", ptrn.text()); // not sure if we may use '\0' in sprintf
  strcpy(m_ofn.lpstrCustomFilter + ptrn.size() + 6, pszWinPat);
  m_ofn.nFilterIndex = 0;
  delete [] pszWinPat;
  
  // save pattern for getPattern();
  delete [] m_pszFoxCustomPat;
  m_pszFoxCustomPat = NULL;
  m_pszFoxCustomPat = new FXchar[ptrn.size()];
  strcpy(m_pszFoxCustomPat, ptrn.text());
  }

/*-----------------------------------------------------------------*\
 * 
 * Get the selected file extension
 *
\*-----------------------------------------------------------------*/
FXString FXFileDialogEx::getPattern() const {
  if (m_ofn.nFilterIndex == 0 && m_pszFoxCustomPat != NULL) {
    return FXString(m_pszFoxCustomPat);
    }
  else if (m_pszFoxPats != NULL && m_pszFoxPats[m_ofn.nFilterIndex - 1] != NULL) {
    return m_pszFoxPats[m_ofn.nFilterIndex -1];
    }
  FXASSERT(FALSE);
  return FXString::null;
  }

/*-----------------------------------------------------------------*\
 * 
 * Set the pattern list
 *
\*-----------------------------------------------------------------*/
void FXFileDialogEx::setPatternList(const FXchar* ptrns[]) {
  FXint i = 0, len = 0, nPat = 0;
  FXbool bHasAll = FALSE;
  
  /*
  ** Remove old items
  */
  for (i = 0; i < m_nFoxPats; i++) {
    delete [] m_pszFoxPats[i];
    }
  delete [] m_pszFoxPats;
  delete [] (FXchar*)m_ofn.lpstrFilter; 
  m_ofn.lpstrFilter = NULL;

  /*
  ** Count patterns && and allocate memory
  */
  for (nPat = 0; ptrns && ptrns[2*nPat] != NULL; nPat++) {
    if (strcmp(ptrns[2*nPat+1], "*") == 0 || strcmp(ptrns[2*nPat+1], "*.*") == 0) bHasAll=TRUE;
    }
  if (!bHasAll) nPat++;
  m_nFoxPats = nPat;
  m_pszFoxPats = new FXchar*[nPat];
  FXchar const** pszWinDesc = new FXchar const*[nPat];
  FXchar** pszWinPats = new FXchar*[nPat];

  /*
  ** Copy translated patterns
  */
  for (i = 0; ptrns && ptrns[2*i] != NULL; i++) {
    pszWinDesc[i] = ptrns[2*i];
    getWinPattern(pszWinPats[i], ptrns[2*i+1]);
    len += strlen(pszWinDesc[i])+1 + strlen(pszWinPats[i])+1;
    
    // save fox pattern
    m_pszFoxPats[i] = new FXchar[strlen(ptrns[2*i+1])+1];
    strcpy(m_pszFoxPats[i], ptrns[2*i+1]); 
    }

  if (!bHasAll) {
    pszWinDesc[i] = "All Files";
    pszWinPats[i] = new FXchar[2];
    strcpy(pszWinPats[i], "*");
    len += strlen(pszWinDesc[i])+ 1 + strlen(pszWinPats[i]) + 1;

    // save fox pattern
    m_pszFoxPats[i] = new FXchar[strlen(pszWinPats[i])+1];
    strcpy(m_pszFoxPats[i], pszWinPats[i]); 
    }

  m_ofn.lpstrFilter = new FXchar[len + 1 /* terminate with "\0\0" */];
  for (i = 0, len = 0; i < nPat; i++) {
    strcpy((FXchar*)m_ofn.lpstrFilter + len, pszWinDesc[i]);
    len += strlen(pszWinDesc[i]) + 1;
    strcpy((FXchar*)m_ofn.lpstrFilter + len, pszWinPats[i]);
    len += strlen(pszWinPats[i]) + 1;
    }
  (FXchar)m_ofn.lpstrFilter[len] = '\0';

  /*
  ** A lot of stuff to delete...
  */
  for (i = 0; i < nPat; i++) {
    delete [] pszWinPats[i];
    }
  delete [] pszWinPats;
  delete [] pszWinDesc;
  }

/*-----------------------------------------------------------------*\
 * 
 * Get index of selected pattern in the list (zero based)
 *
\*-----------------------------------------------------------------*/
FXint FXFileDialogEx::getCurrentPattern() const {
  return m_ofn.nFilterIndex - 1;
  }

/*-----------------------------------------------------------------*\
 * 
 * Select the inital pattern from the list (zero based)
 *
\*-----------------------------------------------------------------*/
void FXFileDialogEx::setCurrentPattern(FXint n) {
  m_ofn.nFilterIndex = n + 1;
  }

/*-----------------------------------------------------------------*\
 * 
 * Get the selected pattern in the list 
 *
\*-----------------------------------------------------------------*/
FXString FXFileDialogEx::getPatternText(FXint patno) const {
  const FXchar* pFilter;
  FXint i;

  // skip over patterns
  for (pFilter = m_ofn.lpstrFilter, i = 0; i != patno && pFilter && *pFilter; i++) {
    pFilter += strlen(pFilter) + 1;
    pFilter += strlen(pFilter) + 1;
    }

  if (pFilter && *pFilter) return FXString(pFilter);
  return FXString::null;
  }

/*-----------------------------------------------------------------*\
 * 
 * Modify pattern list 
 *
\*-----------------------------------------------------------------*/
void FXFileDialogEx::setPatternText(FXint patno, const FXString& text) {
  FXchar* pNewFilter;
  const FXchar* pOldFilter;
  FXchar* pOldList;
  FXint i, len = 0;

  FXASSERT(m_ofn.lpstrFilter != NULL);

  // calculate new size
  pOldFilter = m_ofn.lpstrFilter;
  for (len = 0, i = 0; pOldFilter && *pOldFilter; i++) {
    if (i != patno) len += strlen(pOldFilter) + 1;
    else len += text.size();
    pOldFilter += strlen(pOldFilter) + 1;
    len += strlen(pOldFilter) + 1;
    pOldFilter += strlen(pOldFilter) + 1;
    }
  
  // change pattern list
  pOldList = (FXchar*)m_ofn.lpstrFilter;
  m_ofn.lpstrFilter = new FXchar[len + 1];
  
  pNewFilter = (FXchar*)m_ofn.lpstrFilter;
  pOldFilter = pOldList;
  for (i = 0; pOldFilter && *pOldFilter; i++) {
    if (i != patno) {
      strcpy(pNewFilter, pOldFilter);
      pNewFilter += strlen(pNewFilter) + 1;
      }
    else {
      strcpy(pNewFilter, text.text());
      pNewFilter += text.size();
      }
    
    pOldFilter += strlen(pOldFilter) + 1;
    strcpy(pNewFilter, pOldFilter);
    pNewFilter += strlen(pNewFilter) + 1;
    pOldFilter += strlen(pOldFilter) + 1;
    }
  
  *pNewFilter = '\0';
  delete [] pOldList;
  }

/*-----------------------------------------------------------------*\
 * 
 * Display the file dialog
 *
\*-----------------------------------------------------------------*/
FXuint FXFileDialogEx::execute() {
  if (m_opts & DLGEX_SAVE) return ::GetSaveFileName(&m_ofn);
  else return ::GetOpenFileName(&m_ofn);
  }

/*-----------------------------------------------------------------*\
 * 
 * Translate a fox pattern
 *
\*-----------------------------------------------------------------*/
void FXFileDialogEx::getWinPattern(FXchar*& pszWinPat, const FXchar* pszFoxPat) const {
  FXint   nPerms = 1, nLen = 0, nSubPerms;
  FXint   i, j, k;
  FXchar* pszRedFoxPat;
  FXchar* pTgt = NULL;
  const FXchar* pSrc = NULL;
  const FXchar* pMark = NULL;
  FXbool  bIgnore = FALSE;
  pszRedFoxPat = new FXchar[strlen(pszFoxPat) + 1];
  memset(pszRedFoxPat, 0, strlen(pszFoxPat) + 1);

  /*
  ** Calculate number of permutations and reduce source pattern
  */
  pMark = NULL;
  for (pSrc = pszFoxPat, pTgt = pszRedFoxPat; *pSrc != '\0'; pSrc++) {

    if (pMark == NULL && *pSrc == '[' && *(pSrc+1) != '\0') {
      // start of character range
      if (*(pSrc+1) == '!' || *(pSrc+1) == '^') {
        bIgnore = TRUE; // ignore negations
        *(pTgt++) = '?';
        nLen++;
        FXASSERT(FALSE);
        }
      else {
        *(pTgt++) = '[';
        }
      pMark = pTgt;
      }

    else if (pMark != NULL && *pSrc != ']') {
      // only copy unique characters (ignoring case)
      if (!bIgnore && strchr(pMark, tolower(*pSrc)) == 0) *(pTgt++) = tolower(*pSrc);
      }

    else if (pMark != NULL && *pSrc == ']') {
      // end of character range
      if (bIgnore) bIgnore = FALSE;
      else if (pTgt - pMark == 0) *(--pTgt) = '\0'; // ignore []
      else if (pTgt - pMark == 1) {
        *(pTgt-2) = *(pTgt-1);
        *(--pTgt) = '\0'; // single character
        nLen ++;
        }
      else {
        nPerms *= (pTgt - pMark); // calculate permutations
        nLen ++;
        *(pTgt++) = ']';
        }
      pMark = NULL;
      }

    else {
      *(pTgt++) = tolower(*pSrc);
      nLen++;
      }
    }

  /*
  ** At this point we have the number of permutations in nPerms, 
  ** the length of each pattern in nLen,
  ** and pszRedFoxPat contains a reduced version of the original pattern
  ** (all lowercase, removed duplicates & negated ranges)
  */
  pszWinPat = new FXchar[nPerms * (nLen+1)];    // nLen + 1 because of 
                                                // separator (';') or terminating null
  /*
  ** Generate Windoze pattern
  */
  nSubPerms = nPerms;
  pMark = NULL;
  for (pSrc = pszRedFoxPat, pTgt = pszWinPat; *pSrc != '\0'; pSrc++) {

    if (pMark == NULL && *pSrc == '[' && *(pSrc+1) != '\0') pMark = pSrc + 1;

    else if (pMark != NULL && *pSrc == ']') {
      nSubPerms /= (pSrc - pMark);                // remaining permutations
      for (i = 0; i < nPerms / (nSubPerms * (pSrc - pMark)); i++) {
        for (j = 0; j < (pSrc - pMark); j++) {
          // copy as many times as there are remaining perm's
          for (k = 0; k < nSubPerms; k++) {
            pTgt[(i*(nSubPerms * (pSrc - pMark))+j*nSubPerms+k) * (nLen+1)] = pMark[j];
            }
          }
        }
      pTgt++;
      pMark = NULL;
      }

    else if (pMark == NULL) {
      for (i = 0; i < nPerms; i++) {
        pTgt[i *(nLen + 1)] = *pSrc;
        }
      pTgt++;
      }
    }

  /*
  ** That's it. Insert the separators and we're done.
  */
  for (i = 0; i < nPerms - 1; i++) {
    pTgt[i *(nLen + 1)] = ';';
    }
  pTgt[i *(nLen + 1)] = '\0';
  delete [] pszRedFoxPat;
  }

}
#endif /* WIN32_COMDLG */

