/********************************************************************************
*                                                                               *
*                  Binary log file object                                       *
*                                                                               *
*********************************************************************************
* Copyright (C) 2003 by Mathew Robertson.   All Rights Reserved.                *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
*********************************************************************************/
#include <config.h>
#include <fox/fxver.h>
#include <fox/xincs.h>
#include <fox/fxdefs.h>
#include <fox/FXStream.h>
#include <fox/FXString.h>
#include <fox/FXSize.h>
#include <fox/FXPoint.h>
#include <fox/FXRectangle.h>
#include <fox/FXRegistry.h>
#include <fox/FXApp.h>
#include <fox/FXFont.h>
#include <fox/FXComposite.h>
using namespace FX;
#include "fxexdefs.h"
#include "fxexutils.h"
#include "FXDateTime.h"
#include "FXFileIO.h"
#include "FXBinaryLogger.h"
#include "FXBinaryLogReader.h"
using namespace FXEX;
namespace FXEX {

/*
 * Notes:
 */

#define ENDIAN_MASK  (0x80)

// Swap duplets - shamelessly taken from FOX
static inline void swap2(void *p){
  register FXuchar t;
  t=((FXuchar*)p)[0]; ((FXuchar*)p)[0]=((FXuchar*)p)[1]; ((FXuchar*)p)[1]=t;
  }

// Swap quadruplets - shamelessly taken from FOX
static inline void swap4(FXuchar *p){
  register FXuchar t;
  t=p[0]; p[0]=p[3]; p[3]=t;
  t=p[1]; p[1]=p[2]; p[2]=t;
  }

// ctor
FXBinaryLogReader::FXBinaryLogReader(const FXString& file,FXuint opts) {
  options=0;
  if (opts&BINARYLOGREADER_TRUNCATE) options=FILEIO_TRUNCATE;
  fileio=new FXFileIO(FXApp::instance(),file,NULL,0,options);
  options=opts;
  if (file.length()) open();
  }

// dtor
FXBinaryLogReader::~FXBinaryLogReader(){
  close();
  delete fileio;
  }

// set the logfile filename
FXbool FXBinaryLogReader::name(const FXString& file){
  close();
  fileio->name(file);
  fileio->mode(FXUtils::fxfilemode(FILEPERM_DEFAULT_IO));
  return open();
  }

// get the logfile filename
FXString FXBinaryLogReader::name() {
  return fileio->name();
  }

// return indication if file is open
FXbool FXBinaryLogReader::opened() {
  return fileio->opened();
  }

// open the logfile - if we can
FXbool FXBinaryLogReader::open(){
  if (!fileio->opened()){
    fileio->create();
    if (!fileio->opened()) return FALSE;
    FXuchar data[2]={0};
    if (fileio->read(data,2) < 2) {
      fxwarning("FXBinaryLogReader: short read from file\n");
      return FALSE;
      }
    switch (data[0]) {
      case 1: { // version 1 binary logfile
        options|=data[1];
        } break;
      default: {
        fxwarning("FXBinaryLogReader: unknown binary logfile format\n");
	return FALSE;
        }
      }
    if(options&BINARYLOGGER_SUBSECOND) logentry.subsecond=TRUE;
    else logentry.microseconds=0;
    }
  return TRUE;
  }

// close the logfile
void FXBinaryLogReader::close(){
  if (fileio->opened()){
    fileio->destroy();
    }
  }

// read entries from the file
FXBinaryLogData* FXBinaryLogReader::read() {
  if (!fileio->opened()) return NULL;
  if(options&BINARYLOGGER_SUBSECOND){
    FXuchar data[12];
    if (fileio->read(data,12)<12) return NULL;
    if ((options&ENDIAN_MASK) != FXStream::isLittleEndian()) {
      swap4(&data[0]);
      swap4(&data[4]);
      swap2(&data[8]);
      swap2(&data[10]);
      }
    memcpy(&logentry.seconds,&data[0],4);
    memcpy(&logentry.microseconds,&data[4],4);
    FXshort tmp;
    memcpy(&tmp,&data[8],2);
    logentry.code=tmp;
    memcpy(&tmp,&data[10],2);
    logentry.value=tmp;
    }
  else {
    FXuchar data[8];
    if(fileio->read(data,8)<8) return NULL;
fxmessage("options&ENDIAN_MASK:  %i\n",options&ENDIAN_MASK);
    if ((options&ENDIAN_MASK) != FXStream::isLittleEndian()) {
fxmessage("Swapping bytes\n");
//      swap4(&data[0]);
//      swap2(&data[4]);
//      swap2(&data[6]);
      }
    memcpy(&logentry.seconds,&data[0],4);
    FXshort tmp;
    memcpy(&tmp,&data[4],2);
    logentry.code=tmp;
    memcpy(&tmp,&data[6],2);
    logentry.value=tmp;
    }
  return &logentry;
  }
    

// save to stream
FXStream& operator<<(FXStream& store,const FXBinaryLogReader& b){
  store << b.fileio;
  return store;
  }

// load from stream
FXStream& operator>>(FXStream& store,FXBinaryLogReader& b){
  store >> b.fileio;
  return store;
  }

}

