/********************************************************************************
*                                                                               *
*                   Base type for thread objects                                *
*                                                                               *
*********************************************************************************
* Copyright (C) 2003 by Mathew Robertson.   All Rights Reserved.                *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
*********************************************************************************/
#ifndef FXTHREAD_H
#define FXTHREAD_H

#ifndef FXRUNNABLE_H
#include "FXRunnable.h"
#endif
namespace FXEX {
class FXAtomicInt;

/**
 * Thread base class for worker thread objects.
 *
 * The idea here is that you derive from this class, to create your wroker thread.
 * Facilities are provided to:
 * a) start the thread, from another thread
 * b) request a stop from another thread.
 *
 * Note, the stop functionality only provides a facility to 'request' a stop of a thread.
 * There isn't an implementation that I can think of (in a working program) where forcibly stopping
 * a thread is a good thing...  As such, it is up to the worker thread to periodically
 * poll the status of the 'stopThread' variable, to find out if another thread (ie the main
 * thread) has asked the worker thread to stop.  This is reasonably important, since the
 * owner thread will wait for the worker thread to stop before it will completely destruct.
 *
 * Your derived class only need to implement the 'run()' method.  This method is started in
 * a worker thread when you call start() on the object.  If you need to pass parameters to
 * the worker thread, just create set/get methods in your derived class (which then sets/gets
 * the parameter that you want access to).
 *
 * To facilitate simple access to class local variables, the FXThread class inherits a
 * mutex (call 'mutex') from the FXRunnable class.  Your accessor methods then simply needs
 * to either:
 * a) call lock()/unlock() on the mutex
 * b) use the FXSynchronise object to auto-lock/unlock the mutex
 *
 * Alternatively you could use the FXAtomic data types, which provide implicit lock/unlock
 * calls for each of its accessor methods.  This technique is used internally within the
 * FXAtomicDataTarget class to create a FOX DataTarget which works across threads.
 */
class FXAPI FXThread : public FXRunnable {
  FXDECLARE_ABSTRACT(FXThread)

private:
  FXThread(const FXThread&);
  FXThread& operator=(const FXThread&);

private:
  FXThreadHandle thread;     // the thread

protected:
  FXAtomicInt *stopThread;

public:
  enum {
    ID_STOP=FXRunnable::ID_LAST,
    ID_LAST
    };

public:
  long onStop(FXObject*,FXSelector,void*);
  long onCmdGetIntValue(FXObject*,FXSelector,void*);
  long onCmdSetIntValue(FXObject*,FXSelector,void*);

public:
  /// You dont normally need to call these, they are used internally but must be public

  /// Reset the thread object, so that it can be re-used
  void reset();

  /// you are meant to overload this in your child thread classes...
  virtual void run()=0;

protected:
  /// This is the ctor you need when you sub-class FXThread
  FXThread(FXObject *tgt=NULL,FXSelector sel=0);

  /// yield the thread - allows other threads/processes to access the CPU
  void yield();

public:
  /// start running the object
  void start();

  /// request to stop the thread - the thread can ignore this
  void stop();

  /// is thread curently running?
  FXbool isRunning();

  /// dtor
  virtual ~FXThread();
  };

}; // namespace FXEX
#endif // FXTHREAD_H
 
