/********************************************************************************
*                                                                               *
*                  Hostname resolver object                                     *
*                                                                               *
*********************************************************************************
* Copyright (C) 2003 by Mathew Robertson.   All Rights Reserved.                *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
*********************************************************************************/
#ifndef FXRESOLVER_H
#define FXRESOLVER_H

#ifndef FXTHREAD_H
#include "FXThread.h"
using namespace FX;
#endif
namespace FXEX {
class FXFastMutex;

/**
 * FXResolver is designed to resolve hostnames synchronously or asychronously, so that the
 * application can choose whether it wants to wait for resolution to finish, before returning
 * or whether to wait for the SEL_COMMAND to be sent to indicate if the resolution has finished.
 *
 * The resolver defaults to resolving IPv4 addresses, but can be made to lookup IPv6 addresses.
 * The default socket type used in the lookup, is any type (ie stream or datagram).
 *
 * The class implements a non-threaded lookup of hostnames which correspond to the current
 * machine.  This is to reduce the overhead with the lookup, since a blocking lookup should be
 * quick for most machines that have some form of network stack.
 */
class FXAPI FXResolver : public FXThread {
  FXDECLARE (FXResolver)

  protected:
    FXString        hostname;               // host to lookup
    FXString        officialname;           // canonomical name of resolved host
    FXSocketFamily  family;                 // prefered socket family - defaults to any
    FXSocketType    type;                   // prefered socket type - defaults to any
    FXint           code;                   // status code of lookup
    void           *sock_addr;              // socket address structure
    FXFastMutex    *fastmutex;              // resource lock

  protected:
    /// serialisation
    FXResolver();

    /// worker thread implementation
    void run();

  public:
    enum {
      ID_RESOLVER=FXThread::ID_LAST,
      ID_LAST
      };

  public:
    long onCmdSetValue(FXObject*,FXSelector,void*);
    long onCmdSetStringValue(FXObject*,FXSelector,void*);
    long onCmdGetStringValue(FXObject*,FXSelector,void*);

  public:
    /// Just supply the target and selector (de-serialisation too)
    FXResolver(const FXString& host,FXObject *tgt=NULL,FXSelector sel=0);

    /// get status last hostname lookup
    FXint status();

    /// indicates if we have resolved the hostname
    FXbool resolved();

    /// set resolver to new host name
    void host(const FXString& h);

    /// get currently resolving host name
    FXString host() const { return hostname; }

    /// get the official/canonomical hostname of the resolved name
    FXString officialName();

    /**
     * Tell resolver to resolve the host info
     * - handle a SEL_COMMAND event, ptr contains a pointer to the
     *   array of returned addrinfo structures
     */
    void resolve();

    /// resolve using a blocking implementation
    FXbool blockingResolve();

    /**
     * return reference to last looked up resolved hostname
     * as a socket address struct (ie you need to cast it)
     */
    void* socket_address_struct();

    /**
     * Set the family of the conection type
     * ie you can set it to one of FXSocketFamilyInet6 or FXSocketFamilyInet
     */
    void setFamily(FXSocketFamily family);

    /// return the prefered socket family
    FXSocketFamily getFamily();

    /// set the prefered socket type
    void setType(FXSocketType type);

    /// get the currently prefered socket type
    FXSocketType getType();

    /// dtor
    virtual ~FXResolver();
  };

} // namespace FXEX
#endif // FXRESOLVER_H

