/********************************************************************************
*                                                                               *
*                  Generic I/O object                                           *
*                                                                               *
*********************************************************************************
* Copyright (C) 2003 by Mathew Robertson.   All Rights Reserved.                *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
********************************************************************************/
#ifndef FXIOHANDLE_H
#define FXIOHANDLE_H

#ifndef FXBASEOBJECT_H
#include "FXBaseObject.h"
#endif
namespace FXEX {

/**
 * FXIOhandle's move data around using this data structure
 */
class FXAPI FXIOData {
public:
  FXInputHandle  iohandle;  // the IO handle
  FXuchar       *data;      // raw data
  FXuint         no;        // the amount of data
  FXbool         oob;       // the data is Out-Of-Band data (not valid/used for some IO types)

public:
  /// save structure to stream
  friend FXAPI FXStream& operator<< (FXStream& store,const FXIOData& d);

  /// load structure from stream
  friend FXAPI FXStream& operator>> (FXStream& store,FXIOData& d);

public:
  /// Construct from existing IO handle
  FXIOData(FXInputHandle h=INVALID_HANDLE,FXbool outOfBand=FALSE);

  /// Construct to contain data
  FXIOData(FXuchar *d,FXuint n);

  /// dtor
  virtual ~FXIOData();
  };


/**
 * This is the base class of all IO capabilities (sockets/files/serial-port).  It provides
 * generic IO handling capabilities.
 *
 * Notes:
 * - the write() and read() methods are in the public interface specifically to be flexible
 *   to the end programmer, though they shouldn't really need to use them directly.
 */
class FXAPI FXIOHandle : public FXBaseObject {
  FXDECLARE (FXIOHandle)

  protected:
    FXInputHandle    iohandle;          // ... the IO handle ...
    FXint            code;              // status of object
    FXIOState        state;             // indicates the state of the IO handle
    FXint            readBufSize;       // the size of the read buffer
    FXIOData         writeBuffer;       // data waiting to be written
    FXIOData        *readBuffer;        // data just been read
    FXint            activityPeriod;    // the period between handle activity events

  protected:
    /// helper; to write data
    FXbool writeAgain();

    /// helper; to add this IO handle to the FXApp watch list
    FXbool addInput();

  public:
    enum {
      ID_IOHANDLE=FXBaseObject::ID_LAST,
      ID_WRITE,
      ID_WRITE_OOB,
      ID_LAST
      };

  public:
    long onRead(FXObject*,FXSelector,void*);
    long onWrite(FXObject*,FXSelector,void*);
    long onExcept(FXObject*,FXSelector,void*);
    long onActivityTimeout(FXObject*,FXSelector,void*);
    long onIORead(FXObject*,FXSelector,void*);
    long onIOWrite(FXObject*,FXSelector,void*);
    long onIOExcept(FXObject*,FXSelector,void*);
    long onIOConnect(FXObject*,FXSelector,void*);
    long onCmdWrite(FXObject*,FXSelector,void*);
    long onCmdGetIntValue(FXObject*,FXSelector,void*);
    long onOpened(FXObject*,FXSelector,void*);
    long onClose(FXObject*,FXSelector,void*);
    long onClosed(FXObject*,FXSelector,void*);
    long onDestroy(FXObject*,FXSelector,void*);
    long onData(FXObject*,FXSelector,void*);

  protected:
    /// for de-serialisation
    FXIOHandle();

    /// close the underlying operating system handle
    virtual void closehandle();

    /// instanciates this object
    virtual FXIOHandle* newInstance(FXInputHandle h);

    /** Create an IO handle - child classes should overload the
     *  open(), close(), errorClose(), read() and write()
     *  methods to implement appropriate behaviour
     */
    FXIOHandle(FXApp *a,FXObject *tgt=NULL,FXSelector sel=0);

  public:
    /// IO Handle equality is just the test to see if the OS handles are the same
    friend FXAPI FXbool operator==(const FXIOHandle& h1, const FXIOHandle& h2);
    friend FXAPI FXbool operator!=(const FXIOHandle& h1, const FXIOHandle& h2);

  public:
    /// Use an existing IO handle, adding it to the FXApp event loop
    FXIOHandle(FXInputHandle h,FXApp *a,FXObject *tgt=NULL,FXSelector sel=0);

    /// create resources
    virtual void create();

    /// detach resource
    virtual void detach();

    /// destroy resource
    virtual void destroy();

    /// get the status of the last action (ie returns errno or something similar)
    FXint status() { return code; }

    /// reset the status value
    void resetStatus() { code=FXIOStatusOk; }

    /// get the state of the socket
    FXIOState getState() { return state; }

    /// simple indication if handle is open
    FXbool opened();

    /// simple indication if handle is closed
    FXbool closed();

    /// return the file descriptor / event handle
    virtual FXInputHandle getHandle() { return iohandle; }

    /// set the size of the minimum read chunk
    /// ie use this to optimise the number of reads vs. memory used
    void setBuf(FXint size) { readBufSize=size; }

    /// get the non-blocking state of the connection
    FXbool nonBlocking();

    /// set the connection to the non-blocking state
    FXbool nonBlocking(FXbool nonBlock);

    /// Open-up/create connection
    virtual FXbool open();

    /// Close down the connection
    virtual void close();

    /// Close down the connection, sending the child/target a SEL_DESTROY message
    /// This is used internally when there are errors on the IO handle
    virtual void errorClose();

    /**
     * Get some data from the IO handle
     * derived classes should implement IO specific read()
     */
    virtual FXint read(FXuchar *data,FXint len,FXbool outOfBand=FALSE);

    /**
     * Write some data over the IO handle - can write out of band data if required
     * derived classes should implement IO specific write()
     */
    virtual FXint write(FXuchar *data,FXint n,FXbool outOfBand=FALSE);

    /// read the IO handle and generate FOX events
    FXbool readData(FXbool outOfBand=FALSE);

    /**
     * Write the data out of the IO handle.
     * May not send all the data at once but call always completes without blocking
     * re-schedules any remaing data to send later
     */
    FXbool writeData(FXIOData *d);

    /**
     * Returns the data just received, as a text string
     * Note: this call is _only_ valid 'during the handle' of the SEL_COMMAND event
     *       by the child/target, at all other times this returns FXString::null
     */
    FXString getText() const;

    /// writes the string to the IO handle
    void setText(const FXString& message);

    /**
     * Set an activity timer.  Each time data is read-from / written-to the handle,
     * the timer is reset.  If the timer expires, no activity has occurred.  A value
     * of zero, disables the timer.
     */
    void setActivityTimeout(FXint ms=0);

    /**
     * Duplicate this handle onto another handle, defaults to OS assigned value 
     * ie the user can ask for the duplicate to be mapped onto a specific IO value
     */
    virtual FXIOHandle* duplicate(FXInputHandle newHandle=INVALID_HANDLE);

    /// save object to stream
    virtual void save(FXStream& store) const;

    /// load object from stream
    virtual void load(FXStream& store);

    /// dtor
    virtual ~FXIOHandle();
  };

} // namespace FXEX
#endif // FXIOHANDLE_H
