/********************************************************************************
*                                                                               *
*                  Monitor the status of a file                                 *
*                                                                               *
*********************************************************************************
* Copyright (C) 2003 by Mathew Robertson.   All Rights Reserved.                *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
********************************************************************************/
#ifndef FXFILEMONITOR_H
#define FXFILEMONITOR_H

#ifndef FXBASEOBJECT_H
#include "FXBaseObject.h"
#endif
namespace FXEX {

/**
 * Monitors the status of a file, whether it exists, and its size as it grows.
 * Can optionally specify a timeout for when a file stops growing.
 */
class FXAPI FXFileMonitor : public FXBaseObject {
  FXDECLARE (FXFileMonitor)

  private:
    FXString file;
    FXint growtime;
    FXint endtimeout;
    unsigned long size;
    FXTime modified;

  protected:
    FXFileMonitor(){}

  public:
    enum {
      ID_FILE_CREATE=FXBaseObject::ID_LAST,
      ID_FILE_CHANGED,
      ID_FILE_END,
      ID_LAST
      };

  public:
    long onFileCreate(FXObject*,FXSelector,void*);
    long onFileChange(FXObject*,FXSelector,void*);
    long onFileEnd(FXObject*,FXSelector,void*);
    long onFileCreated(FXObject*,FXSelector,void*);
    long onFileChanged(FXObject*,FXSelector,void*);
    long onFileDeleted(FXObject*,FXSelector,void*);
    long onFileClosed(FXObject*,FXSelector,void*);

  public:
    /**
     * Monitor a file:
     * -> ms   frequency of scanning to see if the file exists or is growing
     * -> to   once the file starts growing, an event is generated if the file stops
     *         growing for at least this timeout. (in secs) Can be <= 0 to stop this
     *         begin generated (ie we detect changes only)
     * Note: If the file grows after the timeout event, further change events will be generated.
     */
    FXFileMonitor(FXApp *a,const FXString& file,FXint ms=2000,FXint to=0,FXObject *tgt=NULL,FXSelector sel=0);

    /// create resource
    virtual void create();

    /// set monitor to new file
    void filename(const FXString& file);

    /// get filename being monitored
    FXString filename() { return file; }

    /// set the scantime
    void setScanTime(FXint ms);

    /// get the scantime
    FXint getScanTime() const { return growtime; }

    /// set the timeout (in secs)
    void setEndTimeout(FXint to);

    /// get the timeout
    FXint getEndTimeout() const { return endtimeout; }

    /// reset the monitor - this causes the events to be regenerated
    void reset();

    /// save object to stream
    virtual void save(FXStream& store) const;

    /// load object from stream
    virtual void load(FXStream& store);

    /// dtor
    virtual ~FXFileMonitor();
  };

} // namespace FXEX
#endif // FXFILEMONITOR_H
