/********************************************************************************
*                                                                               *
*               Dynamically linked library loader (DLL loader)                  *
*                                                                               *
*********************************************************************************
* Copyright (C) 2003 by Mathew Robertson.   All Rights Reserved.                *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
********************************************************************************/
#ifndef FXDLL_H
#define FXDLL_H

namespace FXEX {

/**
 * Dynamically linked library loader which loads the library, based on the filename.
 * Includes a macro so as to make dynamic loading of classes simpler than would be otherwise.
 *
 * Notes:
 * When dynamically loading a dll, most OS's automatically load a default library function if
 * it it exists (DllMain on Win32, _init on Linux).  The problem is two-fold in that
 * a) they are different names for different platforms, b) they are only called once, even
 * if they opened a second time, for a second thread (this is a particularly nasty problem
 * since the second thread wont initialise and thread specific variables).
 *
 * To combat this, I have written the class so that if your library contains a C function
 * FXDLL_load, it will be called the first time a library is loaded (for a given FXDLL object),
 * thus solving the platform problem.  Also, if you load it from a second thread (and thus from
 * a second FXDLL objects), it calls FXDLL_load in that thread too (the OS specific auto-exec
 * functions dont do this), thus correctly initialising thread specific variables.
 *
 * Conversly, when you unload the library, if you have a library function called FXDLL_unload,
 * it is called before unloading the library (for each FXDLL instance).
 */
class FXAPI FXDLL {
protected:
  FXString file;
  FXString loadedFile;
  FXDLLHandle handle;
  FXuint refcount;

protected:
  virtual FXDLLHandle loadLibrary(const FXString& filename);
  virtual FXDLLHandle unloadLibrary();

public:
  /// create a dll object, using specified filename
  FXDLL(const FXString& filename="");

  /// set to new filename - fails silently if library is already loaded.
  void setFilename(const FXString& filename);

  /// get the filename
  FXString getFilename() const { return file; }

  /// get the name of the file actually loaded
  FXString getLoadedFilename() const { return loadedFile; }
  
  /// open library file
  FXbool open();

  /// close library
  void close();

  /// indicates if the library is already open
  FXbool isOpen();

  /// get handle
  FXDLLHandle getHandle() { return handle; }

  /**
  * return a pointer to a function
  * you will need to do something like this:
  *
  *   double (*calc)(double a,double b);
  *   calc = dll.getFunction("power");
  *   result = (*calc)(val1,val2);
  */
  virtual void* getFunction(const FXString& function);

  /// close library
  virtual ~FXDLL();
  };


/**
* A helper routine for implementing dynamic method loading within a class.
* This implementation uses the method parameters: (FXint,void*) or ()
*
* eg:
* FXDLL_METHOD(FXint,libraryMethod,(void *data,FXint size),const);
* or:
* FXDLL_METHOD(FXString,libraryMethod,(),);
*/
#define FXDLL_METHOD(returntype,funcname,params,constness) \
  typedef returntype (CALLBACK* functype_##funcname) params constness; \
  functype_##funcname dll_##funcname=NULL; \
  FXbool isLoaded_##funcname=FALSE;\
  returntype funcname params { \
    if (!getHandle()) return (returntype)NULL; \
    if (!isLoaded_##funcname) { \
      dll_##funcname= (functype_##funcname)getFunction(#funcname); \
      if (dll_##funcname == NULL) return (returntype)NULL; \
      isLoaded_##funcname = TRUE;\
      } \
    return (dll_##funcname)(params); \
    }


/**
* A helper routine for implementing dynamic function loading.
* This implementation uses the function parameters : (FXint,void*)  or ()
*
* eg:
* FXDLL_FUNCTION(dllp,FXint,libraryFunction,(void *data,FXint size),const);
* or:
* FXDLL_FUNCTION(pointer_to_FXDLL,FXbool,libraryFunction,(FXString s),);
*/
#define FXDLL_FUNCTION(fxdllpointer,returntype,funcname,params,constness) \
  typedef returntype (CALLBACK* functype_##funcname) params constness; \
  functype_##funcname dll_##funcname=NULL; \
  FXbool isLoaded_##funcname=FALSE;\
  returntype funcname params { \
    if (!fxdllpointer->getHandle()) return (returntype)NULL; \
    if (!isLoaded_##funcname) { \
      dll_##funcname= (functype_##funcname)fxdllpointer->getFunction(#funcname); \
      if (dll_##funcname == NULL) return (returntype)NULL; \
      isLoaded_##funcname = TRUE;\
      } \
    return (dll_##funcname)(params); \
    }

} // namespace FXEX
#endif // FXDLL_H

