/********************************************************************************
*                                                                               *
*                         T r e e   L i s t   W i d g e t                       *
*                                                                               *
*********************************************************************************
* Copyright (C) 1997,2002 by Jeroen van der Zijp.   All Rights Reserved.        *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
********************************************************************************/
#ifndef FXCHECKTREELIST_H
#define FXCHECKTREELIST_H

#ifndef FXSCROLLAREA_H
#include <fox/FXScrollArea.h>
using namespace FX;
#endif
namespace FXEX {

/// CheckTreeList styles
enum {
  CHECKTREELIST_EXTENDEDSELECT= 0,          /// Extended selection mode allows for drag-selection of ranges of items
  CHECKTREELIST_SINGLESELECT  = 0x00100000, /// Single selection mode allows up to one item to be selected
  CHECKTREELIST_BROWSESELECT  = 0x00200000, /// Browse selection mode enforces one single item to be selected at all times
  CHECKTREELIST_MULTIPLESELECT= 0x00300000, /// Multiple selection mode is used for selection of individual items
  CHECKTREELIST_AUTOSELECT    = 0x00400000, /// Automatically select under cursor
  CHECKTREELIST_SHOWS_LINES   = 0x00800000, /// Lines shown
  CHECKTREELIST_SHOWS_BOXES   = 0x01000000, /// Boxes to expand shown
  CHECKTREELIST_ROOT_BOXES    = 0x02000000, /// Display root boxes also
  CHECKTREELIST_NORMAL        = CHECKTREELIST_EXTENDEDSELECT
  };


/// Tree list Item
class FXAPI FXCheckTreeItem : public FXObject {
  FXDECLARE(FXCheckTreeItem)
  friend class FXCheckTreeList;
  friend class FXDirList;

protected:
  FXCheckTreeItem *prev;
  FXCheckTreeItem *next;
  FXCheckTreeItem *parent;
  FXCheckTreeItem *first;
  FXCheckTreeItem *last;
  FXString    label;
  FXIcon     *openIcon;
  FXIcon     *closedIcon;
  void       *data;
  FXuint      state;
  FXint       x,y;

protected:
  FXCheckTreeItem():prev(NULL),next(NULL),parent(NULL),first(NULL),last(NULL),openIcon(NULL),closedIcon(NULL),data(NULL),state(0),x(0),y(0){}
  virtual void draw(const FXCheckTreeList* list,FXDC& dc,FXint x,FXint y,FXint w,FXint h) const;
  void drawFocus(const FXCheckTreeList* list,FXDC& dc,FXint x,FXint y,FXint w,FXint h) const;
  virtual FXint hitItem(const FXCheckTreeList* list,FXint x,FXint y) const;

protected:
  enum{
    SELECTED        = 1,
    FOCUS           = 2,
    DISABLED        = 4,
    OPENED          = 8,
    EXPANDED        = 16,
    HASITEMS        = 32,
    DRAGGABLE       = 64,
    OPENICONOWNED   = 128,
    CLOSEDICONOWNED = 256,
    CHECKED         = 512
    };

public:
  /// Check tree item
  FXCheckTreeItem(const FXString& text,FXIcon* oi=NULL,FXIcon* ci=NULL,void* ptr=NULL):prev(NULL),next(NULL),parent(NULL),first(NULL),last(NULL),label(text),openIcon(oi),closedIcon(ci),data(ptr),state(0),x(0),y(0){}

  /// get parent item
  FXCheckTreeItem* getParent() const { return parent; }

  /// get next item
  FXCheckTreeItem* getNext() const { return next; }

  /// get previous item
  FXCheckTreeItem* getPrev() const { return prev; }

  /// get first item in branch
  FXCheckTreeItem* getFirst() const { return first; }

  /// get last item in branch
  FXCheckTreeItem* getLast() const { return last; }

  /// get item below this item
  FXCheckTreeItem* getBelow() const;

  /// get item above this item
  FXCheckTreeItem* getAbove() const;

  /// get the number of children that belong to this item
  FXint getNumChildren() const;

  /// set item text
  virtual void setText(const FXString& txt){ label=txt; }

  /// get item text
  FXString getText() const { return label; }

  /// set open icon
  virtual void setOpenIcon(FXIcon* icn){ openIcon=icn; }

  /// get open icon
  FXIcon* getOpenIcon() const { return openIcon; }

  /// set closed icon
  virtual void setClosedIcon(FXIcon* icn){ closedIcon=icn; }

  /// get closed icon
  FXIcon* getClosedIcon() const { return closedIcon; }

  /// set item specific data
  void setData(void* ptr){ data=ptr; }

  /// get item specific data
  void* getData() const { return data; }

  /// set focus on item
  virtual void setFocus(FXbool focus);

  /// item has focus
  FXbool hasFocus() const { return (state&FOCUS)!=0; }

  /// make item selected
  virtual void setSelected(FXbool selected);

  /// is item selected
  FXbool isSelected() const { return (state&SELECTED)!=0; }

  /// make item checked
  virtual void setChecked(FXbool checked);

  /// is item checked
  FXbool isChecked() const { return (state&CHECKED)!=0; }

  /// set item opened
  virtual void setOpened(FXbool opened);

  /// is item opened
  FXbool isOpened() const { return (state&OPENED)!=0; }

  /// expand item
  virtual void setExpanded(FXbool expanded);

  /// is item expaned
  FXbool isExpanded() const { return (state&EXPANDED)!=0; }

  /// enable item
  virtual void setEnabled(FXbool enabled);

  /// is item enabled
  FXbool isEnabled() const { return (state&DISABLED)==0; }

  /// make item draggable
  virtual void setDraggable(FXbool draggable);

  /// is item draggable
  FXbool isDraggable() const { return (state&DRAGGABLE)!=0; }

  /// set ownership of icon
  virtual void setIconOwned(FXuint owned=(OPENICONOWNED|CLOSEDICONOWNED));

  /// get ownership of icon
  FXuint isIconOwned() const { return (state&(OPENICONOWNED|CLOSEDICONOWNED)); }

  /// get width
  virtual FXint getWidth(const FXCheckTreeList* list) const;

  /// get height
  virtual FXint getHeight(const FXCheckTreeList* list) const;

  /// create resources
  virtual void create();

  /// detach resources
  virtual void detach();

  /// destroy resources
  virtual void destroy();

  /// save to stream
  virtual void save(FXStream& store) const;

  /// load from stream
  virtual void load(FXStream& store);

  /// dtor
  virtual ~FXCheckTreeItem();
  };



/// Tree item collate function
typedef FXint (*FXCheckTreeListSortFunc)(const FXCheckTreeItem*,const FXCheckTreeItem*);



/// Tree list Widget
class FXAPI FXCheckTreeList : public FXScrollArea {
  FXDECLARE(FXCheckTreeList)

protected:
  FXCheckTreeItem        *firstitem;         // First root item
  FXCheckTreeItem        *lastitem;          // Last root item
  FXCheckTreeItem        *anchoritem;        // Selection anchor item
  FXCheckTreeItem        *currentitem;       // Current item
  FXCheckTreeItem        *extentitem;        // Selection extent
  FXCheckTreeItem        *cursoritem;        // Item under cursor
  FXFont            *font;              // Font
  FXCheckTreeListSortFunc sortfunc;          // Item sort function
  FXColor            textColor;         // Text color
  FXColor            selbackColor;      // Selected background color
  FXColor            seltextColor;      // Selected text color
  FXColor            lineColor;         // Line color
  FXint              treeWidth;         // Tree width
  FXint              treeHeight;        // Tree height
  FXint              visible;           // Number of visible items
  FXint              indent;            // Parent to child indentation
  FXint              grabx;             // Grab point x
  FXint              graby;             // Grab point y
  FXString           lookup;            // Lookup string
  FXString           help;              // Help string
  FXbool             state;             // State of item

protected:
  /// serialisation
  FXCheckTreeList();

  /// calculate layout
  virtual void layout();

  /// create instance of item
  virtual FXCheckTreeItem* createItem(const FXString& text,FXIcon* oi,FXIcon* ci,void* ptr);

  /// sort items
  void sort(FXCheckTreeItem*& f1,FXCheckTreeItem*& t1,FXCheckTreeItem*& f2,FXCheckTreeItem*& t2,int n);
  void recompute();

private:
  FXCheckTreeList(const FXCheckTreeList&);
  FXCheckTreeList& operator=(const FXCheckTreeList&);

public:
  long onPaint(FXObject*,FXSelector,void*);
  long onEnter(FXObject*,FXSelector,void*);
  long onLeave(FXObject*,FXSelector,void*);
  long onUngrabbed(FXObject*,FXSelector,void*);
  long onMotion(FXObject*,FXSelector,void*);
  long onKeyPress(FXObject*,FXSelector,void*);
  long onKeyRelease(FXObject*,FXSelector,void*);
  long onLeftBtnPress(FXObject*,FXSelector,void*);
  long onLeftBtnRelease(FXObject*,FXSelector,void*);
  long onRightBtnPress(FXObject*,FXSelector,void*);
  long onRightBtnRelease(FXObject*,FXSelector,void*);
  long onQueryTip(FXObject*,FXSelector,void*);
  long onQueryHelp(FXObject*,FXSelector,void*);
  long onTipTimer(FXObject*,FXSelector,void*);
  long onFocusIn(FXObject*,FXSelector,void*);
  long onFocusOut(FXObject*,FXSelector,void*);
  long onAutoScroll(FXObject*,FXSelector,void*);
  long onClicked(FXObject*,FXSelector,void*);
  long onDoubleClicked(FXObject*,FXSelector,void*);
  long onTripleClicked(FXObject*,FXSelector,void*);
  long onCommand(FXObject*,FXSelector,void*);
  long onSelected(FXObject*,FXSelector,void*);
  long onDeselected(FXObject*,FXSelector,void*);
  long onOpened(FXObject*,FXSelector,void*);
  long onClosed(FXObject*,FXSelector,void*);
  long onExpanded(FXObject*,FXSelector,void*);
  long onCollapsed(FXObject*,FXSelector,void*);
  long onLookupTimer(FXObject*,FXSelector,void*);

public:
  /// sort ascending
  static FXint ascending(const FXCheckTreeItem* a,const FXCheckTreeItem* b);

  /// sort descending
  static FXint descending(const FXCheckTreeItem* a,const FXCheckTreeItem* b);

public:
  enum {
    ID_TIPTIMER=FXScrollArea::ID_LAST,
    ID_LOOKUPTIMER,
    ID_LAST
    };

public:
  /// Construct a tree list with nvis visible items; the tree list is initially empty
  FXCheckTreeList(FXComposite *p,FXint nvis,FXObject* tgt=NULL,FXSelector sel=0,FXuint opts=CHECKTREELIST_NORMAL,FXint x=0,FXint y=0,FXint w=0,FXint h=0);

  /// Create server-side resources
  virtual void create();

  /// Detach server-side resources
  virtual void detach();

  /// Return default width
  virtual FXint getDefaultWidth();

  /// Return default height
  virtual FXint getDefaultHeight();

  /// Compute and return content width
  virtual FXint getContentWidth();

  /// Return content height
  virtual FXint getContentHeight();

  /// Recalculate layout
  virtual void recalc();

  /// Tree list can receive focus
  virtual FXbool canFocus() const;

  /// Return number of items
  FXint getNumItems() const;

  /// Return number of visible items
  FXint getNumVisible() const { return visible; }

  /// Change number of visible items
  void setNumVisible(FXint nvis);

  /// REturn first root item
  FXCheckTreeItem* getFirstItem() const { return firstitem; }

  /// Return last root item
  FXCheckTreeItem* getLastItem() const { return lastitem; }

  /// Prepend new [possibly subclassed] item as first child of p
  FXCheckTreeItem* addItemFirst(FXCheckTreeItem* p,FXCheckTreeItem* item,FXbool notify=FALSE);

  /// Prepend new item with given text and optional icon, and user-data pointer as first child of p
  FXCheckTreeItem* addItemFirst(FXCheckTreeItem* p,const FXString& text,FXIcon* oi=NULL,FXIcon* ci=NULL,void* ptr=NULL,FXbool notify=FALSE);

  /// Append new [possibly subclassed] item as last child of p
  FXCheckTreeItem* addItemLast(FXCheckTreeItem* p,FXCheckTreeItem* item,FXbool notify=FALSE);

  /// Append new item with given text and optional icon, and user-data pointer as last child of p
  FXCheckTreeItem* addItemLast(FXCheckTreeItem* p,const FXString& text,FXIcon* oi=NULL,FXIcon* ci=NULL,void* ptr=NULL,FXbool notify=FALSE);

  /// Append new [possibly subclassed] item after to other item
  FXCheckTreeItem* addItemAfter(FXCheckTreeItem* other,FXCheckTreeItem* item,FXbool notify=FALSE);

  /// Append new item with given text and optional icon, and user-data pointer after to other item
  FXCheckTreeItem* addItemAfter(FXCheckTreeItem* other,const FXString& text,FXIcon* oi=NULL,FXIcon* ci=NULL,void* ptr=NULL,FXbool notify=FALSE);

  /// Prepend new [possibly subclassed] item prior to other item
  FXCheckTreeItem* addItemBefore(FXCheckTreeItem* other,FXCheckTreeItem* item,FXbool notify=FALSE);

  /// Prepend new item with given text and optional icon, and user-data pointer prior to other item
  FXCheckTreeItem* addItemBefore(FXCheckTreeItem* other,const FXString& text,FXIcon* oi=NULL,FXIcon* ci=NULL,void* ptr=NULL,FXbool notify=FALSE);

  /// Remove item
  void removeItem(FXCheckTreeItem* item,FXbool notify=FALSE);

  /// Remove items in range [fm, to] inclusively
  void removeItems(FXCheckTreeItem* fm,FXCheckTreeItem* to,FXbool notify=FALSE);

  /// Remove all items from list
  void clearItems(FXbool notify=FALSE);

  /// Return item width
  FXint getItemWidth(const FXCheckTreeItem* item) const { return item->getWidth(this); }

  /// Return item height
  FXint getItemHeight(const FXCheckTreeItem* item) const { return item->getHeight(this); }

  /// Get item at x,y, if any
  virtual FXCheckTreeItem* getItemAt(FXint x,FXint y) const;

  /**
  * Search items for item by name, starting from start item; the
  * flags argument controls the search direction, and case sensitivity.
  */
  FXCheckTreeItem* findItem(const FXString& text,FXCheckTreeItem* start=NULL,FXuint flags=SEARCH_FORWARD|SEARCH_WRAP) const;

  /// Scroll to make item visible
  void makeItemVisible(FXCheckTreeItem* item);

  /// Change item's text
  void setItemText(FXCheckTreeItem* item,const FXString& text);

  /// Return item's text
  FXString getItemText(const FXCheckTreeItem* item) const;

  /// Change item's open icon
  void setItemOpenIcon(FXCheckTreeItem* item,FXIcon* icon);

  /// Return item's open icon
  FXIcon* getItemOpenIcon(const FXCheckTreeItem* item) const;

  /// Chance item's closed icon
  void setItemClosedIcon(FXCheckTreeItem* item,FXIcon* icon);

  /// Return item's closed icon
  FXIcon* getItemClosedIcon(const FXCheckTreeItem* item) const;

  /// Change item user-data pointer
  void setItemData(FXCheckTreeItem* item,void* ptr) const;

  /// Return item user-data pointer
  void* getItemData(const FXCheckTreeItem* item) const;

  /// Return TRUE if item is selected
  FXbool isItemSelected(const FXCheckTreeItem* item) const;

  /// Return TRUE if item is checked
  FXbool isItemChecked(const FXCheckTreeItem* item) const;

  /// Return TRUE if item is current
  FXbool isItemCurrent(const FXCheckTreeItem* item) const;

  /// Return TRUE if item is visible
  FXbool isItemVisible(const FXCheckTreeItem* item) const;

  /// Return TRUE if item opened
  FXbool isItemOpened(const FXCheckTreeItem* item) const;

  /// Return TRUE if item expanded
  FXbool isItemExpanded(const FXCheckTreeItem* item) const;

  /// Return TRUE if item is a leaf-item, i.e. has no children
  FXbool isItemLeaf(const FXCheckTreeItem* item) const;

  /// Return TRUE if item is enabled
  FXbool isItemEnabled(const FXCheckTreeItem* item) const;

  /// Return item hit code: 0 outside, 1 icon, 2 text, 3 box
  /// 4 hit the check
  FXint hitItem(const FXCheckTreeItem* item,FXint x,FXint y) const;

  /// Repaint item
  void updateItem(FXCheckTreeItem* item);

  /// Enable item
  FXbool enableItem(FXCheckTreeItem* item);

  /// Disable item
  FXbool disableItem(FXCheckTreeItem* item);

  /// Select item
  FXbool selectItem(FXCheckTreeItem* item,FXbool notify=FALSE);

  /// Deselect item
  FXbool deselectItem(FXCheckTreeItem* item,FXbool notify=FALSE);

  /// Toggle item selection
  FXbool toggleItem(FXCheckTreeItem* item,FXbool notify=FALSE);

  /// check item
  FXbool checkItem(FXCheckTreeItem* item,FXbool notify=FALSE);

  /// uncheck item
  FXbool uncheckItem(FXCheckTreeItem* item,FXbool notify=FALSE);

  /// Toggle item check state
  FXbool toggleCheckItem(FXCheckTreeItem* item,FXbool notify=FALSE);

  /// Open item
  FXbool openItem(FXCheckTreeItem* item,FXbool notify=FALSE);

  /// Close item
  FXbool closeItem(FXCheckTreeItem* item,FXbool notify=FALSE);

  /// Collapse tree
  FXbool collapseTree(FXCheckTreeItem* tree,FXbool notify=FALSE);

  /// Expand tree
  FXbool expandTree(FXCheckTreeItem* tree,FXbool notify=FALSE);

  /// Reparent item under parent p
  void reparentItem(FXCheckTreeItem* item,FXCheckTreeItem* p);

  /// Change current item
  void setCurrentItem(FXCheckTreeItem* item,FXbool notify=FALSE);

  /// Return current item, if any
  FXCheckTreeItem* getCurrentItem() const { return currentitem; }

  /// Change anchor item
  void setAnchorItem(FXCheckTreeItem* item);

  /// Return anchor item, if any
  FXCheckTreeItem* getAnchorItem() const { return anchoritem; }

  /// Return item under cursor, if any
  FXCheckTreeItem* getCursorItem() const { return cursoritem; }

  /// Extend selection from anchor item to item
  FXbool extendSelection(FXCheckTreeItem* item,FXbool notify=FALSE);

  /// Deselect all items
  FXbool killSelection(FXbool notify=FALSE);

  /// Sort root items
  void sortItems();

  /// Sort children of item
  void sortChildItems(FXCheckTreeItem* item);

  /// Change text font
  void setFont(FXFont* fnt);

  /// Return text font
  FXFont* getFont() const { return font; }

  /// Change parent-child indent amount
  void setIndent(FXint in);

  /// Return parent-child indent amount
  FXint getIndent() const { return indent; }

  /// Return normal text color
  FXColor getTextColor() const { return textColor; }

  /// Change normal text color
  void setTextColor(FXColor clr);

  /// Return selected text background
  FXColor getSelBackColor() const { return selbackColor; }

  /// Change selected text background
  void setSelBackColor(FXColor clr);

  /// Return selected text color
  FXColor getSelTextColor() const { return seltextColor; }

  /// Change selected text color
  void setSelTextColor(FXColor clr);

  /// Return line color
  FXColor getLineColor() const { return lineColor; }

  /// Change line color
  void setLineColor(FXColor clr);

  /// Return sort function
  FXCheckTreeListSortFunc getSortFunc() const { return sortfunc; }

  /// Change sort function
  void setSortFunc(FXCheckTreeListSortFunc func){ sortfunc=func; }

  /// Return list style
  FXuint getListStyle() const;

  /// Change list style
  void setListStyle(FXuint style);

  /// Set the status line help text for this list
  void setHelpText(const FXString& text);

  /// Get the status line help text for this list
  FXString getHelpText() const { return help; }

  /// Save object to a stream
  virtual void save(FXStream& store) const;

  /// Load object from a stream
  virtual void load(FXStream& store);

  /// Destructor
  virtual ~FXCheckTreeList();
  };

} // namespace FXEX
#endif // FXCHECKTREE_LIST_H

