#include "FXSocket.h"
#include <string.h>
#include <errno.h>

static FXint sockets = 0;

FXDEFMAP(FXSocket) FXSocketMap[] = {
  FXMAPFUNC(SEL_IO_READ,    FXSocket::ID_INPUT,   FXSocket::onIORead),
  FXMAPFUNC(SEL_IO_WRITE,   FXSocket::ID_INPUT,   FXSocket::onIOWrite)
};

FXIMPLEMENT(FXSocket, FXObject, FXSocketMap, ARRAYNUMBER(FXSocketMap));

FXSocket::FXSocket(FXApp *app, FXObject *tgt, FXSelector sel) {
  this->app = app;
  this->tgt = tgt;
  this->sel = sel;
  read_buf = NULL;
  write_buf = NULL;
  read_items = -1;
  write_items = -1;
  connected = FALSE;
  sock = INVALID_SOCKET_HANDLE;
  memset(&localSide, 0, sizeof(localSide));
  memset(&remoteSide, 0, sizeof(remoteSide));
  socket_inc_count();
}

FXSocket::~FXSocket() {
  if(connected) disconnect();
  if(read_buf) delete[] read_buf;
  if(write_buf) delete[] write_buf;
  socket_dec_count();
}

void FXSocket::create() {
  FXASSERT(sock==INVALID_SOCKET_HANDLE);
  read_buf = new FXchar[SOCK_BUFFER_SIZE];
  write_buf = new FXchar[SOCK_BUFFER_SIZE];
  read_items = 0;
  write_items = 0;
}

FXSOCKET FXSocket::newSocket(FXSOCKET sock = INVALID_SOCKET_HANDLE) {
  if(isConnected()) return INVALID_SOCKET_HANDLE;
  socket_close(this->sock);
  if(sock == INVALID_SOCKET_HANDLE) sock = socket_create();
  this->sock = sock;
  return sock;
}

long FXSocket::raiseError(FXint errorCode, FXchar *msg, FXint idata) {
  return sendEvent(SOCK_ERROR, idata, msg, errorCode);
}

long FXSocket::sendEvent(FXSocketEventType eventType, FXint idata, void *vdata, FXint errorCode) {
  FXSocketEvent ev;
  ev.eventType = eventType;
  ev.Socket = this;
  ev.idata = idata;
  ev.vdata = vdata;
  ev.errorCode = errorCode;
  return tgt->handle(this, MKUINT(sel, SEL_COMMAND), &ev);
}

long FXSocket::read_data() {
  int r;
  if(read_items>=SOCK_BUFFER_SIZE) return 0;
  r = recvfrom(sock, read_buf+read_items, SOCK_BUFFER_SIZE-read_items, MSG_NOSIGNAL, NULL, NULL);
  if(r>0) {
    read_items += r;
    sendEvent(SOCK_READ, r);
  }
  else if (r<0) {
    raiseError(errno, "Socket read error");
    r = SOCKET_ERROR;
  }
  else if (r==0) disconnect();
  return r;
}

long FXSocket::write_data() {
  int r;

  if(!write_items) return 0;
  sendEvent(SOCK_WRITE, write_items);
  r = sendto(sock, write_buf, write_items, MSG_NOSIGNAL, NULL, 0);
  if(r>0) {
    write_items -= r;
    if(write_items) memmove(write_buf, write_buf+r, write_items);
  } else if(r<0) {
    raiseError(errno, "Socket write error");
    r = SOCKET_ERROR;
  }
  return r;
}

long FXSocket::disconnect() {
  FXint ret;
  if(isConnected()) {
    setConnected(FALSE);
    sendEvent(SOCK_DISCONNECT);
    ret = socket_close(sock);
    sock = INVALID_SOCKET_HANDLE;
    return ret;
  } else return SOCKET_ERROR;
}

void FXSocket::setConnected(FXbool connected, struct sockaddr_in *remoteSide) {
  FXbool old_state = this->connected;
  FXint ret;
  socklen_t namelen;
  
  this->connected = connected;
  FXASSERT(sock!=INVALID_SOCKET_HANDLE);
  if(connected && !old_state) {
    read_items = 0;
    write_items = 0;
    if(remoteSide) this->remoteSide = *remoteSide;
    namelen = sizeof(struct sockaddr_in);
    ret = getsockname(sock, (struct sockaddr *) &localSide, &namelen);
    if(ret == SOCKET_ERROR) memset(&localSide, 0, sizeof(localSide));
    sendEvent(SOCK_CONNECT);
    app->addInput((FXInputHandle) sock, INPUT_READ|INPUT_EXCEPT, this, ID_INPUT);
  } else  if(old_state && !connected) {
    write_items = 0;
    app->removeInput((FXInputHandle) sock, INPUT_READ|INPUT_WRITE|INPUT_EXCEPT);
  }
}

FXint FXSocket::read(void *buf, FXint size) {
  FXint to_read;
  if(size<0) return SOCKET_ERROR;
  if(!read_items) return isConnected() ? 0 : SOCKET_ERROR;
  if(!size) return 0;
  to_read = (size>read_items) ? read_items : size;
  memcpy(buf, read_buf, to_read);
  read_items -= to_read;
  memmove(read_buf, read_buf+to_read, read_items);
  return to_read;
}

FXint FXSocket::write(const void *buf, FXint size) {
  FXint to_write;
  FXint avail;
  if(!isConnected()) return SOCKET_ERROR;
  avail = SOCK_BUFFER_SIZE-write_items;
  to_write = (size>avail) ? avail : size;
  if(!to_write) return 0;
  memcpy(write_buf+write_items, buf, to_write);
  write_items += to_write;
  app->addInput((FXInputHandle) sock, INPUT_WRITE, this, ID_INPUT);
  return to_write;
}

FXint FXSocket::readString(FXString &s) {
  FXint pos;
  FXint len;
  FXchar *c;

  s = "";
  len = read_items;
  if(!len) return isConnected() ? 0 : SOCKET_ERROR;
  c = (FXchar *) read_buf;
  for(pos=0; pos<len; ++pos)
    if(!c[pos]) break;
  if(pos==len) {
    if(pos!=SOCK_BUFFER_SIZE) return SOCKET_RETRY;
    s.insert(0, c, pos);
    s.insert(pos, '\0');
    read_buf = 0;
    return SOCKET_MSGSIZE; 
  }
  ++pos;
  s = c;
  read_items -= pos;
  memmove(read_buf, read_buf+pos, read_items);
  return pos;
}

FXint FXSocket::writeString(const FXString &s) {
  FXint l;
  l = s.length()+1;
  if(l>SOCK_BUFFER_SIZE) return SOCKET_MSGSIZE;
  if((SOCK_BUFFER_SIZE-write_items) < l) return SOCKET_RETRY;
  return write(s.text(), l);
}


FXint FXSocket::socket_close(FXSOCKET socket) {
  FXint ret;
  if(socket!=INVALID_SOCKET_HANDLE) {
#ifdef WIN32
  shutdown(socket, SD_BOTH);
  ret = closesocket(socket);
#else
  shutdown(socket, SHUT_RDWR);
  ret = close(socket);
#endif
  } else ret = SOCKET_ERROR;
  return ret;
}

FXSOCKET FXSocket::socket_create() {
  return socket(PF_INET, SOCK_STREAM, 0);
}

void FXSocket::socket_inc_count() {
  if(!sockets++) {
#ifdef WIN32
    WSAData data;
    WSAStartup(MAKEWORD(1, 1), &data);
#endif
  }
}

void FXSocket::socket_dec_count() {
  --sockets;
  FXASSERT(sockets>=0);
  if(!sockets) {
#ifdef WIN32
    WSACleanup();
#endif
  }
}

long FXSocket::onIORead(FXObject *sender, FXSelector sel, void *data) {
  FXASSERT(sock==(FXSOCKET)data);
  if(isConnected()) read_data();
  return 1;
}

long FXSocket::onIOWrite(FXObject *sender, FXSelector sel, void *data) {
  if(write_items && isConnected()) {
    FXASSERT(sock==(FXSOCKET)data);
    write_data();
  }
  else app->removeInput((FXInputHandle) data, INPUT_WRITE);
  return 1;
}
