#include "FXServerSocket.h"
#include <stdlib.h>
#include <string.h>
#include <errno.h>

#define DEFAULT_NEW_CLIENTS 1
#define DEFAULT_MAX_CLIENTS 8
#define DEFAULT_LISTEN_QUEUE 2

FXDEFMAP(FXServerSocket) FXServerSocketMap[] = {
  FXMAPFUNC(SEL_COMMAND,    FXServerSocket::ID_SOCK,    FXServerSocket::onSock),
  FXMAPFUNC(SEL_IO_READ,    FXServerSocket::ID_INPUT,   FXServerSocket::onIORead)
};

FXIMPLEMENT(FXServerSocket, FXObject, FXServerSocketMap, ARRAYNUMBER(FXServerSocketMap));

FXServerSocket::FXServerSocket(FXApp *app, FXObject *tgt, FXSelector sel) {
  this->app = app;
  this->tgt = tgt;
  this->sel = sel;
  Clients = NULL;
  nClients = 0;
  localPort = -1;
  bindAddress = "0.0.0.0";
  maxClients = DEFAULT_MAX_CLIENTS;
  listenQueue = DEFAULT_LISTEN_QUEUE;
  connected = FALSE;
  server = INVALID_SOCKET_HANDLE;
  FXSocket::socket_inc_count();
}

FXServerSocket::~FXServerSocket() {
  disconnect();
  while(nClients) {
    --nClients;
    delete Clients[nClients];
  }
  if(Clients) FXFREE(&Clients);
  FXSocket::socket_dec_count();
}

FXint FXServerSocket::PackSockets() {
  FXint dst = 0;
  FXint x;
  FXSocket *tmp;

  if(!isConnected()) return 0;
  for(x=0; x<nClients; ++x) {
    if(Clients[x]->isConnected()) {
      if(dst != x) {
        tmp = Clients[dst];
        Clients[dst] = Clients[x];
        Clients[x] = tmp;
      }
      ++dst;
    }
  }
  return dst;
}

FXint FXServerSocket::disconnect() {
  FXint x;
  if(isConnected()) {
    FXASSERT(server!=INVALID_SOCKET_HANDLE);
    connected = FALSE;
    app->removeInput((FXInputHandle) server, INPUT_READ|INPUT_WRITE|INPUT_EXCEPT);
    for(x=0; x<nClients; ++x) Clients[x]->disconnect();
    sendEvent(SOCK_SERVDISCONNECT);
    x = FXSocket::socket_close(server);
    server = INVALID_SOCKET_HANDLE;  
    return x;
  } else return SOCKET_ERROR;
}

FXSocket *FXServerSocket::getFreeSocket() {
  FXint x;
  FXint adjClients;
  
  for(x=0; x<nClients; ++x)
    if(!Clients[x]->isConnected()) return Clients[x];

  adjClients = maxClients-nClients;
  adjClients = (adjClients<DEFAULT_NEW_CLIENTS) ? adjClients : DEFAULT_NEW_CLIENTS;
  if(adjClients <= 0) return NULL;
  FXRESIZE(&Clients, FXSocket *, (nClients+adjClients));
  for(x=0; x<adjClients; ++x) {
    Clients[nClients+x] = new FXSocket(app, this, ID_SOCK);
    Clients[nClients+x]->create();
  }
  x = nClients;
  nClients += adjClients;
  return Clients[x];
}

FXint FXServerSocket::setLocalPort(FXint localPort) {
  if(isConnected() || (localPort<0)||(localPort>65535)) return SOCKET_ERROR;
  this->localPort = localPort;
  return 0;
}

FXint FXServerSocket::setBindAddress(const FXString &bindAddress) {
  struct in_addr addr;
  if(isConnected()) return SOCKET_ERROR;
  if(!inet_aton(bindAddress.text(), &addr)) return SOCKET_ERROR;
  this->bindAddress = bindAddress;
  return 0;
}

FXint FXServerSocket::setMaxClients(FXint maxClients) {
  FXint busy;
  FXint x;

  if(maxClients<0) return SOCKET_ERROR;
  busy = connCount();
  if(busy>maxClients) return SOCKET_ERROR;
  this->maxClients = maxClients;
  if(maxClients < nClients) {
    for(x=maxClients; x<nClients; ++x)
      delete Clients[x];
    nClients = maxClients;
    FXRESIZE(&Clients, FXSocket *, nClients);
  }
  return 0;
}

FXSocket *FXServerSocket::connection(FXint conn) {
  FXint max;
  if(conn<0) return NULL;
  max = PackSockets();
  if(conn>=max) return NULL;
  return Clients[conn];
}

FXint FXServerSocket::connect() {
  FXSOCKET socket = INVALID_SOCKET_HANDLE;
  struct sockaddr_in sockname;
  struct in_addr addr;
  int opt;
  
  if(isConnected() || (localPort<0) ) goto c_fail;
  socket = FXSocket::socket_create();
  if(socket==INVALID_SOCKET_HANDLE) goto c_fail;
  sockname.sin_family = AF_INET;
  sockname.sin_port = htons(localPort);
  if(!inet_aton(bindAddress.text(), &addr)) goto c_fail;
  sockname.sin_addr.s_addr = addr.s_addr;
  opt = 1;
  if(setsockopt(socket, SOL_SOCKET, SO_REUSEADDR, &opt, sizeof(opt))==SOCKET_ERROR) goto c_fail;
  if(bind(socket, (struct sockaddr *) &sockname, sizeof(sockname)) == SOCKET_ERROR) goto c_fail;
  if(listen(socket, listenQueue) == SOCKET_ERROR) goto c_fail;
  server = socket;
  connected = TRUE;
  app->addInput((FXInputHandle) server, INPUT_READ|INPUT_EXCEPT, this, ID_INPUT);
  return 0;
c_fail:
  FXSocket::socket_close(socket);
  return SOCKET_ERROR;
}

long FXServerSocket::sendEvent(FXSocketEventType eventType, FXSocket *Socket, FXint idata, void *vdata, FXint errorCode) {
  FXSocketEvent ev;
  ev.eventType = eventType;
  ev.Socket = Socket;
  ev.idata = idata;
  ev.vdata = vdata;
  ev.errorCode = errorCode;
  return tgt->handle(this, MKUINT(sel, SEL_COMMAND), &ev);
}

long FXServerSocket::raiseError(FXint errorCode, FXchar *msg, FXint idata) {
  return sendEvent(SOCK_ERROR, NULL, idata, msg, errorCode);
}

long FXServerSocket::onSock(FXObject *sender, FXSelector sel, void *data) {
  FXSocketEvent *sock;
  sock = (FXSocketEvent *) data;
  switch(sock->eventType) {
    default:
    return tgt->handle(this, MKUINT(this->sel, SEL_COMMAND), sock);
  }
  return 1;
}

long FXServerSocket::onIORead(FXObject *sender, FXSelector sel, void *data) {
  socklen_t size;
  FXSOCKET newConn;
  FXSocket *newSock;
  struct sockaddr_in clientname;

  if(isConnected()) {
    size = sizeof(clientname);
    newConn = accept(server, (struct sockaddr *) &clientname, &size);
    if(newConn == INVALID_SOCKET_HANDLE) raiseError(errno, "Socket accept error");
    else {
      newSock = getFreeSocket();
      if(!newSock) {
        FXSocket::socket_close(newConn);
        sendEvent(SOCK_CONNREJECT);
      } else {
        newSock->newSocket(newConn);
        newSock->setConnected(TRUE, &clientname);
      }
    }
  }
  return 1;
}
