#include "fxex.h"

/* FXThread demo
 *
 * This demo demonstrates the use of a true thread object.  A subclass of FXThread is created
 * which is capable of signalling the GUI thread when work has completed.
 *
 * Here we use an atomic value to store the result of a calculation, and we use an accessor
 * method getValue() to pass it back to the main thread.
 */

// ***************************** Thread object ***************************
class MyThread : public FXThread {
  FXDECLARE(MyThread)
private:
  FXAtomicInt calculation_result;
public:
  virtual void run();
public:
  MyThread(FXint startVal=0,FXObject *tgt=NULL,FXSelector sel=0);
  FXint getValue();
  void setValue(FXint value);
  };

// MyThread map
// Note: your derived class can quite happily handle the reception of messages from the
//       GUI, but you need to provide some form of locking inside your event handler, when
//       accessing data in the child thread.
FXIMPLEMENT(MyThread,FXThread,NULL,0)

// MyThread ctor - your constructor can take any number of arguments, since at construction
// time, it is essentially still in the main thread.
MyThread::MyThread(FXint startVal,FXObject *tgt,FXSelector sel) : FXThread(tgt,sel) {
  calculation_result=startVal;
  }

// MyThread worker function - the run() method is where you do all your work
// - test the 'stopThread' condition to see if the main thread wants us to stop
void MyThread::run(){
  fxmessage("MyThread started (with value: %i)\n",calculation_result.getValue());

  // some interesting calculation
  while (!stopThread && calculation_result < 1000000){
    calculation_result++;

    // remember to yield, so as to avoid the GUI from becoming unresponsive
    yield();
    }

  // Note, I'm not sure if fxmessage (et.al) is thread safe...?
  fxmessage("MyThread ended (with value: %i)\n",calculation_result.getValue());
  }

// allow other threads to retrieve the value
FXint MyThread::getValue(){
  return calculation_result.getValue();
  }

// assign to new value
void MyThread::setValue(FXint value){
  calculation_result.setValue(value);
  }

// ************************* Main Window *************************************
class MainWindow : public FXMainWindow {
  FXDECLARE(MainWindow)
private:
  MyThread       *thread;
protected:
  MainWindow(){}
  void thread_function();  // the worker thread
public:
  enum {
    ID_MAINWINDOW=FXMainWindow::ID_LAST,
    ID_THREAD,
    ID_THREAD_START,
    ID_SET_VALUE,
    ID_RESET_VALUE,
    ID_TIMER,
    ID_LAST
    };
public:
  long onClose(FXObject*,FXSelector,void*);
  long onThreadEvent(FXObject*,FXSelector,void*);
  long onThreadStart(FXObject*,FXSelector,void*);
  long onCmdSetValue(FXObject*,FXSelector,void*);
  long onUpdSetValue(FXObject*,FXSelector,void*);
  long onCmdResetValue(FXObject*,FXSelector,void*);
  long onTimer(FXObject*,FXSelector,void*);
public:
  MainWindow(FXApp *a);
  virtual void create();
  ~MainWindow();
  };

// map
FXDEFMAP(MainWindow) MainWindowMap[]={
  FXMAPFUNC(SEL_SIGNAL,MainWindow::ID_MAINWINDOW,MainWindow::onClose),
  FXMAPFUNC(SEL_CLOSE,MainWindow::ID_MAINWINDOW,MainWindow::onClose),
  // start the thread
  FXMAPFUNC(SEL_COMMAND,MainWindow::ID_THREAD_START,MainWindow::onThreadStart),
  // set the thread value, to a new value
  FXMAPFUNC(SEL_COMMAND,MainWindow::ID_SET_VALUE,MainWindow::onCmdSetValue),
  // set the textfield, to the value in the thread
  FXMAPFUNC(SEL_UPDATE,MainWindow::ID_SET_VALUE,MainWindow::onUpdSetValue),
  // reset the value
  FXMAPFUNC(SEL_COMMAND,MainWindow::ID_RESET_VALUE,MainWindow::onCmdResetValue),
  // handle thread event
  FXMAPFUNC(SEL_THREAD,MainWindow::ID_THREAD,MainWindow::onThreadEvent),
  // support the demo
  FXMAPFUNC(SEL_TIMEOUT,MainWindow::ID_TIMER,MainWindow::onTimer),
  };
FXIMPLEMENT(MainWindow,FXMainWindow,MainWindowMap,ARRAYNUMBER(MainWindowMap))

// your mainwindow thread is no different from any other FOX app
MainWindow::MainWindow(FXApp *a) : FXMainWindow(a,"FXThread demo") {
  setTarget(this);
  setSelector(ID_MAINWINDOW);
  setX(50);
  setY(50);
  new FXTextField(this,10,this,ID_SET_VALUE,TEXTFIELD_INTEGER|LAYOUT_FILL_X);
  new FXButton(this,"press to start/stop thread",NULL,this,ID_THREAD_START);
  new FXButton(this,"reset value",NULL,this,ID_RESET_VALUE);
  thread = new MyThread(5,this,ID_THREAD);
  }

// dtor
MainWindow::~MainWindow(){
  getApp()->removeTimeout(this,ID_TIMER);
  }

// create resource
void MainWindow::create(){
  FXMainWindow::create();
  show();
  onTimer(NULL,0,NULL);
  }

// we must make sure that the thread has stopped, before exiting
// (good programming practise).  We sleep a little bit to allow the main thread
// to yield the CPU, thus the OS will execute the worker thread, allowing it to run.
// Your worker thread should always check to see if it should exit.
long MainWindow::onClose(FXObject*,FXSelector,void*){
  if (thread->isRunning()) thread->stop();
  while (thread->isRunning()) { fxsleep(200); }
  getApp()->exit();
  return 1;
  }

long MainWindow::onTimer(FXObject*,FXSelector,void*){
  getApp()->addTimeout(this,ID_TIMER,1000);
  return 1;
  }

long MainWindow::onThreadEvent(FXObject*,FXSelector,void*){
  fxmessage("Received message from thread\n");
  return 1;
  }

// start/stop thread.  if a thread is running, the stop() method is just a 'request'
// to stop - the thread must honour this request.
long MainWindow::onThreadStart(FXObject*,FXSelector,void*){
  if (!thread->isRunning()) thread->start();
  else thread->stop();
  return 1;
  }

// handle update event from textfield - connect the text field to the value in the
// worker thread so that the text field shows the new value of the worker thread.
long MainWindow::onUpdSetValue(FXObject *sender,FXSelector,void*){
  FXint i=thread->getValue();
  sender->handle(this,FXSEL(SEL_COMMAND,ID_SETINTVALUE),(void*)&i);
  return 1;
  }

// set thread value from text field
long MainWindow::onCmdSetValue(FXObject*,FXSelector,void *ptr){
  FXString s=(FXchar*)ptr;
  FXint i=FXIntVal(s);
  thread->setValue(i);
  return 1;
  }

// reset the value to something suitable
long MainWindow::onCmdResetValue(FXObject*,FXSelector,void*){
  thread->setValue(2);
  return 1;
  }

/* start it up */
int main(int argc, char *argv[]){
  FXApp app("FXThread","FoxExTest");
  app.init(argc,argv);
  new MainWindow (&app);
  app.create();
  return app.run();
  }

