/********************************************************************************
*                                                                               *
*                      Simple db navigator control                              *
*                                                                               *
*********************************************************************************
* Copyright (C) 2003 by Giancarlo Formicuccia.  All Rights Reserved.            *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
*********************************************************************************/
#include <config.h>
#include <fox/fxver.h>
#include <fox/xincs.h>
#include <fox/fxdefs.h>
#include <fox/FXStream.h>
#include <fox/FXString.h>
#include <fox/FXWString.h>
#include <fox/FXCharset.h>
#include <fox/FXSize.h>
#include <fox/FXPoint.h>
#include <fox/FXRectangle.h>
#include <fox/FXRegistry.h>
#include <fox/FXApp.h>
#include <fox/FXMatrix.h>
#include <fox/FXButton.h>
#include <fox/FXMessageBox.h>
#include <fox/FXInputDialog.h>
using namespace FX;
#include "fxexdefs.h"
#include "FXArray.h"
#include "FXDatabaseField.h"
#include "FXDatabaseQuery.h"
#include "FXdbNavigator.h"
#include "FXExtendedException.h"
using namespace FXEX;
namespace FXEX {

FXDEFMAP(FXdbNavigator) FXdbNavigatorMap[] = {
  FXMAPFUNC(SEL_COMMAND,  FXdbNavigator::ID_FIRST,      FXdbNavigator::onFirst),
  FXMAPFUNC(SEL_COMMAND,  FXdbNavigator::ID_PREV,       FXdbNavigator::onPrev),
  FXMAPFUNC(SEL_COMMAND,  FXdbNavigator::ID_TO,         FXdbNavigator::onTo),
  FXMAPFUNC(SEL_COMMAND,  FXdbNavigator::ID_NEXT,       FXdbNavigator::onNext),
  FXMAPFUNC(SEL_COMMAND,  FXdbNavigator::ID_LST,        FXdbNavigator::onLst),
  FXMAPFUNC(SEL_COMMAND,  FXdbNavigator::ID_ADD,        FXdbNavigator::onAdd),
  FXMAPFUNC(SEL_COMMAND,  FXdbNavigator::ID_EDIT,       FXdbNavigator::onEdit),
  FXMAPFUNC(SEL_COMMAND,  FXdbNavigator::ID_DEL,        FXdbNavigator::onDel),
  FXMAPFUNC(SEL_COMMAND,  FXdbNavigator::ID_OK,         FXdbNavigator::onOk),
  FXMAPFUNC(SEL_COMMAND,  FXdbNavigator::ID_CANC,       FXdbNavigator::onCanc),
  FXMAPFUNC(SEL_EVENT,    FXdbNavigator::ID_DBQUERY,    FXdbNavigator::onDBQuery)
};

FXIMPLEMENT(FXdbNavigator, FXVerticalFrame, FXdbNavigatorMap, ARRAYNUMBER(FXdbNavigatorMap));

FXdbNavigator::FXdbNavigator(FXComposite *p, FXDatabaseQuery *dbq):
  FXVerticalFrame(p, LAYOUT_FILL_X|LAYOUT_FILL_Y|FRAME_SUNKEN) {
  mtx = new FXMatrix(this, 5, MATRIX_BY_COLUMNS|LAYOUT_FILL_X|LAYOUT_FILL_Y|LAYOUT_FILL_ROW|LAYOUT_FILL_COLUMN);
    bFirst = new FXButton(mtx, "<<", NULL, this, ID_FIRST, BUTTON_NORMAL|LAYOUT_FILL_X|LAYOUT_FILL_Y|LAYOUT_FILL_ROW|LAYOUT_FILL_COLUMN);
    bPrev = new FXButton(mtx, "<", NULL, this, ID_PREV, BUTTON_NORMAL|LAYOUT_FILL_X|LAYOUT_FILL_Y|LAYOUT_FILL_ROW|LAYOUT_FILL_COLUMN);
    bTo = new FXButton(mtx, "?", NULL, this, ID_TO, BUTTON_NORMAL|LAYOUT_FILL_X|LAYOUT_FILL_Y|LAYOUT_FILL_ROW|LAYOUT_FILL_COLUMN);
    bNext = new FXButton(mtx, ">", NULL, this, ID_NEXT, BUTTON_NORMAL|LAYOUT_FILL_X|LAYOUT_FILL_Y|LAYOUT_FILL_ROW|LAYOUT_FILL_COLUMN);
    bLst = new FXButton(mtx, ">>", NULL, this, ID_LST, BUTTON_NORMAL|LAYOUT_FILL_X|LAYOUT_FILL_Y|LAYOUT_FILL_ROW|LAYOUT_FILL_COLUMN);
    bAdd = new FXButton(mtx, "add", NULL, this, ID_ADD, BUTTON_NORMAL|LAYOUT_FILL_X|LAYOUT_FILL_Y|LAYOUT_FILL_ROW|LAYOUT_FILL_COLUMN);
    bEdit = new FXButton(mtx, "edit", NULL, this, ID_EDIT, BUTTON_NORMAL|LAYOUT_FILL_X|LAYOUT_FILL_Y|LAYOUT_FILL_ROW|LAYOUT_FILL_COLUMN);
    bDel = new FXButton(mtx, "del", NULL, this, ID_DEL, BUTTON_NORMAL|LAYOUT_FILL_X|LAYOUT_FILL_Y|LAYOUT_FILL_ROW|LAYOUT_FILL_COLUMN);
    bOk = new FXButton(mtx, "ok", NULL, this, ID_OK, BUTTON_NORMAL|LAYOUT_FILL_X|LAYOUT_FILL_Y|LAYOUT_FILL_ROW|LAYOUT_FILL_COLUMN);
    bCanc = new FXButton(mtx, "canc", NULL, this, ID_CANC, BUTTON_NORMAL|LAYOUT_FILL_X|LAYOUT_FILL_Y|LAYOUT_FILL_ROW|LAYOUT_FILL_COLUMN);
    this->dbq = dbq;
}

void FXdbNavigator::create() {
  FXVerticalFrame::create();
  setEditMode(FALSE);
  dbq ? enable() : disable();
}

void FXdbNavigator::setQuery(FXDatabaseQuery *dbq) {
  if(this->dbq!=dbq) {
    this->dbq = dbq;
    setEditMode(FALSE);
    dbq ? enable() : disable();
  }
}

long FXdbNavigator::onFirst(FXObject *, FXSelector, void *) {
  if(!dbq) return 1;
  dbq->moveFirst();
  return 1;
}

long FXdbNavigator::onPrev(FXObject *, FXSelector, void *) {
  if(!dbq) return 1;
  dbq->movePrevious();
  return 1;
}

long FXdbNavigator::onTo(FXObject *, FXSelector, void *) {
  FXint val = 0;
  if(!dbq) return 1;
  if(FXInputDialog::getInteger(val, this, "Select row to fetch", "New row: ")) {
    dbq->moveTo(val);
  }
  return 1;
}

long FXdbNavigator::onNext(FXObject *, FXSelector, void *) {
  if(!dbq) return 1;
  dbq->moveNext();
  return 1;
}

long FXdbNavigator::onLst(FXObject *, FXSelector, void *) {
  if(!dbq) return 1;
  dbq->moveLast();
  return 1;
}

long FXdbNavigator::onAdd(FXObject *, FXSelector, void *) {
  if(!dbq) return 1;
  try {
    dbq->addNew();
  } catch(const FXDatabaseException &e) {
    FXMessageBox::error(this, MBOX_OK, "Error", "%s", e.what());
    return 1;
  }
  setEditMode(TRUE);
  return 1;
}

long FXdbNavigator::onEdit(FXObject *, FXSelector, void *) {
  if(!dbq) return 1;
  try {
    dbq->Edit();
  } catch(const FXDatabaseException &e) {
    FXMessageBox::error(this, MBOX_OK, "Error", "%s", e.what());
    return 1;
  }
  setEditMode(TRUE);
  return 1;
}

long FXdbNavigator::onDel(FXObject *, FXSelector, void *) {
  if(!dbq) return 1;
  try {
    dbq->Delete();
  } catch(const FXDatabaseException &e) {
    FXMessageBox::error(this, MBOX_OK, "Error", "%s", e.what());
    return 1;
  }
  return 1;
}

long FXdbNavigator::onOk(FXObject *, FXSelector, void *) {
  FXASSERT(dbq);
  try {
    dbq->Update();
  } catch(const FXDatabaseException &e) {
    FXMessageBox::error(this, MBOX_OK, "Error", "%s", e.what());
    return 1;
  }
  setEditMode(FALSE);
  return 1;
}

long FXdbNavigator::onCanc(FXObject *, FXSelector, void *) {
  FXASSERT(dbq);
  try {
    dbq->CancelUpdate();
  } catch(const FXDatabaseException &e) {
    FXMessageBox::error(this, MBOX_OK, "Unexpected Error!", "%s", e.what());
//    return 1;
  }
  setEditMode(FALSE);
  return 1;
}

long FXdbNavigator::onDBQuery(FXObject *, FXSelector, void *data) {
  FXdbEvent *ev = (FXdbEvent *) data;
  switch(ev->type) {
    case dbConnect:
    case dbDisconnect:
    case dbDestroy:
      setQuery(NULL);
      break;
    case dbClose:
      setEditMode(FALSE);
      disable();
      break;
    case dbOpen:
      setEditMode(FALSE);
      enable();
      break;
    default: /* ignore */
      break;
  }
  if(target) target->handle(this, MKUINT(message, SEL_EVENT), data);
  return 1;
}

void FXdbNavigator::setEditMode(FXbool mode) {
  if(mode) {
    bFirst->disable();
    bPrev->disable();
    bTo->disable();
    bNext->disable();
    bLst->disable();
    bAdd->disable();
    bEdit->disable();
    bDel->disable();
    bOk->enable();
    bCanc->enable();
    bOk->setDefault();
  } else {
    bOk->setDefault(FALSE);
    bFirst->enable();
    bPrev->enable();
    bTo->enable();
    bNext->enable();
    bLst->enable();
    bAdd->enable();
    bEdit->enable();
    bDel->enable();
    bOk->disable();
    bCanc->disable();
  }
}

}
