/********************************************************************************
*                                                                               *
*                  Serial port I/O object                                       *
*                                                                               *
*********************************************************************************
* Copyright (C) 2003 by Mathew Robertson.   All Rights Reserved.                *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
********************************************************************************/
#include <config.h>
#include <fox/fxver.h>
#include <fox/xincs.h>
#include <fox/fxdefs.h>
#include <fox/FXStream.h>
#include <fox/FXString.h>
#include <fox/FXSize.h>
#include <fox/FXPoint.h>
#include <fox/FXRectangle.h>
#include <fox/FXRegistry.h>
#include <fox/FXApp.h>
using namespace FX;
#include "fxexdefs.h"
#include "FXSerialPort.h"
using namespace FXEX;
namespace FXEX {

/* Notes:
 * - to implement for Win32 use CreateFile(),OpenFile(),ReadFile(),CloseHandle()
 */

#ifndef WIN32
#  define DEVICE_STRING  "/dev/ttyS"
#else
#  define DEVICE_SRTING  "\\\\.\\com"
#  define errno GetLastError()
#endif

FXIMPLEMENT(FXSerialPort,FXIOHandle,NULL,0)

// device number specified ctor
FXSerialPort::FXSerialPort(FXApp *a,FXint no,FXObject *tgt,FXSelector sel) : FXIOHandle(a,tgt,sel){
  if(no<0) fxerror("%s: must specify non-negative port number.\n",getClassName());
  portNumber=no;
  }

// use an existing open device
FXSerialPort::FXSerialPort(FXInputHandle port,FXApp *a,FXObject *tgt,FXSelector sel) : FXIOHandle(port,a,tgt,sel){
  portNumber=-1;
  // FIXME need to query device for its name
  }

// dtor
FXSerialPort::~FXSerialPort(){}

// create resources
void FXSerialPort::create(){
  if (state!=FXIOStateNone) return;
  FXTRACE((100,"%s::create %p\n",getClassName(),this));
  FXIOHandle::create();
  if (code!=FXIOStatusOk) { fxerror("%s: create() error\n",getClassName()); return; }
  open();
  }

// save resources
void FXSerialPort::save(FXStream &store) const {
  FXIOHandle::save(store);
  store << portNumber;
  }

// load resources
void FXSerialPort::load(FXStream &store) {
  FXIOHandle::load(store);
  store >> portNumber;
  }

// helper routine, to set the port parameters if the port is open
FXbool FXSerialPort::setPortParameters(){
  if (iohandle == INVALID_HANDLE) { code=EINVAL; return FALSE; }
#ifndef WIN32
  // FIXME: implement for Unix
#else
  FXchar p;
  switch(parity) {
    case ParityEven: p='E'; break;
    case ParityOdd:  p='O'; break;
    default:         p='N';
    }
  FXString params="baud="+ FXStringVal(speed) +" data="+ FXStringVal(databits) +" stop="+ FXStringVal(stopbits) +" parity="+ p;
  DCB devicestate;
  BuildCommDCB(params.text(),&devicestate);
  if (! SetCommState(iohandle,&devicestate)){ code=errno; return FALSE; }
#endif
  return TRUE;
  }

// set the port number
void FXSerialPort::setPortNumber(FXint no){
  if (state==FXIOStateNone) portNumber=no;
  }

// set the baud rate of the port
void FXSerialPort::setSpeed(FXint sp){
  speed=sp;
  setPortParameters();
  }

// set the number of databits
void FXSerialPort::setDataBits(FXint db){
  databits=db;
  setPortParameters();
  }

// set the number of stop bits
void FXSerialPort::setStopBits(FXint sb){
  stopbits=sb;
  setPortParameters();
  }

// set the serial port parity mode
void FXSerialPort::setParity(FXint p){
  parity=p;
  setPortParameters();
  }

// open the serial port
FXbool FXSerialPort::open(){
 
  // make sure we haven't already opened the serial port
  if (state != FXIOStateNone) return FALSE;

  // see what parent says about our handle
  if (!FXIOHandle::open()) return FALSE;

  // open the serial port
  FXString device;
#ifndef WIN32
  device=DEVICE_STRING + FXStringVal(portNumber);
  iohandle=::open(device.text(),O_RDWR|O_EXCL);
#else
  device=DEVICE_STRING + FXStringVal(portNumber+1);
  iohandle=::CreateFile(device.text(),FILE_ALL_ACCESS,0,NULL,OPEN_EXISTING,FILE_ATTRIBUTE_NORMAL,NULL);
#endif
  if (iohandle==INVALID_HANDLE) { code=errno; return FALSE; }
  if (! (setPortParameters() &&
         getApp()->addInput(iohandle,INPUT_READ|INPUT_EXCEPT,this,ID_IOHANDLE) ) ){
    code=EBADR;
    errorClose();
    return FALSE;
    }
  FXTRACE((100,"Serial port opened"));
  state=FXIOStateOk;
  handle(this,FXSEL(SEL_OPENED,0),(void*)this);
  return TRUE;
  }

/*
 * Create a serial port handle which is effectively a duplicate of this one
 */
FXSerialPort* FXSerialPort::duplicate(FXInputHandle newHandle){
  return dynamic_cast<FXSerialPort*>(FXIOHandle::duplicate(newHandle));
  }

// implement the instance creation
FXSerialPort* FXSerialPort::newInstance(FXInputHandle h){
  return new FXSerialPort(h,getApp(),target,message);
  }

}

