/********************************************************************************
*                                                                               *
*               F o n t    r o t a t i o n    o b j e c t                       *
*                                                                               *
*********************************************************************************
* Copyright (C) 2002 by Richard L. Herbst.          All Rights Reserved.        *
*********************************************************************************
* This program is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
********************************************************************************/
#include <config.h>
#include <fox/fxver.h>
#include <fox/xincs.h>
#include <fox/fxdefs.h>
#include <fox/FXStream.h>
#include <fox/FXString.h>
#include <fox/FXSize.h>
#include <fox/FXPoint.h>
#include <fox/FXRectangle.h>
#include <fox/FXRegistry.h>
#include <fox/FXApp.h>
#include <fox/FXDC.h>
using namespace FX;
#include "FXRotatableFont.h"
#include <fox/xincs.h>
using namespace FXEX;
namespace FXEX {

// See P.109 in Prog Supp for Rel 6
#define DPI_DIV   72.27

/***************** FXFontMatrix implementation *********************/
// ctor
FXFontMatrix::FXFontMatrix(FXfloat FontSz){
  setFontSz(FontSz);
  }

// Copy constructor
FXFontMatrix::FXFontMatrix(const FXFontMatrix &M) {
  a=M.a; b=M.b; c=M.c; d=M.d;
  }

// get sign 
FXString FXFontMatrix::getSign(FXfloat x){
  return (x>0.0) ? FXString(" ") : FXString(" ~");
  }

// adjust font size
void FXFontMatrix::setFontSz(FXfloat FontSz) {
  a=FontSz;
  b=0.0;
  c=0.0;
  d=FontSz;
  }

// return string'ised name of font
FXString FXFontMatrix::getXLFDStr(void) {
  const char *fmt = "%6.4f";
  FXString XLFDStr;
  XLFDStr  = getSign(a) + FXStringFormat(fmt, fabs(a));
  XLFDStr += getSign(b) + FXStringFormat(fmt, fabs(b));
  XLFDStr += getSign(c) + FXStringFormat(fmt, fabs(c));
  XLFDStr += getSign(d) + FXStringFormat(fmt, fabs(d));
  return XLFDStr;
  }

// rotation angle
void FXFontMatrix::setRotate(FXfloat T) {
  a=cos(T*(M_PI/180));
  b=sin(T*(M_PI/180));
  c=-sin(T*(M_PI/180));
  d=cos(T*(M_PI/180));
  }

// shear metric
void FXFontMatrix::setShear(FXfloat T) {
  a=1.0;
  b=0.0;
  c=-tan(T*(M_PI/180));
  d=1.0;
  }

// set font to anamorphic
void FXFontMatrix::setAnamorphic(FXfloat a1, FXfloat d1) {
  a=a1;
  b=0.0;
  c=0.0;
  d=d1;
  }

/// set font to mirror about X
void FXFontMatrix::setMirrorX(){
  a=-1.0;
  b=0.0;
  c=0.0;
  d=1.0;
  }

/// set font to mirror about Y
void FXFontMatrix::setMirrorY(){
  a=1.0;
  b=0.0;
  c=0.0;
  d=-1.0;
  }

// M3 = M2 * M1  Note: Order of multiplication is probably important
FXFontMatrix operator * (FXFontMatrix M2, FXFontMatrix M1) {
  FXFontMatrix M;
  M.a = M2.a * M1.a + M2.b * M1.c;
  M.b = M2.a * M1.b + M2.b * M1.d;
  M.c = M2.c * M1.a + M2.d * M1.c;
  M.d = M2.c * M1.b + M2.d * M1.d;
  return M;
  }


/***************** FXRotatableFont implementation *********************/

// construct from font description
FXRotatableFont::FXRotatableFont(FXApp* a,const FXFontDesc& fontdesc,FXfloat ang):FXFont(a,fontdesc){
  angle=ang;
  }

// construct from provided specifications
FXRotatableFont::FXRotatableFont(FXApp* a,const FXString& face,FXuint sz,FXfloat ang,FXuint wt,FXuint sl,FXuint enc,FXuint setw,FXuint h):FXFont(a,face,sz,wt,sl,enc,setw,h) {
  angle=ang;
  }

// Initialise the FXRotatableFont. size=size in points,angle=angle of rotation.
void FXRotatableFont::create() {
#ifdef WIN32
  FXFont::create();
#else
  if (angle<0.01 && angle>=-0.01) FXFont::create();
  else {  
    if(!xid && getApp()->isInitialized()) {

      // Avoid bitmappy text at 90,180,270 degrees for certain fonts
      hints|=FONTHINT_SCALABLE;
      FXString xname=getFontStr();

      // BaseFontStr Ex. = "-adobe-helvetica-medium-r-normal-*-0-[";
      FXString baseFontStr = xname.before('-', 7) + "-0-[";
      FXString strTst = xname.after('-', 9);
      FXString strResol = strTst.before('-', 1);
      pixelMul = FXFloatVal(strResol)/DPI_DIV;

      // EndFontStr  Ex. = "]-75-75-p-0-iso8859-1";
      FXString endFontStr = "]-" + strTst.before('-',3) + "-0-" + strTst.after('-',4);
      FXFontMatrix M(size); 
      matrix.setRotate(angle);
      matrix=matrix*M;

      // set the name of the font to be our new XLFDMatrix name.
      setXName(baseFontStr+matrix.getXLFDStr()+endFontStr);
      FXFont::create();
      }
    }
#endif
  }

// set to new angle
void FXRotatableFont::setAngle(FXfloat ang) {
  if (xid) fxerror("%s: cannot set font angle if font is created (ie you must destroy() it first)\n",getClassName());
  angle=ang;
  }

#ifndef WIN32
// We need to get this info for the character spacing (From: getTextHeight())
FXint FXRotatableFont::getFontAttributes(const FXchar *text, FXuint n) {
  if(font) {
    XCharStruct chst; FXint dir,asc,desc;
    XTextExtents((XFontStruct*)font,text,n,&dir,&asc,&desc,&chst);
    return chst.attributes;
    }
  return 0;
  }

// ask X-server for name of font
FXString FXRotatableFont::getFontStr(void) {
  const char* Family = name.text();
  char FontName[1024];
  char *Match = findmatch(FontName, Family);
  FXString n=FontName;
  return n; 
  }

// set to X based font
void FXRotatableFont::setXName(const FXString& n) {
  name=n;
  encoding=FONTENCODING_DEFAULT;
  setwidth=FONTSETWIDTH_DONTCARE;
  hints=FONTHINT_X11;
  font=NULL;
  }
#endif


// This is where the actual font drawing takes place
void FXRotatableFont::displayText(FXDC& dc,const FXString& str,FXint xstart,FXint ystart) {
   dc.setFont(this);
#ifndef WIN32
  if (angle>0.01 || angle<-0.01) {
    const FXchar *chr = str.text();
    FXint j, Attributes, xpos=xstart, ypos=ystart;
    // draw the individual characters
    for (j = 0; j < str.length(); j++) { 
      dc.drawText(xpos, ypos, chr, 1);
      Attributes = getFontAttributes(chr, 1); // Char spacing, we need this
      xpos+=(FXint)((matrix.a * pixelMul * (FXfloat)Attributes)/1000.0);
      ypos+=(FXint)((-matrix.b * pixelMul * (FXfloat)Attributes)/1000.0); 
      chr++;
      }
    }
  else {
    dc.drawText(xstart,ystart,str.text(),str.length());
    }
#else
  dc.drawText(xstart,ystart,str.text(),str.length());
#endif
  }

}
