/********************************************************************************
*                                                                               *
*                  Child process object/interface                               *
*                                                                               *
*********************************************************************************
* Copyright (C) 2003 by Mathew Robertson.   All Rights Reserved.                *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
*********************************************************************************/
#include <config.h>
#include <fox/fxver.h>
#include <fox/xincs.h>
#include <fox/fxdefs.h>
#include <fox/FXStream.h>
#include <fox/FXString.h>
#include <fox/FXSize.h>
#include <fox/FXPoint.h>
#include <fox/FXRectangle.h>
#include <fox/FXRegistry.h>
#include <fox/FXApp.h>
using namespace FX;
#include "exincs.h"
#include "fxexdefs.h"
#include "FXBaseObject.h"
#include "FXArray.h"
#include "FXIOHandle.h"
#include "FXStdIO.h"
#include "FXProcessManager.h"
#include "FXProcess.h"
using namespace FXEX;
namespace FXEX {

/* Notes:
 * - We could add an 'addChannel()' method so as to provide more communication channels to the
 *   child process?
 */

// map
FXDEFMAP(FXProcess) FXProcessMap[]={
  FXMAPFUNC(SEL_SIGNAL,0,FXProcess::onChildDeath),
  FXMAPFUNC(SEL_CLOSED,0,FXProcess::onTerminate),
  FXMAPFUNC(SEL_IO_READ,FXProcess::ID_STDIN,FXProcess::onStdin),
  FXMAPFUNC(SEL_IO_READ,FXProcess::ID_STDOUT,FXProcess::onStdout),
  FXMAPFUNC(SEL_IO_READ,FXProcess::ID_STDERR,FXProcess::onStderr)
  };
FXIMPLEMENT(FXProcess,FXBaseObject,FXProcessMap,ARRAYNUMBER(FXProcessMap));

// serialisation
FXProcess::FXProcess() : FXBaseObject() {}

// ctor
FXProcess::FXProcess(FXApp* a,FXObject* tgt,FXSelector sel) : FXBaseObject(a,tgt,sel) {
  exitcode=0;
  code=0;
  pid=-1;
  child=FALSE;
  }
 
// ctor
FXProcess::FXProcess(FXApp* a,const FXString& cmd,const FXStringList& args,FXObject* tgt,FXSelector sel) : FXBaseObject(a,tgt,sel) {
  command=cmd;
  arguments=args;
  exitcode=0;
  code=0;
  pid=-1;
  child=FALSE;
  }
 
// dtor
FXProcess::~FXProcess() {
  if (isRunning()) terminate();
  }

// start the child process
// FIXME: we may change this implementation from using pipe()s, to socketpair()s, since this
//        may be more suitable under win32
FXbool FXProcess::start() {
  if (isRunning()) fxerror("%s: Cannot start multiple times.\n",getClassName());

#ifndef WIN32
  FXInputHandle inPipe[2];
  FXInputHandle outPipe[2];
  FXInputHandle errPipe[2];
  if (::pipe(inPipe)<0) fxerror("%s: failed to create stdin pipe\n",getClassName());
  if (::pipe(outPipe)<0) fxerror("%s: failed to create stdout pipe\n",getClassName());
  if (::pipe(errPipe)<0) fxerror("%s: failed to create stderr pipe\n",getClassName());
  pid = ::fork();

  // fork failed
  if (pid < 0) { code=errno; return FALSE; }

  // in parent process
  if (pid) {
    ::close(inPipe[1]);
    ::close(outPipe[1]);
    ::close(errPipe[1]);
    in = new FXIOHandle(inPipe[0],getApp(),this,ID_STDIN);
    out = new FXIOHandle(outPipe[0],getApp(),this,ID_STDOUT);
    err = new FXIOHandle(errPipe[0],getApp(),this,ID_STDERR);
    in->create();
    out->create();
    err->create();
    if (in->status()!=FXIOStatusOk || out->status()!=FXIOStatusOk || err->status()!=FXIOStatusOk)
      fxerror("%s: failure creating stdio connections.\n",getClassName());
    in->nonBlocking(TRUE);
    out->nonBlocking(TRUE);
    err->nonBlocking(TRUE);
    getProcessManager()->add(this);
    return TRUE;
    }
  
  // in child process
  child=TRUE;
  ::close(inPipe[0]);
  ::close(outPipe[0]);
  ::close(errPipe[0]);
  ::close(0);
  ::close(1);
  ::close(2);
  ::dup2(inPipe[1],0);
  ::dup2(outPipe[1],1);
  ::dup2(errPipe[1],2);
  ::close(inPipe[1]);
  ::close(outPipe[1]);
  ::close(errPipe[1]);

  // if supplied command string, replace child process image, with new process image.
  if (command.length()){ 
    char *args[arguments.size()+1];
    FXint i=0;
    for (; i< arguments.size(); i++){
      args[i] = (char*) arguments[i].text();
      }
    args[i]=NULL;
    ::execvp(command.text(),args);
    fxerror("%s: child exited prematurely (errno: %i).\n",getClassName(),errno);
    }

  // use this new'ly duplicated process
  else {
    in = new FXStdin(getApp(),this,ID_STDIN);
    out = new FXStdout(getApp(),this,ID_STDOUT);
    err = new FXStderr(getApp(),this,ID_STDERR);
    in->create();
    out->create();
    err->create();
    if (in->status()!=FXIOStatusOk || out->status()!=FXIOStatusOk || err->status()!=FXIOStatusOk)
      fxerror("%s: failed creating stdio connections.\n",getClassName());
    in->nonBlocking(TRUE);
    out->nonBlocking(TRUE);
    err->nonBlocking(TRUE);
    pid=fxgetpid();
    return TRUE;
    }

#else
#endif
  return TRUE;
  }

// is the child process running
FXbool FXProcess::isRunning() const {
  return (pid > -1) ? TRUE : FALSE;
  }

// terminate the child process, using the specified signal code
FXbool FXProcess::terminate(FXint code) {
  if (!isRunning()) return FALSE;
#ifndef WIN32
  FXint result = ::kill(pid,code);
  if (result != 0) { code=errno; return FALSE; }
#else
  FXint result = TerminateProcess(phnd,code);
  if (result != 0) { code=GetLastError(); return FALSE; }
#endif
  pid=-1;
  return TRUE;
  }

// return handle to the process manager
FXProcessManager* FXProcess::getProcessManager() {
  return FXProcessManager::instance();
  }

// Either:
// - read some binary data from parent process, via child's stdin
// - read some binary data from child process, via child's stdout
FXint FXProcess::read(FXuchar* data,FXuint size,FXbool outOfBand) {
  if (child) return in->read(data,size);
  else {
    if (outOfBand) return err->read(data,size);
    else return out->read(data,size);
    }
  }

// Either:
// - send some binary data to the child process, via child's stdin
// - send some binary data to the parent process, via child's stdout
void FXProcess::write(FXuchar *data,FXuint size,FXbool outOfBand) {
  if (child) {
    if (outOfBand) err->write(data,size);
    else out->write(data,size);
    }
  else in->write(data,size);
  }

// send some text to the child process
void FXProcess::setText(const FXString& message){
  in->setText(message);
  }

// handle a child death signal
long FXProcess::onChildDeath(FXObject*,FXSelector,void *ptr) {
  getProcessManager()->remove(this);
  pid=-1;
  exitcode = (FXint)ptr;
  handle(this,FXSEL(SEL_CLOSED,0),ptr);
  delete in;
  delete out;
  delete err;
  in=out=err=(FXIOHandle*)-1;
  return 1;
  }

// handle data from stdin from parent process
long FXProcess::onStdin(FXObject*,FXSelector,void *ptr) {
  return target && target->handle(this,FXSEL(SEL_INPUT,message),ptr);
  }

// handle data from stdout from child process
long FXProcess::onStdout(FXObject*,FXSelector,void *ptr) {
  return target && target->handle(this,FXSEL(SEL_OUTPUT,message),ptr);
  }

// handle data from stderr from child process
long FXProcess::onStderr(FXObject*,FXSelector,void *ptr) {
  return target && target->handle(this,FXSEL(SEL_ERROR,message),ptr);
  }

// handle child process termination
long FXProcess::onTerminate(FXObject*,FXSelector,void *ptr) {
  return target && target->handle(this,FXSEL(SEL_CLOSED,message),ptr);
  }

}

