/********************************************************************************
*                                                                               *
*                     ODBC query connector                                      *
*                                                                               *
*********************************************************************************
* Copyright (C) 2003 by Mathew Robertson.       All Rights Reserved.            *
* Copyright (C) 2003 by Giancarlo Formicuccia.  All Rights Reserved.            *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
*********************************************************************************/
#include <config.h>
#include <fox/fxver.h>
#include <fox/xincs.h>
#include <fox/fxdefs.h>
#include <fox/FXStream.h>
#include <fox/FXString.h>
#include <fox/FXWString.h>
#include <fox/FXCharset.h>
#include <fox/FXSize.h>
#include <fox/FXPoint.h>
#include <fox/FXRectangle.h>
#include <fox/FXRegistry.h>
#include <fox/FXApp.h>
using namespace FX;
#include "exincs.h"
#include "fxexdefs.h"
#include "FXArray.h"
#include "FXDatabaseField.h"
#include "FXODBCInterface.h"
#include "FXODBCQuery.h"

/*
 * myodbc gift with OPTION&1==0
 */
#define HACK_MAX_VARCHAR_SIZE

/*
 * Try converting unknown sql types to string
 */
#define HACK_TYPE_UNKNOWN

/*
 * Workaround broken SQLDescribeCol/SQLColAttribute not working with bookmark column
 */
#define HACK_DESC_BOOKMARK

/*
 * Another myodbc gift: SQLBulkOperations out of specs
 */
#define HACK_SQL_COLUMN_IGNORE

/*
 * n-th myodbc gift with OPTION&32!=0
 */
#define HACK_SQL_SCROLL_OPTIONS


#ifdef DEBUG
/* Simple code for overflow checking */
#define CK_ADD  sizeof(unsigned int)
#define CK_SIGN (0xa7fa11ed)
#define CK_ADDR(ptr, len) ((unsigned int *) ((char *)(ptr)+(len)))
#define CK_FILL(ptr, len) do { *CK_ADDR(ptr, len) = CK_SIGN; } while(0)
#define CK_OFL(ptr, len) (*CK_ADDR(ptr, len)==CK_SIGN)
#else
#define CK_ADD (0)
#define CK_FILL(ptr, len) do { } while(0)
#define CK_OFL(ptr, len) (TRUE)
#endif

using namespace FXEX;
namespace FXEX {

static FXuint cursor_index_name = 0;

FXDEFMAP(FXODBCQuery) FXODBCQueryMap[] = {
  FXMAPFUNC(SEL_EVENT,    FXDatabaseQuery::ID_DISCONNECT,   FXODBCQuery::onDisconnect),
  FXMAPFUNC(SEL_EVENT,    FXDatabaseQuery::ID_CONNECT,      FXODBCQuery::onConnect)
};

FXIMPLEMENT(FXODBCQuery, FXDatabaseQuery, FXODBCQueryMap, ARRAYNUMBER(FXODBCQueryMap));

FXODBCQuery::FXODBCQuery(FXODBCInterface *dbi, FXObject *tgt, FXSelector sel): FXDatabaseQuery(dbi, tgt, sel) {
  hStmt = NULL;
  UseBookmarks = FALSE;
  bookmarkField.raw = NULL;
  capabilities = dbi ? &dbi->getCapabilities() : NULL;
}

FXODBCQuery::~FXODBCQuery() {
  FXODBCQuery::deleteFields();
  FreeHandle();
  if(database) database->handle(this, MKUINT(FXDatabaseInterface::ID_QRYDETACH, SEL_EVENT), NULL);
}

long FXODBCQuery::onDisconnect(FXObject *sender, FXSelector sel, void *data) {
  FXDatabaseQuery::onDisconnect(sender, sel, data);
  deleteFields();
  FreeHandle();
  capabilities = NULL;
  return 1;
}

long FXODBCQuery::onConnect(FXObject *sender, FXSelector sel, void *data) {
  FXDatabaseQuery::onConnect(sender, sel, data);
  FXASSERT(database);
  capabilities = &((FXODBCInterface *)database)->getCapabilities();
  return 1;
}

void FXODBCQuery::save(FXStream &store) const {
  FXODBCQuery::save(store);
  /* TODO... */
}

void FXODBCQuery::load(FXStream &store) {
  FXODBCQuery::load(store);
  /* TODO... */
}

void FXODBCQuery::FreeHandle() {
  if(hStmt) {
    if(UseBookmarks) {
      SQLBindCol(hStmt, 0, bookmarkField.cType, NULL, 0, NULL);
      SQLSetStmtAttr(hStmt, SQL_ATTR_FETCH_BOOKMARK_PTR, NULL,
          bookmarkField.fieldSize);
    }
    SQLFreeStmt(hStmt, SQL_UNBIND);
    SQLFreeStmt(hStmt, SQL_CLOSE);
    SQLFreeHandle(SQL_HANDLE_STMT, hStmt);
    hStmt = NULL;
  }
}

void FXODBCQuery::beforeBulkUpdate() {
#ifdef HACK_SQL_COLUMN_IGNORE
  FXuint nCols = rawFields.no();
  FXuint x;
  SQLRETURN res;
  for(x=0; x<nCols; ++x) {
    if(rawFields[x].slen==SQL_COLUMN_IGNORE) {
      res = SQLBindCol(hStmt, x+1, rawFields[x].cType, NULL, 0, NULL);
      if(!VALID(res)) FXODBCInterface::odbcThrow(SQL_HANDLE_STMT, hStmt);
    }
  }
#endif
}

void FXODBCQuery::afterBulkUpdate() {
#ifdef HACK_SQL_COLUMN_IGNORE
  FXuint nCols = rawFields.no();
  FXuint x;
  SQLRETURN res;
  for(x=0; x<nCols; ++x) {
    if(rawFields[x].slen==SQL_COLUMN_IGNORE) {
      res = SQLBindCol(hStmt, x+1, rawFields[x].cType, rawFields[x].raw,
        rawFields[x].fieldSize, &rawFields[x].slen);
      if(!VALID(res)) FXODBCInterface::odbcThrow(SQL_HANDLE_STMT, hStmt);
    }
  }
#endif
}

/*
  Retrieve data from FXDatabaseFields, xlat to odbc,
  set odbcField->slen to SQL_ROW_IGNORE if this field doesn't need
  to be updated, or to the field db size if modified.
*/
FXbool FXODBCQuery::var2odbc(FXint col) {
  FXodbcField *odbcField = &rawFields[col];
  FXDatabaseField *dbField = fields[col];
  FXdbEvent ev;
  FXVariant v;
  FXint l;
  FXbool sign;
  SQLRETURN ret;
  void *raw = odbcField->raw;
  ev.type = dbAskData;
  ev.state = state;
  ev.dbi = database;
  ev.dbq = this;
  ev.dbf = NULL;
  ev.data.setType(FXVariant::DT_UNASSIGNED);
  dbField->handle(this, MKUINT(FXDatabaseField::ID_EVENT, SEL_EVENT), &ev);
  v = ev.data;
  if(v.isUnassigned()) {
  /* field not modified */
    odbcField->slen = SQL_COLUMN_IGNORE;
    return TRUE;
  }
  if(v.isNull()) {
    FXASSERT(odbcField->type.nullable);
    odbcField->slen = SQL_NULL_DATA;
    return TRUE;
  } else {
    l = v.getSize();
    sign = odbcField->withSign;
    if(!odbcField->type.maxSize) {
      if(odbcField->fieldSize<l) {
        FXTRACE((20, "Column %d needs resizing\n", col));
        odbcField->fieldSize = l;
        FXRESIZE(&odbcField->raw, FXuchar, odbcField->fieldSize+CK_ADD);
        raw = odbcField->raw;
        CK_FILL(raw, odbcField->fieldSize);
        ret = SQLBindCol(hStmt, col+1, odbcField->cType, raw,
        odbcField->fieldSize, &odbcField->slen);
        if(!VALID(ret)) FXODBCInterface::odbcThrow(SQL_HANDLE_STMT, hStmt);
        odbcField->slen = odbcField->fieldSize;
      }
    }
    if(l>odbcField->fieldSize) {
      fxwarning("Warning: truncating data\n");
      l = odbcField->fieldSize;
    }
    odbcField->slen = l;
    switch(odbcField->odbcType) {
      case SQL_VARCHAR:
      case SQL_LONGVARCHAR:
      case SQL_CHAR:
        strncpy((char *) raw, v.asFXString().text(), odbcField->fieldSize);
        if(odbcField->fieldSize) *((char *) raw+odbcField->fieldSize-1) = '\0';
        break;
      case SQL_BIGINT:
        if(sign) *(FXlong *) raw = v; else *(FXulong *) raw = v;
        break;
      case SQL_INTEGER:
        if(sign) *(FXint *) raw = v; else *(FXuint *) raw = v;
        break;
      case SQL_SMALLINT:
        if(sign) *(FXshort *) raw = v; else *(FXushort *) raw = v;
        break;
      case SQL_TINYINT:
        if(sign) *(FXchar *) raw = v; else *(FXuchar *) raw = v;
        break;
      case SQL_REAL:
        *(FXfloat *) raw = v;
        break;
      case SQL_FLOAT:
      case SQL_DOUBLE:
        *(FXdouble *) raw = v;
        break;
      case SQL_BIT:
        *(FXint *) raw = ((bool) v) ? 1 : 0;
        break;
      case SQL_NUMERIC:
      case SQL_DECIMAL:
      case SQL_DATE: // Same as SQL_DATETIME
      case SQL_TIME: // Same as SQL_INTERVAL
      case SQL_TIMESTAMP:
      case SQL_BINARY:
      case SQL_VARBINARY:
      case SQL_LONGVARBINARY:
      case SQL_GUID:
      default:
#ifdef HACK_TYPE_UNKNOWN
        strncpy((char *) raw, v.asFXString().text(), odbcField->fieldSize);
        if(odbcField->fieldSize) *((char *) raw+odbcField->fieldSize-1) = '\0';
#else
        fxwarning("FIXME!\n");
        odbcField->slen = SQL_ROW_IGNORE;
        return FALSE;
#endif
    }
  }
  FXASSERT(CK_OFL(raw, odbcField->fieldSize));
  return TRUE;
}

/*
  xlat odbc data to FXVariant, then store the result
  into FXDatabaseField
*/
FXbool FXODBCQuery::odbc2var(FXint col) {
  FXodbcField *odbcField = &rawFields[col];
  FXDatabaseField *dbField = fields[col];
  FXVariant v;
  FXbool ret = TRUE;
  FXbool sign;
  void *raw = odbcField->raw;
  FXASSERT(CK_OFL(raw, odbcField->fieldSize));
  if(odbcField->slen==SQL_NO_DATA) {
    v.setType(FXVariant::DT_UNASSIGNED);
  } else if(odbcField->slen==SQL_NULL_DATA) {
    FXASSERT(odbcField->type.nullable);
    v.setType(FXVariant::DT_NULL);
  } else {
    sign = odbcField->withSign;
    switch(odbcField->odbcType) {
      case SQL_VARCHAR:
      case SQL_LONGVARCHAR:
      case SQL_CHAR:
        if(odbcField->fieldSize) *((FXchar *)raw+odbcField->fieldSize-1) = '\0';
        if(odbcField->slen>0) {
          if(odbcField->slen<odbcField->fieldSize) *((FXchar *)raw+odbcField->slen) = '\0';
          v = FXString((FXchar *) raw);
        } else v = FXString("");
        break;
      case SQL_BIGINT:
        if(sign) v = *(FXlong *) raw; else v = *(FXulong *) raw;
        break;
      case SQL_INTEGER:
        if(sign) v = *(FXint *) raw; else v = *(FXuint *) raw;
        break;
      case SQL_SMALLINT:
        if(sign) v = *(FXshort *) raw; else v = *(FXushort *) raw;
        break;
      case SQL_TINYINT:
        if(sign) v = *(FXchar *) raw; else v = *(FXuchar *) raw;
        break;
      case SQL_REAL:
        v = *(FXfloat *) raw;
        break;
      case SQL_FLOAT:
      case SQL_DOUBLE:
        v = *(FXdouble *) raw;
        break;
      case SQL_BIT:
        v = 0!=*(FXint *) raw;
        break;
      case SQL_NUMERIC:
      case SQL_DECIMAL:
      case SQL_DATE: // Same as SQL_DATETIME
      case SQL_TIME: // Same as SQL_INTERVAL
      case SQL_TIMESTAMP:
      case SQL_BINARY:
      case SQL_VARBINARY:
      case SQL_LONGVARBINARY:
      case SQL_GUID:
      default:
#ifdef HACK_TYPE_UNKNOWN
        if(odbcField->fieldSize) *((FXchar *)raw+odbcField->fieldSize-1) = '\0';
        if(odbcField->slen>0) {
          if(odbcField->slen<odbcField->fieldSize) *((FXchar *)raw+odbcField->slen) = '\0';
          v = FXString((FXchar *) raw);
        } else v = FXString("");
        break;
#else
        fxwarning("FIXME!\n");
        v.setType(FXVariant::DT_UNASSIGNED);
        ret = FALSE;
#endif
    }
  }
  dbField->handle(this, MKUINT(FXDatabaseField::ID_SETDATA, SEL_EVENT), &v);
  return ret;
}

/*
  xlat odbc data, then notify FXDatabaseFields the data has changed
*/
void FXODBCQuery::notifyMove(FXuint) {
  FXint x;
  FXint cnt = rawFields.no();
  lastValidPos = currentPos();
  for(x=0; x<cnt; ++x) odbc2var(x);
  FXDatabaseQuery::notifyMove(lastValidPos);
}

void FXODBCQuery::AllocHandle(FXbool &readOnly) {
  SQLRETURN res;
  SQLUINTEGER cursor_type;
  SQLINTEGER unused;
  FXuint sql_scroll_options;
  if(!database) FXDatabaseInterface::dbThrow("Unable to allocate resource", -1);
  checkOpen(FALSE);
  if(capabilities->sql_data_source_read_only && !readOnly) {
    fxwarning("Warning: Recordset is read-only\n");
    readOnly = TRUE;
  }
  database->handle(this, MKUINT(FXODBCInterface::ID_ALLOCHANDLE, SEL_EVENT), &hStmt);
  sql_scroll_options = capabilities->sql_scroll_options;
#ifdef HACK_SQL_SCROLL_OPTIONS
  if(capabilities->sql_dynamic_cursor_attributes1) sql_scroll_options |= SQL_SO_DYNAMIC;
  if(capabilities->sql_keyset_cursor_attributes1) sql_scroll_options |= SQL_SO_KEYSET_DRIVEN;
  if(capabilities->sql_static_cursor_attributes1) sql_scroll_options |= SQL_SO_STATIC;
  if(capabilities->sql_forward_only_cursor_attributes1) sql_scroll_options |= SQL_SO_FORWARD_ONLY;
#endif
  try {
/* Use scrollable cursor if available */
    SQLSetStmtAttr(hStmt, SQL_ATTR_CURSOR_SCROLLABLE, (SQLPOINTER) SQL_SCROLLABLE, 0);
    if(readOnly) {
      if(sql_scroll_options & SQL_SO_STATIC) {
        res = SQLSetStmtAttr(hStmt, SQL_ATTR_CURSOR_TYPE, (SQLPOINTER) SQL_CURSOR_STATIC, 0);
      } else if(sql_scroll_options & (SQL_SO_KEYSET_DRIVEN|SQL_SO_MIXED)) {
        res = SQLSetStmtAttr(hStmt, SQL_ATTR_CURSOR_TYPE, (SQLPOINTER) SQL_CURSOR_KEYSET_DRIVEN, 0);
      } else if(sql_scroll_options & SQL_SO_DYNAMIC) {
        res = SQLSetStmtAttr(hStmt, SQL_ATTR_CURSOR_TYPE, (SQLPOINTER) SQL_CURSOR_DYNAMIC, 0);
      } else if (sql_scroll_options & SQL_SO_FORWARD_ONLY) {
        res = SQLSetStmtAttr(hStmt, SQL_ATTR_CURSOR_TYPE, (SQLPOINTER) SQL_CURSOR_FORWARD_ONLY, 0);
      } else res = SQLSetStmtAttr(hStmt, SQL_ATTR_CURSOR_TYPE, (SQLPOINTER) SQL_CURSOR_DYNAMIC, 0);
    } else {
      if(sql_scroll_options & (SQL_SO_KEYSET_DRIVEN|SQL_SO_MIXED)) {
        res = SQLSetStmtAttr(hStmt, SQL_ATTR_CURSOR_TYPE, (SQLPOINTER) SQL_CURSOR_KEYSET_DRIVEN, 0);
      } else if(sql_scroll_options & SQL_SO_DYNAMIC) {
        res = SQLSetStmtAttr(hStmt, SQL_ATTR_CURSOR_TYPE, (SQLPOINTER) SQL_CURSOR_DYNAMIC, 0);
      } else if(sql_scroll_options & SQL_SO_STATIC) {
        res = SQLSetStmtAttr(hStmt, SQL_ATTR_CURSOR_TYPE, (SQLPOINTER) SQL_CURSOR_STATIC, 0);
      } else if (sql_scroll_options & SQL_SO_FORWARD_ONLY) {
        res = SQLSetStmtAttr(hStmt, SQL_ATTR_CURSOR_TYPE, (SQLPOINTER) SQL_CURSOR_FORWARD_ONLY, 0);
      } else res = SQLSetStmtAttr(hStmt, SQL_ATTR_CURSOR_TYPE, (SQLPOINTER) SQL_CURSOR_DYNAMIC, 0);
    }
    if(!VALID(res)) fxwarning("Warning: unable to set a suitable cursor type\n");
/* Use bookmarks if available */
    res = SQLSetStmtAttr(hStmt, SQL_ATTR_USE_BOOKMARKS, (SQLPOINTER) SQL_UB_VARIABLE, 0);
    UseBookmarks = VALID(res);
/* Grab the chosen cursor type */
    res = SQLGetStmtAttr(hStmt, SQL_ATTR_CURSOR_TYPE, &cursor_type, sizeof(cursor_type), &unused);
    if(!VALID(res)) FXODBCInterface::odbcThrow(SQL_HANDLE_STMT, hStmt);
    this->cursor_type = cursor_type;
    switch(cursor_type) {
      case SQL_CURSOR_FORWARD_ONLY:
        cursor_attr1 = capabilities->sql_forward_only_cursor_attributes1;
        cursor_attr2 = capabilities->sql_forward_only_cursor_attributes2;
        break;
      case SQL_CURSOR_STATIC:
        cursor_attr1 = capabilities->sql_static_cursor_attributes1;
        cursor_attr2 = capabilities->sql_static_cursor_attributes2;
        break;
      case SQL_CURSOR_KEYSET_DRIVEN:
        cursor_attr1 = capabilities->sql_keyset_cursor_attributes1;
        cursor_attr2 = capabilities->sql_keyset_cursor_attributes2;
        break;
      case SQL_CURSOR_DYNAMIC:
        cursor_attr1 = capabilities->sql_dynamic_cursor_attributes1;
        cursor_attr2 = capabilities->sql_dynamic_cursor_attributes2;
        break;
      default:
        FXDatabaseInterface::dbThrow("Unknown cursor type", -1);
    }

    if(!readOnly && !(cursor_attr1 & (SQL_CA1_POS_UPDATE|SQL_CA1_POS_DELETE|
        SQL_CA1_BULK_ADD|SQL_CA1_BULK_UPDATE_BY_BOOKMARK|SQL_CA1_BULK_DELETE_BY_BOOKMARK))) {
      fxwarning("Warning: unable to write on this cursor\n");
      readOnly = TRUE;
    }

/* Set the cursor concurrency if supported */
    if(cursor_attr2 & (SQL_CA2_LOCK_CONCURRENCY|SQL_CA2_OPT_ROWVER_CONCURRENCY|
        SQL_CA2_OPT_VALUES_CONCURRENCY|SQL_CA2_READ_ONLY_CONCURRENCY)) {
      if(readOnly) {
        FXASSERT(cursor_attr2 & SQL_CA2_READ_ONLY_CONCURRENCY);
        SQLSetStmtAttr(hStmt, SQL_ATTR_CONCURRENCY, (SQLPOINTER) SQL_CONCUR_READ_ONLY, 0);
      } else {
        if(cursor_attr2 & SQL_CA2_OPT_VALUES_CONCURRENCY) {
          res = SQLSetStmtAttr(hStmt, SQL_ATTR_CONCURRENCY, (SQLPOINTER) SQL_CONCUR_VALUES, 0);
        } else if(cursor_attr2 & SQL_CA2_OPT_ROWVER_CONCURRENCY) {
          res = SQLSetStmtAttr(hStmt, SQL_ATTR_CONCURRENCY, (SQLPOINTER) SQL_CONCUR_ROWVER, 0);
        } else if(cursor_attr2 & SQL_CA2_LOCK_CONCURRENCY) {
          res = SQLSetStmtAttr(hStmt, SQL_ATTR_CONCURRENCY, (SQLPOINTER) SQL_CONCUR_LOCK, 0);
        } else {
          res = SQLSetStmtAttr(hStmt, SQL_ATTR_CONCURRENCY, (SQLPOINTER) SQL_CONCUR_READ_ONLY, 0);
          fxwarning("Warning: unable to set a suitable concurrency value\n");
        }
        FXASSERT(VALID(res));
      }
    }
  } catch(...) {
    SQLFreeHandle(SQL_HANDLE_STMT, hStmt);
    hStmt = NULL;
    UseBookmarks = FALSE;
    throw;
  }
/* This should help myodbc v 2.x */
  SQLSetCursorName(hStmt, (SQLCHAR *) FXStringVal(cursor_index_name++).text(), SQL_NTS);
}

/*
  Cleanup of various field-related objects
*/
void FXODBCQuery::deleteFields() {
  FXint x;
  FXint cnt = rawFields.no();
  for(x=0; x<cnt; ++x) FXFREE(&rawFields[x].raw);
  rawFields.clear();
  FXFREE(&bookmarkField.raw);
  FXDatabaseQuery::deleteFields();
}

void FXODBCQuery::Open(const FXString &q, FXbool readOnly) {
  SQLRETURN res = SQL_NO_DATA;
  SQLSMALLINT nCols;
  FXDatabaseField *f;
  FXodbcField rawf;
  SQLSMALLINT x;
  SQLSMALLINT unused;
  SQLSMALLINT dataType;
  SQLSMALLINT decimalDigits;
  SQLSMALLINT nullable;
  SQLINTEGER val;
  BOOL uns;
  SQLUINTEGER fieldSize;
  SQLUINTEGER allocSize;
  FXchar name[4096];
  FXint fxType;
  SQLINTEGER cType;
  FXint start;

  AllocHandle(readOnly);
  bookmarkReady = FALSE;
  lastValidPos = 0;
  try {
    res = SQLPrepare(hStmt, (SQLCHAR *)q.text(), SQL_NTS);
    if(!VALID(res)) FXODBCInterface::odbcThrow(SQL_HANDLE_STMT, hStmt);
    res = SQLExecute(hStmt);
    if(!VALID(res)) FXODBCInterface::odbcThrow(SQL_HANDLE_STMT, hStmt);
    res = SQLNumResultCols(hStmt, &nCols);
    if(!VALID(res)) FXODBCInterface::odbcThrow(SQL_HANDLE_STMT, hStmt);
    deleteFields();
    start = UseBookmarks ? -1 : 0;
    for(x=start; x<nCols; ++x) {
      /* General field info... */
      if(x<0) {
        res = SQLDescribeCol(hStmt, 0, (SQLCHAR *) name, sizeof(name), &unused,
          &dataType, &fieldSize, &decimalDigits, &nullable);
#ifdef HACK_DESC_BOOKMARK
        if(!VALID(res)) {
          strcpy(name, "bookmark");
          dataType = SQL_INTEGER;
          fieldSize = sizeof(SQLINTEGER);
          decimalDigits = 0;
          nullable = 0;
          uns = SQL_TRUE;
        } else {
          res = SQLColAttribute(hStmt, 0, SQL_DESC_UNSIGNED, NULL, 0, NULL, &uns);
          if(!VALID(res)) uns = SQL_TRUE;
        }
#else
        if(!VALID(res)) FXODBCInterface::odbcThrow(SQL_HANDLE_STMT, hStmt);
        res = SQLColAttribute(hStmt, 0, SQL_DESC_UNSIGNED, NULL, 0, NULL, &uns);
        if(!VALID(res)) uns = SQL_TRUE;
#endif
        if(dataType==SQL_INTEGER) cType = uns ? SQL_C_ULONG : SQL_C_SLONG;
        else {
          FXASSERT(dataType==SQL_BINARY);
          cType = SQL_C_BINARY;
        }
        bookmarkField.type.maxSize = (FXuint) fieldSize;
        if(fieldSize<sizeof(SQLINTEGER)) fieldSize = sizeof(SQLINTEGER);
        bookmarkField.type.name = name;
        bookmarkField.type.index = -1;
        bookmarkField.type.fxType = FXVariant::DT_RAW;
        bookmarkField.cType = cType;
        bookmarkField.fieldSize = fieldSize;
        bookmarkField.withSign = (uns==SQL_FALSE);
        bookmarkField.type.nullable = FALSE;
        bookmarkField.odbcType = dataType;
        bookmarkField.terminatorLength = 0;
        bookmarkField.type.varSize = FALSE;
        bookmarkField.type.counter = FALSE;
        FXMALLOC(&bookmarkField.raw, FXchar, bookmarkField.fieldSize);
        memset(bookmarkField.raw, 0, bookmarkField.fieldSize);
        res = SQLBindCol(hStmt, 0, bookmarkField.cType, bookmarkField.raw,
        bookmarkField.fieldSize, &bookmarkField.slen);
        if(VALID(res) && (cursor_attr1 & SQL_CA1_BOOKMARK)) {
          res = SQLSetStmtAttr(hStmt, SQL_ATTR_FETCH_BOOKMARK_PTR, bookmarkField.raw,
            bookmarkField.fieldSize);
        }
        if(!VALID(res)) {
          fxwarning("Bookmark disabled\n");
          UseBookmarks = FALSE;
          SQLBindCol(hStmt, 0, bookmarkField.cType, NULL, 0, NULL);
        } else {
          FXTRACE((20, "Bookmark field enabled\n"));
        }
        continue;
      }
      res = SQLDescribeCol(hStmt, x+1, (SQLCHAR *) name, sizeof(name), &unused,
        &dataType, &fieldSize, &decimalDigits, &nullable);
      if(!VALID(res)) FXODBCInterface::odbcThrow(SQL_HANDLE_STMT, hStmt);
      res = SQLColAttribute(hStmt, x+1, SQL_DESC_UNSIGNED, NULL, 0,
        NULL, &uns);
      if(!VALID(res)) uns = SQL_FALSE;
      rawf.withSign = (uns==SQL_FALSE);
      res = SQLColAttribute(hStmt, x+1, SQL_DESC_AUTO_UNIQUE_VALUE, NULL, 0,
        NULL, &val);
      if(!VALID(res)) val = SQL_FALSE;
      rawf.type.counter = (val==SQL_TRUE);
      res = SQLColAttribute(hStmt, x+1, SQL_DESC_UPDATABLE, NULL, 0,
        NULL, &val);
      if(!VALID(res) || val==SQL_ATTR_READWRITE_UNKNOWN) val = SQL_ATTR_WRITE;
      rawf.type.readOnly = rawf.type.counter || (val==SQL_ATTR_READONLY);
      name[sizeof(name)-1] = '\0';
      rawf.type.name = name;
      rawf.type.index = (FXint) x;
      rawf.type.nullable = (nullable!=0) || (capabilities->sql_non_nullable_columns==SQL_NNC_NULL);
      rawf.type.varSize = FALSE;
      rawf.odbcType = dataType;
      rawf.decimalDigits = decimalDigits;
      rawf.terminatorLength = 0;
      cType = SQL_C_DEFAULT;
      allocSize = fieldSize;
      switch(dataType) {
/*
  Type-conversion saga.
  We have three data types here:
  rawf.odbcType: It's the sql data type as reported from the driver
  rawf.cType: Each sql type can be converted to a specific "c" type; we choose the most appropiate
  rawf.type.fxType: The fox type equivalent to cType, as FXVariant::DT_*. This is the only type applications will see.
 */
        case SQL_VARCHAR:
        case SQL_LONGVARCHAR:
          rawf.type.varSize = TRUE;
#ifdef HACK_MAX_VARCHAR_SIZE
          fieldSize = 0;
#endif
        case SQL_CHAR:
          cType = SQL_C_CHAR;
          fxType = FXVariant::DT_FXSTRING;
          rawf.terminatorLength = sizeof(FXchar);
          break;
        case SQL_BIGINT:
          cType = uns ? SQL_C_UBIGINT : SQL_C_SBIGINT;
          fxType = uns ? FXVariant::DT_FXULONG : FXVariant::DT_FXLONG;
          fieldSize = sizeof(FXlong);
          allocSize = fieldSize;
          FXASSERT(fieldSize==sizeof(SQLBIGINT));
          break;
        case SQL_INTEGER:
          cType = uns ? SQL_C_ULONG : SQL_C_SLONG;
          fxType = uns ? FXVariant::DT_FXUINT : FXVariant::DT_FXINT;
          fieldSize = sizeof(FXint);
          allocSize = fieldSize;
          FXASSERT(fieldSize==sizeof(SQLINTEGER));
          break;
        case SQL_SMALLINT:
          cType = uns ? SQL_C_USHORT : SQL_C_SSHORT;
          fxType = uns ? FXVariant::DT_FXUSHORT : FXVariant::DT_FXSHORT;
          fieldSize = sizeof(FXshort);
          allocSize = fieldSize;
          FXASSERT(fieldSize==sizeof(SQLSMALLINT));
          break;
        case SQL_TINYINT:
          cType = uns ? SQL_C_UTINYINT : SQL_C_STINYINT;
          fxType = uns ? FXVariant::DT_FXUCHAR : FXVariant::DT_FXCHAR;
          fieldSize = sizeof(FXchar);
          allocSize = fieldSize;
          FXASSERT(fieldSize==sizeof(SQLCHAR));
          break;
        case SQL_REAL:
          cType = SQL_C_FLOAT;
          fxType = FXVariant::DT_FXFLOAT;
          fieldSize = sizeof(FXfloat);
          allocSize = fieldSize;
          FXASSERT(fieldSize==sizeof(SQLREAL));
          break;
        case SQL_FLOAT: /* Yes, it is... */
        case SQL_DOUBLE:
          cType = SQL_C_DOUBLE;
          fxType = FXVariant::DT_FXDOUBLE;
          fieldSize = sizeof(FXdouble);
          allocSize = fieldSize;
          FXASSERT(fieldSize==sizeof(SQLDOUBLE));
          break;
        case SQL_BIT:
//          cType = SQL_C_BIT;
          cType = SQL_C_SLONG;
          fxType = FXVariant::DT_BOOL;
          fieldSize = sizeof(FXint); /* CHECKME: dunno if it is BOOL(==int) or SQLCHAR(==uns char) */
          allocSize = fieldSize;
          break;
#if 0
/* The following types are TODO */
        case SQL_NUMERIC:
          cType = SQL_C_NUMERIC;
//          fxType = FXVariant::DT_TODO;
          fieldSize = sizeof(SQL_NUMERIC_STRUCT);
          allocSize = fieldSize;
        case SQL_DECIMAL:
        /* dunno what's this...it returns SQLCHAR* */
          cType = SQL_C_CHAR;
//          fxType = FXVariant::DT_FXSTRING (not really true I guess);
//        fieldSize = default?
        case SQL_DATE: // Same as SQL_DATETIME
          cType = SQL_C_TYPE_DATE;
//          fxType = FXVariant::DT_TODO;
          fieldSize = sizeof(SQL_DATE_STRUCT);
        case SQL_TIME: // Same as SQL_INTERVAL
#if 0 /* if SQL_INTERVAL */
          cType = SQL_C_INTERVAL;
          fieldSize = sizeof(SQL_INTERVAL_STRUCT);
          allocSize = fieldSize;
#endif
          cType = SQL_C_TYPE_TIME;
//          fxType = FXVariant::DT_TODO;
          fieldSize = sizeof(SQL_TIME_STRUCT);
          allocSize = fieldSize;
        case SQL_TIMESTAMP:
          cType = SQL_C_TYPE_TIMESTAMP;
//          fxType = FXVariant::DT_TODO;
          fieldSize = sizeof(SQL_TIMESTAMP_STRUCT);
          allocSize = fieldSize;
        case SQL_BINARY:
          cType = SQL_C_BINARY;
          fxType = FXVariant::DT_RAW;
        case SQL_VARBINARY:
        case SQL_LONGVARBINARY:
          rawf.type.varSize = TRUE;
        case SQL_GUID:
          cType = SQL_C_GUID;
//          fxType = FXVariant::DT_TODO;
          fieldSize = sizeof(SQLGUID);
          allocSize = fieldSize;
#endif
        default:
          fxwarning("FIXME: unhandled type conversion!\n");
#ifdef HACK_TYPE_UNKNOWN
// Fall back to string and see what happens...
          fxType = FXVariant::DT_FXSTRING;
          fieldSize = 0;
          ++allocSize;
          rawf.terminatorLength = sizeof(FXchar);
          cType = SQL_C_CHAR;
#else
          fxType = FXVariant::DT_UNASSIGNED;
#endif
          break;
      }
      if(fieldSize) fieldSize += rawf.terminatorLength;
      allocSize += rawf.terminatorLength;
      rawf.type.maxSize = (FXint) fieldSize;  /* Field size (!=0 if known) */
      rawf.fieldSize = (FXint) allocSize;     /* buffer allocated size */
      FXMALLOC(&rawf.raw, FXuchar, rawf.fieldSize+CK_ADD);
      CK_FILL(rawf.raw, rawf.fieldSize);
      rawf.type.fxType = fxType;
      rawf.cType = cType;
      f = new FXDatabaseField(this);
      f->create();
      f->handle(this, MKUINT(FXDatabaseField::ID_SETFIELDTYPE, SEL_EVENT), &rawf.type);
      rawFields.append(rawf);
      fields.append(f);
      FXTRACE((20, "Column %s: init done\n", rawf.type.name.text()));
    }
    /* Bind'em all */
    for(x=0; x<nCols; ++x) {
      res = SQLBindCol(hStmt, x+1, rawFields[x].cType, rawFields[x].raw,
        rawFields[x].fieldSize, &rawFields[x].slen);
      if(!VALID(res)) FXODBCInterface::odbcThrow(SQL_HANDLE_STMT, hStmt);
    }
    res = SQLSetStmtAttr(hStmt, SQL_ATTR_ROW_STATUS_PTR, &row_status, sizeof(row_status));
    if(!VALID(res)) fxwarning("Warning: row status will be unavailable\n");
    FXTRACE((20, "All columns bound successfully\n"));
    /* Fetch first row */
    res = SQLRowCount(hStmt, &val); /* recordCount() */
    if(!VALID(res)) FXODBCInterface::odbcThrow(SQL_HANDLE_STMT, hStmt);
    if(val>0) { /* Fetch first row */
      if(cursor_attr1 & SQL_CA1_NEXT) {
        res = SQLFetchScroll(hStmt, SQL_FETCH_NEXT, 0);
      } else if(cursor_attr1 & SQL_CA1_ABSOLUTE) {
        res = SQLFetchScroll(hStmt, SQL_FETCH_FIRST, 0);
      } else FXDatabaseInterface::dbThrow("Action not supported", -1);
      if(res!=SQL_NO_DATA) bookmarkReady = TRUE;
    }
    else res = SQL_NO_DATA;
    if(res!=SQL_NO_DATA && !VALID(res)) FXODBCInterface::odbcThrow(SQL_HANDLE_STMT, hStmt);
    if(res==SQL_NO_DATA) {
      for(x=0; x<nCols; ++x) rawFields[x].slen = SQL_NO_DATA;
    } else if(res==SQL_SUCCESS_WITH_INFO) reFetch();
  } catch(...) {
    FreeHandle();
    throw;
  }
  FXDatabaseQuery::Open(q, readOnly);
  notifyMove();
}

/*
 * Check for size overflow, and re-get data if needed.
 * Must be called after a SQLFetchScroll returning SQL_SUCCESS_WITH_INFO.
 */
void FXODBCQuery::reFetch() {
  SQLRETURN ret;
  FXint sz = rawFields.no();
  FXint x;
  SQLINTEGER slen;
  for(x=0; x<sz; ++x) {
    if(!rawFields[x].type.maxSize) {
      slen = rawFields[x].slen;
      if((slen!=SQL_NULL_DATA) && (slen != SQL_NO_TOTAL)) slen += rawFields[x].terminatorLength;
      FXASSERT(slen!=SQL_NO_TOTAL);
      if((slen!=SQL_NULL_DATA) && (slen != SQL_NO_TOTAL) && (slen>rawFields[x].fieldSize)) {
        FXTRACE((20, "Column %d needs resizing\n", x));
        rawFields[x].fieldSize = slen;
        FXRESIZE(&rawFields[x].raw, FXuchar, rawFields[x].fieldSize+CK_ADD);
        CK_FILL(rawFields[x].raw, rawFields[x].fieldSize);
        ret = SQLBindCol(hStmt, x+1, rawFields[x].cType, rawFields[x].raw,
        rawFields[x].fieldSize, &rawFields[x].slen);
        if(!VALID(ret)) FXODBCInterface::odbcThrow(SQL_HANDLE_STMT, hStmt);
        if(capabilities->sql_getdata_extensions & SQL_GD_BOUND) {
          ret = SQLGetData(hStmt, x+1, rawFields[x].cType, rawFields[x].raw,
            rawFields[x].fieldSize, &rawFields[x].slen);
        } else if(UseBookmarks && bookmarkReady && (cursor_attr1 & SQL_FETCH_BY_BOOKMARK)) {
          ret = SQLFetchScroll(hStmt, SQL_FETCH_BOOKMARK, 0);
        } else if(cursor_attr1 & SQL_CA1_RELATIVE) {
          ret = SQLFetchScroll(hStmt, SQL_FETCH_RELATIVE, 0);
        } else if(UseBookmarks && bookmarkReady && (cursor_attr1 & SQL_CA1_BULK_UPDATE_BY_BOOKMARK)) {
          ret = SQLBulkOperations(hStmt, SQL_FETCH_BY_BOOKMARK);
        } else if(cursor_attr1 & SQL_CA1_ABSOLUTE) {
          FXuint pos = currentPos();
          if(pos) ret = SQLFetchScroll(hStmt, SQL_FETCH_ABSOLUTE, pos);
          else ret = SQL_SUCCESS_WITH_INFO;
        } else if(cursor_attr1 & SQL_CA1_POS_REFRESH) {
          ret = SQLSetPos(hStmt, SQL_REFRESH, 1, SQL_LOCK_NO_CHANGE);
        } else ret = SQL_SUCCESS_WITH_INFO;
        if(!VALID(ret)) FXODBCInterface::odbcThrow(SQL_HANDLE_STMT, hStmt);
        FXASSERT(ret==SQL_SUCCESS);
      }
    }
  }
}

void FXODBCQuery::Free() {
  FreeHandle();
  FXDatabaseQuery::Free();
}

void FXODBCQuery::Close() {
  checkOpen(TRUE);
  if(isEditable()) CancelUpdate();
  FreeHandle();
  FXDatabaseQuery::Close();
}

FXbool FXODBCQuery::moveFirst() {
  SQLRETURN res;
  checkEditable(FALSE);
  if(cursor_attr1 & SQL_CA1_ABSOLUTE) {
    res = SQLFetchScroll(hStmt, SQL_FETCH_FIRST, 0);
  } else FXDatabaseInterface::dbThrow("Action not supported", -1);
  if(res==SQL_NO_DATA) {
    /* a failed moveFirst is assumed as "empty recordset" */
    lastValidPos = 0;
    bookmarkReady = FALSE;
    FXuint nCols = fields.no();
    FXuint x;
    for(x=0; x<nCols; ++x) rawFields[x].slen = SQL_NO_DATA;
    notifyMove();
    return FALSE;
  }
  if(!VALID(res)) FXODBCInterface::odbcThrow(SQL_HANDLE_STMT, hStmt);
  if(res==SQL_SUCCESS_WITH_INFO) reFetch();
  bookmarkReady = TRUE;
  notifyMove();
  return TRUE;
}

FXbool FXODBCQuery::moveNext() {
  SQLRETURN res;
  FXuint pos;
  checkEditable(FALSE);
  pos = currentPos();
/* Choose an available method */
  if(cursor_attr1 & SQL_CA1_NEXT) {
    res = SQLFetchScroll(hStmt, SQL_FETCH_NEXT, 0);
  } else if(cursor_attr1 & SQL_CA1_ABSOLUTE) {
    res = SQLFetchScroll(hStmt, SQL_FETCH_ABSOLUTE, currentPos()+1);
  } else FXDatabaseInterface::dbThrow("Action not supported", -1);
/* Rollback if needed */
  if(res==SQL_NO_DATA) {
    if(UseBookmarks && (cursor_attr1 & SQL_CA1_BOOKMARK) && bookmarkReady) {
      res = SQLFetchScroll(hStmt, SQL_FETCH_BOOKMARK, 0);
    } else if(UseBookmarks && (cursor_attr1 & SQL_CA1_BULK_UPDATE_BY_BOOKMARK) && bookmarkReady) {
      res = SQLBulkOperations(hStmt, SQL_FETCH_BY_BOOKMARK);
    } else if((cursor_attr1 & SQL_CA1_ABSOLUTE) && pos) {
      res = SQLFetchScroll(hStmt, SQL_FETCH_ABSOLUTE, pos);
    } else if(cursor_attr1 & SQL_CA1_RELATIVE) {
      res = SQLFetchScroll(hStmt, SQL_FETCH_PRIOR, 0);
    }
    if(res!=SQL_NO_DATA && !VALID(res)) FXODBCInterface::odbcThrow(SQL_HANDLE_STMT, hStmt);
    if(res==SQL_SUCCESS_WITH_INFO) reFetch();
    if(res==SQL_NO_DATA) {
      lastValidPos = 0;
      bookmarkReady = FALSE;
    }
    return FALSE;
  }
  if(!VALID(res)) FXODBCInterface::odbcThrow(SQL_HANDLE_STMT, hStmt);
  if(res==SQL_SUCCESS_WITH_INFO) reFetch();
  bookmarkReady = TRUE;
  notifyMove();
  return TRUE;
}

FXbool FXODBCQuery::movePrevious() {
  SQLRETURN res;
  FXuint pos;
  checkEditable(FALSE);
/* Choose an available method */
  pos = currentPos();
  if(cursor_attr1 & SQL_CA1_RELATIVE) {
    res = SQLFetchScroll(hStmt, SQL_FETCH_PRIOR, 0);
  } else if (cursor_attr1 & SQL_CA1_ABSOLUTE) {
      if(pos>1) res = SQLFetchScroll(hStmt, SQL_FETCH_ABSOLUTE, pos-1);
      else return FALSE;
  } else FXDatabaseInterface::dbThrow("Action not supported", -1);
/* Rollback if needed */
  if(res==SQL_NO_DATA) {
    if(UseBookmarks && (cursor_attr1 & SQL_CA1_BOOKMARK) && bookmarkReady) {
      res = SQLFetchScroll(hStmt, SQL_FETCH_BOOKMARK, 0);
    } else if(UseBookmarks && (cursor_attr1 & SQL_CA1_BULK_UPDATE_BY_BOOKMARK) && bookmarkReady) {
      res = SQLBulkOperations(hStmt, SQL_FETCH_BY_BOOKMARK);
    } else if((cursor_attr1 & SQL_CA1_ABSOLUTE) && pos) {
      res = SQLFetchScroll(hStmt, SQL_FETCH_ABSOLUTE, pos);
    } else if(cursor_attr1 & SQL_CA1_NEXT) {
      res = SQLFetchScroll(hStmt, SQL_FETCH_NEXT, 0);
    }
    if(res!=SQL_NO_DATA && !VALID(res)) FXODBCInterface::odbcThrow(SQL_HANDLE_STMT, hStmt);
    if(res==SQL_SUCCESS_WITH_INFO) reFetch();
    if(res==SQL_NO_DATA) {
      lastValidPos = 0;
      bookmarkReady = FALSE;
    }
    return FALSE;
  }
  if(!VALID(res)) FXODBCInterface::odbcThrow(SQL_HANDLE_STMT, hStmt);
  if(res==SQL_SUCCESS_WITH_INFO) reFetch();
  bookmarkReady = TRUE;
  notifyMove();
  return TRUE;
}

FXbool FXODBCQuery::moveLast() {
  SQLRETURN res;
  FXuint pos;
  checkEditable(FALSE);
  pos = currentPos();
  if(cursor_attr1 & SQL_CA1_ABSOLUTE) {
    res = SQLFetchScroll(hStmt, SQL_FETCH_LAST, 0);
  } else FXDatabaseInterface::dbThrow("Action not supported", -1);
/* Rollback if needed */
  if(res==SQL_NO_DATA) {
    /* CHECKME: should we assume an empty recordset? */
    if(UseBookmarks && (cursor_attr1 & SQL_CA1_BOOKMARK) && bookmarkReady) {
      res = SQLFetchScroll(hStmt, SQL_FETCH_BOOKMARK, 0);
    } else if(UseBookmarks && (cursor_attr1 & SQL_CA1_BULK_UPDATE_BY_BOOKMARK) && bookmarkReady) {
      res = SQLBulkOperations(hStmt, SQL_FETCH_BY_BOOKMARK);
    } else if((cursor_attr1 & SQL_CA1_ABSOLUTE) && pos) {
      res = SQLFetchScroll(hStmt, SQL_FETCH_ABSOLUTE, pos);
    }
    if(res!=SQL_NO_DATA && !VALID(res)) FXODBCInterface::odbcThrow(SQL_HANDLE_STMT, hStmt);
    if(res==SQL_SUCCESS_WITH_INFO) reFetch();
    if(res==SQL_NO_DATA) {
      lastValidPos = 0;
      bookmarkReady = FALSE;
    }
    return FALSE;
  }
  if(!VALID(res)) FXODBCInterface::odbcThrow(SQL_HANDLE_STMT, hStmt);
  if(res==SQL_SUCCESS_WITH_INFO) reFetch();
  bookmarkReady = TRUE;
  notifyMove();
  return TRUE;
}

FXbool FXODBCQuery::moveTo(FXuint where) {
  SQLRETURN res;
  FXuint pos;
  checkEditable(FALSE);
  pos = currentPos();
  if(cursor_attr1 & SQL_CA1_ABSOLUTE) {
    res = SQLFetchScroll(hStmt, SQL_FETCH_ABSOLUTE, where);
  } else FXDatabaseInterface::dbThrow("Action not supported", -1);
/* Rollback if needed */
  if(res==SQL_NO_DATA) {
    if(UseBookmarks && (cursor_attr1 & SQL_CA1_BOOKMARK) && bookmarkReady) {
      res = SQLFetchScroll(hStmt, SQL_FETCH_BOOKMARK, 0);
    } else if(UseBookmarks && (cursor_attr1 & SQL_CA1_BULK_UPDATE_BY_BOOKMARK) && bookmarkReady) {
      res = SQLBulkOperations(hStmt, SQL_FETCH_BY_BOOKMARK);
    } else if((cursor_attr1 & SQL_CA1_ABSOLUTE) && pos) {
      res = SQLFetchScroll(hStmt, SQL_FETCH_ABSOLUTE, pos);
    }
    if(res!=SQL_NO_DATA && !VALID(res)) FXODBCInterface::odbcThrow(SQL_HANDLE_STMT, hStmt);
    if(res==SQL_SUCCESS_WITH_INFO) reFetch();
    if(res==SQL_NO_DATA) {
      lastValidPos = 0;
      bookmarkReady = FALSE;
    }
    return FALSE;
  }
  if(!VALID(res)) FXODBCInterface::odbcThrow(SQL_HANDLE_STMT, hStmt);
  if(res==SQL_SUCCESS_WITH_INFO) reFetch();
  bookmarkReady = TRUE;
  notifyMove();
  return TRUE;
}

FXbool FXODBCQuery::moveOf(FXint displacement) {
  SQLRETURN res;
  FXuint pos;
  checkEditable(FALSE);
  pos = currentPos();
  if(cursor_attr1 & SQL_CA1_RELATIVE) {
      res = SQLFetchScroll(hStmt, SQL_FETCH_RELATIVE, displacement);
  } else FXDatabaseInterface::dbThrow("Action not supported", -1);
/* Rollback if needed */
  if(res==SQL_NO_DATA) {
    if(UseBookmarks && (cursor_attr1 & SQL_CA1_BOOKMARK) && bookmarkReady) {
      res = SQLFetchScroll(hStmt, SQL_FETCH_BOOKMARK, 0);
    } else if(UseBookmarks && (cursor_attr1 & SQL_CA1_BULK_UPDATE_BY_BOOKMARK) && bookmarkReady) {
      res = SQLBulkOperations(hStmt, SQL_FETCH_BY_BOOKMARK);
    } else if((cursor_attr1 & SQL_CA1_ABSOLUTE) && pos) {
      res = SQLFetchScroll(hStmt, SQL_FETCH_ABSOLUTE, pos);
    }
    if(res!=SQL_NO_DATA && !VALID(res)) FXODBCInterface::odbcThrow(SQL_HANDLE_STMT, hStmt);
    if(res==SQL_SUCCESS_WITH_INFO) reFetch();
    if(res==SQL_NO_DATA) {
      lastValidPos = 0;
      bookmarkReady = FALSE;
    }
    return FALSE;
  }
  if(!VALID(res)) FXODBCInterface::odbcThrow(SQL_HANDLE_STMT, hStmt);
  if(res==SQL_SUCCESS_WITH_INFO) reFetch();
  bookmarkReady = TRUE;
  notifyMove();
  return TRUE;
}

FXuint FXODBCQuery::currentPos() {
  SQLRETURN res;
  FXuint ret;
  SQLINTEGER unused;
  checkOpen(TRUE);
  if(!recordCount()) return 0;
  res = SQLGetStmtAttr(hStmt, SQL_ATTR_ROW_NUMBER, &ret, sizeof(ret), &unused);
  if(!VALID(res)) FXODBCInterface::odbcThrow(SQL_HANDLE_STMT, hStmt);
  return ret;
}

FXuint FXODBCQuery::recordCount() {
  SQLRETURN res;
  FXuint ret;
  checkOpen(TRUE);
  res = SQLRowCount(hStmt, (SQLINTEGER *) &ret);
  if(!VALID(res)) FXODBCInterface::odbcThrow(SQL_HANDLE_STMT, hStmt);
  return ret;
}

void FXODBCQuery::Edit() {
  checkEditable(FALSE);
  if(readOnly) FXDatabaseInterface::dbThrow("Read-only recordset", -1);
  if(!(UseBookmarks && (cursor_attr1 & SQL_CA1_BULK_UPDATE_BY_BOOKMARK)) &&
    !(cursor_attr1 & SQL_CA1_POS_UPDATE))
    FXDatabaseInterface::dbThrow("This driver doesn't support 'edit'", -1);
  updatingRow = currentPos(); /* Must be a valid position */
  if(!updatingRow) FXDatabaseInterface::dbThrow("No current row", -1);
  FXDatabaseQuery::Edit();
}

void FXODBCQuery::addNew() {
  checkEditable(FALSE);
  if(readOnly) FXDatabaseInterface::dbThrow("Read-only recordset", -1);
  if(!(cursor_attr1 & SQL_CA1_BULK_ADD))
    FXDatabaseInterface::dbThrow("This driver doesn't support 'add'", -1);
  FXDatabaseQuery::addNew();
}

void FXODBCQuery::Delete() {
  SQLRETURN res;
  checkEditable(FALSE);
  FXuint to_del;
  if(readOnly) FXDatabaseInterface::dbThrow("Read-only recordset", -1);
  to_del = currentPos(); /* Must be a valid position */
  if(!to_del) FXDatabaseInterface::dbThrow("No current row", -1);
  if(UseBookmarks && (cursor_attr1 & SQL_CA1_BULK_DELETE_BY_BOOKMARK) && bookmarkReady) {
    res = SQLBulkOperations(hStmt, SQL_DELETE_BY_BOOKMARK);
  } else if(cursor_attr1 & SQL_CA1_POS_DELETE) {
    res = SQLSetPos(hStmt, 1, SQL_DELETE, SQL_LOCK_NO_CHANGE);
  } else FXDatabaseInterface::dbThrow("This driver doesn't support 'delete'", -1);
  if(!VALID(res)) FXODBCInterface::odbcThrow(SQL_HANDLE_STMT, hStmt);
  FXDatabaseQuery::Delete();
  doRefresh(to_del, !(cursor_attr2 & SQL_CA2_SENSITIVITY_DELETIONS));
}

void FXODBCQuery::CancelUpdate() {
  checkEditable(TRUE);
  FXDatabaseQuery::CancelUpdate();
}

void FXODBCQuery::Update() {
  FXint x;
  FXint no;
  SQLRETURN res;
  checkEditable(TRUE);
  no = rawFields.no();
  for(x=0; x<no; ++x) var2odbc(x);
  switch(state) {
    case rsModify:
      FXASSERT(updatingRow);
      if(UseBookmarks && (cursor_attr1 & SQL_CA1_BULK_UPDATE_BY_BOOKMARK)) {
          try {
            beforeBulkUpdate();
            res = SQLBulkOperations(hStmt, SQL_UPDATE_BY_BOOKMARK);
            if(!VALID(res)) FXODBCInterface::odbcThrow(SQL_HANDLE_STMT, hStmt);
            afterBulkUpdate();
          } catch(...) {
            afterBulkUpdate();
            throw;
          }
      } else if(cursor_attr1 & SQL_CA1_POS_UPDATE) {
        res = SQLSetPos(hStmt, 1, SQL_UPDATE, SQL_LOCK_NO_CHANGE);
        if(!VALID(res)) FXODBCInterface::odbcThrow(SQL_HANDLE_STMT, hStmt);
      } else FXDatabaseInterface::dbThrow("This driver doesn't support 'update'", -1);
      FXDatabaseQuery::Update();
      doRefresh(updatingRow, !(cursor_attr2 & SQL_CA2_SENSITIVITY_UPDATES));
      break;
    case rsAddNew:
        if(cursor_attr1 & SQL_CA1_BULK_ADD) {
          try {
            beforeBulkUpdate();
            res = SQLBulkOperations(hStmt, SQL_ADD);
            if(!VALID(res)) FXODBCInterface::odbcThrow(SQL_HANDLE_STMT, hStmt);
            afterBulkUpdate();
          } catch(...) {
            afterBulkUpdate();
            throw;
          }
        } else FXDatabaseInterface::dbThrow("Unsupported method", -1);
      FXDatabaseQuery::Update();
      doRefresh((FXuint)-1, !(cursor_attr2 & SQL_CA2_SENSITIVITY_ADDITIONS));
      break;
    default:
      FXDatabaseInterface::dbThrow("Internal error!", -1);
  }
}

void FXODBCQuery::doRequery() {
  SQLRETURN res;
  bookmarkReady = FALSE;
  lastValidPos = 0;
  res = SQLFreeStmt(hStmt, SQL_CLOSE);
  if(!VALID(res)) FXODBCInterface::odbcThrow(SQL_HANDLE_STMT, hStmt);
  res = SQLExecute(hStmt);
  if(!VALID(res)) FXODBCInterface::odbcThrow(SQL_HANDLE_STMT, hStmt);
}

void FXODBCQuery::Requery() {
  FXuint x;
  FXuint nCols;
  checkEditable(FALSE);
  doRequery();
  if(!moveNext()) {
    nCols = rawFields.no();
    for(x=0; x<nCols; ++x) rawFields[x].slen = SQL_NO_DATA;
    notifyMove();
  }
}

void FXODBCQuery::doRefresh(FXuint pos, FXbool forceRequery) {
  SQLRETURN res;
  FXuint x;
  FXuint nCols;
  nCols = rawFields.no();
  if(forceRequery) doRequery();
  bookmarkReady = FALSE;
  res = SQL_ERROR;
  if(pos && (cursor_attr1 & SQL_CA1_ABSOLUTE)) {
    if(pos==(FXuint)-1) { /* moveLast */
      res = SQLFetchScroll(hStmt, SQL_FETCH_LAST, 0);
    } else if(pos>1) { /* moveTo */
      res = SQLFetchScroll(hStmt, SQL_FETCH_ABSOLUTE, pos);
    }
    if(!VALID(res)) { /* moveFirst */
      res = SQLFetchScroll(hStmt, SQL_FETCH_FIRST, 0);
      if(res!=SQL_NO_DATA && !VALID(res)) FXODBCInterface::odbcThrow(SQL_HANDLE_STMT, hStmt);
    }
  } else if(pos && cursor_attr1 & SQL_CA1_NEXT) {
    res = SQLFetchScroll(hStmt, SQL_FETCH_NEXT, 0);
    if(res!=SQL_NO_DATA && !VALID(res)) FXODBCInterface::odbcThrow(SQL_HANDLE_STMT, hStmt);
  }
  if(VALID(res)) bookmarkReady = TRUE;
  else { /* No rows/unable to move */
    for(x=0; x<nCols; ++x) rawFields[x].slen = SQL_NO_DATA;
  }
  notifyMove();
}

}
