/********************************************************************************
*                                                                               *
*                  ODBC Database connector                                      *
*                                                                               *
*********************************************************************************
* Copyright (C) 2002 by Mathew Robertson.       All Rights Reserved.            *
* Copyright (C) 2002 by Giancarlo Formicuccia.  All Rights Reserved.            *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
*********************************************************************************/
#include <config.h>
#include <fox/fxver.h>
#include <fox/xincs.h>
#include <fox/fxdefs.h>
#include <fox/FXStream.h>
#include <fox/FXString.h>
#include <fox/FXWString.h>
#include <fox/FXCharset.h>
#include <fox/FXSize.h>
#include <fox/FXPoint.h>
#include <fox/FXRectangle.h>
#include <fox/FXRegistry.h>
#include <fox/FXApp.h>
using namespace FX;
#include "exincs.h"
#include "fxexdefs.h"
#include "FXArray.h"
#include "FXDatabaseField.h"
#include "FXODBCInterface.h"
#include "FXODBCQuery.h"
using namespace FXEX;
namespace FXEX {

static SQLHENV hEnv = NULL;

FXDEFMAP(FXODBCInterface) FXODBCInterfaceMap[] = {
  FXMAPFUNC(SEL_EVENT,      FXODBCInterface::ID_ALLOCHANDLE,   FXODBCInterface::onAllocHandle)
};

FXIMPLEMENT(FXODBCInterface, FXDatabaseInterface, FXODBCInterfaceMap, ARRAYNUMBER(FXODBCInterfaceMap))

FXODBCInterface::FXODBCInterface(FXApp *a,FXObject *tgt,FXSelector sel) : FXDatabaseInterface(a,tgt,sel){
  SQLRETURN res;

  hConn = NULL;

  if(!hEnv) {
    res = SQLAllocHandle(SQL_HANDLE_ENV, SQL_NULL_HANDLE, &hEnv);
    if(!VALID(res)) {
      hEnv = NULL;
      dbThrow("SQLAllocHandle(SQL_HANDLE_ENV) failed", -1);
    }
  }
  res = SQLSetEnvAttr(hEnv, SQL_ATTR_ODBC_VERSION, (SQLPOINTER)SQL_OV_ODBC3, 0);
  if(!VALID(res)) odbcThrow(SQL_HANDLE_ENV, hEnv);
  SQLSetEnvAttr(hEnv, SQL_ATTR_OUTPUT_NTS, (SQLPOINTER)SQL_TRUE, 0);
}

FXODBCInterface::~FXODBCInterface(){
  if(connected) FXODBCInterface::disconnect();
}

FXString FXODBCInterface::xlatState(const FXString &state) {
  FXString r = state;
  if(state=="00000") r = "";
  else if(state=="01000") r = "General warning";
  else if(state=="01004") r = "String data, right truncated";
  else if(state=="01S02") r = "Option value changed";
  else if(state=="07005") r = "Prepared statement not a cursor-specification";
  else if(state=="07009") r = "Invalid descriptor index";
  else if(state=="08S01") r = "Communication link failed";
  else if(state=="08002") r = "Connection name in use";
  else if(state=="08003") r = "Connection does not exist";
  else if(state=="24000") r = "Invalid cursor state";
  else if(state=="3D000") r = "Invalid catalog name";
  else if(state=="HY000") r = "General error";
  else if(state=="HY001") r = "Memory allocation error";
  else if(state=="HY009") r = "Invalid use of NULL pointer";
  else if(state=="HY010") r = "Function sequence error";
  else if(state=="HY011") r = "Attribute cannot be set now";
  else if(state=="HY013") r = "memory management error";
  else if(state=="HY017") r = "Invalid use of an automatically allocated descriptor handle";
  else if(state=="HY024") r = "Invalid attribute value";
  else if(state=="HY090") r = "Invalid string or buffer length";
  else if(state=="HY091") r = "Invalid descriptor field identifier";
  else if(state=="HY092") r = "Invalid attribute/option identifier";
  else if(state=="HY107") r = "Row value out of range";
  else if(state=="HY109") r = "Invalid cursor position";
  else if(state=="HYC00") r = "Optional feature not implemented";
  else if(state=="HYT01") r = "Connection timeout expired";
  else if(state=="IM001") r = "Driver does not support this function";
  else if(state=="IM009") r = "Unable to load translation library";
  return r;
}

void FXODBCInterface::getError(SQLSMALLINT HandleType, SQLHANDLE Handle, FXString &e, FXint &v) {
  SQLCHAR *buf = NULL;
  SQLSMALLINT buflen = 0;
  SQLSMALLINT reqbuflen = 127;
  SQLINTEGER sqlerr = 0;
  SQLCHAR state[6];
  SQLRETURN res;
  FXString s;

  memset(state, 0, sizeof(state));
  while((buflen+1)<reqbuflen) {
    FXFREE(&buf);
    buflen = reqbuflen+1;
    fxmalloc((void **) &buf, buflen);
    res = SQLGetDiagRec(HandleType, Handle, 1, state, &sqlerr, buf, buflen, &reqbuflen);
    if(!VALID(res)) {
      FXFREE(&buf);
      e = "FXODBC error: <unable to get error description>";
      v = -1;
      return;
    }
  }
  FXASSERT(!state[sizeof(state)-1]);
  s = xlatState((FXchar *) state);
  e = (FXchar *) buf;
  if(s!="") e += " ("+s+")";
  v = (FXint) sqlerr;
  FXFREE(&buf);
}

void FXODBCInterface::odbcThrow(SQLSMALLINT HandleType, SQLHANDLE Handle) {
  FXString e;
  FXint v;
  getError(HandleType, Handle, e, v);
  dbThrow(e, v);
}

void FXODBCInterface::save(FXStream& store) const {
  FXDatabaseInterface::save(store);
  /* TODO... */
}

void FXODBCInterface::load(FXStream& store){
  FXDatabaseInterface::load(store);
  /* TODO... */
}

void FXODBCInterface::connect(FXbool readOnly) {
  SQLRETURN res;
  SQLHDBC hConn;
  checkStatus(FALSE);
  res = SQLAllocHandle(SQL_HANDLE_DBC, hEnv, &hConn);
  if(!VALID(res)) odbcThrow(SQL_HANDLE_ENV, hEnv);
  try {
    if(readOnly) SQLSetConnectAttr(hConn, SQL_ATTR_ACCESS_MODE, (SQLPOINTER) SQL_MODE_READ_ONLY, 0);
    else {
      SQLSetConnectAttr(hConn, SQL_ATTR_ACCESS_MODE, (SQLPOINTER) SQL_MODE_READ_WRITE, 0);
      SQLSetConnectAttr(hConn, SQL_ATTR_AUTOCOMMIT, (SQLPOINTER) SQL_AUTOCOMMIT_ON, 0);
    }
    SQLSetConnectAttr(hConn, SQL_ATTR_ODBC_CURSORS, (SQLPOINTER) SQL_CUR_USE_IF_NEEDED, 0);
    res = SQLConnect(hConn, (SQLCHAR*) database.text(), SQL_NTS,
      (SQLCHAR*) username.text(), SQL_NTS, (SQLCHAR*) password.text(), SQL_NTS);
    if(!VALID(res)) odbcThrow(SQL_HANDLE_DBC, hConn);
  } catch(...) {
    SQLFreeHandle(SQL_HANDLE_DBC, hConn);
    throw;
  }
  this->hConn = hConn;
  initCapabilities();
  connected = TRUE;
  notifyAllQueries(this, MKUINT(FXDatabaseQuery::ID_CONNECT, SEL_EVENT), this);
}

FXuint FXODBCInterface::capUint(SQLUSMALLINT cap, FXuint defval) {
  SQLRETURN res;
  SQLUINTEGER val;
  SQLSMALLINT unused;
  res = SQLGetInfo(hConn, cap, &val, sizeof(val), &unused);
  if(res!=SQL_SUCCESS) return defval;
  else return (FXuint) val;
}

FXushort FXODBCInterface::capUshort(SQLUSMALLINT cap, FXushort defval) {
  SQLRETURN res;
  SQLSMALLINT val;
  SQLSMALLINT unused;
  res = SQLGetInfo(hConn, cap, &val, sizeof(val), &unused);
  if(res!=SQL_SUCCESS) return defval;
  else return (FXushort) val;
}

FXString FXODBCInterface::capString(SQLUSMALLINT cap, FXString defval) {
  FXchar *ret = NULL;
  SQLSMALLINT sz = 0;
  SQLSMALLINT len = 0;
  FXString retval;
  SQLRETURN res;
  FXMALLOC(&ret, FXchar, sz);
  while(len>=sz) {
    sz += 128;
    FXRESIZE(&ret, FXchar, sz);
    res = SQLGetInfo(hConn, cap, (SQLCHAR *) ret, sz, &len);
    if(!VALID(res)) {
      FXFREE(&ret);
      return defval;
    }
  }
  retval = ret;
  FXFREE(&ret);
  return retval;
}

FXbool FXODBCInterface::capBool(SQLUSMALLINT cap, FXbool defval) {
  FXString ch;
  ch = capString(cap, defval ? "Y" : "N");
  return (ch=="Y" || ch=="y");
}


void FXODBCInterface::initCapabilities() {
  capabilities.sql_async_mode = capUint(SQL_ASYNC_MODE, SQL_AM_NONE);
  capabilities.sql_cursor_sensitivity = capUint(SQL_CURSOR_SENSITIVITY, SQL_UNSPECIFIED);
  capabilities.sql_data_source_name = capString(SQL_DATA_SOURCE_NAME, "");
  capabilities.sql_data_source_read_only = capBool(SQL_DATA_SOURCE_READ_ONLY, TRUE);
  capabilities.sql_database_name = capString(SQL_DATABASE_NAME, "");
  capabilities.sql_dbms_name = capString(SQL_DBMS_NAME, "");
  capabilities.sql_dbms_ver = capString(SQL_DBMS_VER, "");
  capabilities.sql_dm_ver = capString(SQL_DM_VER, "");
  capabilities.sql_driver_name = capString(SQL_DRIVER_NAME, "");
  capabilities.sql_driver_odbc_ver = capString(SQL_DRIVER_ODBC_VER, "");
  capabilities.sql_driver_ver = capString(SQL_DRIVER_VER, "");
  capabilities.sql_dynamic_cursor_attributes1 = capUint(SQL_DYNAMIC_CURSOR_ATTRIBUTES1, 0);
  capabilities.sql_dynamic_cursor_attributes2 = capUint(SQL_DYNAMIC_CURSOR_ATTRIBUTES2, 0);
  capabilities.sql_keyset_cursor_attributes1 = capUint(SQL_KEYSET_CURSOR_ATTRIBUTES1, 0);
  capabilities.sql_keyset_cursor_attributes2 = capUint(SQL_KEYSET_CURSOR_ATTRIBUTES2, 0);
  capabilities.sql_static_cursor_attributes1 = capUint(SQL_STATIC_CURSOR_ATTRIBUTES1, 0);
  capabilities.sql_static_cursor_attributes2 = capUint(SQL_STATIC_CURSOR_ATTRIBUTES2, 0);
  capabilities.sql_forward_only_cursor_attributes1 = capUint(SQL_FORWARD_ONLY_CURSOR_ATTRIBUTES1, 0);
  capabilities.sql_forward_only_cursor_attributes2 = capUint(SQL_FORWARD_ONLY_CURSOR_ATTRIBUTES2, 0);
  capabilities.sql_getdata_extensions = capUint(SQL_GETDATA_EXTENSIONS, 0);
  capabilities.sql_txn_capable = capUshort(SQL_TXN_CAPABLE, SQL_TC_NONE);
  capabilities.sql_default_txn_isolation = capUint(SQL_DEFAULT_TXN_ISOLATION, 0);
  capabilities.sql_cursor_commit_behavior = capUshort(SQL_CURSOR_COMMIT_BEHAVIOR, 0);
  capabilities.sql_cursor_rollback_behavior = capUshort(SQL_CURSOR_ROLLBACK_BEHAVIOR, 0);
  capabilities.sql_multiple_active_txn = capBool(SQL_MULTIPLE_ACTIVE_TXN, FALSE);
  capabilities.sql_non_nullable_columns = capUshort(SQL_NON_NULLABLE_COLUMNS, SQL_NNC_NON_NULL);
  capabilities.sql_integrity = capBool(SQL_INTEGRITY, FALSE);
  capabilities.sql_odbc_interface_conformance = capUint(SQL_ODBC_INTERFACE_CONFORMANCE, SQL_OIC_CORE);
  capabilities.sql_odbc_ver = capString(SQL_ODBC_VER, "");
  capabilities.sql_scroll_options = capUint(SQL_SCROLL_OPTIONS, 0);
  capabilities.sql_server_name = capString(SQL_SERVER_NAME, "");
  capabilities.sql_sql_conformance = capUint(SQL_SQL_CONFORMANCE, 0);
  FXTRACE((20, "*** driver capabilities/properties ***\n"));
  FXTRACE((20, "sql_dm_ver                           : %s\n", capabilities.sql_dm_ver.text()));
  FXTRACE((20, "sql_odbc_ver                         : %s\n", capabilities.sql_odbc_ver.text()));
  FXTRACE((20, "sql_driver_name                      : %s\n", capabilities.sql_driver_name.text()));
  FXTRACE((20, "sql_driver_odbc_ver                  : %s\n", capabilities.sql_driver_odbc_ver.text()));
  FXTRACE((20, "sql_driver_ver                       : %s\n", capabilities.sql_driver_ver.text()));
  FXTRACE((20, "sql_dbms_name                        : %s\n", capabilities.sql_dbms_name.text()));
  FXTRACE((20, "sql_dbms_ver                         : %s\n", capabilities.sql_dbms_ver.text()));
  FXTRACE((20, "sql_server_name                      : %s\n", capabilities.sql_server_name.text()));
  FXTRACE((20, "sql_data_source_name                 : %s\n", capabilities.sql_data_source_name.text()));
  FXTRACE((20, "sql_data_source_read_only            : %s\n", capabilities.sql_data_source_read_only ? "Y" : "N"));
  FXTRACE((20, "sql_database_name                    : %s\n", capabilities.sql_database_name.text()));
  FXTRACE((20, "sql_odbc_interface_conformance       : %08x\n", capabilities.sql_odbc_interface_conformance));
  FXTRACE((20, "sql_sql_conformance                  : %08x\n", capabilities.sql_sql_conformance));
  FXTRACE((20, "sql_async_mode                       : %08x\n", capabilities.sql_async_mode));
  FXTRACE((20, "sql_cursor_sensitivity               : %08x\n", capabilities.sql_cursor_sensitivity));
  FXTRACE((20, "sql_dynamic_cursor_attributes1       : %08x\n", capabilities.sql_dynamic_cursor_attributes1));
  FXTRACE((20, "sql_dynamic_cursor_attributes2       : %08x\n", capabilities.sql_dynamic_cursor_attributes2));
  FXTRACE((20, "sql_keyset_cursor_attributes1        : %08x\n", capabilities.sql_keyset_cursor_attributes1));
  FXTRACE((20, "sql_keyset_cursor_attributes2        : %08x\n", capabilities.sql_keyset_cursor_attributes2));
  FXTRACE((20, "sql_static_cursor_attributes1        : %08x\n", capabilities.sql_static_cursor_attributes1));
  FXTRACE((20, "sql_static_cursor_attributes2        : %08x\n", capabilities.sql_static_cursor_attributes2));
  FXTRACE((20, "sql_forward_only_cursor_attributes1  : %08x\n", capabilities.sql_forward_only_cursor_attributes1));
  FXTRACE((20, "sql_forward_only_cursor_attributes2  : %08x\n", capabilities.sql_forward_only_cursor_attributes2));
  FXTRACE((20, "sql_getdata_extensions               : %08x\n", capabilities.sql_getdata_extensions));
  FXTRACE((20, "sql_txn_capable                      : %04x\n", capabilities.sql_txn_capable));
  FXTRACE((20, "sql_default_txn_isolation            : %08x\n", capabilities.sql_default_txn_isolation));
  FXTRACE((20, "sql_cursor_commit_behavior           : %04x\n", capabilities.sql_cursor_commit_behavior));
  FXTRACE((20, "sql_cursor_rollback_behavior         : %04x\n", capabilities.sql_cursor_rollback_behavior));
  FXTRACE((20, "sql_multiple_active_txn              : %s\n", capabilities.sql_multiple_active_txn ? "Y" : "N"));
  FXTRACE((20, "sql_non_nullable_columns             : %04x\n", capabilities.sql_non_nullable_columns));
  FXTRACE((20, "sql_integrity                        : %s\n", capabilities.sql_integrity ? "Y" : "N"));
  FXTRACE((20, "sql_scroll_options                   : %08x\n", capabilities.sql_scroll_options));
}

void FXODBCInterface::disconnect() {
  checkStatus(TRUE);
  notifyAllQueries(this, MKUINT(FXDatabaseQuery::ID_DISCONNECT, SEL_EVENT), this);
  SQLDisconnect(hConn);
  SQLFreeHandle(SQL_HANDLE_DBC, hConn);
  hConn = NULL;
  connected = FALSE;
}

FXStringList FXODBCInterface::getTables() {
  FXStringList ret;
  SQLHSTMT hStmt;
  SQLSMALLINT nCols;
  SQLSMALLINT unused[4];
  SQLUINTEGER csize;
  FXchar *buf = NULL;
  long res;
  checkStatus(TRUE);
  res = SQLAllocHandle(SQL_HANDLE_STMT, hConn, &hStmt);
  if(!VALID(res)) odbcThrow(SQL_HANDLE_DBC, hConn);

  try {
    res = SQLTables(hStmt,  NULL, 0, NULL, 0, NULL, 0, (SQLCHAR *) "TABLE", SQL_NTS);
    if(!VALID(res)) odbcThrow(SQL_HANDLE_STMT, hStmt);
    res = SQLNumResultCols(hStmt, &nCols);
    if(!VALID(res)) odbcThrow(SQL_HANDLE_STMT, hStmt);
    if(nCols<0) dbThrow("Internal error", -1);

    res = SQLDescribeCol(hStmt, 3, NULL, 0, &unused[0], &unused[1], &csize, &unused[2], &unused[3]);
    if(!VALID(res)) odbcThrow(SQL_HANDLE_STMT, hStmt);

    FXMALLOC(&buf, FXchar, ++csize);
    res = SQLBindCol(hStmt, 3, SQL_C_CHAR, (SQLPOINTER) buf, (SQLINTEGER) csize, (SQLINTEGER *)&unused[0]);
    if(!VALID(res)) odbcThrow(SQL_HANDLE_STMT, hStmt);

    res = SQLFetch(hStmt);
    while(res!=SQL_NO_DATA) {
      if(!VALID(res)) odbcThrow(SQL_HANDLE_STMT, hStmt);
      ret.append(FXString(buf));
      res = SQLFetch(hStmt);
    }
  } catch(...) {
    SQLFreeHandle(SQL_HANDLE_STMT, hStmt);
    FXFREE(&buf);
    throw;
  }

  SQLFreeHandle(SQL_HANDLE_STMT, hStmt);
  FXFREE(&buf);
  return ret;
}

void FXODBCInterface::executeDirect(const FXString &command) {
  SQLHSTMT hStmt;
  long res;
  handle(this, MKUINT(ID_ALLOCHANDLE, SEL_EVENT), &hStmt);
  FXASSERT(hStmt);
  try {
    res = SQLExecDirect(hStmt, (SQLCHAR *) command.text(), SQL_NTS);
    if(!VALID(res)) odbcThrow(SQL_HANDLE_STMT, hStmt);
  } catch(...) {
    SQLFreeHandle(SQL_HANDLE_STMT, hStmt);
    throw;
  }
  SQLFreeHandle(SQL_HANDLE_STMT, hStmt);
}

long FXODBCInterface::onAllocHandle(FXObject *, FXSelector, void *data) {
  SQLHSTMT hStmt;
  long res;
  checkStatus(TRUE);
  res = SQLAllocHandle(SQL_HANDLE_STMT, hConn, &hStmt);
  if(!VALID(res)) odbcThrow(SQL_HANDLE_DBC, hConn);
  *(SQLHSTMT *) data = hStmt;
  return 1;
}

FXDatabaseQuery *FXODBCInterface::createQuery() {
  FXDatabaseQuery *ret;
  checkStatus(TRUE);
  ret = new FXODBCQuery(this, NULL, 0);
  queries.append(ret);
  return ret;
}

void FXODBCInterface::BeginTrans() {
  SQLRETURN res;
  SQLUINTEGER val;
  SQLINTEGER unused;
  checkStatus(TRUE);
  if(capabilities.sql_txn_capable==SQL_TC_NONE)
    dbThrow("This driver doesn't support transactions", -1);
  res = SQLGetConnectAttr(hConn, SQL_ATTR_AUTOCOMMIT, &val, sizeof(val), &unused);
  if(!VALID(res)) odbcThrow(SQL_HANDLE_DBC, hConn);
  if(val==SQL_AUTOCOMMIT_OFF) dbThrow("Transaction already in progress", -1);
  res = SQLSetConnectAttr(hConn, SQL_ATTR_AUTOCOMMIT, (SQLPOINTER) SQL_AUTOCOMMIT_OFF, 0);
  if(!VALID(res)) odbcThrow(SQL_HANDLE_DBC, hConn);
}

void FXODBCInterface::Commit() {
  SQLRETURN res;
  SQLUINTEGER val;
  SQLINTEGER unused;
  checkStatus(TRUE);
  if(capabilities.sql_txn_capable==SQL_TC_NONE)
    dbThrow("This driver doesn't support transactions", -1);
  res = SQLGetConnectAttr(hConn, SQL_ATTR_AUTOCOMMIT, &val, sizeof(val), &unused);
  if(!VALID(res)) odbcThrow(SQL_HANDLE_DBC, hConn);
  if(val==SQL_AUTOCOMMIT_ON) dbThrow("No transaction to commit", -1);
  res = SQLEndTran(SQL_HANDLE_DBC, hConn, SQL_COMMIT);
  if(!VALID(res)) odbcThrow(SQL_HANDLE_DBC, hConn);
  res = SQLSetConnectAttr(hConn, SQL_ATTR_AUTOCOMMIT, (SQLPOINTER) SQL_AUTOCOMMIT_ON, 0);
  if(!VALID(res)) odbcThrow(SQL_HANDLE_DBC, hConn);
}

void FXODBCInterface::Rollback() {
  SQLRETURN res;
  SQLUINTEGER val;
  SQLINTEGER unused;
  checkStatus(TRUE);
  if(capabilities.sql_txn_capable==SQL_TC_NONE)
    dbThrow("This driver doesn't support transactions", -1);
  res = SQLGetConnectAttr(hConn, SQL_ATTR_AUTOCOMMIT, &val, sizeof(val), &unused);
  if(!VALID(res)) odbcThrow(SQL_HANDLE_DBC, hConn);
  if(val==SQL_AUTOCOMMIT_ON) dbThrow("No transaction to rollback", -1);
  res = SQLEndTran(SQL_HANDLE_DBC, hConn, SQL_ROLLBACK);
  if(!VALID(res)) odbcThrow(SQL_HANDLE_DBC, hConn);
  res = SQLSetConnectAttr(hConn, SQL_ATTR_AUTOCOMMIT, (SQLPOINTER) SQL_AUTOCOMMIT_ON, 0);
  if(!VALID(res)) odbcThrow(SQL_HANDLE_DBC, hConn);
}


}
