/********************************************************************************
*                                                                               *
*                  Log file object                                              *
*                                                                               *
*********************************************************************************
* Copyright (C) 2003 by Mathew Robertson.   All Rights Reserved.                *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
*********************************************************************************/
#include <config.h>
#include <fox/fxver.h>
#include <fox/xincs.h>
#include <fox/fxdefs.h>
#include <fox/FXStream.h>
#include <fox/FXString.h>
#include <fox/FXSize.h>
#include <fox/FXPoint.h>
#include <fox/FXRectangle.h>
#include <fox/FXRegistry.h>
#include <fox/FXApp.h>
#include <fox/FXException.h>
#include <fox/FXFile.h>
using namespace FX;
#include "FXExtendedException.h"
#include "FXDateTime.h"
#include "FXLogger.h"
using namespace FXEX;
namespace FXEX {

/*
 * Notes:
 * - Currently  only allows for file open for read and write - it is missing append.  As such,
 *   it will create a new log file on startup, rather than append to an existing log file.
 * - The stream operator<< dont append a timestamp to the log file correctly.  It only works
 *   correctly when the first stream item is an FXchar...
 * - File permission enums' are not currently part of FOX, but is included in this header
 */

FXLogger* FXLogger::thisLog = NULL;

// ctor
FXLogger::FXLogger(const FXString& file,const FXlong size,const FXuint mode,const FXbool timestamp) : FXFileStream(),filename(file),filesize(size),filemode(mode),enabled(FALSE),stamp(timestamp),nextStamp(TRUE) {
  trim();
  if (open()) thisLog=this;
  }

// dtor
FXLogger::~FXLogger(){
  close();
  thisLog=(FXLogger*)-1;
  }

// instance
FXLogger& FXLogger::instance(){
  if (!thisLog) {
    FXString name=FXApp::instance()->getAppName();
    name.append(".log");
    new FXLogger(name);
    }
  return *thisLog;
  }

// accessors
void FXLogger::log(const FXchar* msg, ...){
  if (enabled && filesize>0 && trim()){
    FXString message;
    va_list args;
    va_start(args,msg);
    message.vformat(msg, args);
    va_end(args);
    if (stamp) {
      FXString timestring = FXDateTime::convert(FXDateTime::now());
      timestring.append(" ");
      message.prepend(timestring);
      }
    if (message.right(1) != "\n") message.append("\n");
    saveItems(message.text(),message.length());
    }
  }

void FXLogger::log(const FXString& msg){
  log("%s",msg.text());
  }

void FXLogger::log(const FXException* e){
  log( "EXC > %s",e->what() );
  }

void FXLogger::log(const FXExtendedException* e){
  log( "EXC > %s (%d)",e->what(),e->error() );
  }

void FXLogger::status(const FXString& code, const FXString& msg){
  log( "%s > %s",code.text(),msg.text() );
  }

void FXLogger::warning(const FXString& msg){
  log( "WRN > %s",msg.text() );
  }

void FXLogger::error(const FXString& msg){
  log( "ERR > %s",msg.text() );
  }

void FXLogger::debug(const FXString& msg){
  log( "DBG > %s",msg.text() );
  }

// set the logfile filename
FXbool FXLogger::name(const FXString& file){
  close();
  filename=file;
  return open();
  }

// open the logfile - if we can
FXbool FXLogger::open(){
  if (filesize>0){
    enabled = FXFileStream::open (filename,FXStreamSave);
    if (enabled){
      FXFile::mode(filename, filemode);
      log("=== Logfile open: %s ===");
      }
    }
  return enabled;
  }

// close the logfile - should always return FALSE
FXbool FXLogger::close(){
  if (enabled){
    log ("=== Logfile close ===");
    enabled = FXFileStream::close();
    }
  return enabled;
  }

// checks the current filesize against the required size
FXbool FXLogger::trim(){
  if ( FXFile::size(filename) > filesize ){
    close();
    FXFile::move(filename, FXStringFormat("%s.old",filename.text()), TRUE);
    open();
    }
  return enabled;
  }

// when using the << stream operators, we check if we are ready for a new timestamp
void FXLogger::checkStreamStamp(const FXchar& v){
  if (nextStamp){
    nextStamp=FALSE;
    trim();
    FXString timestring= FXDateTime::convert(FXDateTime::now());
    timestring.append(" ");
    saveItems(timestring.text(),timestring.length());
    }
  else if (v == '\n') nextStamp=TRUE;
  }

// handle incoming char's
FXStream& FXLogger::operator<<(const FXchar& v){
  checkStreamStamp(v);
  return FXFileStream::operator<<(v);
  }

// handle incoming uchar's
FXStream& FXLogger::operator<<(const FXuchar& v){
  // FIXME hmm... this is probably not the best way to do this
  checkStreamStamp((const FXchar&) v);
  return FXFileStream::operator<<(v);
  }

}

