/********************************************************************************
*                                                                               *
*                  Stream Buffering object                                      *
*                                                                               *
*********************************************************************************
* Copyright (C) 2003 by Mathew Robertson.   All Rights Reserved.                *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
*********************************************************************************/
#include <config.h>
#include <fox/fxver.h>
#include <fox/fxdefs.h>
#include <fox/FXStream.h>
#include <fox/FXObject.h>
using namespace FX;
#include "FXBufferedStream.h"
using namespace FXEX;
namespace FXEX {

FXBufferedStream::FXBufferedStream(FXStream *s,const FXObject* cont):FXMemoryStream(cont){
  stream=s;
  }

// Save items to buffer - if and only if the buffer is full:
// - call bufferSave() so that child classes can do some useful stuff (ie compress the data)
// - reset the buffer so that subsequent saveItems() can put the data somewhere
// - since the buffer is full, the last saveItems() failed, so re-call the saveItems()
void FXBufferedStream::saveItems(const void *buf,unsigned long n){
  FXMemoryStream::saveItems(buf,n);
  if (code==FXStreamFull){
    if (bufferSave()){ FXMemoryStream::saveItems(buf,n); }
    }
  }

// Load items out of buffer - if we have retrieved everything, we have to
// - call bufferLoad() so that child classes can do some useful stuff (ie uncompress data from file)
// - reset the buffer so the subsequent loadItems() can get the data from somewhere
// - since the buffer was at the end, the last loadItems() failed, so re-call the loadItems()
void FXBufferedStream::loadItems(void *buf,unsigned long n){
  FXMemoryStream::loadItems(buf,n);
  if (code==FXStreamEnd){
    if (bufferLoad()){ FXMemoryStream::loadItems(buf,n); }
    }
  }

FXStream& FXBufferedStream::operator<<(const FXuchar& v){
  FXMemoryStream::operator<<(v);
  if (code==FXStreamFull){
    if (bufferSave()){ FXMemoryStream::operator<<(v); }
    }
  return *this;
  }

FXStream& FXBufferedStream::operator<<(const FXchar& v){
  FXMemoryStream::operator<<(v);
  if (code==FXStreamFull){
    if (bufferSave()){ FXMemoryStream::operator<<(v); }
    }
  return *this;
  }

FXStream& FXBufferedStream::operator>>(FXuchar& v){
  FXMemoryStream::operator>>(v);
  if (code==FXStreamEnd){
    if (bufferLoad()){ FXMemoryStream::operator>>(v); }
    }
  return *this;
  }

FXStream& FXBufferedStream::operator>>(FXchar& v){
  FXMemoryStream::operator>>(v);
  if (code==FXStreamEnd){
    if (bufferLoad()){ FXMemoryStream::operator>>(v); }
    }
  return *this;
  }

// save the buffer to the target stream
// - child classes mung the data before calling this method (ie its superclass method)
FXbool FXBufferedStream::bufferSave(){
  if (stream->status()==FXStreamOK){
    stream->save((FXuchar*)ptr,pos);
    if (stream->status()==FXStreamOK){
      code=FXStreamOK;
      return position(0);
      }
    }
  return FALSE;
  }

// load the target in from the target stream
// - child class first calls this method (ie its superclass) before un-munging the buffer
FXbool FXBufferedStream::bufferLoad(){
  if (stream->status()==FXStreamOK){
    stream->load((FXuchar*)ptr,pos);
    if (stream->status()==FXStreamOK){
      code=FXStreamOK;
      return position(0);
      }
    }
  return FALSE;
  }

// Child classes overload these to do something useful
// When we open a buffer we allow the child class to allocate a buffer of appropriate size
// ie 1. to read a compressed stream, the child class needs to figure out how much space it
//       wants the superclass to manage
//    2. to write to a compressed stream the child class allocates a size dependant on some
//       factor (such as to maximise compression ratio, speed, etc)
// => basically we dont have to do a whole lot here
FXbool FXBufferedStream::open(FXStreamDirection save_or_load){
  return FXMemoryStream::open(NULL,save_or_load);
  }

FXbool FXBufferedStream::open(FXuint sp,FXStreamDirection save_or_load){
  return FXMemoryStream::open(NULL,sp,save_or_load);
  }

// On stream load, we dont care if our application hasn't fully read in the the data contained
// in the buffer (though we probably should care...)
// On stream save, there may be data in our buffer which hasn't yet been handled by our subclass
// so just call bufferSave()
FXbool FXBufferedStream::close(){
  if (dir==FXStreamSave && pos) {
    if (!bufferSave()) return FALSE;
    }
  return FXMemoryStream::close();
  }

// this just pushes the data into the base stream
void FXBufferedStream::flush(){
  if (dir==FXStreamSave && pos) bufferSave();
  }

// just close the stream on destroy
FXBufferedStream::~FXBufferedStream(){
  close();
  }

}

