/********************************************************************************
*                                                                               *
*                  XML event generator object                                   *
*                                                                               *
*********************************************************************************
* Copyright (C) 2003 by Mathew Robertson.   All Rights Reserved.                *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
*********************************************************************************/
#ifndef FXXMLPARSER_H
#define FXXMLPARSER_H

#ifndef FXBASEOBJECT_H
#include "FXBaseObject.h"
#endif
namespace FXEX {
class FXXmlReader;

/**
 * This XML parser is designed to absorb FOX XML tag events.
 *
 * Each time a tag is encountered, the XmlReader embedded in FXXmlParser, forwards tag
 * events to this class.  We then do some handling of the tag and content by transposing
 * embedded &...; characters, etc.
 */
class FXAPI FXXmlParser : public FXBaseObject {
  FXDECLARE(FXXmlParser)

private:
  FXXmlReader *reader;
  FXbool       state;
  FXbool       whitespace;
  FXuint       stopPolling;

protected:
  /// for serialization
  FXXmlParser() : FXBaseObject() {}

  /// removes specified characters, provided they are not embedded in double quotes
  void stripChar(FXString& s,FXchar c='\n');

  /**
   * takes a string containing &...; embeded characters, replacing
   * them with the real thing
   */
  void replaceEmbedded(FXString& s);

  /// lookup table that takes the incoming string, transposing it to the outgoing string
  virtual FXString lookup(const FXString& s);

  /// removes any white space found between tags, which space is (' ', \r, \n)
  void removeWhitespace(FXString& s);

public:
  enum {
    ID_XML_PARSER=FXBaseObject::ID_LAST,
    ID_LAST
    };

public:
  long onTimeout(FXObject*,FXSelector,void*);
  long onChore(FXObject*,FXSelector,void*);
  long onTag(FXObject*,FXSelector,void*);
  long onContent(FXObject*,FXSelector,void*);
  long onReadError(FXObject*,FXSelector,void*);

public:
  /// create a reader for the incoming stream
  FXXmlParser(FXApp* a,FXStream* s,FXObject* tgt,FXSelector sel);

  /// create resources
  void create();

  /// get the current state of parsing
  FXbool getState() { return state; }

  /// set to new stream
  void setStream(FXStream *s);

  /// returns the stream
  FXStream* getStream();

  /// should we automatically discard whitespace
  void keepWhitespace(FXbool ws) { whitespace=ws; }

  /// get the state of whitespace discarding
  FXbool keepWhitespace() { return whitespace; }

  /// parse the stream
  FXbool parse();

  /// stop parsing the input stream and close all remaining tags
  void stop() { stopPolling=1; state=FALSE; }

  /// cleanup
  virtual ~FXXmlParser();
  };

} // namespace FXEX
#endif // FXXMLPARSER_H
