/********************************************************************************
*                                                                               *
*                  Primary Thread manager                                       *
*                                                                               *
*********************************************************************************
* Copyright (C) 2003 by Mathew Robertson.   All Rights Reserved.                *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
********************************************************************************/
#ifndef FXTHREADMANAGER_H
#define FXTHREADMANAGER_H

#ifndef FXBASEOBJECT_H
#include "FXBaseObject.h"
#endif
namespace FXEX {
class FXRunnable;
class FXFastMutex;

/// An FXThreadList is a list of threads running in the current process
typedef FXArray<FXRunnable*> FXThreadList;

/**
 * A Thread Manager is required to be created in the main thread, for a few reasons:
 *
 * 1. It allows worker thread to call 'wait()' on themselves to do non-busy thread waiting.
 *    This then means that if we exit our main thread, the worker thread will be woken,
 *    thus allowing the main thread to exit.
 *
 * 2. It allows worker thread to communicate with the main thread using the 'signal()' method.
 *    Internally the signal() method uses a form of mainthread->handle(..), thus allowing
 *    worker threads to communication with the main thread using the standard FOX event
 *    handling technique.
 *
 *    Note, this means that worker threads cant do a mainthread->handle(..).  The reason for
 *    this is that this would require a change to FOX at the lowest level which would
 *    severly impact performance for single threaded applications.
 *
 * FXThreadManager is a singleton object, ie you should only create one of them.
 */
class FXAPI FXThreadManager : public FXBaseObject {
  FXDECLARE(FXThreadManager)

private:
  FXThreadHandle     mainthread;  // the main thread
  FXThreadList       threadlist;  // list of currently executing threads
  FXFastMutex       *mutex;       // control access to various resoures (non-recursive)
  FXint              last;        // last pointed-to thread

  static FXThreadManager* manager;  // thread manager instance

protected:
  FXThreadManager();

public:
  /// create a thread manager
  FXThreadManager(FXApp *a,FXObject *tgt=NULL,FXSelector sel=0);

  /// get the current thread manager instance
  static FXThreadManager* instance();

  /// add a thread to the list of managed threads
  void add(FXRunnable *r);

  /// remove an existing thread
  void remove(FXRunnable *r);

  /// return the number of active threads (not including the main thread)
  FXint size();

  /// return the next available thread
  FXRunnable* next();

  /// dtor
  virtual ~FXThreadManager();
  };

} // namespace FXEX
#endif // FXTHREADMANAGER_H
