/********************************************************************************
*                                                                               *
*                  Database query-result interface                              *
*                                                                               *
*********************************************************************************
* Copyright (C) 2002 by Mathew Robertson.       All Rights Reserved.            *
* Copyright (C) 2003 by Giancarlo Formicuccia.  All Rights Reserved.            *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
*********************************************************************************/
#ifndef FXDATABASEQUERY_H
#define FXDATABASEQUERY_H

#ifndef FXBASEOBJECT_H
#include "FXBaseObject.h"
#endif
namespace FXEX {
class FXDatabaseInterface;

/**
 * Interface to data returned from a database query
 */
class FXAPI FXDatabaseQuery : public FXBaseObject {
  FXDECLARE_ABSTRACT(FXDatabaseQuery)

typedef struct {
  FXObject *tgt;
  FXSelector msg;
  FXString fldName;
  FXint fldPos;
} FXfldTarget;

typedef FXArray<FXfldTarget> FXfldTargetList;

protected:
  FXString              query;      // query performed
  FXDatabaseInterface   *database;  // points to real database instance
  FXDatabaseFieldList   fields;
  FXrsState             state;
  FXfldTargetList       fldTargets;
  FXbool                readOnly;

  /// build a query on a specific database
  FXDatabaseQuery(FXDatabaseInterface *dbi, FXObject *tgt=NULL, FXSelector sel=0);

  FXDatabaseQuery();
  void checkState(FXrsState should_be) const;
  void checkOpen(FXbool open) const;
  void checkEditable(FXbool editable) const;
  void broadcastMessage(FXdbEvType ev, void *data = NULL);
  virtual void notifyMove(FXuint currentPos); /* dispatch dbRefresh to targets */
  FXint findFldTarget(FXObject *tgt);
  FXbool bindTarget(FXint pos);
  FXbool unbindTarget(FXint pos);
  void initTarget(FXint tgt);
  virtual void deleteFields();
public:
  enum {
    ID_DISCONNECT = FXBaseObject::ID_LAST,
    ID_CONNECT,
    ID_DESTROY,
    ID_LAST
    };

public:
  long onDisconnect(FXObject *, FXSelector, void *);
  long onConnect(FXObject *, FXSelector, void *);
  long onDestroy(FXObject *, FXSelector, void *);
  long onFree(FXObject *, FXSelector, void *);

public:
  /// get database interface
  FXDatabaseInterface* getDatabase() const { return database; }

  /// get the last query
  FXString getQuery() const { checkOpen(TRUE); return query; }

  /// Add fld targets
  void addFldTarget(FXObject *tgt, FXSelector msg, const FXString &fld);
  void addFldTarget(FXObject *tgt, FXSelector msg, FXint fld);
  void removeFldTarget(FXObject *tgt);
  FXrsState getState() const { return state; }
  FXbool isOpen() const;
  FXbool isEditable() const;
  /* WARNING: derivated classes MUST call FXDatabaseQuery::these_methods
     if (and only if) the operation is successul.
     These functions perform message delivery to various targets. */
  virtual void Open(const FXString &query, FXbool readOnly=TRUE);
  virtual void Close();
  virtual void addNew();
  virtual void Edit();
  virtual void Delete();
//  virtual void Delete();
  virtual void Update();
  virtual void CancelUpdate();
  virtual void Free(); /* Self-destruction. Don't call ~FXDatabaseQuery directly. */

  /* Remember to call notifyMove() in derivated classes on success */
  virtual void Requery() = '\0';
  virtual FXbool moveFirst() = '\0';
  virtual FXbool moveNext() = '\0';
  virtual FXbool movePrevious() = '\0';
  virtual FXbool moveLast() = '\0';
  virtual FXbool moveTo(FXuint where) = '\0';
  virtual FXbool moveOf(FXint displacement) = '\0';

  /// Current position
  virtual FXuint currentPos() = '\0';

  /// Estimate record count
  virtual FXuint recordCount() = '\0';

  /// Field count
  FXint fieldCount();
  FXDatabaseField &operator[](FXint pos);

  /// Stream save and load
  virtual void save(FXStream& store) const;
  virtual void load(FXStream& store);

  /// dtor
  virtual ~FXDatabaseQuery();
  };

typedef FXObjectListOf<FXDatabaseQuery> FXDatabaseQueryList;

}  // namespace FXEX
#endif // FXDATABASEQUERY_H
