/********************************************************************
 *
 * NAME:     
 *   FXExThread - multiplatform threadding objects
 *
 * AUTHORS:
 *   Daniel Gehriger (gehriger@linkcad.com)
 *   fixes by Josh
 *   adapted to FOX from the omniThread library.
 *
 *   Copyright (c) 2000 by Daniel Gehriger.
 *   Copyright (c) 1994-1997 Olivetti & Oracle Research Laboratory
 *   Copyright (c) 1994-1999 AT&T Laboratories Cambridge
 *
 * PUPROSE:
 *   Thread abstraction class
 *
 * NOTE
 *
 * This library is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU Library General Public   
 * License as published by the Free Software Foundation; either  
 * version 2 of the License, or (at your option) any later version.
 *                                                                 
 * This library is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of  
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.                 
 *                                                                  
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free       
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 * $Id: FXExThread.cpp,v 1.8 2000/11/30 01:44:42 dgehrige Exp $
 *
 ********************************************************************/

#include <fx.h>
#include "FXExThread.h"

#include <stdlib.h>
#include <errno.h>
#define NoNanoSleep
#define __USE_XOPEN_EXTENDED
#if !defined(WIN32)
#define __timespec_defined 1
#define __USE_POSIX199309 1
//int nanosleep(const struct timespec *req, struct timespec *rem);

struct timespec {
  long int tv_sec;
  long int tv_nsec;
};

# include <time.h>

# if (defined(__GLIBC__) && __GLIBC__ >= 2)
//      || defined(__SCO_VERSION__)
//      || defined(__aix__)
#   include <sys/time.h>
// typedef of struct timeval and gettimeofday();
# endif

# if defined(__linux__) && defined(_MIT_POSIX_THREADS)
#   include <pthread/mit/sys/timers.h>
# endif

# if defined(__irix__) && defined(PthreadSupportThreadPriority)
#   if _POSIX_THREAD_PRIORITY_SCHEDULING
#     include <sched.h>
#   endif
# endif

# if (PthreadDraftVersion <= 6)
#   define ERRNO(x) (((x) != 0) ? (errno) : 0)
#   define THROW_ERRORS(x) { if ((x) != 0) throw FXExThreadFatal(errno); }
# else
#   define ERRNO(x) (x)
#   define THROW_ERRORS(x) { int rc = (x); if (rc != 0) throw FXExThreadFatal(rc); }
# endif
#else                 // defined(WIN32)
# include <process.h>
#endif                // defined(WIN32)



#ifdef WIN32
static void get_time_now(unsigned long* abs_sec, unsigned long* abs_nsec);
#endif                // defined(WIN32)

///////////////////////////////////////////////////////////////////////////
//
// Mutex
//
///////////////////////////////////////////////////////////////////////////

//------------------------------------------------------------------
// 
// FXExMutex constructor
//
//------------------------------------------------------------------
FXExMutex::FXExMutex(void)
{
#if !defined(WIN32)
# if (PthreadDraftVersion == 4)
  THROW_ERRORS(pthread_mutex_init(&posix_mutex, pthread_mutexattr_default));
# else
  THROW_ERRORS(pthread_mutex_init(&posix_mutex, 0));
# endif
#else                 // defined(WIN32)
  InitializeCriticalSection(&crit);
#endif                // defined(WIN32)
}

//------------------------------------------------------------------
// 
// FXExMutex destructor
//
//------------------------------------------------------------------
FXExMutex::~FXExMutex(void)
{
#if !defined(WIN32)
  THROW_ERRORS(pthread_mutex_destroy(&posix_mutex));
#else                 // defined(WIN32)
  DeleteCriticalSection(&crit);
#endif                // defined(WIN32)
}

//------------------------------------------------------------------
// 
// lock
//
//------------------------------------------------------------------
void FXExMutex::lock(void)
{
#if !defined(WIN32)
  THROW_ERRORS(pthread_mutex_lock(&posix_mutex));
#else                 // defined(WIN32)
  EnterCriticalSection(&crit);
#endif                // defined(WIN32)
}

//------------------------------------------------------------------
// 
// unlock
//
//------------------------------------------------------------------
void FXExMutex::unlock(void)
{
#if !defined(WIN32)
  THROW_ERRORS(pthread_mutex_unlock(&posix_mutex));
#else                 // defined(WIN32)
  LeaveCriticalSection(&crit);
#endif                // defined(WIN32)
}


///////////////////////////////////////////////////////////////////////////
//
// Condition variable
//
//
// Condition variables are tricky to implement using NT synchronisation
// primitives, since none of them have the atomic "release mutex and wait to be
// signalled" which is central to the idea of a condition variable.  To get
// around this the solution is to record which threads are waiting and
// explicitly wake up those threads.
//
// Here we implement a condition variable using a list of waiting threads
// (protected by a critical section), and a per-thread semaphore (which
// actually only needs to be a binary semaphore).
//
// To wait on the cv, a thread puts itself on the list of waiting threads for
// that cv, then releases the mutex and waits on its own personal semaphore.  A
// signalling thread simply takes a thread from the head of the list and kicks
// that thread's semaphore.  Broadcast is simply implemented by kicking the
// semaphore of each waiting thread.
//
// The only other tricky part comes when a thread gets a timeout from a timed
// wait on its semaphore.  Between returning with a timeout from the wait and
// entering the critical section, a signalling thread could get in, kick the
// waiting thread's semaphore and remove it from the list.  If this happens,
// the waiting thread's semaphore is now out of step so it needs resetting, and
// the thread should indicate that it was signalled rather than that it timed
// out.
//
///////////////////////////////////////////////////////////////////////////

#ifdef WIN32
//------------------------------------------------------------------
// 
// _internal_fxexThread_helper
//
// It is possible that the thread calling wait or timedwait is not a
// FXExThread. In this case we have to provide a temporary data structure,
// i.e. for the duration of the call, for the thread to link itself on the
// list of waiting threads. _internal_fxexThread_dummy provides such
// a data structure and _internal_fxexThread_helper is a helper class to
// deal with this special case for wait() and timedwait(). Once created,
// the _internal_fxexThread_dummy is cached for use by the next wait() or
// timedwait() call from a non-FXExThread. This is probably worth doing
// because creating a Semaphore is quite heavy weight.
//
//------------------------------------------------------------------
class _internal_fxexThread_helper;

class _internal_fxexThread_dummy : public FXExThread {
public:
  inline _internal_fxexThread_dummy() : next(0) { }
  inline ~_internal_fxexThread_dummy() { }
  friend class _internal_fxexThread_helper;
private:
  _internal_fxexThread_dummy* next;
};

class _internal_fxexThread_helper {
public:
  inline _internal_fxexThread_helper()  { 
    d = 0;
    t = FXExThread::self();
    if (!t) {
      FXExMutexLock sync(cachelock);
      if (cache) {
        d = cache;
        cache = cache->next;
      }
      else {
        d = new _internal_fxexThread_dummy;
      }
      t = d;
    }
  }
  inline ~_internal_fxexThread_helper() { 
    if (d) {
      FXExMutexLock sync(cachelock);
      d->next = cache;
      cache = d;
    }
  }
  inline operator FXExThread* () { return t; }
  inline FXExThread* operator->() { return t; }
  
  static _internal_fxexThread_dummy* cache;
  static FXExMutex                   cachelock;
  
private:
  _internal_fxexThread_dummy* d;
  FXExThread*                 t;
};

_internal_fxexThread_dummy* _internal_fxexThread_helper::cache = 0;
FXExMutex                   _internal_fxexThread_helper::cachelock;

#endif                // defined(WIN32)


//------------------------------------------------------------------
// 
// FXExCondition constructor
//
//------------------------------------------------------------------
FXExCondition::FXExCondition(FXExMutex* m) : mutex(m)
{
#if !defined(WIN32)
# if (PthreadDraftVersion == 4)
  THROW_ERRORS(pthread_cond_init(&posix_cond, pthread_condattr_default));
# else
  THROW_ERRORS(pthread_cond_init(&posix_cond, 0));
# endif
#else                 // defined(WIN32)
  InitializeCriticalSection(&crit);
  waiting_head = waiting_tail = NULL;
#endif                // defined(WIN32)
}

//------------------------------------------------------------------
// 
// FXExCondition destructor
//
//------------------------------------------------------------------
FXExCondition::~FXExCondition(void)
{
#if !defined(WIN32)
  THROW_ERRORS(pthread_cond_destroy(&posix_cond));
#else                 // defined(WIN32)
  DeleteCriticalSection(&crit);
#endif                // defined(WIN32)
}

//------------------------------------------------------------------
// 
// wait
//
//------------------------------------------------------------------
void FXExCondition::wait(void)
{
#if !defined(WIN32)
  THROW_ERRORS(pthread_cond_wait(&posix_cond, &mutex->posix_mutex));
#else                 // defined(WIN32)
  _internal_fxexThread_helper me;
  
  EnterCriticalSection(&crit);
  
  me->cond_next = NULL;
  me->cond_prev = waiting_tail;
  if (waiting_head == NULL)
    waiting_head = me;
  else
    waiting_tail->cond_next = me;
  waiting_tail = me;
  me->cond_waiting = TRUE;
  
  LeaveCriticalSection(&crit);
  
  mutex->unlock();
  
  DWORD result = WaitForSingleObject(me->cond_semaphore, INFINITE);
  
  mutex->lock();
  
  if (result != WAIT_OBJECT_0)
    throw FXExThreadFatal(GetLastError());
#endif                // defined(WIN32)
}

//------------------------------------------------------------------
// 
// timedwait
//
//------------------------------------------------------------------
int FXExCondition::timedwait(unsigned long secs, unsigned long nanosecs)
{
#if !defined(WIN32)

  struct timespec rqts = { secs, nanosecs };

again:
  int rc = ERRNO(pthread_cond_timedwait(&posix_cond,
                    &mutex->posix_mutex, &rqts));
  switch (rc)
  {
  case 0:           // signaled
    return 1;

  case ETIMEDOUT:   // timed out
    return 0;

# if (PthreadDraftVersion <= 6)
  case EAGAIN:      // timed out
    return 0;
# endif

  case EINTR:
    // Some versions of unix produces this errno when the wait was
    // interrupted by a unix signal or fork.
    // Some versions of the glibc 2.0.x produces this errno when the 
    // program is debugged under gdb. Straightly speaking this is non-posix
    // compliant. We catch this here to make debugging possible.
    goto again;

  default:          // invalid errno
    throw FXExThreadFatal(rc);
  }
    
#else                 // defined(WIN32)
  
  _internal_fxexThread_helper me;
  
  EnterCriticalSection(&crit);
  
  me->cond_next = NULL;
  me->cond_prev = waiting_tail;
  if (waiting_head == NULL)
    waiting_head = me;
  else
    waiting_tail->cond_next = me;
  waiting_tail = me;
  me->cond_waiting = TRUE;
  
  LeaveCriticalSection(&crit);
  
  mutex->unlock();
  
  unsigned long now_sec, now_nsec;
  get_time_now(&now_sec, &now_nsec);
  
  DWORD timeout;
  if ((secs <= now_sec) && ((secs < now_sec) || (nanosecs < now_nsec)))
    timeout = 0;
  else {
    timeout = (secs-now_sec) * 1000;
    
    if( nanosecs < now_nsec )  
      timeout -= (now_nsec-nanosecs) / 1000000;
    else                       timeout += (nanosecs-now_nsec) / 1000000;
  }
  
  DWORD result = WaitForSingleObject(me->cond_semaphore, timeout);
  
  if (result == WAIT_TIMEOUT) {
    EnterCriticalSection(&crit);
    
    if (me->cond_waiting) {
      if (me->cond_prev != NULL)
        me->cond_prev->cond_next = me->cond_next;
      else
        waiting_head = me->cond_next;
      if (me->cond_next != NULL)
        me->cond_next->cond_prev = me->cond_prev;
      else
        waiting_tail = me->cond_prev;
      me->cond_waiting = FALSE;
      
      LeaveCriticalSection(&crit);
      
      mutex->lock();
      return 0;
    }
    
    //
    // We timed out but another thread still signalled us.  Wait for
    // the semaphore (it _must_ have been signalled) to decrement it
    // again.  Return that we were signalled, not that we timed out.
    //
    
    LeaveCriticalSection(&crit);
    
    result = WaitForSingleObject(me->cond_semaphore, INFINITE);
  }
  
  if (result != WAIT_OBJECT_0)
    throw FXExThreadFatal(GetLastError());
  
  mutex->lock();
  return 1;
#endif                // defined(WIN32)
}

//------------------------------------------------------------------
// 
// signal
//
//------------------------------------------------------------------
void FXExCondition::signal(void)
{
#if !defined(WIN32)
  THROW_ERRORS(pthread_cond_signal(&posix_cond));
#else                 // defined(WIN32)
  EnterCriticalSection(&crit);
  
  if (waiting_head != NULL) {
    FXExThread* t = waiting_head;
    waiting_head = t->cond_next;
    if (waiting_head == NULL)
      waiting_tail = NULL;
    else
      waiting_head->cond_prev = NULL;
    t->cond_waiting = FALSE;
    
    if (!ReleaseSemaphore(t->cond_semaphore, 1, NULL)) {
      int rc = GetLastError();
      LeaveCriticalSection(&crit);
      throw FXExThreadFatal(rc);
    }
  }
  
  LeaveCriticalSection(&crit);
#endif                // defined(WIN32)
}

//------------------------------------------------------------------
// 
// broadcast
//
//------------------------------------------------------------------
void FXExCondition::broadcast(void)
{
#if !defined(WIN32)
  THROW_ERRORS(pthread_cond_broadcast(&posix_cond));
#else                 // defined(WIN32)
  EnterCriticalSection(&crit);
  
  while (waiting_head != NULL) {
    FXExThread* t = waiting_head;
    waiting_head = t->cond_next;
    if (waiting_head == NULL)
      waiting_tail = NULL;
    else
      waiting_head->cond_prev = NULL;
    t->cond_waiting = FALSE;
    
    if (!ReleaseSemaphore(t->cond_semaphore, 1, NULL)) {
      int rc = GetLastError();
      LeaveCriticalSection(&crit);
      throw FXExThreadFatal(rc);
    }
  }
  
  LeaveCriticalSection(&crit);
#endif                // defined(WIN32)
}



///////////////////////////////////////////////////////////////////////////
//
// Counting semaphore
//
///////////////////////////////////////////////////////////////////////////
#if defined(WIN32)
#define SEMAPHORE_MAX 0x7fffffff
#endif                // defined(WIN32)


//------------------------------------------------------------------
// 
// FXExSemaphore constructor
//
//------------------------------------------------------------------
FXExSemaphore::FXExSemaphore(unsigned int initial) 
#if !defined(WIN32)
  : c(&m)
#endif                // defined(WIN32)
{
#if !defined(WIN32)
  value = initial;
#else                 // defined(WIN32)
  nt_sem = CreateSemaphore(NULL, initial, SEMAPHORE_MAX, NULL);
  
  if (nt_sem == NULL) {
    throw FXExThreadFatal(GetLastError());
  }
#endif                // defined(WIN32)
}

//------------------------------------------------------------------
// 
// FXExSemaphore destructor
//
//------------------------------------------------------------------
FXExSemaphore::~FXExSemaphore(void)
{
#ifdef WIN32
  if (!CloseHandle(nt_sem)) {
    throw FXExThreadFatal(GetLastError());
  }
#endif                // defined(WIN32)
}

//------------------------------------------------------------------
// 
// wait
//
//------------------------------------------------------------------
void FXExSemaphore::wait(void)
{
#if !defined(WIN32)
  //  FXExMutex_lock l(m);
  
  while (value == 0)
    c.wait();
  
  value--;
#else                 // defined(WIN32)
  if (WaitForSingleObject(nt_sem, INFINITE) != WAIT_OBJECT_0)
    throw FXExThreadFatal(GetLastError());
#endif                // defined(WIN32)
}

//------------------------------------------------------------------
// 
// trywait
//
//------------------------------------------------------------------
int FXExSemaphore::trywait(void)
{
#if !defined(WIN32)
  FXExMutexLock l(m);
  
  if (value == 0)
  {
    return 0;
  }
  else
  {
    --value;
    return 1;
  }
  
#else                 // defined(WIN32)
  switch (WaitForSingleObject(nt_sem, 0)) 
  {    
  case WAIT_OBJECT_0:
    return 1;
    
  case WAIT_TIMEOUT:
    return 0;

  default:
    throw FXExThreadFatal(GetLastError());
  }
  
  return 0; /* keep msvc++ happy */
#endif                // defined(WIN32)
}

//------------------------------------------------------------------
// 
// post
//
//------------------------------------------------------------------
void FXExSemaphore::post(void)
{
#if !defined(WIN32)
  {
    FXExMutexLock l(m);
    value++;
  }
  
  c.signal();
#else                 // defined(WIN32)
  if (!ReleaseSemaphore(nt_sem, 1, NULL))
    throw FXExThreadFatal(GetLastError());
#endif                // defined(WIN32)
}


///////////////////////////////////////////////////////////////////////////
//
// Thread
//
///////////////////////////////////////////////////////////////////////////

//
// Static variables
//
int FXExThread::init_t::count = 0;
FXExMutex* FXExThread::next_id_mutex;
int FXExThread::next_id = 0;

#if !defined(WIN32)

static pthread_key_t self_key;
static size_t stack_size = 0;

# ifdef PthreadSupportThreadPriority
static int lowest_priority;
static int normal_priority;
static int highest_priority;
# endif

#else                 // defined(WIN32)

static DWORD self_tls_index;
static unsigned int stack_size = 0;

#endif                // defined(WIN32)

//------------------------------------------------------------------
// 
// Initialisation function (gets called before any user code).
//
//------------------------------------------------------------------
FXExThread::init_t::init_t(void)
{
  if (count++ != 0)	// only do it once however many objects get created.
    return;
  
#if !defined(WIN32)
    
# ifdef NeedPthreadInit
  pthread_init();
# endif
  
# if (PthreadDraftVersion == 4)
  THROW_ERRORS(pthread_keycreate(&self_key, NULL));
# else
  THROW_ERRORS(pthread_key_create(&self_key, NULL));
# endif
  
# ifdef PthreadSupportThreadPriority
  
#   if defined(__osf1__) && defined(__alpha__) || defined(__VMS)
  
  lowest_priority = PRI_OTHER_MIN;
  highest_priority = PRI_OTHER_MAX;
  
#   elif defined(__hpux__)
  lowest_priority = PRI_OTHER_MIN;
  highest_priority = PRI_OTHER_MAX;
  
#   elif defined(__sunos__) && (__OSVERSION__ == 5)
  
  // a bug in pthread_attr_setschedparam means lowest priority is 1 not 0
  
  lowest_priority  = 1;
  highest_priority = 3;
  
#   else
  
  lowest_priority = sched_get_priority_min(SCHED_FIFO);
  highest_priority = sched_get_priority_max(SCHED_FIFO);
  
#   endif
  
  switch (highest_priority - lowest_priority) {
    
  case 0:
  case 1:
    normal_priority = lowest_priority;
    break;
    
  default:
    normal_priority = lowest_priority + 1;
    break;
  }
  
# endif   /* PthreadSupportThreadPriority */
  
    
  next_id_mutex = new FXExMutex;
  
  //
  // Create object for this (i.e. initial) thread.
  //
  
  FXExThread* t = new FXExThread;
  
  t->_state = STATE_RUNNING;
  t->posix_thread = pthread_self ();
    
  THROW_ERRORS(pthread_setspecific(self_key, (void*)t));
  
# ifdef PthreadSupportThreadPriority
  
#   if (PthreadDraftVersion == 4)
  
  THROW_ERRORS(pthread_setprio(t->posix_thread,
    posix_priority(PRIORITY_NORMAL)));
  
#   elif (PthreadDraftVersion == 6)
  
  pthread_attr_t attr;
  pthread_attr_init(&attr);
  
  THROW_ERRORS(pthread_attr_setprio(&attr, posix_priority(PRIORITY_NORMAL)));
  
  THROW_ERRORS(pthread_setschedattr(t->posix_thread, attr));
  
#   else
  
  struct sched_param sparam;
  
  sparam.sched_priority = posix_priority(PRIORITY_NORMAL);
  
  THROW_ERRORS(pthread_setschedparam(t->posix_thread, SCHED_OTHER, &sparam));
  
#   endif   /* PthreadDraftVersion */
# endif   /* PthreadSupportThreadPriority */

//
//
#else                 // defined(WIN32)
//
//
   
  self_tls_index = TlsAlloc();
  if (self_tls_index == 0xffffffff)
    throw FXExThreadFatal(GetLastError());
    
  next_id_mutex = new FXExMutex;
  
  //
  // Create object for this (i.e. initial) thread.
  //
  
  FXExThread* t = new FXExThread;
  
  t->_state = STATE_RUNNING;
  if (!DuplicateHandle(GetCurrentProcess(), GetCurrentThread(),
			 GetCurrentProcess(), &t->handle,
       0, FALSE, DUPLICATE_SAME_ACCESS))
       throw FXExThreadFatal(GetLastError());
  
  t->nt_id = GetCurrentThreadId();
    
  if (!TlsSetValue(self_tls_index, (LPVOID)t))
    throw FXExThreadFatal(GetLastError());
  
  if (!SetThreadPriority(t->handle, nt_priority(PRIORITY_NORMAL)))
    throw FXExThreadFatal(GetLastError());
#endif                // defined(WIN32)
}


//------------------------------------------------------------------
// 
// Wrapper for thread creation.
//
//------------------------------------------------------------------
extern "C" FXEX_THREAD_WRAPPER
{
  FXExThread* me = (FXExThread*)ptr;
    
#if !defined(WIN32)
  THROW_ERRORS(pthread_setspecific(self_key, me));
#else                 // defined(WIN32)
  if (!TlsSetValue(self_tls_index, (LPVOID)me))
    throw FXExThreadFatal(GetLastError());
#endif                // defined(WIN32)
  
  //
  // Now invoke the thread function with the given argument.
  //
  
  if (me->fn_void != NULL) {
    (*me->fn_void)(me->thread_arg);
    FXExThread::exit();
  }
  
  if (me->fn_ret != NULL) {
    void* return_value = (*me->fn_ret)(me->thread_arg);
    FXExThread::exit(return_value);
  }
  
  if (me->detached) {
    me->run(me->thread_arg);
    FXExThread::exit();
  } else {
    void* return_value = me->run_undetached(me->thread_arg);
    FXExThread::exit(return_value);
  }
  
  // should never get here.
#if !defined(WIN32)
  return NULL;
#else                 // defined(WIN32)
# ifndef __BCPLUSPLUS__
  return 0;
# endif
#endif                // defined(WIN32)
}

//------------------------------------------------------------------
// 
// construct a detached thread running a given function.
//
//------------------------------------------------------------------
FXExThread::FXExThread(void (*fn)(void*), void* arg, priority_t pri)
{
  common_constructor(arg, pri, 1);
  fn_void = fn;
  fn_ret = NULL;
}

//------------------------------------------------------------------
// 
// construct an undetached thread running a given function.
//
//------------------------------------------------------------------
FXExThread::FXExThread(void* (*fn)(void*), void* arg, priority_t pri)
{
  common_constructor(arg, pri, 0);
  fn_void = NULL;
  fn_ret = fn;
}

//------------------------------------------------------------------
// 
// construct a thread which will run either run() or run_undetached().
//
//------------------------------------------------------------------
FXExThread::FXExThread(void* arg, priority_t pri)
{
  common_constructor(arg, pri, 1);
  fn_void = NULL;
  fn_ret = NULL;
}

//------------------------------------------------------------------
// 
// common part of all constructors.
//
//------------------------------------------------------------------
void FXExThread::common_constructor(void* arg, priority_t pri, int det)
{
  _state = STATE_NEW;
  _priority = pri;
  
  next_id_mutex->lock();
  _id = next_id++;
  next_id_mutex->unlock();
  
  thread_arg = arg;
  detached = det;	// may be altered in start_undetached()
  // posix_thread is set up in initialisation routine or start().
  
#if defined(WIN32)
  cond_semaphore = CreateSemaphore(NULL, 0, SEMAPHORE_MAX, NULL);
  
  if (cond_semaphore == NULL)
    throw FXExThreadFatal(GetLastError());
  
  cond_next = cond_prev = NULL;
  cond_waiting = FALSE;
  
  handle = NULL;
#endif                // defined(WIN32)
}


//------------------------------------------------------------------
// 
// Destructor for FXExThread
//
//------------------------------------------------------------------
FXExThread::~FXExThread(void)
{
#if defined(WIN32)
  if (handle && !CloseHandle(handle))
    throw FXExThreadFatal(GetLastError());
  if (cond_semaphore && !CloseHandle(cond_semaphore))
    throw FXExThreadFatal(GetLastError());
#endif                // defined(WIN32)
}


//------------------------------------------------------------------
//
// Start the thread
//
//------------------------------------------------------------------
void FXExThread::start(void)
{  
  FXExMutexLock l(mutex);
 
  if (_state != STATE_NEW)
    throw FXExThreadInvalid();
 
#if !defined(WIN32)

  pthread_attr_t attr;
 
# if (PthreadDraftVersion == 4)
  pthread_attr_create(&attr);
# else
  pthread_attr_init(&attr);
# endif
 
# if (PthreadDraftVersion == 8)
  //  pthread_attr_setdetachstate(&attr,PTHREAD_CREATE_UNDETACHED);
# endif
 
# ifdef PthreadSupportThreadPriority
 
#   if (PthreadDraftVersion <= 6)
 
  THROW_ERRORS(pthread_attr_setprio(&attr, posix_priority(_priority)));
 
#   else
 
  struct sched_param sparam;
 
  sparam.sched_priority = posix_priority(_priority);
 
  THROW_ERRORS(pthread_attr_setschedparam(&attr, &sparam));
 
#   endif   /* PthreadDraftVersion */
 
# endif   /* PthreadSupportThreadPriority */
 
# if !defined(__linux__)
  if (stack_size) {
    THROW_ERRORS(pthread_attr_setstacksize(&attr, stack_size));
  }
# endif
 

# if (PthreadDraftVersion == 4)
  THROW_ERRORS(pthread_create(&posix_thread, attr, fxexThreadWrapper,
            (void*)this));
  pthread_attr_delete(&attr);
# else
  THROW_ERRORS(pthread_create(&posix_thread, &attr, fxexThreadWrapper,
            (void*)this));
  pthread_attr_destroy(&attr);
# endif
 
  _state = STATE_RUNNING;
 
  if (detached) {
# if (PthreadDraftVersion <= 6)
    THROW_ERRORS(pthread_detach(&posix_thread));
# else
    THROW_ERRORS(pthread_detach(posix_thread));
# endif
  }

//
//  
#else                 // defined(WIN32)
//
//  
# ifndef __BCPLUSPLUS__
  // MSVC++ or compatiable
  unsigned int t;
  handle = (HANDLE)_beginthreadex(
    NULL,
    stack_size,
    fxexThreadWrapper,
    (LPVOID)this,
    CREATE_SUSPENDED,
    &t);
  nt_id = t;
  if (handle == NULL)
    throw FXExThreadFatal(GetLastError());
# else
  // Borland C++
  handle = (HANDLE)_beginthreadNT(fxexThreadWrapper,
    stack_size,
    (void*)this,
    NULL,
    CREATE_SUSPENDED,
    &nt_id);
  if (handle == INVALID_HANDLE_VALUE)
    throw FXExThreadFatal(errno);
# endif
 
  if (!SetThreadPriority(handle, nt_priority(_priority)))
    throw FXExThreadFatal(GetLastError());
 
  if (ResumeThread(handle) == 0xffffffff)
    throw FXExThreadFatal(GetLastError());
 
  _state = STATE_RUNNING;
  
#endif                // defined(WIN32)
}


//------------------------------------------------------------------
//
// Start a thread which will run the member function run_undetached().
//
//------------------------------------------------------------------
void FXExThread::start_undetached(void)
{
  if ((fn_void != NULL) || (fn_ret != NULL))
    throw FXExThreadInvalid();
  
  detached = 0;
  start();
}

//------------------------------------------------------------------
//
// join - simply check error conditions & call 
//        pthread_join / WaitForSingleObject
//
//------------------------------------------------------------------
void FXExThread::join(void** status)
{
  mutex.lock();
  
  if ((_state != STATE_RUNNING) && (_state != STATE_TERMINATED)) {
    mutex.unlock();
    throw FXExThreadInvalid();
  }
  
  mutex.unlock();
  
  if (this == self())
    throw FXExThreadInvalid();
  
  if (detached)
    throw FXExThreadInvalid();
    
#if !defined(WIN32)
  
  THROW_ERRORS(pthread_join(posix_thread, status));
# if (PthreadDraftVersion == 4)
  // With draft 4 pthreads implementations (HPUX 10.x and
  // Digital Unix 3.2), have to detach the thread after 
  // join. If not, the storage for the thread will not be
  // be reclaimed.
  THROW_ERRORS(pthread_detach(&posix_thread));
# endif
  
#else                 // defined(WIN32)
  
  if (WaitForSingleObject(handle, INFINITE) != WAIT_OBJECT_0)
    throw FXExThreadFatal(GetLastError());
  
  if (status)
    *status = return_val;
  
#endif                // defined(WIN32)
  
  delete this;
}


//------------------------------------------------------------------
//
// Change this thread's priority.
//
//------------------------------------------------------------------
void FXExThread::set_priority(priority_t pri)
{
  FXExMutexLock l(mutex);
  
  if (_state != STATE_RUNNING)
    throw FXExThreadInvalid();
  
  _priority = pri;
  
#if !defined(WIN32)
# ifdef PthreadSupportThreadPriority
  
#   if (PthreadDraftVersion == 4)
  
  THROW_ERRORS(pthread_setprio(posix_thread, posix_priority(pri)));
  
#   elif (PthreadDraftVersion == 6)
  
  pthread_attr_t attr;
  pthread_attr_init(&attr);
  
  THROW_ERRORS(pthread_attr_setprio(&attr, posix_priority(pri)));
  
  THROW_ERRORS(pthread_setschedattr(posix_thread, attr));
  
#   else
  
  struct sched_param sparam;
  
  sparam.sched_priority = posix_priority(pri);
  
  THROW_ERRORS(pthread_setschedparam(posix_thread, SCHED_OTHER, &sparam));
  
#   endif   /* PthreadDraftVersion */
  
# endif   /* PthreadSupportThreadPriority */
  
#else                 // defined(WIN32)
  
  if (!SetThreadPriority(handle, nt_priority(pri)))
    throw FXExThreadFatal(GetLastError());
#endif                // defined(WIN32)
}

//------------------------------------------------------------------
//
// create - construct a new thread object and start it running.  Returns thread
// object if successful, null pointer if not.
//
// detached version
//
//------------------------------------------------------------------
FXExThread* FXExThread::create(void (*fn)(void*), void* arg, priority_t pri)
{
  FXExThread* t = new FXExThread(fn, arg, pri);
  
  t->start();
  return t;
}

//------------------------------------------------------------------
//
// create - construct a new thread object and start it running.  Returns thread
// object if successful, null pointer if not.
//
// undetached version
//
//------------------------------------------------------------------
FXExThread* FXExThread::create(void* (*fn)(void*), void* arg, priority_t pri)
{
  FXExThread* t = new FXExThread(fn, arg, pri);

  t->start();
  return t;
}


//------------------------------------------------------------------
//
// exit() _must_ lock the mutex even in the case of a detached thread.  This is
// because a thread may run to completion before the thread that created it has
// had a chance to get out of start().  By locking the mutex we ensure that the
// creating thread must have reached the end of start() before we delete the
// thread object.  Of course, once the call to start() returns, the user can
// still incorrectly refer to the thread object, but that's their problem.
//
//------------------------------------------------------------------
void FXExThread::exit(void* return_value)
{
  FXExThread* me = self();
  
#if !defined(WIN32)
  if (me)
  {
    me->mutex.lock();
    
    me->_state = STATE_TERMINATED;
    
    me->mutex.unlock();
        
    if (me->detached)
      delete me;
  }
  
  pthread_exit(return_value);
#else                 // defined(WIN32)
  if (me)
  {
    me->mutex.lock();
    
    me->_state = STATE_TERMINATED;
    
    me->mutex.unlock();
        
    if (me->detached)
    {
      delete me;
    }
    else 
    {
      me->return_val = return_value;
    }
  }
  else
  {
    FXTRACE((100, "FXExThread::exit: called with a FXExThread. Exit quietly."));
  }
 
# ifndef __BCPLUSPLUS__
  // MSVC++ or compatiable
  //   _endthreadex() does not automatically close the thread handle.
  //   The FXExThread dtor closes the thread handle.
  _endthreadex(0);
# else
  // Borland C++
  //   _endthread() does not automatically close the thread handle.
  //   _endthreadex() is only available if __MFC_COMPAT__ is defined and
  //   all it does is to call _endthread().
  _endthread();
# endif
#endif                // defined(WIN32)
}


//------------------------------------------------------------------
//
// self
//
//------------------------------------------------------------------
FXExThread* FXExThread::self(void)
{
#if !defined(WIN32)
  
  FXExThread* me;
  
# if (PthreadDraftVersion <= 6)
  THROW_ERRORS(pthread_getspecific(self_key, (void**)&me));
# else
  me = (FXExThread *)pthread_getspecific(self_key);
# endif
    
  return me;
  
#else                 // defined(WIN32)
  
  LPVOID me;
  me = TlsGetValue(self_tls_index);
  return (FXExThread*)me;
#endif                // defined(WIN32)
}


//------------------------------------------------------------------
//
// yield
//
//------------------------------------------------------------------
void FXExThread::yield(void)
{
#if !defined(WIN32)
# if (PthreadDraftVersion == 6)
  
  pthread_yield(NULL);
  
# elif (PthreadDraftVersion < 9)
  
  //  pthread_yield();
  
# else
  
  THROW_ERRORS(sched_yield());
  
# endif
#else                 // defined(WIN32)
  Sleep(0);
#endif                // defined(WIN32)
}


#ifdef WIN32
#define MAX_SLEEP_SECONDS (DWORD)4294966	// (2**32-2)/1000
#endif                // defined(WIN32)
//------------------------------------------------------------------
//
// sleep
//
//------------------------------------------------------------------
void FXExThread::sleep(unsigned long secs, unsigned long nanosecs)
{
#if !defined(WIN32)

  timespec rqts = { secs, nanosecs };
 
# ifndef NoNanoSleep
 
  timespec remain;
  while (nanosleep(&rqts, &remain)) {
    if (errno == EINTR) {
      rqts.tv_sec  = remain.tv_sec;
      rqts.tv_nsec = remain.tv_nsec;
      continue;
    }
    else
      throw FXExThreadFatal(errno);
  }
# else
 
#   if defined(__osf1__) && defined(__alpha__) \
        || defined(__hpux__) && (__OSVERSION__ == 10) \
        || defined(__VMS) || defined(__SINIX__) || defined (__POSIX_NT__)
 
  if (pthread_delay_np(&rqts) != 0)
    throw FXExThreadFatal(errno);
 
#   elif defined(__linux__) || defined(__aix__)
 
  if (secs > 2000) {
    while ((secs = ::sleep(secs))) ;
  } else {
    usleep(secs * 1000000 + (nanosecs / 1000));
  }

#   else
 
  throw FXExThreadInvalid();
 
#   endif
# endif   /* NoNanoSleep */
  
#else                 // defined(WIN32)

  if (secs <= MAX_SLEEP_SECONDS) {
    Sleep(secs * 1000 + nanosecs / 1000000);
    return;
  }

  DWORD no_of_max_sleeps = secs / MAX_SLEEP_SECONDS;

  for (DWORD i = 0; i < no_of_max_sleeps; i++)
    Sleep(MAX_SLEEP_SECONDS * 1000);

  Sleep((secs % MAX_SLEEP_SECONDS) * 1000 + nanosecs / 1000000);

#endif                // defined(WIN32)
}


//------------------------------------------------------------------
//
// sleep
//
//------------------------------------------------------------------
void FXExThread::get_time(unsigned long* abs_sec, 
                          unsigned long* abs_nsec,
                          unsigned long rel_sec, 
                          unsigned long rel_nsec)
{
#if !defined(WIN32)
  timespec abs;
  
# if defined(__osf1__) && defined(__alpha__) \
      || defined(__hpux__) && (__OSVERSION__ == 10) \
      || defined(__VMS) || defined(__SINIX__) || defined(__POSIX_NT__)
    
  timespec rel;
  rel.tv_sec = rel_sec;
  rel.tv_nsec = rel_nsec;
  THROW_ERRORS(pthread_get_expiration_np(&rel, &abs));
  
# else
  
#   if defined(__linux__) || defined(__aix__) || defined(__SCO_VERSION__)
  
  struct timeval tv;
  gettimeofday(&tv, NULL); 
  abs.tv_sec = tv.tv_sec;
  abs.tv_nsec = tv.tv_usec * 1000;
  
#   else	/* __linux__ || __aix__ */
  
  clock_gettime(CLOCK_REALTIME, &abs);
  
#   endif	/* __linux__ || __aix__ */
  
  abs.tv_nsec += rel_nsec;
  abs.tv_sec += rel_sec + abs.tv_nsec / 1000000000;
  abs.tv_nsec = abs.tv_nsec % 1000000000;
  
# endif	/* __osf1__ && __alpha__ */
  
  *abs_sec = abs.tv_sec;
  *abs_nsec = abs.tv_nsec;
#else                 // defined(WIN32)
  get_time_now(abs_sec, abs_nsec);
  *abs_nsec += rel_nsec;
  *abs_sec += rel_sec + *abs_nsec / 1000000000;
  *abs_nsec = *abs_nsec % 1000000000;
#endif                // defined(WIN32)
}


#if !defined(WIN32)
//------------------------------------------------------------------
// 
// posix_priority
//
//------------------------------------------------------------------
int FXExThread::posix_priority(priority_t pri)
{
# ifdef PthreadSupportThreadPriority
  switch (pri) {

  case PRIORITY_LOW:
    return lowest_priority;

  case PRIORITY_NORMAL:
    return normal_priority;

  case PRIORITY_HIGH:
    return highest_priority;

  }
# endif
  throw FXExThreadInvalid();
  
# ifdef _MSC_VER
  return 0; /* keep msvc++ happy */
# endif
}
  
#else                 // defined(WIN32)
//------------------------------------------------------------------
// 
// nt_priority
//
//------------------------------------------------------------------
int FXExThread::nt_priority(priority_t pri)
{
  switch (pri) 
  {
  case PRIORITY_LOW:
    return THREAD_PRIORITY_LOWEST;
    
  case PRIORITY_NORMAL:
    return THREAD_PRIORITY_NORMAL;
    
  case PRIORITY_HIGH:
    return THREAD_PRIORITY_HIGHEST;
  }

  throw FXExThreadInvalid();
  return 0; /* keep msvc++ happy */
}
#endif

#if defined(WIN32)
//------------------------------------------------------------------
// 
// get_time_now
//
//------------------------------------------------------------------
static void get_time_now(unsigned long* abs_sec, unsigned long* abs_nsec)
{
  static int days_in_preceding_months[12]
    = { 0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334 };
  static int days_in_preceding_months_leap[12]
    = { 0, 31, 60, 91, 121, 152, 182, 213, 244, 274, 305, 335 };
  
  SYSTEMTIME st;
  
  GetSystemTime(&st);
  *abs_nsec = st.wMilliseconds * 1000000;
  
  // this formula should work until 1st March 2100
  
  DWORD days = ((st.wYear - 1970) * 365 + (st.wYear - 1969) / 4
		  + ((st.wYear % 4)
      ? days_in_preceding_months[st.wMonth - 1]
      : days_in_preceding_months_leap[st.wMonth - 1])
      + st.wDay - 1);
  
  *abs_sec = st.wSecond + 60 * (st.wMinute + 60 * (st.wHour + 24 * days));
}
#endif                // defined(WIN32)

//------------------------------------------------------------------
// 
// stacksize
//
//------------------------------------------------------------------
void FXExThread::stacksize(unsigned long sz)
{
  stack_size = sz;
}

//------------------------------------------------------------------
// 
// stacksize
//
//------------------------------------------------------------------
unsigned long FXExThread::stacksize()
{
  return stack_size;
}
