// Copyright 2013 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef MEDIA_BLINK_WEBCONTENTDECRYPTIONMODULESESSION_IMPL_H_
#define MEDIA_BLINK_WEBCONTENTDECRYPTIONMODULESESSION_IMPL_H_

#include <string>
#include <vector>

#include "base/basictypes.h"
#include "base/callback.h"
#include "base/memory/ref_counted.h"
#include "base/memory/weak_ptr.h"
#include "media/base/media_keys.h"
#include "third_party/WebKit/public/platform/WebContentDecryptionModuleSession.h"
#include "third_party/WebKit/public/platform/WebString.h"

namespace media {

class CdmSessionAdapter;
class MediaKeys;

class WebContentDecryptionModuleSessionImpl
    : public blink::WebContentDecryptionModuleSession {
 public:
  WebContentDecryptionModuleSessionImpl(
      const scoped_refptr<CdmSessionAdapter>& adapter);
  virtual ~WebContentDecryptionModuleSessionImpl();

  // blink::WebContentDecryptionModuleSession implementation.
  virtual void setClientInterface(Client* client);
  virtual blink::WebString sessionId() const;

  virtual void initializeNewSession(
      blink::WebEncryptedMediaInitDataType init_data_type,
      const unsigned char* initData,
      size_t initDataLength,
      blink::WebEncryptedMediaSessionType session_type,
      blink::WebContentDecryptionModuleResult result);
  virtual void load(const blink::WebString& session_id,
                    blink::WebContentDecryptionModuleResult result);
  virtual void update(const uint8* response,
                      size_t response_length,
                      blink::WebContentDecryptionModuleResult result);
  virtual void close(blink::WebContentDecryptionModuleResult result);
  virtual void remove(blink::WebContentDecryptionModuleResult result);

  // Callbacks.
  void OnSessionMessage(MediaKeys::MessageType message_type,
                        const std::vector<uint8>& message);
  void OnSessionKeysChange(bool has_additional_usable_key,
                           CdmKeysInfo keys_info);
  void OnSessionExpirationUpdate(const base::Time& new_expiry_time);
  void OnSessionClosed();

 private:
  // Called when a new session is created.
  blink::WebContentDecryptionModuleResult::SessionStatus OnSessionInitialized(
      const std::string& session_id);

  scoped_refptr<CdmSessionAdapter> adapter_;

  // Non-owned pointer.
  Client* client_;

  // Session ID is the app visible ID for this session generated by the CDM.
  // This value is not set until the CDM resolves the initializeNewSession()
  // promise.
  std::string session_id_;

  // Don't pass more than 1 close() event to blink::
  // TODO(jrummell): Remove this once blink tests handle close() promise and
  // closed() event.
  bool is_closed_;

  // Since promises will live until they are fired, use a weak reference when
  // creating a promise in case this class disappears before the promise
  // actually fires.
  base::WeakPtrFactory<WebContentDecryptionModuleSessionImpl> weak_ptr_factory_;

  DISALLOW_COPY_AND_ASSIGN(WebContentDecryptionModuleSessionImpl);
};

}  // namespace media

#endif  // MEDIA_BLINK_WEBCONTENTDECRYPTIONMODULESESSION_IMPL_H_
