// Copyright 2016 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "services/ui/ws/frame_generator.h"

#include <memory>

#include "base/test/test_message_loop.h"
#include "cc/quads/render_pass.h"
#include "cc/quads/shared_quad_state.h"
#include "services/ui/ws/ids.h"
#include "services/ui/ws/platform_display_init_params.h"
#include "services/ui/ws/server_window.h"
#include "services/ui/ws/server_window_compositor_frame_sink_manager.h"
#include "services/ui/ws/test_server_window_delegate.h"
#include "services/ui/ws/test_utils.h"
#include "testing/gtest/include/gtest/gtest.h"

namespace ui {
namespace ws {
namespace test {
namespace {

// Typical id for the display root ServerWindow.
constexpr WindowId kRootDisplayId(0, 2);
const base::UnguessableToken kArbitraryToken = base::UnguessableToken::Create();

// Makes the window visible and creates the default surface for it.
void InitWindow(ServerWindow* window) {
  window->SetVisible(true);
  ServerWindowCompositorFrameSinkManager* compositor_frame_sink_manager =
      window->GetOrCreateCompositorFrameSinkManager();
  compositor_frame_sink_manager->SetLatestSurfaceInfo(
      mojom::CompositorFrameSinkType::DEFAULT,
      cc::SurfaceId(
          cc::FrameSinkId(
              WindowIdToTransportId(window->id()),
              static_cast<uint32_t>(mojom::CompositorFrameSinkType::DEFAULT)),
          cc::LocalFrameId(1u, kArbitraryToken)),
      gfx::Size(100, 100));
}

}  // namespace

class FrameGeneratorTest : public testing::Test {
 public:
  FrameGeneratorTest()
      : root_window_(base::MakeUnique<ServerWindow>(&window_delegate_,
                                                    kRootDisplayId)) {}
  ~FrameGeneratorTest() override {}

  // Calls DrawWindowTree() on |frame_generator_|
  void DrawWindowTree(cc::RenderPass* pass);

  ServerWindow* root_window() { return root_window_.get(); }

  TestServerWindowDelegate* test_window_delegate() { return &window_delegate_; }

 private:
  // testing::Test:
  void SetUp() override;
  void TearDown() override;

  std::unique_ptr<FrameGenerator> frame_generator_;
  std::unique_ptr<TestFrameGeneratorDelegate> frame_generator_delegate_;
  TestServerWindowDelegate window_delegate_;
  std::unique_ptr<ServerWindow> root_window_;

  // Needed so that Mojo classes can be initialized for ServerWindow use.
  base::TestMessageLoop message_loop_;

  DISALLOW_COPY_AND_ASSIGN(FrameGeneratorTest);
};

void FrameGeneratorTest::DrawWindowTree(cc::RenderPass* pass) {
  frame_generator_->DrawWindowTree(pass, root_window_.get(), gfx::Vector2d(),
                                   1.0f);
}

void FrameGeneratorTest::SetUp() {
  testing::Test::SetUp();
  frame_generator_delegate_ = base::MakeUnique<TestFrameGeneratorDelegate>();
  PlatformDisplayInitParams init_params;
  frame_generator_ = base::MakeUnique<FrameGenerator>(
      frame_generator_delegate_.get(), root_window_.get());
  InitWindow(root_window());
}

void FrameGeneratorTest::TearDown() {
  frame_generator_.reset();
  frame_generator_delegate_.reset();
}

// Tests correctness of the SharedQuadStateList generated by
// FrameGenerator::DrawWindowTree().
TEST_F(FrameGeneratorTest, DrawWindowTree) {
  ServerWindow child_window(test_window_delegate(), WindowId(1, 1));
  root_window()->Add(&child_window);
  InitWindow(&child_window);
  const float root_opacity = .5f;
  const float child_opacity = .4f;
  root_window()->SetOpacity(root_opacity);
  child_window.SetOpacity(child_opacity);

  std::unique_ptr<cc::RenderPass> render_pass = cc::RenderPass::Create();
  DrawWindowTree(render_pass.get());
  cc::SharedQuadStateList* quad_state_list =
      &render_pass->shared_quad_state_list;

  // Both child and root have a DEFAULT Surface and no underlay Surfaces, so
  // there should be two SharedQuadStates in the list.
  EXPECT_EQ(2u, quad_state_list->size());
  cc::SharedQuadState* root_sqs = quad_state_list->back();
  cc::SharedQuadState* child_sqs = quad_state_list->front();
  EXPECT_EQ(root_opacity, root_sqs->opacity);
  // Child's SharedQuadState contains the effective opacity of the child layer,
  // which should be a product of the child and the parent opacity.
  EXPECT_EQ(child_opacity * root_opacity, child_sqs->opacity);

  // Pretend to create the UNDERLAY Surface for the child window, and confirm
  // that this creates an extra SharedQuadState in the CompositorFrame.
  child_window.GetOrCreateCompositorFrameSinkManager()->SetLatestSurfaceInfo(
      mojom::CompositorFrameSinkType::UNDERLAY,
      cc::SurfaceId(
          cc::FrameSinkId(
              WindowIdToTransportId(child_window.id()),
              static_cast<uint32_t>(mojom::CompositorFrameSinkType::UNDERLAY)),
          cc::LocalFrameId(1u, kArbitraryToken)),
      gfx::Size(100, 100));

  render_pass = cc::RenderPass::Create();
  DrawWindowTree(render_pass.get());
  quad_state_list = &render_pass->shared_quad_state_list;
  EXPECT_EQ(3u, quad_state_list->size());
  auto it = quad_state_list->begin();
  EXPECT_EQ(child_opacity * root_opacity, (*it)->opacity);
  EXPECT_EQ(child_opacity * root_opacity, (*++it)->opacity);
  EXPECT_EQ(root_opacity, (*++it)->opacity);
}

}  // namespace test
}  // namespace ws
}  // namespace ui
