/*
 * Copyright (C) 2004, 2005, 2006, 2009 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE COMPUTER, INC. ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL APPLE COMPUTER, INC. OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef PlatformWheelEvent_h
#define PlatformWheelEvent_h

#include "platform/PlatformEvent.h"
#include "platform/PlatformMouseEvent.h"
#include "platform/geometry/IntPoint.h"

namespace blink {

enum PlatformWheelEventGranularity {
  // Indicates that the wheel event should scroll an entire page. In this case
  // WebCore's built in paging behavior is used to page up and down (you get the
  // same behavior as if the user was clicking in a scrollbar track to page up
  // or page down).
  ScrollByPageWheelEvent,
  // A fine-grained event that specifies the precise number of pixels to scroll.
  // It is sent directly by MacBook touchpads on OS X, and synthesized in other
  // cases where platforms generate line-by-line scrolling events.
  ScrollByPixelWheelEvent,
};

#if OS(MACOSX)
enum PlatformWheelEventPhase {
  PlatformWheelEventPhaseNone = 0,
  PlatformWheelEventPhaseBegan = 1 << 0,
  PlatformWheelEventPhaseStationary = 1 << 1,
  PlatformWheelEventPhaseChanged = 1 << 2,
  PlatformWheelEventPhaseEnded = 1 << 3,
  PlatformWheelEventPhaseCancelled = 1 << 4,
  PlatformWheelEventPhaseMayBegin = 1 << 5,
};
#endif

class PlatformWheelEvent : public PlatformMouseEvent {
 public:
  PlatformWheelEvent()
      : PlatformMouseEvent(PlatformEvent::Wheel),
        m_deltaX(0),
        m_deltaY(0),
        m_wheelTicksX(0),
        m_wheelTicksY(0),
        m_granularity(ScrollByPixelWheelEvent),
        m_hasPreciseScrollingDeltas(false),
        m_resendingPluginId(-1),
        m_railsMode(RailsModeFree),
        m_dispatchType(Blocking)
#if OS(MACOSX)
        ,
        m_phase(PlatformWheelEventPhaseNone),
        m_momentumPhase(PlatformWheelEventPhaseNone)
#endif
  {
  }

  float deltaX() const { return m_deltaX; }
  float deltaY() const { return m_deltaY; }

  float wheelTicksX() const { return m_wheelTicksX; }
  float wheelTicksY() const { return m_wheelTicksY; }

  PlatformWheelEventGranularity granularity() const { return m_granularity; }

  bool hasPreciseScrollingDeltas() const { return m_hasPreciseScrollingDeltas; }
  void setHasPreciseScrollingDeltas(bool b) { m_hasPreciseScrollingDeltas = b; }
  int resendingPluginId() const { return m_resendingPluginId; }
  RailsMode getRailsMode() const { return m_railsMode; }
  DispatchType dispatchType() const { return m_dispatchType; }
  bool cancelable() const { return m_dispatchType == PlatformEvent::Blocking; }
#if OS(MACOSX)
  PlatformWheelEventPhase phase() const { return m_phase; }
  PlatformWheelEventPhase momentumPhase() const { return m_momentumPhase; }
#endif

 protected:
  float m_deltaX;
  float m_deltaY;
  float m_wheelTicksX;
  float m_wheelTicksY;
  PlatformWheelEventGranularity m_granularity;
  bool m_hasPreciseScrollingDeltas;
  int m_resendingPluginId;
  RailsMode m_railsMode;
  DispatchType m_dispatchType;
#if OS(MACOSX)
  PlatformWheelEventPhase m_phase;
  PlatformWheelEventPhase m_momentumPhase;
#endif
};

}  // namespace blink

#endif  // PlatformWheelEvent_h
