//! Qore namespace
namespace Qore {
//! Oracle namespace
namespace Oracle {
/** @defgroup oracle_functions Oracle Functions
///@{
 */

//! returns a value to be used to bind a Qore value as an "IN OUT" variable for PL/SQL
/** @par Code Flags:
    @ref CONSTANT

    @param value any Qore variable
    @param type an optional parameter to enforce the type of the variable. It is obtained automatically fromn the \c value when it is omitted.

    @return a has which can be used for IN OUT binding. Keys:
    - \c "type" : a string type name of the variable (if it is provided as an argument)
    - \c "value" : a variable/value itself

    @par Example:
    @code
my string $str = "lorem ipsum";
my int $i = 10;
my date $d = now();

my hash $res = $ds.exec("begin qore_test.do_simple_inout(:ostr, :oint, :odt); end;",
                        Oracle::bindInOut($str),
                        Oracle::bindInOut($i),
                        Oracle::bindInOut($d)
                       );
    @endcode
 */
hash bindInOut(auto value, __7_ string type);

//! returns a value to be used to bind a Qore value as an Oracle collection
/** @par Code Flags:
    @ref CONSTANT

    @param tname the name of the Oracle collection type to bind
    @param values a list of values for the collection

    @return a hash value which can be used to bind the given value as the given collection in an SQL statement with the following keys:
    - \c "type": a constant string \c "OracleCollection"
    - \c "^oratype^": the value of the \c tname argument
    - \c "^values^": the value of the \c values argument

    @par Example:
    @code
$db.exec("insert into object_table (obj) values (%v)", bindOracleObject("MY_OBJECT", $h));
    @endcode
 */
hash bindOracleCollection(string tname, list<auto> values);

//! returns a value to be used to bind a Qore value as an Oracle object
/** @par Code Flags:
    @ref CONSTANT

    @param tname the name of the Oracle type to bind
    @param values a hash where the keys are Oracle object attributes

    @return a hash value which can be used to bind the given value as the given object in an SQL statement with the following keys:
    - \c "type": a constant string \c "OracleObject"
    - \c "^oratype^": the value of the \c tname argument
    - \c "^values^": the value of the \c values argument

    @par Example:
    @code
$db.exec("insert into object_table (obj) values (%v)", bindOracleObject("MY_OBJECT", $h));
    @endcode
 */
hash bindOracleObject(string tname, hash<auto> values);

//! returns a value to be used to describe a placeholder buffer for retrieving an Oracle collection as an output value
/** @par Code Flags:
    @ref CONSTANT

    @param tname the name of the Oracle collection type to bind for the placeholder buffer

    @return a hash value which can be used to describe an output placeholder buffer as a collection buffer of the given type in an SQL statement with the following keys:
    - \c "type": a constant string \c "OracleCollection"
    - \c "value": the value of the \c tname argument

    @par Example:
    @code
my hash $h = $db.exec("begin my_pkg.get_collection(:retval); end;", placeholderOracleCollection("MY_COLLECTION")).retval;
    @endcode
 */
hash placeholderOracleCollection(string tname);

//! returns a value to be used to describe a placeholder buffer for retrieving an Oracle object as an output value
/** @par Code Flags:
    @ref CONSTANT

    @param tname the name of the Oracle object type to bind for the placeholder buffer

    @return a hash value which can be used to describe an output placeholder buffer as an object buffer of the given type in an SQL statement with the following keys:
    - \c "type": a constant string \c "OracleObject"
    - \c "value": the value of the \c tname argument

    @par Example:
    @code
my hash $h = $db.exec("begin my_pkg.get_obj(:retval); end;", placeholderOracleObject("MY_OBJECT")).retval;
    @endcode
 */
hash placeholderOracleObject(string tname);

///@}
}
}
