/* -*- mode: c++; indent-tabs-mode: nil -*- */
/*
    QC_LdapClient.qpp

    Qore Programming Language

    Copyright 2003 - 2018 Qore Technologies, s.r.o.

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include "openldap-module.h"

#include "QoreLdapClient.h"

QoreLdapParseResultHelper::QoreLdapParseResultHelper(const char *n_meth, const char* n_f, QoreLdapClient* n_l, LDAPMessage* msg, ExceptionSink* xs) : meth(n_meth), f(n_f), l(n_l), xsink(xs), err(0), matched(0), text(0), refs(0) {
   l->checkLdapError(meth, f, ldap_parse_result(l->ldp, msg, &err, &matched, &text, &refs, 0, 1), xsink);
}

int QoreLdapParseResultHelper::check() const {
   if (err == LDAP_SUCCESS)
      return 0;

   QoreStringNode* desc = l->getErrorText(meth, f, err);
   if (text)
      desc->sprintf(": %s", text);
   if (matched)
      desc->sprintf(" (matched: '%s')", matched);
   //if (refs) { }

   xsink->raiseException("LDAP-RESULT-ERROR", desc);
   return -1;
}

class LdapClientHolder {
protected:
   QoreLdapClient* ldap;
   ExceptionSink* xsink;

public:
   DLLLOCAL LdapClientHolder(QoreLdapClient* l, ExceptionSink* xs) : ldap(l), xsink(xs) {
   }

   DLLLOCAL ~LdapClientHolder() {
      if (ldap) {
         ldap->destructor(xsink);
         ldap->deref(xsink);
      }
   }

   DLLLOCAL QoreLdapClient* release() {
      QoreLdapClient* l = ldap;
      ldap = 0;
      return l;
   }
};
/* Qore class OpenLdap::LdapClient */

qore_classid_t CID_LDAPCLIENT;
QoreClass* QC_LDAPCLIENT;

// nothing LdapClient::add(string dn, hash attrs, *timeout timeout_ms){}
static QoreValue LdapClient_add_VsVhNt(QoreObject* self, QoreLdapClient* ldap, const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    const QoreStringNode* dn = HARD_QORE_VALUE_STRING(args, 0);
    const QoreHashNode* attrs = HARD_QORE_VALUE_HASH(args, 1);
    int64 timeout_ms = HARD_QORE_VALUE_INT(args, 2);
# 269 "QC_LdapClient.qpp"
    ldap->add(xsink, dn, attrs, timeout_ms);
    return QoreValue();
}

// nothing LdapClient::bind(hash bind, *timeout timeout_ms){}
static QoreValue LdapClient_bind_VhNt(QoreObject* self, QoreLdapClient* ldap, const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    const QoreHashNode* bind = HARD_QORE_VALUE_HASH(args, 0);
    int64 timeout_ms = HARD_QORE_VALUE_INT(args, 1);
# 198 "QC_LdapClient.qpp"
    ldap->bind(xsink, *bind, timeout_ms);
    return QoreValue();
}

// bool LdapClient::compare(string dn, string attr, softlist vals, *timeout timeout_ms){}
static QoreValue LdapClient_compare_VsVsvlNt(QoreObject* self, QoreLdapClient* ldap, const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    const QoreStringNode* dn = HARD_QORE_VALUE_STRING(args, 0);
    const QoreStringNode* attr = HARD_QORE_VALUE_STRING(args, 1);
    const QoreListNode* vals = HARD_QORE_VALUE_LIST(args, 2);
    int64 timeout_ms = HARD_QORE_VALUE_INT(args, 3);
# 335 "QC_LdapClient.qpp"
    return ldap->compare(xsink, dn, attr, vals, timeout_ms);
}

// LdapClient::constructor(string uri, *hash options) {}
static void LdapClient_constructor_VsNh(QoreObject* self, const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    const QoreStringNode* uri = HARD_QORE_VALUE_STRING(args, 0);
    const QoreHashNode* options = get_param_value(args, 1).get<const QoreHashNode>();
# 146 "QC_LdapClient.qpp"
LdapClientHolder ldap(new QoreLdapClient(uri, options, xsink), xsink);
   if (!*xsink)
      self->setPrivate(CID_LDAPCLIENT, ldap.release());
}

// LdapClient::copy() {}
static void LdapClient_copy(QoreObject* self, QoreObject* old, QoreLdapClient* ldap, ExceptionSink* xsink) {
# 172 "QC_LdapClient.qpp"
LdapClientHolder ldp(new QoreLdapClient(*ldap, xsink), xsink);
   if (!*xsink)
      self->setPrivate(CID_LDAPCLIENT, ldp.release());
}

// nothing LdapClient::del(string dn, *timeout timeout_ms){}
static QoreValue LdapClient_del_VsNt(QoreObject* self, QoreLdapClient* ldap, const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    const QoreStringNode* dn = HARD_QORE_VALUE_STRING(args, 0);
    int64 timeout_ms = HARD_QORE_VALUE_INT(args, 1);
# 312 "QC_LdapClient.qpp"
    ldap->del(xsink, dn, timeout_ms);
    return QoreValue();
}

// LdapClient::destructor() {}
static void LdapClient_destructor(QoreObject* self, QoreLdapClient* ldap, ExceptionSink* xsink) {
# 158 "QC_LdapClient.qpp"
ldap->destructor(xsink);
   ldap->deref(xsink);
}

// string LdapClient::getUri(){}
static QoreValue LdapClient_getUri(QoreObject* self, QoreLdapClient* ldap, const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
# 390 "QC_LdapClient.qpp"
    return ldap->getUriStr();
}

// bool LdapClient::isSecure(){}
static QoreValue LdapClient_isSecure(QoreObject* self, QoreLdapClient* ldap, const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
# 404 "QC_LdapClient.qpp"
    return ldap->isSecure(xsink);
}

// nothing LdapClient::modify(string dn, softlist mods, *timeout timeout_ms){}
static QoreValue LdapClient_modify_VsvlNt(QoreObject* self, QoreLdapClient* ldap, const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    const QoreStringNode* dn = HARD_QORE_VALUE_STRING(args, 0);
    const QoreListNode* mods = HARD_QORE_VALUE_LIST(args, 1);
    int64 timeout_ms = HARD_QORE_VALUE_INT(args, 2);
# 293 "QC_LdapClient.qpp"
    ldap->modify(xsink, dn, mods, timeout_ms);
    return QoreValue();
}

// nothing LdapClient::passwd(string dn, string oldpwd, string newpwd, *timeout timeout_ms){}
static QoreValue LdapClient_passwd_VsVsVsNt(QoreObject* self, QoreLdapClient* ldap, const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    const QoreStringNode* dn = HARD_QORE_VALUE_STRING(args, 0);
    const QoreStringNode* oldpwd = HARD_QORE_VALUE_STRING(args, 1);
    const QoreStringNode* newpwd = HARD_QORE_VALUE_STRING(args, 2);
    int64 timeout_ms = HARD_QORE_VALUE_INT(args, 3);
# 378 "QC_LdapClient.qpp"
    ldap->passwd(xsink, dn, oldpwd, newpwd, timeout_ms);
    return QoreValue();
}

// nothing LdapClient::rename(string dn, string newrdn, string newparent, softbool deleteoldrdn = True, *timeout timeout_ms){}
static QoreValue LdapClient_rename_VsVsVsvbNt(QoreObject* self, QoreLdapClient* ldap, const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    const QoreStringNode* dn = HARD_QORE_VALUE_STRING(args, 0);
    const QoreStringNode* newrdn = HARD_QORE_VALUE_STRING(args, 1);
    const QoreStringNode* newparent = HARD_QORE_VALUE_STRING(args, 2);
    bool deleteoldrdn = HARD_QORE_VALUE_BOOL(args, 3);
    int64 timeout_ms = HARD_QORE_VALUE_INT(args, 4);
# 357 "QC_LdapClient.qpp"
    ldap->rename(xsink, dn, newrdn, newparent, deleteoldrdn, timeout_ms);
    return QoreValue();
}

// hash LdapClient::search(hash h, *timeout timeout_ms){}
static QoreValue LdapClient_search_VhNt(QoreObject* self, QoreLdapClient* ldap, const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    const QoreHashNode* h = HARD_QORE_VALUE_HASH(args, 0);
    int64 timeout_ms = HARD_QORE_VALUE_INT(args, 1);
# 223 "QC_LdapClient.qpp"
    const QoreStringNode* base = check_hash_key<QoreStringNode>(xsink, *h, "base", "LDAP-SEARCH-ERROR");
    const QoreStringNode* filter = check_hash_key<QoreStringNode>(xsink, *h, "filter", "LDAP-SEARCH-ERROR");
    QoreValue n = h->getKeyValue("attributes");
    ReferenceHolder<QoreListNode> attrl(xsink);
    if (n) {
        if (n.getType() == NT_STRING) {
            attrl = new QoreListNode(autoTypeInfo);
            attrl->push(n.refSelf(), xsink);
        }
        else if (n.getType() == NT_LIST)
            attrl = n.get<const QoreListNode>()->listRefSelf();
        else {
            xsink->raiseException("LDAPCLIENT-SEARCH-ERROR", "the 'attributes' key of the search hash contains type '%s' (expecting 'list' or 'string')", n.getTypeName());
            return 0;
        }
    }

    // get scope
    n = h->getKeyValue("scope");
    int scope = n.getAsBigInt();
    if (!scope)
        scope = LDAP_SCOPE_SUBTREE;

    return ldap->search(xsink, base, scope, filter, *attrl, false, timeout_ms);
}

// static nothing LdapClient::getInfo(){}
static QoreValue static_LdapClient_getInfo(const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
# 417 "QC_LdapClient.qpp"
return QoreLdapClient::getInfo();
}

DLLLOCAL void preinitLdapClientClass() {
    QC_LDAPCLIENT = new QoreClass("LdapClient", "::OpenLdap::LdapClient", QDOM_NETWORK);
    CID_LDAPCLIENT = QC_LDAPCLIENT->getID();
    QC_LDAPCLIENT->setSystem();
}

DLLLOCAL QoreClass* initLdapClientClass(QoreNamespace& ns) {
    if (!QC_LDAPCLIENT)
        preinitLdapClientClass();

    // nothing LdapClient::add(string dn, hash attrs, *timeout timeout_ms){}
    QC_LDAPCLIENT->addMethod("add", (q_method_n_t)LdapClient_add_VsVhNt, Public, QCF_NO_FLAGS, QDOM_DEFAULT, nothingTypeInfo, 3, stringTypeInfo, QORE_PARAM_NO_ARG, "dn", hashTypeInfo, QORE_PARAM_NO_ARG, "attrs", timeoutOrNothingTypeInfo, QORE_PARAM_NO_ARG, "timeout_ms");

    // nothing LdapClient::bind(hash bind, *timeout timeout_ms){}
    QC_LDAPCLIENT->addMethod("bind", (q_method_n_t)LdapClient_bind_VhNt, Public, QCF_NO_FLAGS, QDOM_DEFAULT, nothingTypeInfo, 2, hashTypeInfo, QORE_PARAM_NO_ARG, "bind", timeoutOrNothingTypeInfo, QORE_PARAM_NO_ARG, "timeout_ms");

    // bool LdapClient::compare(string dn, string attr, softlist vals, *timeout timeout_ms){}
    QC_LDAPCLIENT->addMethod("compare", (q_method_n_t)LdapClient_compare_VsVsvlNt, Public, QCF_NO_FLAGS, QDOM_DEFAULT, boolTypeInfo, 4, stringTypeInfo, QORE_PARAM_NO_ARG, "dn", stringTypeInfo, QORE_PARAM_NO_ARG, "attr", softListTypeInfo, QORE_PARAM_NO_ARG, "vals", timeoutOrNothingTypeInfo, QORE_PARAM_NO_ARG, "timeout_ms");

    // LdapClient::constructor(string uri, *hash options) {}
    QC_LDAPCLIENT->addConstructor(LdapClient_constructor_VsNh, Public, QCF_NO_FLAGS, QDOM_DEFAULT, 2, stringTypeInfo, QORE_PARAM_NO_ARG, "uri", hashOrNothingTypeInfo, QORE_PARAM_NO_ARG, "options");

    // LdapClient::copy() {}
    QC_LDAPCLIENT->setCopy((q_copy_t)LdapClient_copy);

    // nothing LdapClient::del(string dn, *timeout timeout_ms){}
    QC_LDAPCLIENT->addMethod("del", (q_method_n_t)LdapClient_del_VsNt, Public, QCF_NO_FLAGS, QDOM_DEFAULT, nothingTypeInfo, 2, stringTypeInfo, QORE_PARAM_NO_ARG, "dn", timeoutOrNothingTypeInfo, QORE_PARAM_NO_ARG, "timeout_ms");

    // LdapClient::destructor() {}
    QC_LDAPCLIENT->setDestructor((q_destructor_t)LdapClient_destructor);

    // string LdapClient::getUri(){}
    QC_LDAPCLIENT->addMethod("getUri", (q_method_n_t)LdapClient_getUri, Public, QCF_CONSTANT, QDOM_DEFAULT, stringTypeInfo);

    // bool LdapClient::isSecure(){}
    QC_LDAPCLIENT->addMethod("isSecure", (q_method_n_t)LdapClient_isSecure, Public, QCF_RET_VALUE_ONLY, QDOM_DEFAULT, boolTypeInfo);

    // nothing LdapClient::modify(string dn, softlist mods, *timeout timeout_ms){}
    QC_LDAPCLIENT->addMethod("modify", (q_method_n_t)LdapClient_modify_VsvlNt, Public, QCF_NO_FLAGS, QDOM_DEFAULT, nothingTypeInfo, 3, stringTypeInfo, QORE_PARAM_NO_ARG, "dn", softListTypeInfo, QORE_PARAM_NO_ARG, "mods", timeoutOrNothingTypeInfo, QORE_PARAM_NO_ARG, "timeout_ms");

    // nothing LdapClient::passwd(string dn, string oldpwd, string newpwd, *timeout timeout_ms){}
    QC_LDAPCLIENT->addMethod("passwd", (q_method_n_t)LdapClient_passwd_VsVsVsNt, Public, QCF_NO_FLAGS, QDOM_DEFAULT, nothingTypeInfo, 4, stringTypeInfo, QORE_PARAM_NO_ARG, "dn", stringTypeInfo, QORE_PARAM_NO_ARG, "oldpwd", stringTypeInfo, QORE_PARAM_NO_ARG, "newpwd", timeoutOrNothingTypeInfo, QORE_PARAM_NO_ARG, "timeout_ms");

    // nothing LdapClient::rename(string dn, string newrdn, string newparent, softbool deleteoldrdn = True, *timeout timeout_ms){}
    QC_LDAPCLIENT->addMethod("rename", (q_method_n_t)LdapClient_rename_VsVsVsvbNt, Public, QCF_NO_FLAGS, QDOM_DEFAULT, nothingTypeInfo, 5, stringTypeInfo, QORE_PARAM_NO_ARG, "dn", stringTypeInfo, QORE_PARAM_NO_ARG, "newrdn", stringTypeInfo, QORE_PARAM_NO_ARG, "newparent", softBoolTypeInfo, QoreSimpleValue().assign(true), "deleteoldrdn", timeoutOrNothingTypeInfo, QORE_PARAM_NO_ARG, "timeout_ms");

    // hash LdapClient::search(hash h, *timeout timeout_ms){}
    QC_LDAPCLIENT->addMethod("search", (q_method_n_t)LdapClient_search_VhNt, Public, QCF_NO_FLAGS, QDOM_DEFAULT, hashTypeInfo, 2, hashTypeInfo, QORE_PARAM_NO_ARG, "h", timeoutOrNothingTypeInfo, QORE_PARAM_NO_ARG, "timeout_ms");

    // static nothing LdapClient::getInfo(){}
    QC_LDAPCLIENT->addStaticMethod("getInfo", (q_func_n_t)static_LdapClient_getInfo, Public, QCF_CONSTANT, QDOM_DEFAULT, nothingTypeInfo);
# 71 "QC_LdapClient.qpp"
# 76 "QC_LdapClient.qpp"
# 79 "QC_LdapClient.qpp"
# 82 "QC_LdapClient.qpp"
    ns.addConstant("LDAP_SCOPE_BASE", ((int64)LDAP_SCOPE_BASE));
    ns.addConstant("LDAP_SCOPE_CHILDREN", ((int64)LDAP_SCOPE_CHILDREN));
    ns.addConstant("LDAP_SCOPE_ONELEVEL", ((int64)LDAP_SCOPE_ONELEVEL));
    ns.addConstant("LDAP_SCOPE_SUBTREE", ((int64)LDAP_SCOPE_SUBTREE));
# 91 "QC_LdapClient.qpp"
# 98 "QC_LdapClient.qpp"
    ns.addConstant("LDAP_MOD_ADD", new QoreStringNode("add"));
    ns.addConstant("LDAP_MOD_DELETE", new QoreStringNode("delete"));
    ns.addConstant("LDAP_MOD_REPLACE", new QoreStringNode("replace"));
# 105 "QC_LdapClient.qpp"
    ns.addConstant("DefaultTimeout", DateTimeNode::makeRelative(0, 0, 0, 0, 0, 60, 0));

    return QC_LDAPCLIENT;
}
